(function(){'use strict';// ::- Persistent data structure representing an ordered mapping from
// strings to values, with some convenient update methods.
function OrderedMap(content) {
  this.content = content;
}

OrderedMap.prototype = {
  constructor: OrderedMap,

  find: function(key) {
    for (var i = 0; i < this.content.length; i += 2)
      { if (this.content[i] === key) { return i } }
    return -1
  },

  // :: (string) → ?any
  // Retrieve the value stored under `key`, or return undefined when
  // no such key exists.
  get: function(key) {
    var found = this.find(key);
    return found == -1 ? undefined : this.content[found + 1]
  },

  // :: (string, any, ?string) → OrderedMap
  // Create a new map by replacing the value of `key` with a new
  // value, or adding a binding to the end of the map. If `newKey` is
  // given, the key of the binding will be replaced with that key.
  update: function(key, value, newKey) {
    var self = newKey && newKey != key ? this.remove(newKey) : this;
    var found = self.find(key), content = self.content.slice();
    if (found == -1) {
      content.push(newKey || key, value);
    } else {
      content[found + 1] = value;
      if (newKey) { content[found] = newKey; }
    }
    return new OrderedMap(content)
  },

  // :: (string) → OrderedMap
  // Return a map with the given key removed, if it existed.
  remove: function(key) {
    var found = this.find(key);
    if (found == -1) { return this }
    var content = this.content.slice();
    content.splice(found, 2);
    return new OrderedMap(content)
  },

  // :: (string, any) → OrderedMap
  // Add a new key to the start of the map.
  addToStart: function(key, value) {
    return new OrderedMap([key, value].concat(this.remove(key).content))
  },

  // :: (string, any) → OrderedMap
  // Add a new key to the end of the map.
  addToEnd: function(key, value) {
    var content = this.remove(key).content.slice();
    content.push(key, value);
    return new OrderedMap(content)
  },

  // :: (string, string, any) → OrderedMap
  // Add a key after the given key. If `place` is not found, the new
  // key is added to the end.
  addBefore: function(place, key, value) {
    var without = this.remove(key), content = without.content.slice();
    var found = without.find(place);
    content.splice(found == -1 ? content.length : found, 0, key, value);
    return new OrderedMap(content)
  },

  // :: ((key: string, value: any))
  // Call the given function for each key/value pair in the map, in
  // order.
  forEach: function(f) {
    for (var i = 0; i < this.content.length; i += 2)
      { f(this.content[i], this.content[i + 1]); }
  },

  // :: (union<Object, OrderedMap>) → OrderedMap
  // Create a new map by prepending the keys in this map that don't
  // appear in `map` before the keys in `map`.
  prepend: function(map) {
    map = OrderedMap.from(map);
    if (!map.size) { return this }
    return new OrderedMap(map.content.concat(this.subtract(map).content))
  },

  // :: (union<Object, OrderedMap>) → OrderedMap
  // Create a new map by appending the keys in this map that don't
  // appear in `map` after the keys in `map`.
  append: function(map) {
    map = OrderedMap.from(map);
    if (!map.size) { return this }
    return new OrderedMap(this.subtract(map).content.concat(map.content))
  },

  // :: (union<Object, OrderedMap>) → OrderedMap
  // Create a map containing all the keys in this map that don't
  // appear in `map`.
  subtract: function(map) {
    var result = this;
    map = OrderedMap.from(map);
    for (var i = 0; i < map.content.length; i += 2)
      { result = result.remove(map.content[i]); }
    return result
  },

  // :: () → Object
  // Turn ordered map into a plain object.
  toObject: function() {
    var result = {};
    this.forEach(function(key, value) { result[key] = value; });
    return result
  },

  // :: number
  // The amount of keys in this map.
  get size() {
    return this.content.length >> 1
  }
};

// :: (?union<Object, OrderedMap>) → OrderedMap
// Return a map with the given content. If null, create an empty
// map. If given an ordered map, return that map itself. If given an
// object, create a map from the object's properties.
OrderedMap.from = function(value) {
  if (value instanceof OrderedMap) { return value }
  var content = [];
  if (value) { for (var prop in value) { content.push(prop, value[prop]); } }
  return new OrderedMap(content)
};function findDiffStart(a, b, pos) {
    for (var i = 0;; i++) {
        if (i == a.childCount || i == b.childCount)
            { return a.childCount == b.childCount ? null : pos; }
        var childA = a.child(i), childB = b.child(i);
        if (childA == childB) {
            pos += childA.nodeSize;
            continue;
        }
        if (!childA.sameMarkup(childB))
            { return pos; }
        if (childA.isText && childA.text != childB.text) {
            for (var j = 0; childA.text[j] == childB.text[j]; j++)
                { pos++; }
            return pos;
        }
        if (childA.content.size || childB.content.size) {
            var inner = findDiffStart(childA.content, childB.content, pos + 1);
            if (inner != null)
                { return inner; }
        }
        pos += childA.nodeSize;
    }
}
function findDiffEnd(a, b, posA, posB) {
    for (var iA = a.childCount, iB = b.childCount;;) {
        if (iA == 0 || iB == 0)
            { return iA == iB ? null : { a: posA, b: posB }; }
        var childA = a.child(--iA), childB = b.child(--iB), size = childA.nodeSize;
        if (childA == childB) {
            posA -= size;
            posB -= size;
            continue;
        }
        if (!childA.sameMarkup(childB))
            { return { a: posA, b: posB }; }
        if (childA.isText && childA.text != childB.text) {
            var same = 0, minSize = Math.min(childA.text.length, childB.text.length);
            while (same < minSize && childA.text[childA.text.length - same - 1] == childB.text[childB.text.length - same - 1]) {
                same++;
                posA--;
                posB--;
            }
            return { a: posA, b: posB };
        }
        if (childA.content.size || childB.content.size) {
            var inner = findDiffEnd(childA.content, childB.content, posA - 1, posB - 1);
            if (inner)
                { return inner; }
        }
        posA -= size;
        posB -= size;
    }
}

/**
A fragment represents a node's collection of child nodes.

Like nodes, fragments are persistent data structures, and you
should not mutate them or their content. Rather, you create new
instances whenever needed. The API tries to make this easy.
*/
var Fragment = function Fragment(
  /**
  @internal
  */
  content, size) {
      this.content = content;
      this.size = size || 0;
      if (size == null)
          { for (var i = 0; i < content.length; i++)
              { this.size += content[i].nodeSize; } }
  };

var prototypeAccessors$c = { firstChild: { configurable: true },lastChild: { configurable: true },childCount: { configurable: true } };
  /**
  Invoke a callback for all descendant nodes between the given two
  positions (relative to start of this fragment). Doesn't descend
  into a node when the callback returns `false`.
  */
  Fragment.prototype.nodesBetween = function nodesBetween (from, to, f, nodeStart, parent) {
        if ( nodeStart === void 0 ) nodeStart = 0;

      for (var i = 0, pos = 0; pos < to; i++) {
          var child = this.content[i], end = pos + child.nodeSize;
          if (end > from && f(child, nodeStart + pos, parent || null, i) !== false && child.content.size) {
              var start = pos + 1;
              child.nodesBetween(Math.max(0, from - start), Math.min(child.content.size, to - start), f, nodeStart + start);
          }
          pos = end;
      }
  };
  /**
  Call the given callback for every descendant node. `pos` will be
  relative to the start of the fragment. The callback may return
  `false` to prevent traversal of a given node's children.
  */
  Fragment.prototype.descendants = function descendants (f) {
      this.nodesBetween(0, this.size, f);
  };
  /**
  Extract the text between `from` and `to`. See the same method on
  [`Node`](https://prosemirror.net/docs/ref/#model.Node.textBetween).
  */
  Fragment.prototype.textBetween = function textBetween (from, to, blockSeparator, leafText) {
      var text = "", first = true;
      this.nodesBetween(from, to, function (node, pos) {
          var nodeText = node.isText ? node.text.slice(Math.max(from, pos) - pos, to - pos)
              : !node.isLeaf ? ""
                  : leafText ? (typeof leafText === "function" ? leafText(node) : leafText)
                      : node.type.spec.leafText ? node.type.spec.leafText(node)
                          : "";
          if (node.isBlock && (node.isLeaf && nodeText || node.isTextblock) && blockSeparator) {
              if (first)
                  { first = false; }
              else
                  { text += blockSeparator; }
          }
          text += nodeText;
      }, 0);
      return text;
  };
  /**
  Create a new fragment containing the combined content of this
  fragment and the other.
  */
  Fragment.prototype.append = function append (other) {
      if (!other.size)
          { return this; }
      if (!this.size)
          { return other; }
      var last = this.lastChild, first = other.firstChild, content = this.content.slice(), i = 0;
      if (last.isText && last.sameMarkup(first)) {
          content[content.length - 1] = last.withText(last.text + first.text);
          i = 1;
      }
      for (; i < other.content.length; i++)
          { content.push(other.content[i]); }
      return new Fragment(content, this.size + other.size);
  };
  /**
  Cut out the sub-fragment between the two given positions.
  */
  Fragment.prototype.cut = function cut (from, to) {
        if ( to === void 0 ) to = this.size;

      if (from == 0 && to == this.size)
          { return this; }
      var result = [], size = 0;
      if (to > from)
          { for (var i = 0, pos = 0; pos < to; i++) {
              var child = this.content[i], end = pos + child.nodeSize;
              if (end > from) {
                  if (pos < from || end > to) {
                      if (child.isText)
                          { child = child.cut(Math.max(0, from - pos), Math.min(child.text.length, to - pos)); }
                      else
                          { child = child.cut(Math.max(0, from - pos - 1), Math.min(child.content.size, to - pos - 1)); }
                  }
                  result.push(child);
                  size += child.nodeSize;
              }
              pos = end;
          } }
      return new Fragment(result, size);
  };
  /**
  @internal
  */
  Fragment.prototype.cutByIndex = function cutByIndex (from, to) {
      if (from == to)
          { return Fragment.empty; }
      if (from == 0 && to == this.content.length)
          { return this; }
      return new Fragment(this.content.slice(from, to));
  };
  /**
  Create a new fragment in which the node at the given index is
  replaced by the given node.
  */
  Fragment.prototype.replaceChild = function replaceChild (index, node) {
      var current = this.content[index];
      if (current == node)
          { return this; }
      var copy = this.content.slice();
      var size = this.size + node.nodeSize - current.nodeSize;
      copy[index] = node;
      return new Fragment(copy, size);
  };
  /**
  Create a new fragment by prepending the given node to this
  fragment.
  */
  Fragment.prototype.addToStart = function addToStart (node) {
      return new Fragment([node].concat(this.content), this.size + node.nodeSize);
  };
  /**
  Create a new fragment by appending the given node to this
  fragment.
  */
  Fragment.prototype.addToEnd = function addToEnd (node) {
      return new Fragment(this.content.concat(node), this.size + node.nodeSize);
  };
  /**
  Compare this fragment to another one.
  */
  Fragment.prototype.eq = function eq (other) {
      if (this.content.length != other.content.length)
          { return false; }
      for (var i = 0; i < this.content.length; i++)
          { if (!this.content[i].eq(other.content[i]))
              { return false; } }
      return true;
  };
  /**
  The first child of the fragment, or `null` if it is empty.
  */
  prototypeAccessors$c.firstChild.get = function () { return this.content.length ? this.content[0] : null; };
  /**
  The last child of the fragment, or `null` if it is empty.
  */
  prototypeAccessors$c.lastChild.get = function () { return this.content.length ? this.content[this.content.length - 1] : null; };
  /**
  The number of child nodes in this fragment.
  */
  prototypeAccessors$c.childCount.get = function () { return this.content.length; };
  /**
  Get the child node at the given index. Raise an error when the
  index is out of range.
  */
  Fragment.prototype.child = function child (index) {
      var found = this.content[index];
      if (!found)
          { throw new RangeError("Index " + index + " out of range for " + this); }
      return found;
  };
  /**
  Get the child node at the given index, if it exists.
  */
  Fragment.prototype.maybeChild = function maybeChild (index) {
      return this.content[index] || null;
  };
  /**
  Call `f` for every child node, passing the node, its offset
  into this parent node, and its index.
  */
  Fragment.prototype.forEach = function forEach (f) {
      for (var i = 0, p = 0; i < this.content.length; i++) {
          var child = this.content[i];
          f(child, p, i);
          p += child.nodeSize;
      }
  };
  /**
  Find the first position at which this fragment and another
  fragment differ, or `null` if they are the same.
  */
  Fragment.prototype.findDiffStart = function findDiffStart$1 (other, pos) {
        if ( pos === void 0 ) pos = 0;

      return findDiffStart(this, other, pos);
  };
  /**
  Find the first position, searching from the end, at which this
  fragment and the given fragment differ, or `null` if they are
  the same. Since this position will not be the same in both
  nodes, an object with two separate positions is returned.
  */
  Fragment.prototype.findDiffEnd = function findDiffEnd$1 (other, pos, otherPos) {
        if ( pos === void 0 ) pos = this.size;
        if ( otherPos === void 0 ) otherPos = other.size;

      return findDiffEnd(this, other, pos, otherPos);
  };
  /**
  Find the index and inner offset corresponding to a given relative
  position in this fragment. The result object will be reused
  (overwritten) the next time the function is called. @internal
  */
  Fragment.prototype.findIndex = function findIndex (pos, round) {
        if ( round === void 0 ) round = -1;

      if (pos == 0)
          { return retIndex(0, pos); }
      if (pos == this.size)
          { return retIndex(this.content.length, pos); }
      if (pos > this.size || pos < 0)
          { throw new RangeError(("Position " + pos + " outside of fragment (" + (this) + ")")); }
      for (var i = 0, curPos = 0;; i++) {
          var cur = this.child(i), end = curPos + cur.nodeSize;
          if (end >= pos) {
              if (end == pos || round > 0)
                  { return retIndex(i + 1, end); }
              return retIndex(i, curPos);
          }
          curPos = end;
      }
  };
  /**
  Return a debugging string that describes this fragment.
  */
  Fragment.prototype.toString = function toString () { return "<" + this.toStringInner() + ">"; };
  /**
  @internal
  */
  Fragment.prototype.toStringInner = function toStringInner () { return this.content.join(", "); };
  /**
  Create a JSON-serializeable representation of this fragment.
  */
  Fragment.prototype.toJSON = function toJSON () {
      return this.content.length ? this.content.map(function (n) { return n.toJSON(); }) : null;
  };
  /**
  Deserialize a fragment from its JSON representation.
  */
  Fragment.fromJSON = function fromJSON (schema, value) {
      if (!value)
          { return Fragment.empty; }
      if (!Array.isArray(value))
          { throw new RangeError("Invalid input for Fragment.fromJSON"); }
      return new Fragment(value.map(schema.nodeFromJSON));
  };
  /**
  Build a fragment from an array of nodes. Ensures that adjacent
  text nodes with the same marks are joined together.
  */
  Fragment.fromArray = function fromArray (array) {
      if (!array.length)
          { return Fragment.empty; }
      var joined, size = 0;
      for (var i = 0; i < array.length; i++) {
          var node = array[i];
          size += node.nodeSize;
          if (i && node.isText && array[i - 1].sameMarkup(node)) {
              if (!joined)
                  { joined = array.slice(0, i); }
              joined[joined.length - 1] = node
                  .withText(joined[joined.length - 1].text + node.text);
          }
          else if (joined) {
              joined.push(node);
          }
      }
      return new Fragment(joined || array, size);
  };
  /**
  Create a fragment from something that can be interpreted as a
  set of nodes. For `null`, it returns the empty fragment. For a
  fragment, the fragment itself. For a node or array of nodes, a
  fragment containing those nodes.
  */
  Fragment.from = function from (nodes) {
      if (!nodes)
          { return Fragment.empty; }
      if (nodes instanceof Fragment)
          { return nodes; }
      if (Array.isArray(nodes))
          { return this.fromArray(nodes); }
      if (nodes.attrs)
          { return new Fragment([nodes], nodes.nodeSize); }
      throw new RangeError("Can not convert " + nodes + " to a Fragment" +
          (nodes.nodesBetween ? " (looks like multiple versions of prosemirror-model were loaded)" : ""));
  };

Object.defineProperties( Fragment.prototype, prototypeAccessors$c );
/**
An empty fragment. Intended to be reused whenever a node doesn't
contain anything (rather than allocating a new empty fragment for
each leaf node).
*/
Fragment.empty = new Fragment([], 0);
var found = { index: 0, offset: 0 };
function retIndex(index, offset) {
    found.index = index;
    found.offset = offset;
    return found;
}

function compareDeep(a, b) {
    if (a === b)
        { return true; }
    if (!(a && typeof a == "object") ||
        !(b && typeof b == "object"))
        { return false; }
    var array = Array.isArray(a);
    if (Array.isArray(b) != array)
        { return false; }
    if (array) {
        if (a.length != b.length)
            { return false; }
        for (var i = 0; i < a.length; i++)
            { if (!compareDeep(a[i], b[i]))
                { return false; } }
    }
    else {
        for (var p in a)
            { if (!(p in b) || !compareDeep(a[p], b[p]))
                { return false; } }
        for (var p$1 in b)
            { if (!(p$1 in a))
                { return false; } }
    }
    return true;
}

/**
A mark is a piece of information that can be attached to a node,
such as it being emphasized, in code font, or a link. It has a
type and optionally a set of attributes that provide further
information (such as the target of the link). Marks are created
through a `Schema`, which controls which types exist and which
attributes they have.
*/
var Mark = function Mark(
  /**
  The type of this mark.
  */
  type, 
  /**
  The attributes associated with this mark.
  */
  attrs) {
      this.type = type;
      this.attrs = attrs;
  };
  /**
  Given a set of marks, create a new set which contains this one as
  well, in the right position. If this mark is already in the set,
  the set itself is returned. If any marks that are set to be
  [exclusive](https://prosemirror.net/docs/ref/#model.MarkSpec.excludes) with this mark are present,
  those are replaced by this one.
  */
  Mark.prototype.addToSet = function addToSet (set) {
      var copy, placed = false;
      for (var i = 0; i < set.length; i++) {
          var other = set[i];
          if (this.eq(other))
              { return set; }
          if (this.type.excludes(other.type)) {
              if (!copy)
                  { copy = set.slice(0, i); }
          }
          else if (other.type.excludes(this.type)) {
              return set;
          }
          else {
              if (!placed && other.type.rank > this.type.rank) {
                  if (!copy)
                      { copy = set.slice(0, i); }
                  copy.push(this);
                  placed = true;
              }
              if (copy)
                  { copy.push(other); }
          }
      }
      if (!copy)
          { copy = set.slice(); }
      if (!placed)
          { copy.push(this); }
      return copy;
  };
  /**
  Remove this mark from the given set, returning a new set. If this
  mark is not in the set, the set itself is returned.
  */
  Mark.prototype.removeFromSet = function removeFromSet (set) {
      for (var i = 0; i < set.length; i++)
          { if (this.eq(set[i]))
              { return set.slice(0, i).concat(set.slice(i + 1)); } }
      return set;
  };
  /**
  Test whether this mark is in the given set of marks.
  */
  Mark.prototype.isInSet = function isInSet (set) {
      for (var i = 0; i < set.length; i++)
          { if (this.eq(set[i]))
              { return true; } }
      return false;
  };
  /**
  Test whether this mark has the same type and attributes as
  another mark.
  */
  Mark.prototype.eq = function eq (other) {
      return this == other ||
          (this.type == other.type && compareDeep(this.attrs, other.attrs));
  };
  /**
  Convert this mark to a JSON-serializeable representation.
  */
  Mark.prototype.toJSON = function toJSON () {
      var obj = { type: this.type.name };
      for (var _ in this.attrs) {
          obj.attrs = this.attrs;
          break;
      }
      return obj;
  };
  /**
  Deserialize a mark from JSON.
  */
  Mark.fromJSON = function fromJSON (schema, json) {
      if (!json)
          { throw new RangeError("Invalid input for Mark.fromJSON"); }
      var type = schema.marks[json.type];
      if (!type)
          { throw new RangeError(("There is no mark type " + (json.type) + " in this schema")); }
      var mark = type.create(json.attrs);
      type.checkAttrs(mark.attrs);
      return mark;
  };
  /**
  Test whether two sets of marks are identical.
  */
  Mark.sameSet = function sameSet (a, b) {
      if (a == b)
          { return true; }
      if (a.length != b.length)
          { return false; }
      for (var i = 0; i < a.length; i++)
          { if (!a[i].eq(b[i]))
              { return false; } }
      return true;
  };
  /**
  Create a properly sorted mark set from null, a single mark, or an
  unsorted array of marks.
  */
  Mark.setFrom = function setFrom (marks) {
      if (!marks || Array.isArray(marks) && marks.length == 0)
          { return Mark.none; }
      if (marks instanceof Mark)
          { return [marks]; }
      var copy = marks.slice();
      copy.sort(function (a, b) { return a.type.rank - b.type.rank; });
      return copy;
  };
/**
The empty set of marks.
*/
Mark.none = [];

/**
Error type raised by [`Node.replace`](https://prosemirror.net/docs/ref/#model.Node.replace) when
given an invalid replacement.
*/
var ReplaceError = /*@__PURE__*/(function (Error) {
  function ReplaceError () {
    Error.apply(this, arguments);
  }if ( Error ) ReplaceError.__proto__ = Error;
  ReplaceError.prototype = Object.create( Error && Error.prototype );
  ReplaceError.prototype.constructor = ReplaceError;

  

  return ReplaceError;
}(Error));
/*
ReplaceError = function(this: any, message: string) {
  let err = Error.call(this, message)
  ;(err as any).__proto__ = ReplaceError.prototype
  return err
} as any

ReplaceError.prototype = Object.create(Error.prototype)
ReplaceError.prototype.constructor = ReplaceError
ReplaceError.prototype.name = "ReplaceError"
*/
/**
A slice represents a piece cut out of a larger document. It
stores not only a fragment, but also the depth up to which nodes on
both side are ‘open’ (cut through).
*/
var Slice = function Slice(
  /**
  The slice's content.
  */
  content, 
  /**
  The open depth at the start of the fragment.
  */
  openStart, 
  /**
  The open depth at the end.
  */
  openEnd) {
      this.content = content;
      this.openStart = openStart;
      this.openEnd = openEnd;
  };

var prototypeAccessors$1$3 = { size: { configurable: true } };
  /**
  The size this slice would add when inserted into a document.
  */
  prototypeAccessors$1$3.size.get = function () {
      return this.content.size - this.openStart - this.openEnd;
  };
  /**
  @internal
  */
  Slice.prototype.insertAt = function insertAt (pos, fragment) {
      var content = insertInto(this.content, pos + this.openStart, fragment);
      return content && new Slice(content, this.openStart, this.openEnd);
  };
  /**
  @internal
  */
  Slice.prototype.removeBetween = function removeBetween (from, to) {
      return new Slice(removeRange(this.content, from + this.openStart, to + this.openStart), this.openStart, this.openEnd);
  };
  /**
  Tests whether this slice is equal to another slice.
  */
  Slice.prototype.eq = function eq (other) {
      return this.content.eq(other.content) && this.openStart == other.openStart && this.openEnd == other.openEnd;
  };
  /**
  @internal
  */
  Slice.prototype.toString = function toString () {
      return this.content + "(" + this.openStart + "," + this.openEnd + ")";
  };
  /**
  Convert a slice to a JSON-serializable representation.
  */
  Slice.prototype.toJSON = function toJSON () {
      if (!this.content.size)
          { return null; }
      var json = { content: this.content.toJSON() };
      if (this.openStart > 0)
          { json.openStart = this.openStart; }
      if (this.openEnd > 0)
          { json.openEnd = this.openEnd; }
      return json;
  };
  /**
  Deserialize a slice from its JSON representation.
  */
  Slice.fromJSON = function fromJSON (schema, json) {
      if (!json)
          { return Slice.empty; }
      var openStart = json.openStart || 0, openEnd = json.openEnd || 0;
      if (typeof openStart != "number" || typeof openEnd != "number")
          { throw new RangeError("Invalid input for Slice.fromJSON"); }
      return new Slice(Fragment.fromJSON(schema, json.content), openStart, openEnd);
  };
  /**
  Create a slice from a fragment by taking the maximum possible
  open value on both side of the fragment.
  */
  Slice.maxOpen = function maxOpen (fragment, openIsolating) {
        if ( openIsolating === void 0 ) openIsolating = true;

      var openStart = 0, openEnd = 0;
      for (var n = fragment.firstChild; n && !n.isLeaf && (openIsolating || !n.type.spec.isolating); n = n.firstChild)
          { openStart++; }
      for (var n$1 = fragment.lastChild; n$1 && !n$1.isLeaf && (openIsolating || !n$1.type.spec.isolating); n$1 = n$1.lastChild)
          { openEnd++; }
      return new Slice(fragment, openStart, openEnd);
  };

Object.defineProperties( Slice.prototype, prototypeAccessors$1$3 );
/**
The empty slice.
*/
Slice.empty = new Slice(Fragment.empty, 0, 0);
function removeRange(content, from, to) {
    var ref = content.findIndex(from);
    var index = ref.index;
    var offset = ref.offset;
    var child = content.maybeChild(index);
    var ref$1 = content.findIndex(to);
    var indexTo = ref$1.index;
    var offsetTo = ref$1.offset;
    if (offset == from || child.isText) {
        if (offsetTo != to && !content.child(indexTo).isText)
            { throw new RangeError("Removing non-flat range"); }
        return content.cut(0, from).append(content.cut(to));
    }
    if (index != indexTo)
        { throw new RangeError("Removing non-flat range"); }
    return content.replaceChild(index, child.copy(removeRange(child.content, from - offset - 1, to - offset - 1)));
}
function insertInto(content, dist, insert, parent) {
    var ref = content.findIndex(dist);
    var index = ref.index;
    var offset = ref.offset;
    var child = content.maybeChild(index);
    if (offset == dist || child.isText) {
        if (parent && !parent.canReplace(index, index, insert))
            { return null; }
        return content.cut(0, dist).append(insert).append(content.cut(dist));
    }
    var inner = insertInto(child.content, dist - offset - 1, insert);
    return inner && content.replaceChild(index, child.copy(inner));
}
function replace$3($from, $to, slice) {
    if (slice.openStart > $from.depth)
        { throw new ReplaceError("Inserted content deeper than insertion position"); }
    if ($from.depth - slice.openStart != $to.depth - slice.openEnd)
        { throw new ReplaceError("Inconsistent open depths"); }
    return replaceOuter($from, $to, slice, 0);
}
function replaceOuter($from, $to, slice, depth) {
    var index = $from.index(depth), node = $from.node(depth);
    if (index == $to.index(depth) && depth < $from.depth - slice.openStart) {
        var inner = replaceOuter($from, $to, slice, depth + 1);
        return node.copy(node.content.replaceChild(index, inner));
    }
    else if (!slice.content.size) {
        return close(node, replaceTwoWay($from, $to, depth));
    }
    else if (!slice.openStart && !slice.openEnd && $from.depth == depth && $to.depth == depth) { // Simple, flat case
        var parent = $from.parent, content = parent.content;
        return close(parent, content.cut(0, $from.parentOffset).append(slice.content).append(content.cut($to.parentOffset)));
    }
    else {
        var ref = prepareSliceForReplace(slice, $from);
        var start = ref.start;
        var end = ref.end;
        return close(node, replaceThreeWay($from, start, end, $to, depth));
    }
}
function checkJoin(main, sub) {
    if (!sub.type.compatibleContent(main.type))
        { throw new ReplaceError("Cannot join " + sub.type.name + " onto " + main.type.name); }
}
function joinable$1($before, $after, depth) {
    var node = $before.node(depth);
    checkJoin(node, $after.node(depth));
    return node;
}
function addNode(child, target) {
    var last = target.length - 1;
    if (last >= 0 && child.isText && child.sameMarkup(target[last]))
        { target[last] = child.withText(target[last].text + child.text); }
    else
        { target.push(child); }
}
function addRange$1($start, $end, depth, target) {
    var node = ($end || $start).node(depth);
    var startIndex = 0, endIndex = $end ? $end.index(depth) : node.childCount;
    if ($start) {
        startIndex = $start.index(depth);
        if ($start.depth > depth) {
            startIndex++;
        }
        else if ($start.textOffset) {
            addNode($start.nodeAfter, target);
            startIndex++;
        }
    }
    for (var i = startIndex; i < endIndex; i++)
        { addNode(node.child(i), target); }
    if ($end && $end.depth == depth && $end.textOffset)
        { addNode($end.nodeBefore, target); }
}
function close(node, content) {
    node.type.checkContent(content);
    return node.copy(content);
}
function replaceThreeWay($from, $start, $end, $to, depth) {
    var openStart = $from.depth > depth && joinable$1($from, $start, depth + 1);
    var openEnd = $to.depth > depth && joinable$1($end, $to, depth + 1);
    var content = [];
    addRange$1(null, $from, depth, content);
    if (openStart && openEnd && $start.index(depth) == $end.index(depth)) {
        checkJoin(openStart, openEnd);
        addNode(close(openStart, replaceThreeWay($from, $start, $end, $to, depth + 1)), content);
    }
    else {
        if (openStart)
            { addNode(close(openStart, replaceTwoWay($from, $start, depth + 1)), content); }
        addRange$1($start, $end, depth, content);
        if (openEnd)
            { addNode(close(openEnd, replaceTwoWay($end, $to, depth + 1)), content); }
    }
    addRange$1($to, null, depth, content);
    return new Fragment(content);
}
function replaceTwoWay($from, $to, depth) {
    var content = [];
    addRange$1(null, $from, depth, content);
    if ($from.depth > depth) {
        var type = joinable$1($from, $to, depth + 1);
        addNode(close(type, replaceTwoWay($from, $to, depth + 1)), content);
    }
    addRange$1($to, null, depth, content);
    return new Fragment(content);
}
function prepareSliceForReplace(slice, $along) {
    var extra = $along.depth - slice.openStart, parent = $along.node(extra);
    var node = parent.copy(slice.content);
    for (var i = extra - 1; i >= 0; i--)
        { node = $along.node(i).copy(Fragment.from(node)); }
    return { start: node.resolveNoCache(slice.openStart + extra),
        end: node.resolveNoCache(node.content.size - slice.openEnd - extra) };
}

/**
You can [_resolve_](https://prosemirror.net/docs/ref/#model.Node.resolve) a position to get more
information about it. Objects of this class represent such a
resolved position, providing various pieces of context
information, and some helper methods.

Throughout this interface, methods that take an optional `depth`
parameter will interpret undefined as `this.depth` and negative
numbers as `this.depth + value`.
*/
var ResolvedPos = function ResolvedPos(
  /**
  The position that was resolved.
  */
  pos, 
  /**
  @internal
  */
  path, 
  /**
  The offset this position has into its parent node.
  */
  parentOffset) {
      this.pos = pos;
      this.path = path;
      this.parentOffset = parentOffset;
      this.depth = path.length / 3 - 1;
  };

var prototypeAccessors$2$4 = { parent: { configurable: true },doc: { configurable: true },textOffset: { configurable: true },nodeAfter: { configurable: true },nodeBefore: { configurable: true } };
  /**
  @internal
  */
  ResolvedPos.prototype.resolveDepth = function resolveDepth (val) {
      if (val == null)
          { return this.depth; }
      if (val < 0)
          { return this.depth + val; }
      return val;
  };
  /**
  The parent node that the position points into. Note that even if
  a position points into a text node, that node is not considered
  the parent—text nodes are ‘flat’ in this model, and have no content.
  */
  prototypeAccessors$2$4.parent.get = function () { return this.node(this.depth); };
  /**
  The root node in which the position was resolved.
  */
  prototypeAccessors$2$4.doc.get = function () { return this.node(0); };
  /**
  The ancestor node at the given level. `p.node(p.depth)` is the
  same as `p.parent`.
  */
  ResolvedPos.prototype.node = function node (depth) { return this.path[this.resolveDepth(depth) * 3]; };
  /**
  The index into the ancestor at the given level. If this points
  at the 3rd node in the 2nd paragraph on the top level, for
  example, `p.index(0)` is 1 and `p.index(1)` is 2.
  */
  ResolvedPos.prototype.index = function index (depth) { return this.path[this.resolveDepth(depth) * 3 + 1]; };
  /**
  The index pointing after this position into the ancestor at the
  given level.
  */
  ResolvedPos.prototype.indexAfter = function indexAfter (depth) {
      depth = this.resolveDepth(depth);
      return this.index(depth) + (depth == this.depth && !this.textOffset ? 0 : 1);
  };
  /**
  The (absolute) position at the start of the node at the given
  level.
  */
  ResolvedPos.prototype.start = function start (depth) {
      depth = this.resolveDepth(depth);
      return depth == 0 ? 0 : this.path[depth * 3 - 1] + 1;
  };
  /**
  The (absolute) position at the end of the node at the given
  level.
  */
  ResolvedPos.prototype.end = function end (depth) {
      depth = this.resolveDepth(depth);
      return this.start(depth) + this.node(depth).content.size;
  };
  /**
  The (absolute) position directly before the wrapping node at the
  given level, or, when `depth` is `this.depth + 1`, the original
  position.
  */
  ResolvedPos.prototype.before = function before (depth) {
      depth = this.resolveDepth(depth);
      if (!depth)
          { throw new RangeError("There is no position before the top-level node"); }
      return depth == this.depth + 1 ? this.pos : this.path[depth * 3 - 1];
  };
  /**
  The (absolute) position directly after the wrapping node at the
  given level, or the original position when `depth` is `this.depth + 1`.
  */
  ResolvedPos.prototype.after = function after (depth) {
      depth = this.resolveDepth(depth);
      if (!depth)
          { throw new RangeError("There is no position after the top-level node"); }
      return depth == this.depth + 1 ? this.pos : this.path[depth * 3 - 1] + this.path[depth * 3].nodeSize;
  };
  /**
  When this position points into a text node, this returns the
  distance between the position and the start of the text node.
  Will be zero for positions that point between nodes.
  */
  prototypeAccessors$2$4.textOffset.get = function () { return this.pos - this.path[this.path.length - 1]; };
  /**
  Get the node directly after the position, if any. If the position
  points into a text node, only the part of that node after the
  position is returned.
  */
  prototypeAccessors$2$4.nodeAfter.get = function () {
      var parent = this.parent, index = this.index(this.depth);
      if (index == parent.childCount)
          { return null; }
      var dOff = this.pos - this.path[this.path.length - 1], child = parent.child(index);
      return dOff ? parent.child(index).cut(dOff) : child;
  };
  /**
  Get the node directly before the position, if any. If the
  position points into a text node, only the part of that node
  before the position is returned.
  */
  prototypeAccessors$2$4.nodeBefore.get = function () {
      var index = this.index(this.depth);
      var dOff = this.pos - this.path[this.path.length - 1];
      if (dOff)
          { return this.parent.child(index).cut(0, dOff); }
      return index == 0 ? null : this.parent.child(index - 1);
  };
  /**
  Get the position at the given index in the parent node at the
  given depth (which defaults to `this.depth`).
  */
  ResolvedPos.prototype.posAtIndex = function posAtIndex (index, depth) {
      depth = this.resolveDepth(depth);
      var node = this.path[depth * 3], pos = depth == 0 ? 0 : this.path[depth * 3 - 1] + 1;
      for (var i = 0; i < index; i++)
          { pos += node.child(i).nodeSize; }
      return pos;
  };
  /**
  Get the marks at this position, factoring in the surrounding
  marks' [`inclusive`](https://prosemirror.net/docs/ref/#model.MarkSpec.inclusive) property. If the
  position is at the start of a non-empty node, the marks of the
  node after it (if any) are returned.
  */
  ResolvedPos.prototype.marks = function marks () {
      var parent = this.parent, index = this.index();
      // In an empty parent, return the empty array
      if (parent.content.size == 0)
          { return Mark.none; }
      // When inside a text node, just return the text node's marks
      if (this.textOffset)
          { return parent.child(index).marks; }
      var main = parent.maybeChild(index - 1), other = parent.maybeChild(index);
      // If the `after` flag is true of there is no node before, make
      // the node after this position the main reference.
      if (!main) {
          var tmp = main;
          main = other;
          other = tmp;
      }
      // Use all marks in the main node, except those that have
      // `inclusive` set to false and are not present in the other node.
      var marks = main.marks;
      for (var i = 0; i < marks.length; i++)
          { if (marks[i].type.spec.inclusive === false && (!other || !marks[i].isInSet(other.marks)))
              { marks = marks[i--].removeFromSet(marks); } }
      return marks;
  };
  /**
  Get the marks after the current position, if any, except those
  that are non-inclusive and not present at position `$end`. This
  is mostly useful for getting the set of marks to preserve after a
  deletion. Will return `null` if this position is at the end of
  its parent node or its parent node isn't a textblock (in which
  case no marks should be preserved).
  */
  ResolvedPos.prototype.marksAcross = function marksAcross ($end) {
      var after = this.parent.maybeChild(this.index());
      if (!after || !after.isInline)
          { return null; }
      var marks = after.marks, next = $end.parent.maybeChild($end.index());
      for (var i = 0; i < marks.length; i++)
          { if (marks[i].type.spec.inclusive === false && (!next || !marks[i].isInSet(next.marks)))
              { marks = marks[i--].removeFromSet(marks); } }
      return marks;
  };
  /**
  The depth up to which this position and the given (non-resolved)
  position share the same parent nodes.
  */
  ResolvedPos.prototype.sharedDepth = function sharedDepth (pos) {
      for (var depth = this.depth; depth > 0; depth--)
          { if (this.start(depth) <= pos && this.end(depth) >= pos)
              { return depth; } }
      return 0;
  };
  /**
  Returns a range based on the place where this position and the
  given position diverge around block content. If both point into
  the same textblock, for example, a range around that textblock
  will be returned. If they point into different blocks, the range
  around those blocks in their shared ancestor is returned. You can
  pass in an optional predicate that will be called with a parent
  node to see if a range into that parent is acceptable.
  */
  ResolvedPos.prototype.blockRange = function blockRange (other, pred) {
        if ( other === void 0 ) other = this;

      if (other.pos < this.pos)
          { return other.blockRange(this); }
      for (var d = this.depth - (this.parent.inlineContent || this.pos == other.pos ? 1 : 0); d >= 0; d--)
          { if (other.pos <= this.end(d) && (!pred || pred(this.node(d))))
              { return new NodeRange(this, other, d); } }
      return null;
  };
  /**
  Query whether the given position shares the same parent node.
  */
  ResolvedPos.prototype.sameParent = function sameParent (other) {
      return this.pos - this.parentOffset == other.pos - other.parentOffset;
  };
  /**
  Return the greater of this and the given position.
  */
  ResolvedPos.prototype.max = function max (other) {
      return other.pos > this.pos ? other : this;
  };
  /**
  Return the smaller of this and the given position.
  */
  ResolvedPos.prototype.min = function min (other) {
      return other.pos < this.pos ? other : this;
  };
  /**
  @internal
  */
  ResolvedPos.prototype.toString = function toString () {
      var str = "";
      for (var i = 1; i <= this.depth; i++)
          { str += (str ? "/" : "") + this.node(i).type.name + "_" + this.index(i - 1); }
      return str + ":" + this.parentOffset;
  };
  /**
  @internal
  */
  ResolvedPos.resolve = function resolve (doc, pos) {
      if (!(pos >= 0 && pos <= doc.content.size))
          { throw new RangeError("Position " + pos + " out of range"); }
      var path = [];
      var start = 0, parentOffset = pos;
      for (var node = doc;;) {
          var ref = node.content.findIndex(parentOffset);
            var index = ref.index;
            var offset = ref.offset;
          var rem = parentOffset - offset;
          path.push(node, index, start + offset);
          if (!rem)
              { break; }
          node = node.child(index);
          if (node.isText)
              { break; }
          parentOffset = rem - 1;
          start += offset + 1;
      }
      return new ResolvedPos(pos, path, parentOffset);
  };
  /**
  @internal
  */
  ResolvedPos.resolveCached = function resolveCached (doc, pos) {
      var cache = resolveCache.get(doc);
      if (cache) {
          for (var i = 0; i < cache.elts.length; i++) {
              var elt = cache.elts[i];
              if (elt.pos == pos)
                  { return elt; }
          }
      }
      else {
          resolveCache.set(doc, cache = new ResolveCache);
      }
      var result = cache.elts[cache.i] = ResolvedPos.resolve(doc, pos);
      cache.i = (cache.i + 1) % resolveCacheSize;
      return result;
  };

Object.defineProperties( ResolvedPos.prototype, prototypeAccessors$2$4 );
var ResolveCache = function ResolveCache() {
      this.elts = [];
      this.i = 0;
  };
var resolveCacheSize = 12, resolveCache = new WeakMap();
/**
Represents a flat range of content, i.e. one that starts and
ends in the same node.
*/
var NodeRange = function NodeRange(
  /**
  A resolved position along the start of the content. May have a
  `depth` greater than this object's `depth` property, since
  these are the positions that were used to compute the range,
  not re-resolved positions directly at its boundaries.
  */
  $from, 
  /**
  A position along the end of the content. See
  caveat for [`$from`](https://prosemirror.net/docs/ref/#model.NodeRange.$from).
  */
  $to, 
  /**
  The depth of the node that this range points into.
  */
  depth) {
      this.$from = $from;
      this.$to = $to;
      this.depth = depth;
  };

var prototypeAccessors$3$3 = { start: { configurable: true },end: { configurable: true },parent: { configurable: true },startIndex: { configurable: true },endIndex: { configurable: true } };
  /**
  The position at the start of the range.
  */
  prototypeAccessors$3$3.start.get = function () { return this.$from.before(this.depth + 1); };
  /**
  The position at the end of the range.
  */
  prototypeAccessors$3$3.end.get = function () { return this.$to.after(this.depth + 1); };
  /**
  The parent node that the range points into.
  */
  prototypeAccessors$3$3.parent.get = function () { return this.$from.node(this.depth); };
  /**
  The start index of the range in the parent node.
  */
  prototypeAccessors$3$3.startIndex.get = function () { return this.$from.index(this.depth); };
  /**
  The end index of the range in the parent node.
  */
  prototypeAccessors$3$3.endIndex.get = function () { return this.$to.indexAfter(this.depth); };

Object.defineProperties( NodeRange.prototype, prototypeAccessors$3$3 );

var emptyAttrs = Object.create(null);
/**
This class represents a node in the tree that makes up a
ProseMirror document. So a document is an instance of `Node`, with
children that are also instances of `Node`.

Nodes are persistent data structures. Instead of changing them, you
create new ones with the content you want. Old ones keep pointing
at the old document shape. This is made cheaper by sharing
structure between the old and new data as much as possible, which a
tree shape like this (without back pointers) makes easy.

**Do not** directly mutate the properties of a `Node` object. See
[the guide](/docs/guide/#doc) for more information.
*/
var Node$1 = function Node(
  /**
  The type of node that this is.
  */
  type, 
  /**
  An object mapping attribute names to values. The kind of
  attributes allowed and required are
  [determined](https://prosemirror.net/docs/ref/#model.NodeSpec.attrs) by the node type.
  */
  attrs, 
  // A fragment holding the node's children.
  content, 
  /**
  The marks (things like whether it is emphasized or part of a
  link) applied to this node.
  */
  marks) {
      if ( marks === void 0 ) marks = Mark.none;

      this.type = type;
      this.attrs = attrs;
      this.marks = marks;
      this.content = content || Fragment.empty;
  };

var prototypeAccessors$4$3 = { nodeSize: { configurable: true },childCount: { configurable: true },textContent: { configurable: true },firstChild: { configurable: true },lastChild: { configurable: true },isBlock: { configurable: true },isTextblock: { configurable: true },inlineContent: { configurable: true },isInline: { configurable: true },isText: { configurable: true },isLeaf: { configurable: true },isAtom: { configurable: true } };
  /**
  The size of this node, as defined by the integer-based [indexing
  scheme](/docs/guide/#doc.indexing). For text nodes, this is the
  amount of characters. For other leaf nodes, it is one. For
  non-leaf nodes, it is the size of the content plus two (the
  start and end token).
  */
  prototypeAccessors$4$3.nodeSize.get = function () { return this.isLeaf ? 1 : 2 + this.content.size; };
  /**
  The number of children that the node has.
  */
  prototypeAccessors$4$3.childCount.get = function () { return this.content.childCount; };
  /**
  Get the child node at the given index. Raises an error when the
  index is out of range.
  */
  Node$1.prototype.child = function child (index) { return this.content.child(index); };
  /**
  Get the child node at the given index, if it exists.
  */
  Node$1.prototype.maybeChild = function maybeChild (index) { return this.content.maybeChild(index); };
  /**
  Call `f` for every child node, passing the node, its offset
  into this parent node, and its index.
  */
  Node$1.prototype.forEach = function forEach (f) { this.content.forEach(f); };
  /**
  Invoke a callback for all descendant nodes recursively between
  the given two positions that are relative to start of this
  node's content. The callback is invoked with the node, its
  position relative to the original node (method receiver),
  its parent node, and its child index. When the callback returns
  false for a given node, that node's children will not be
  recursed over. The last parameter can be used to specify a
  starting position to count from.
  */
  Node$1.prototype.nodesBetween = function nodesBetween (from, to, f, startPos) {
        if ( startPos === void 0 ) startPos = 0;

      this.content.nodesBetween(from, to, f, startPos, this);
  };
  /**
  Call the given callback for every descendant node. Doesn't
  descend into a node when the callback returns `false`.
  */
  Node$1.prototype.descendants = function descendants (f) {
      this.nodesBetween(0, this.content.size, f);
  };
  /**
  Concatenates all the text nodes found in this fragment and its
  children.
  */
  prototypeAccessors$4$3.textContent.get = function () {
      return (this.isLeaf && this.type.spec.leafText)
          ? this.type.spec.leafText(this)
          : this.textBetween(0, this.content.size, "");
  };
  /**
  Get all text between positions `from` and `to`. When
  `blockSeparator` is given, it will be inserted to separate text
  from different block nodes. If `leafText` is given, it'll be
  inserted for every non-text leaf node encountered, otherwise
  [`leafText`](https://prosemirror.net/docs/ref/#model.NodeSpec^leafText) will be used.
  */
  Node$1.prototype.textBetween = function textBetween (from, to, blockSeparator, leafText) {
      return this.content.textBetween(from, to, blockSeparator, leafText);
  };
  /**
  Returns this node's first child, or `null` if there are no
  children.
  */
  prototypeAccessors$4$3.firstChild.get = function () { return this.content.firstChild; };
  /**
  Returns this node's last child, or `null` if there are no
  children.
  */
  prototypeAccessors$4$3.lastChild.get = function () { return this.content.lastChild; };
  /**
  Test whether two nodes represent the same piece of document.
  */
  Node$1.prototype.eq = function eq (other) {
      return this == other || (this.sameMarkup(other) && this.content.eq(other.content));
  };
  /**
  Compare the markup (type, attributes, and marks) of this node to
  those of another. Returns `true` if both have the same markup.
  */
  Node$1.prototype.sameMarkup = function sameMarkup (other) {
      return this.hasMarkup(other.type, other.attrs, other.marks);
  };
  /**
  Check whether this node's markup correspond to the given type,
  attributes, and marks.
  */
  Node$1.prototype.hasMarkup = function hasMarkup (type, attrs, marks) {
      return this.type == type &&
          compareDeep(this.attrs, attrs || type.defaultAttrs || emptyAttrs) &&
          Mark.sameSet(this.marks, marks || Mark.none);
  };
  /**
  Create a new node with the same markup as this node, containing
  the given content (or empty, if no content is given).
  */
  Node$1.prototype.copy = function copy (content) {
        if ( content === void 0 ) content = null;

      if (content == this.content)
          { return this; }
      return new Node$1(this.type, this.attrs, content, this.marks);
  };
  /**
  Create a copy of this node, with the given set of marks instead
  of the node's own marks.
  */
  Node$1.prototype.mark = function mark (marks) {
      return marks == this.marks ? this : new Node$1(this.type, this.attrs, this.content, marks);
  };
  /**
  Create a copy of this node with only the content between the
  given positions. If `to` is not given, it defaults to the end of
  the node.
  */
  Node$1.prototype.cut = function cut (from, to) {
        if ( to === void 0 ) to = this.content.size;

      if (from == 0 && to == this.content.size)
          { return this; }
      return this.copy(this.content.cut(from, to));
  };
  /**
  Cut out the part of the document between the given positions, and
  return it as a `Slice` object.
  */
  Node$1.prototype.slice = function slice (from, to, includeParents) {
        if ( to === void 0 ) to = this.content.size;
        if ( includeParents === void 0 ) includeParents = false;

      if (from == to)
          { return Slice.empty; }
      var $from = this.resolve(from), $to = this.resolve(to);
      var depth = includeParents ? 0 : $from.sharedDepth(to);
      var start = $from.start(depth), node = $from.node(depth);
      var content = node.content.cut($from.pos - start, $to.pos - start);
      return new Slice(content, $from.depth - depth, $to.depth - depth);
  };
  /**
  Replace the part of the document between the given positions with
  the given slice. The slice must 'fit', meaning its open sides
  must be able to connect to the surrounding content, and its
  content nodes must be valid children for the node they are placed
  into. If any of this is violated, an error of type
  [`ReplaceError`](https://prosemirror.net/docs/ref/#model.ReplaceError) is thrown.
  */
  Node$1.prototype.replace = function replace$1 (from, to, slice) {
      return replace$3(this.resolve(from), this.resolve(to), slice);
  };
  /**
  Find the node directly after the given position.
  */
  Node$1.prototype.nodeAt = function nodeAt (pos) {
      for (var node = this;;) {
          var ref = node.content.findIndex(pos);
            var index = ref.index;
            var offset = ref.offset;
          node = node.maybeChild(index);
          if (!node)
              { return null; }
          if (offset == pos || node.isText)
              { return node; }
          pos -= offset + 1;
      }
  };
  /**
  Find the (direct) child node after the given offset, if any,
  and return it along with its index and offset relative to this
  node.
  */
  Node$1.prototype.childAfter = function childAfter (pos) {
      var ref = this.content.findIndex(pos);
        var index = ref.index;
        var offset = ref.offset;
      return { node: this.content.maybeChild(index), index: index, offset: offset };
  };
  /**
  Find the (direct) child node before the given offset, if any,
  and return it along with its index and offset relative to this
  node.
  */
  Node$1.prototype.childBefore = function childBefore (pos) {
      if (pos == 0)
          { return { node: null, index: 0, offset: 0 }; }
      var ref = this.content.findIndex(pos);
        var index = ref.index;
        var offset = ref.offset;
      if (offset < pos)
          { return { node: this.content.child(index), index: index, offset: offset }; }
      var node = this.content.child(index - 1);
      return { node: node, index: index - 1, offset: offset - node.nodeSize };
  };
  /**
  Resolve the given position in the document, returning an
  [object](https://prosemirror.net/docs/ref/#model.ResolvedPos) with information about its context.
  */
  Node$1.prototype.resolve = function resolve (pos) { return ResolvedPos.resolveCached(this, pos); };
  /**
  @internal
  */
  Node$1.prototype.resolveNoCache = function resolveNoCache (pos) { return ResolvedPos.resolve(this, pos); };
  /**
  Test whether a given mark or mark type occurs in this document
  between the two given positions.
  */
  Node$1.prototype.rangeHasMark = function rangeHasMark (from, to, type) {
      var found = false;
      if (to > from)
          { this.nodesBetween(from, to, function (node) {
              if (type.isInSet(node.marks))
                  { found = true; }
              return !found;
          }); }
      return found;
  };
  /**
  True when this is a block (non-inline node)
  */
  prototypeAccessors$4$3.isBlock.get = function () { return this.type.isBlock; };
  /**
  True when this is a textblock node, a block node with inline
  content.
  */
  prototypeAccessors$4$3.isTextblock.get = function () { return this.type.isTextblock; };
  /**
  True when this node allows inline content.
  */
  prototypeAccessors$4$3.inlineContent.get = function () { return this.type.inlineContent; };
  /**
  True when this is an inline node (a text node or a node that can
  appear among text).
  */
  prototypeAccessors$4$3.isInline.get = function () { return this.type.isInline; };
  /**
  True when this is a text node.
  */
  prototypeAccessors$4$3.isText.get = function () { return this.type.isText; };
  /**
  True when this is a leaf node.
  */
  prototypeAccessors$4$3.isLeaf.get = function () { return this.type.isLeaf; };
  /**
  True when this is an atom, i.e. when it does not have directly
  editable content. This is usually the same as `isLeaf`, but can
  be configured with the [`atom` property](https://prosemirror.net/docs/ref/#model.NodeSpec.atom)
  on a node's spec (typically used when the node is displayed as
  an uneditable [node view](https://prosemirror.net/docs/ref/#view.NodeView)).
  */
  prototypeAccessors$4$3.isAtom.get = function () { return this.type.isAtom; };
  /**
  Return a string representation of this node for debugging
  purposes.
  */
  Node$1.prototype.toString = function toString () {
      if (this.type.spec.toDebugString)
          { return this.type.spec.toDebugString(this); }
      var name = this.type.name;
      if (this.content.size)
          { name += "(" + this.content.toStringInner() + ")"; }
      return wrapMarks$1(this.marks, name);
  };
  /**
  Get the content match in this node at the given index.
  */
  Node$1.prototype.contentMatchAt = function contentMatchAt (index) {
      var match = this.type.contentMatch.matchFragment(this.content, 0, index);
      if (!match)
          { throw new Error("Called contentMatchAt on a node with invalid content"); }
      return match;
  };
  /**
  Test whether replacing the range between `from` and `to` (by
  child index) with the given replacement fragment (which defaults
  to the empty fragment) would leave the node's content valid. You
  can optionally pass `start` and `end` indices into the
  replacement fragment.
  */
  Node$1.prototype.canReplace = function canReplace (from, to, replacement, start, end) {
        if ( replacement === void 0 ) replacement = Fragment.empty;
        if ( start === void 0 ) start = 0;
        if ( end === void 0 ) end = replacement.childCount;

      var one = this.contentMatchAt(from).matchFragment(replacement, start, end);
      var two = one && one.matchFragment(this.content, to);
      if (!two || !two.validEnd)
          { return false; }
      for (var i = start; i < end; i++)
          { if (!this.type.allowsMarks(replacement.child(i).marks))
              { return false; } }
      return true;
  };
  /**
  Test whether replacing the range `from` to `to` (by index) with
  a node of the given type would leave the node's content valid.
  */
  Node$1.prototype.canReplaceWith = function canReplaceWith (from, to, type, marks) {
      if (marks && !this.type.allowsMarks(marks))
          { return false; }
      var start = this.contentMatchAt(from).matchType(type);
      var end = start && start.matchFragment(this.content, to);
      return end ? end.validEnd : false;
  };
  /**
  Test whether the given node's content could be appended to this
  node. If that node is empty, this will only return true if there
  is at least one node type that can appear in both nodes (to avoid
  merging completely incompatible nodes).
  */
  Node$1.prototype.canAppend = function canAppend (other) {
      if (other.content.size)
          { return this.canReplace(this.childCount, this.childCount, other.content); }
      else
          { return this.type.compatibleContent(other.type); }
  };
  /**
  Check whether this node and its descendants conform to the
  schema, and raise an exception when they do not.
  */
  Node$1.prototype.check = function check () {
      this.type.checkContent(this.content);
      this.type.checkAttrs(this.attrs);
      var copy = Mark.none;
      for (var i = 0; i < this.marks.length; i++) {
          var mark = this.marks[i];
          mark.type.checkAttrs(mark.attrs);
          copy = mark.addToSet(copy);
      }
      if (!Mark.sameSet(copy, this.marks))
          { throw new RangeError(("Invalid collection of marks for node " + (this.type.name) + ": " + (this.marks.map(function (m) { return m.type.name; })))); }
      this.content.forEach(function (node) { return node.check(); });
  };
  /**
  Return a JSON-serializeable representation of this node.
  */
  Node$1.prototype.toJSON = function toJSON () {
      var obj = { type: this.type.name };
      for (var _ in this.attrs) {
          obj.attrs = this.attrs;
          break;
      }
      if (this.content.size)
          { obj.content = this.content.toJSON(); }
      if (this.marks.length)
          { obj.marks = this.marks.map(function (n) { return n.toJSON(); }); }
      return obj;
  };
  /**
  Deserialize a node from its JSON representation.
  */
  Node$1.fromJSON = function fromJSON (schema, json) {
      if (!json)
          { throw new RangeError("Invalid input for Node.fromJSON"); }
      var marks = undefined;
      if (json.marks) {
          if (!Array.isArray(json.marks))
              { throw new RangeError("Invalid mark data for Node.fromJSON"); }
          marks = json.marks.map(schema.markFromJSON);
      }
      if (json.type == "text") {
          if (typeof json.text != "string")
              { throw new RangeError("Invalid text node in JSON"); }
          return schema.text(json.text, marks);
      }
      var content = Fragment.fromJSON(schema, json.content);
      var node = schema.nodeType(json.type).create(json.attrs, content, marks);
      node.type.checkAttrs(node.attrs);
      return node;
  };

Object.defineProperties( Node$1.prototype, prototypeAccessors$4$3 );
Node$1.prototype.text = undefined;
var TextNode$1 = /*@__PURE__*/(function (Node) {
  function TextNode(type, attrs, content, marks) {
        Node.call(this, type, attrs, null, marks);
        if (!content)
            { throw new RangeError("Empty text nodes are not allowed"); }
        this.text = content;
    }

  if ( Node ) TextNode.__proto__ = Node;
  TextNode.prototype = Object.create( Node && Node.prototype );
  TextNode.prototype.constructor = TextNode;

  var prototypeAccessors$5 = { textContent: { configurable: true },nodeSize: { configurable: true } };
    TextNode.prototype.toString = function toString () {
        if (this.type.spec.toDebugString)
            { return this.type.spec.toDebugString(this); }
        return wrapMarks$1(this.marks, JSON.stringify(this.text));
    };
    prototypeAccessors$5.textContent.get = function () { return this.text; };
    TextNode.prototype.textBetween = function textBetween (from, to) { return this.text.slice(from, to); };
    prototypeAccessors$5.nodeSize.get = function () { return this.text.length; };
    TextNode.prototype.mark = function mark (marks) {
        return marks == this.marks ? this : new TextNode(this.type, this.attrs, this.text, marks);
    };
    TextNode.prototype.withText = function withText (text) {
        if (text == this.text)
            { return this; }
        return new TextNode(this.type, this.attrs, text, this.marks);
    };
    TextNode.prototype.cut = function cut (from, to) {
        if ( from === void 0 ) from = 0;
        if ( to === void 0 ) to = this.text.length;

        if (from == 0 && to == this.text.length)
            { return this; }
        return this.withText(this.text.slice(from, to));
    };
    TextNode.prototype.eq = function eq (other) {
        return this.sameMarkup(other) && this.text == other.text;
    };
    TextNode.prototype.toJSON = function toJSON () {
        var base = Node.prototype.toJSON.call(this);
        base.text = this.text;
        return base;
    };

  Object.defineProperties( TextNode.prototype, prototypeAccessors$5 );

  return TextNode;
}(Node$1));
function wrapMarks$1(marks, str) {
    for (var i = marks.length - 1; i >= 0; i--)
        { str = marks[i].type.name + "(" + str + ")"; }
    return str;
}

/**
Instances of this class represent a match state of a node type's
[content expression](https://prosemirror.net/docs/ref/#model.NodeSpec.content), and can be used to
find out whether further content matches here, and whether a given
position is a valid end of the node.
*/
var ContentMatch = function ContentMatch(
  /**
  True when this match state represents a valid end of the node.
  */
  validEnd) {
      this.validEnd = validEnd;
      /**
      @internal
      */
      this.next = [];
      /**
      @internal
      */
      this.wrapCache = [];
  };

var prototypeAccessors$6$2 = { inlineContent: { configurable: true },defaultType: { configurable: true },edgeCount: { configurable: true } };
  /**
  @internal
  */
  ContentMatch.parse = function parse (string, nodeTypes) {
      var stream = new TokenStream(string, nodeTypes);
      if (stream.next == null)
          { return ContentMatch.empty; }
      var expr = parseExpr(stream);
      if (stream.next)
          { stream.err("Unexpected trailing text"); }
      var match = dfa(nfa(expr));
      checkForDeadEnds(match, stream);
      return match;
  };
  /**
  Match a node type, returning a match after that node if
  successful.
  */
  ContentMatch.prototype.matchType = function matchType (type) {
      for (var i = 0; i < this.next.length; i++)
          { if (this.next[i].type == type)
              { return this.next[i].next; } }
      return null;
  };
  /**
  Try to match a fragment. Returns the resulting match when
  successful.
  */
  ContentMatch.prototype.matchFragment = function matchFragment (frag, start, end) {
        if ( start === void 0 ) start = 0;
        if ( end === void 0 ) end = frag.childCount;

      var cur = this;
      for (var i = start; cur && i < end; i++)
          { cur = cur.matchType(frag.child(i).type); }
      return cur;
  };
  /**
  @internal
  */
  prototypeAccessors$6$2.inlineContent.get = function () {
      return this.next.length != 0 && this.next[0].type.isInline;
  };
  /**
  Get the first matching node type at this match position that can
  be generated.
  */
  prototypeAccessors$6$2.defaultType.get = function () {
      for (var i = 0; i < this.next.length; i++) {
          var ref = this.next[i];
            var type = ref.type;
          if (!(type.isText || type.hasRequiredAttrs()))
              { return type; }
      }
      return null;
  };
  /**
  @internal
  */
  ContentMatch.prototype.compatible = function compatible (other) {
      for (var i = 0; i < this.next.length; i++)
          { for (var j = 0; j < other.next.length; j++)
              { if (this.next[i].type == other.next[j].type)
                  { return true; } } }
      return false;
  };
  /**
  Try to match the given fragment, and if that fails, see if it can
  be made to match by inserting nodes in front of it. When
  successful, return a fragment of inserted nodes (which may be
  empty if nothing had to be inserted). When `toEnd` is true, only
  return a fragment if the resulting match goes to the end of the
  content expression.
  */
  ContentMatch.prototype.fillBefore = function fillBefore (after, toEnd, startIndex) {
        if ( toEnd === void 0 ) toEnd = false;
        if ( startIndex === void 0 ) startIndex = 0;

      var seen = [this];
      function search(match, types) {
          var finished = match.matchFragment(after, startIndex);
          if (finished && (!toEnd || finished.validEnd))
              { return Fragment.from(types.map(function (tp) { return tp.createAndFill(); })); }
          for (var i = 0; i < match.next.length; i++) {
              var ref = match.next[i];
                var type = ref.type;
                var next = ref.next;
              if (!(type.isText || type.hasRequiredAttrs()) && seen.indexOf(next) == -1) {
                  seen.push(next);
                  var found = search(next, types.concat(type));
                  if (found)
                      { return found; }
              }
          }
          return null;
      }
      return search(this, []);
  };
  /**
  Find a set of wrapping node types that would allow a node of the
  given type to appear at this position. The result may be empty
  (when it fits directly) and will be null when no such wrapping
  exists.
  */
  ContentMatch.prototype.findWrapping = function findWrapping (target) {
      for (var i = 0; i < this.wrapCache.length; i += 2)
          { if (this.wrapCache[i] == target)
              { return this.wrapCache[i + 1]; } }
      var computed = this.computeWrapping(target);
      this.wrapCache.push(target, computed);
      return computed;
  };
  /**
  @internal
  */
  ContentMatch.prototype.computeWrapping = function computeWrapping (target) {
      var seen = Object.create(null), active = [{ match: this, type: null, via: null }];
      while (active.length) {
          var current = active.shift(), match = current.match;
          if (match.matchType(target)) {
              var result = [];
              for (var obj = current; obj.type; obj = obj.via)
                  { result.push(obj.type); }
              return result.reverse();
          }
          for (var i = 0; i < match.next.length; i++) {
              var ref = match.next[i];
                var type = ref.type;
                var next = ref.next;
              if (!type.isLeaf && !type.hasRequiredAttrs() && !(type.name in seen) && (!current.type || next.validEnd)) {
                  active.push({ match: type.contentMatch, type: type, via: current });
                  seen[type.name] = true;
              }
          }
      }
      return null;
  };
  /**
  The number of outgoing edges this node has in the finite
  automaton that describes the content expression.
  */
  prototypeAccessors$6$2.edgeCount.get = function () {
      return this.next.length;
  };
  /**
  Get the _n_​th outgoing edge from this node in the finite
  automaton that describes the content expression.
  */
  ContentMatch.prototype.edge = function edge (n) {
      if (n >= this.next.length)
          { throw new RangeError(("There's no " + n + "th edge in this content match")); }
      return this.next[n];
  };
  /**
  @internal
  */
  ContentMatch.prototype.toString = function toString () {
      var seen = [];
      function scan(m) {
          seen.push(m);
          for (var i = 0; i < m.next.length; i++)
              { if (seen.indexOf(m.next[i].next) == -1)
                  { scan(m.next[i].next); } }
      }
      scan(this);
      return seen.map(function (m, i) {
          var out = i + (m.validEnd ? "*" : " ") + " ";
          for (var i$1 = 0; i$1 < m.next.length; i$1++)
              { out += (i$1 ? ", " : "") + m.next[i$1].type.name + "->" + seen.indexOf(m.next[i$1].next); }
          return out;
      }).join("\n");
  };

Object.defineProperties( ContentMatch.prototype, prototypeAccessors$6$2 );
/**
@internal
*/
ContentMatch.empty = new ContentMatch(true);
var TokenStream = function TokenStream(string, nodeTypes) {
      this.string = string;
      this.nodeTypes = nodeTypes;
      this.inline = null;
      this.pos = 0;
      this.tokens = string.split(/\s*(?=\b|\W|$)/);
      if (this.tokens[this.tokens.length - 1] == "")
          { this.tokens.pop(); }
      if (this.tokens[0] == "")
          { this.tokens.shift(); }
  };

var prototypeAccessors$7$2 = { next: { configurable: true } };
  prototypeAccessors$7$2.next.get = function () { return this.tokens[this.pos]; };
  TokenStream.prototype.eat = function eat (tok) { return this.next == tok && (this.pos++ || true); };
  TokenStream.prototype.err = function err (str) { throw new SyntaxError(str + " (in content expression '" + this.string + "')"); };

Object.defineProperties( TokenStream.prototype, prototypeAccessors$7$2 );
function parseExpr(stream) {
    var exprs = [];
    do {
        exprs.push(parseExprSeq(stream));
    } while (stream.eat("|"));
    return exprs.length == 1 ? exprs[0] : { type: "choice", exprs: exprs };
}
function parseExprSeq(stream) {
    var exprs = [];
    do {
        exprs.push(parseExprSubscript(stream));
    } while (stream.next && stream.next != ")" && stream.next != "|");
    return exprs.length == 1 ? exprs[0] : { type: "seq", exprs: exprs };
}
function parseExprSubscript(stream) {
    var expr = parseExprAtom(stream);
    for (;;) {
        if (stream.eat("+"))
            { expr = { type: "plus", expr: expr }; }
        else if (stream.eat("*"))
            { expr = { type: "star", expr: expr }; }
        else if (stream.eat("?"))
            { expr = { type: "opt", expr: expr }; }
        else if (stream.eat("{"))
            { expr = parseExprRange(stream, expr); }
        else
            { break; }
    }
    return expr;
}
function parseNum(stream) {
    if (/\D/.test(stream.next))
        { stream.err("Expected number, got '" + stream.next + "'"); }
    var result = Number(stream.next);
    stream.pos++;
    return result;
}
function parseExprRange(stream, expr) {
    var min = parseNum(stream), max = min;
    if (stream.eat(",")) {
        if (stream.next != "}")
            { max = parseNum(stream); }
        else
            { max = -1; }
    }
    if (!stream.eat("}"))
        { stream.err("Unclosed braced range"); }
    return { type: "range", min: min, max: max, expr: expr };
}
function resolveName(stream, name) {
    var types = stream.nodeTypes, type = types[name];
    if (type)
        { return [type]; }
    var result = [];
    for (var typeName in types) {
        var type$1 = types[typeName];
        if (type$1.groups.indexOf(name) > -1)
            { result.push(type$1); }
    }
    if (result.length == 0)
        { stream.err("No node type or group '" + name + "' found"); }
    return result;
}
function parseExprAtom(stream) {
    if (stream.eat("(")) {
        var expr = parseExpr(stream);
        if (!stream.eat(")"))
            { stream.err("Missing closing paren"); }
        return expr;
    }
    else if (!/\W/.test(stream.next)) {
        var exprs = resolveName(stream, stream.next).map(function (type) {
            if (stream.inline == null)
                { stream.inline = type.isInline; }
            else if (stream.inline != type.isInline)
                { stream.err("Mixing inline and block content"); }
            return { type: "name", value: type };
        });
        stream.pos++;
        return exprs.length == 1 ? exprs[0] : { type: "choice", exprs: exprs };
    }
    else {
        stream.err("Unexpected token '" + stream.next + "'");
    }
}
/**
Construct an NFA from an expression as returned by the parser. The
NFA is represented as an array of states, which are themselves
arrays of edges, which are `{term, to}` objects. The first state is
the entry state and the last node is the success state.

Note that unlike typical NFAs, the edge ordering in this one is
significant, in that it is used to contruct filler content when
necessary.
*/
function nfa(expr) {
    var nfa = [[]];
    connect(compile(expr, 0), node());
    return nfa;
    function node() { return nfa.push([]) - 1; }
    function edge(from, to, term) {
        var edge = { term: term, to: to };
        nfa[from].push(edge);
        return edge;
    }
    function connect(edges, to) {
        edges.forEach(function (edge) { return edge.to = to; });
    }
    function compile(expr, from) {
        if (expr.type == "choice") {
            return expr.exprs.reduce(function (out, expr) { return out.concat(compile(expr, from)); }, []);
        }
        else if (expr.type == "seq") {
            for (var i = 0;; i++) {
                var next = compile(expr.exprs[i], from);
                if (i == expr.exprs.length - 1)
                    { return next; }
                connect(next, from = node());
            }
        }
        else if (expr.type == "star") {
            var loop = node();
            edge(from, loop);
            connect(compile(expr.expr, loop), loop);
            return [edge(loop)];
        }
        else if (expr.type == "plus") {
            var loop$1 = node();
            connect(compile(expr.expr, from), loop$1);
            connect(compile(expr.expr, loop$1), loop$1);
            return [edge(loop$1)];
        }
        else if (expr.type == "opt") {
            return [edge(from)].concat(compile(expr.expr, from));
        }
        else if (expr.type == "range") {
            var cur = from;
            for (var i$1 = 0; i$1 < expr.min; i$1++) {
                var next$1 = node();
                connect(compile(expr.expr, cur), next$1);
                cur = next$1;
            }
            if (expr.max == -1) {
                connect(compile(expr.expr, cur), cur);
            }
            else {
                for (var i$2 = expr.min; i$2 < expr.max; i$2++) {
                    var next$2 = node();
                    edge(cur, next$2);
                    connect(compile(expr.expr, cur), next$2);
                    cur = next$2;
                }
            }
            return [edge(cur)];
        }
        else if (expr.type == "name") {
            return [edge(from, undefined, expr.value)];
        }
        else {
            throw new Error("Unknown expr type");
        }
    }
}
function cmp(a, b) { return b - a; }
// Get the set of nodes reachable by null edges from `node`. Omit
// nodes with only a single null-out-edge, since they may lead to
// needless duplicated nodes.
function nullFrom(nfa, node) {
    var result = [];
    scan(node);
    return result.sort(cmp);
    function scan(node) {
        var edges = nfa[node];
        if (edges.length == 1 && !edges[0].term)
            { return scan(edges[0].to); }
        result.push(node);
        for (var i = 0; i < edges.length; i++) {
            var ref = edges[i];
            var term = ref.term;
            var to = ref.to;
            if (!term && result.indexOf(to) == -1)
                { scan(to); }
        }
    }
}
// Compiles an NFA as produced by `nfa` into a DFA, modeled as a set
// of state objects (`ContentMatch` instances) with transitions
// between them.
function dfa(nfa) {
    var labeled = Object.create(null);
    return explore(nullFrom(nfa, 0));
    function explore(states) {
        var out = [];
        states.forEach(function (node) {
            nfa[node].forEach(function (ref) {
                var term = ref.term;
                var to = ref.to;

                if (!term)
                    { return; }
                var set;
                for (var i = 0; i < out.length; i++)
                    { if (out[i][0] == term)
                        { set = out[i][1]; } }
                nullFrom(nfa, to).forEach(function (node) {
                    if (!set)
                        { out.push([term, set = []]); }
                    if (set.indexOf(node) == -1)
                        { set.push(node); }
                });
            });
        });
        var state = labeled[states.join(",")] = new ContentMatch(states.indexOf(nfa.length - 1) > -1);
        for (var i = 0; i < out.length; i++) {
            var states$1 = out[i][1].sort(cmp);
            state.next.push({ type: out[i][0], next: labeled[states$1.join(",")] || explore(states$1) });
        }
        return state;
    }
}
function checkForDeadEnds(match, stream) {
    for (var i = 0, work = [match]; i < work.length; i++) {
        var state = work[i], dead = !state.validEnd, nodes = [];
        for (var j = 0; j < state.next.length; j++) {
            var ref = state.next[j];
            var type = ref.type;
            var next = ref.next;
            nodes.push(type.name);
            if (dead && !(type.isText || type.hasRequiredAttrs()))
                { dead = false; }
            if (work.indexOf(next) == -1)
                { work.push(next); }
        }
        if (dead)
            { stream.err("Only non-generatable nodes (" + nodes.join(", ") + ") in a required position (see https://prosemirror.net/docs/guide/#generatable)"); }
    }
}

// For node types where all attrs have a default value (or which don't
// have any attributes), build up a single reusable default attribute
// object, and use it for all nodes that don't specify specific
// attributes.
function defaultAttrs(attrs) {
    var defaults = Object.create(null);
    for (var attrName in attrs) {
        var attr = attrs[attrName];
        if (!attr.hasDefault)
            { return null; }
        defaults[attrName] = attr.default;
    }
    return defaults;
}
function computeAttrs(attrs, value) {
    var built = Object.create(null);
    for (var name in attrs) {
        var given = value && value[name];
        if (given === undefined) {
            var attr = attrs[name];
            if (attr.hasDefault)
                { given = attr.default; }
            else
                { throw new RangeError("No value supplied for attribute " + name); }
        }
        built[name] = given;
    }
    return built;
}
function checkAttrs(attrs, values, type, name) {
    for (var name$1 in values)
        { if (!(name$1 in attrs))
            { throw new RangeError(("Unsupported attribute " + name$1 + " for " + type + " of type " + name$1)); } }
    for (var name$2 in attrs) {
        var attr = attrs[name$2];
        if (attr.validate)
            { attr.validate(values[name$2]); }
    }
}
function initAttrs(typeName, attrs) {
    var result = Object.create(null);
    if (attrs)
        { for (var name in attrs)
            { result[name] = new Attribute(typeName, name, attrs[name]); } }
    return result;
}
/**
Node types are objects allocated once per `Schema` and used to
[tag](https://prosemirror.net/docs/ref/#model.Node.type) `Node` instances. They contain information
about the node type, such as its name and what kind of node it
represents.
*/
var NodeType$2 = function NodeType(
  /**
  The name the node type has in this schema.
  */
  name, 
  /**
  A link back to the `Schema` the node type belongs to.
  */
  schema, 
  /**
  The spec that this type is based on
  */
  spec) {
      this.name = name;
      this.schema = schema;
      this.spec = spec;
      /**
      The set of marks allowed in this node. `null` means all marks
      are allowed.
      */
      this.markSet = null;
      this.groups = spec.group ? spec.group.split(" ") : [];
      this.attrs = initAttrs(name, spec.attrs);
      this.defaultAttrs = defaultAttrs(this.attrs);
      this.contentMatch = null;
      this.inlineContent = null;
      this.isBlock = !(spec.inline || name == "text");
      this.isText = name == "text";
  };

var prototypeAccessors$8$1 = { isInline: { configurable: true },isTextblock: { configurable: true },isLeaf: { configurable: true },isAtom: { configurable: true },whitespace: { configurable: true } };
  /**
  True if this is an inline type.
  */
  prototypeAccessors$8$1.isInline.get = function () { return !this.isBlock; };
  /**
  True if this is a textblock type, a block that contains inline
  content.
  */
  prototypeAccessors$8$1.isTextblock.get = function () { return this.isBlock && this.inlineContent; };
  /**
  True for node types that allow no content.
  */
  prototypeAccessors$8$1.isLeaf.get = function () { return this.contentMatch == ContentMatch.empty; };
  /**
  True when this node is an atom, i.e. when it does not have
  directly editable content.
  */
  prototypeAccessors$8$1.isAtom.get = function () { return this.isLeaf || !!this.spec.atom; };
  /**
  The node type's [whitespace](https://prosemirror.net/docs/ref/#model.NodeSpec.whitespace) option.
  */
  prototypeAccessors$8$1.whitespace.get = function () {
      return this.spec.whitespace || (this.spec.code ? "pre" : "normal");
  };
  /**
  Tells you whether this node type has any required attributes.
  */
  NodeType$2.prototype.hasRequiredAttrs = function hasRequiredAttrs () {
      for (var n in this.attrs)
          { if (this.attrs[n].isRequired)
              { return true; } }
      return false;
  };
  /**
  Indicates whether this node allows some of the same content as
  the given node type.
  */
  NodeType$2.prototype.compatibleContent = function compatibleContent (other) {
      return this == other || this.contentMatch.compatible(other.contentMatch);
  };
  /**
  @internal
  */
  NodeType$2.prototype.computeAttrs = function computeAttrs$1 (attrs) {
      if (!attrs && this.defaultAttrs)
          { return this.defaultAttrs; }
      else
          { return computeAttrs(this.attrs, attrs); }
  };
  /**
  Create a `Node` of this type. The given attributes are
  checked and defaulted (you can pass `null` to use the type's
  defaults entirely, if no required attributes exist). `content`
  may be a `Fragment`, a node, an array of nodes, or
  `null`. Similarly `marks` may be `null` to default to the empty
  set of marks.
  */
  NodeType$2.prototype.create = function create (attrs, content, marks) {
        if ( attrs === void 0 ) attrs = null;

      if (this.isText)
          { throw new Error("NodeType.create can't construct text nodes"); }
      return new Node$1(this, this.computeAttrs(attrs), Fragment.from(content), Mark.setFrom(marks));
  };
  /**
  Like [`create`](https://prosemirror.net/docs/ref/#model.NodeType.create), but check the given content
  against the node type's content restrictions, and throw an error
  if it doesn't match.
  */
  NodeType$2.prototype.createChecked = function createChecked (attrs, content, marks) {
        if ( attrs === void 0 ) attrs = null;

      content = Fragment.from(content);
      this.checkContent(content);
      return new Node$1(this, this.computeAttrs(attrs), content, Mark.setFrom(marks));
  };
  /**
  Like [`create`](https://prosemirror.net/docs/ref/#model.NodeType.create), but see if it is
  necessary to add nodes to the start or end of the given fragment
  to make it fit the node. If no fitting wrapping can be found,
  return null. Note that, due to the fact that required nodes can
  always be created, this will always succeed if you pass null or
  `Fragment.empty` as content.
  */
  NodeType$2.prototype.createAndFill = function createAndFill (attrs, content, marks) {
        if ( attrs === void 0 ) attrs = null;

      attrs = this.computeAttrs(attrs);
      content = Fragment.from(content);
      if (content.size) {
          var before = this.contentMatch.fillBefore(content);
          if (!before)
              { return null; }
          content = before.append(content);
      }
      var matched = this.contentMatch.matchFragment(content);
      var after = matched && matched.fillBefore(Fragment.empty, true);
      if (!after)
          { return null; }
      return new Node$1(this, attrs, content.append(after), Mark.setFrom(marks));
  };
  /**
  Returns true if the given fragment is valid content for this node
  type.
  */
  NodeType$2.prototype.validContent = function validContent (content) {
      var result = this.contentMatch.matchFragment(content);
      if (!result || !result.validEnd)
          { return false; }
      for (var i = 0; i < content.childCount; i++)
          { if (!this.allowsMarks(content.child(i).marks))
              { return false; } }
      return true;
  };
  /**
  Throws a RangeError if the given fragment is not valid content for this
  node type.
  @internal
  */
  NodeType$2.prototype.checkContent = function checkContent (content) {
      if (!this.validContent(content))
          { throw new RangeError(("Invalid content for node " + (this.name) + ": " + (content.toString().slice(0, 50)))); }
  };
  /**
  @internal
  */
  NodeType$2.prototype.checkAttrs = function checkAttrs$1 (attrs) {
      checkAttrs(this.attrs, attrs, "node", this.name);
  };
  /**
  Check whether the given mark type is allowed in this node.
  */
  NodeType$2.prototype.allowsMarkType = function allowsMarkType (markType) {
      return this.markSet == null || this.markSet.indexOf(markType) > -1;
  };
  /**
  Test whether the given set of marks are allowed in this node.
  */
  NodeType$2.prototype.allowsMarks = function allowsMarks (marks) {
      if (this.markSet == null)
          { return true; }
      for (var i = 0; i < marks.length; i++)
          { if (!this.allowsMarkType(marks[i].type))
              { return false; } }
      return true;
  };
  /**
  Removes the marks that are not allowed in this node from the given set.
  */
  NodeType$2.prototype.allowedMarks = function allowedMarks (marks) {
      if (this.markSet == null)
          { return marks; }
      var copy;
      for (var i = 0; i < marks.length; i++) {
          if (!this.allowsMarkType(marks[i].type)) {
              if (!copy)
                  { copy = marks.slice(0, i); }
          }
          else if (copy) {
              copy.push(marks[i]);
          }
      }
      return !copy ? marks : copy.length ? copy : Mark.none;
  };
  /**
  @internal
  */
  NodeType$2.compile = function compile (nodes, schema) {
      var result = Object.create(null);
      nodes.forEach(function (name, spec) { return result[name] = new NodeType$2(name, schema, spec); });
      var topType = schema.spec.topNode || "doc";
      if (!result[topType])
          { throw new RangeError("Schema is missing its top node type ('" + topType + "')"); }
      if (!result.text)
          { throw new RangeError("Every schema needs a 'text' type"); }
      for (var _ in result.text.attrs)
          { throw new RangeError("The text node type should not have attributes"); }
      return result;
  };

Object.defineProperties( NodeType$2.prototype, prototypeAccessors$8$1 );
function validateType(typeName, attrName, type) {
    var types = type.split("|");
    return function (value) {
        var name = value === null ? "null" : typeof value;
        if (types.indexOf(name) < 0)
            { throw new RangeError(("Expected value of type " + types + " for attribute " + attrName + " on type " + typeName + ", got " + name)); }
    };
}
// Attribute descriptors
var Attribute = function Attribute(typeName, attrName, options) {
      this.hasDefault = Object.prototype.hasOwnProperty.call(options, "default");
      this.default = options.default;
      this.validate = typeof options.validate == "string" ? validateType(typeName, attrName, options.validate) : options.validate;
  };

var prototypeAccessors$9$1 = { isRequired: { configurable: true } };
  prototypeAccessors$9$1.isRequired.get = function () {
      return !this.hasDefault;
  };

Object.defineProperties( Attribute.prototype, prototypeAccessors$9$1 );
// Marks
/**
Like nodes, marks (which are associated with nodes to signify
things like emphasis or being part of a link) are
[tagged](https://prosemirror.net/docs/ref/#model.Mark.type) with type objects, which are
instantiated once per `Schema`.
*/
var MarkType = function MarkType(
  /**
  The name of the mark type.
  */
  name, 
  /**
  @internal
  */
  rank, 
  /**
  The schema that this mark type instance is part of.
  */
  schema, 
  /**
  The spec on which the type is based.
  */
  spec) {
      this.name = name;
      this.rank = rank;
      this.schema = schema;
      this.spec = spec;
      this.attrs = initAttrs(name, spec.attrs);
      this.excluded = null;
      var defaults = defaultAttrs(this.attrs);
      this.instance = defaults ? new Mark(this, defaults) : null;
  };
  /**
  Create a mark of this type. `attrs` may be `null` or an object
  containing only some of the mark's attributes. The others, if
  they have defaults, will be added.
  */
  MarkType.prototype.create = function create (attrs) {
        if ( attrs === void 0 ) attrs = null;

      if (!attrs && this.instance)
          { return this.instance; }
      return new Mark(this, computeAttrs(this.attrs, attrs));
  };
  /**
  @internal
  */
  MarkType.compile = function compile (marks, schema) {
      var result = Object.create(null), rank = 0;
      marks.forEach(function (name, spec) { return result[name] = new MarkType(name, rank++, schema, spec); });
      return result;
  };
  /**
  When there is a mark of this type in the given set, a new set
  without it is returned. Otherwise, the input set is returned.
  */
  MarkType.prototype.removeFromSet = function removeFromSet (set) {
      for (var i = 0; i < set.length; i++)
          { if (set[i].type == this) {
              set = set.slice(0, i).concat(set.slice(i + 1));
              i--;
          } }
      return set;
  };
  /**
  Tests whether there is a mark of this type in the given set.
  */
  MarkType.prototype.isInSet = function isInSet (set) {
      for (var i = 0; i < set.length; i++)
          { if (set[i].type == this)
              { return set[i]; } }
  };
  /**
  @internal
  */
  MarkType.prototype.checkAttrs = function checkAttrs$2 (attrs) {
      checkAttrs(this.attrs, attrs, "mark", this.name);
  };
  /**
  Queries whether a given mark type is
  [excluded](https://prosemirror.net/docs/ref/#model.MarkSpec.excludes) by this one.
  */
  MarkType.prototype.excludes = function excludes (other) {
      return this.excluded.indexOf(other) > -1;
  };
/**
A document schema. Holds [node](https://prosemirror.net/docs/ref/#model.NodeType) and [mark
type](https://prosemirror.net/docs/ref/#model.MarkType) objects for the nodes and marks that may
occur in conforming documents, and provides functionality for
creating and deserializing such documents.

When given, the type parameters provide the names of the nodes and
marks in this schema.
*/
var Schema = function Schema(spec) {
      /**
      The [linebreak
      replacement](https://prosemirror.net/docs/ref/#model.NodeSpec.linebreakReplacement) node defined
      in this schema, if any.
      */
      this.linebreakReplacement = null;
      /**
      An object for storing whatever values modules may want to
      compute and cache per schema. (If you want to store something
      in it, try to use property names unlikely to clash.)
      */
      this.cached = Object.create(null);
      var instanceSpec = this.spec = {};
      for (var prop in spec)
          { instanceSpec[prop] = spec[prop]; }
      instanceSpec.nodes = OrderedMap.from(spec.nodes),
          instanceSpec.marks = OrderedMap.from(spec.marks || {}),
          this.nodes = NodeType$2.compile(this.spec.nodes, this);
      this.marks = MarkType.compile(this.spec.marks, this);
      var contentExprCache = Object.create(null);
      for (var prop$1 in this.nodes) {
          if (prop$1 in this.marks)
              { throw new RangeError(prop$1 + " can not be both a node and a mark"); }
          var type = this.nodes[prop$1], contentExpr = type.spec.content || "", markExpr = type.spec.marks;
          type.contentMatch = contentExprCache[contentExpr] ||
              (contentExprCache[contentExpr] = ContentMatch.parse(contentExpr, this.nodes));
          type.inlineContent = type.contentMatch.inlineContent;
          if (type.spec.linebreakReplacement) {
              if (this.linebreakReplacement)
                  { throw new RangeError("Multiple linebreak nodes defined"); }
              if (!type.isInline || !type.isLeaf)
                  { throw new RangeError("Linebreak replacement nodes must be inline leaf nodes"); }
              this.linebreakReplacement = type;
          }
          type.markSet = markExpr == "_" ? null :
              markExpr ? gatherMarks(this, markExpr.split(" ")) :
                  markExpr == "" || !type.inlineContent ? [] : null;
      }
      for (var prop$2 in this.marks) {
          var type$1 = this.marks[prop$2], excl = type$1.spec.excludes;
          type$1.excluded = excl == null ? [type$1] : excl == "" ? [] : gatherMarks(this, excl.split(" "));
      }
      this.nodeFromJSON = this.nodeFromJSON.bind(this);
      this.markFromJSON = this.markFromJSON.bind(this);
      this.topNodeType = this.nodes[this.spec.topNode || "doc"];
      this.cached.wrappings = Object.create(null);
  };
  /**
  Create a node in this schema. The `type` may be a string or a
  `NodeType` instance. Attributes will be extended with defaults,
  `content` may be a `Fragment`, `null`, a `Node`, or an array of
  nodes.
  */
  Schema.prototype.node = function node (type, attrs, content, marks) {
        if ( attrs === void 0 ) attrs = null;

      if (typeof type == "string")
          { type = this.nodeType(type); }
      else if (!(type instanceof NodeType$2))
          { throw new RangeError("Invalid node type: " + type); }
      else if (type.schema != this)
          { throw new RangeError("Node type from different schema used (" + type.name + ")"); }
      return type.createChecked(attrs, content, marks);
  };
  /**
  Create a text node in the schema. Empty text nodes are not
  allowed.
  */
  Schema.prototype.text = function text (text$1, marks) {
      var type = this.nodes.text;
      return new TextNode$1(type, type.defaultAttrs, text$1, Mark.setFrom(marks));
  };
  /**
  Create a mark with the given type and attributes.
  */
  Schema.prototype.mark = function mark (type, attrs) {
      if (typeof type == "string")
          { type = this.marks[type]; }
      return type.create(attrs);
  };
  /**
  Deserialize a node from its JSON representation. This method is
  bound.
  */
  Schema.prototype.nodeFromJSON = function nodeFromJSON (json) {
      return Node$1.fromJSON(this, json);
  };
  /**
  Deserialize a mark from its JSON representation. This method is
  bound.
  */
  Schema.prototype.markFromJSON = function markFromJSON (json) {
      return Mark.fromJSON(this, json);
  };
  /**
  @internal
  */
  Schema.prototype.nodeType = function nodeType (name) {
      var found = this.nodes[name];
      if (!found)
          { throw new RangeError("Unknown node type: " + name); }
      return found;
  };
function gatherMarks(schema, marks) {
    var found = [];
    for (var i = 0; i < marks.length; i++) {
        var name = marks[i], mark = schema.marks[name], ok = mark;
        if (mark) {
            found.push(mark);
        }
        else {
            for (var prop in schema.marks) {
                var mark$1 = schema.marks[prop];
                if (name == "_" || (mark$1.spec.group && mark$1.spec.group.split(" ").indexOf(name) > -1))
                    { found.push(ok = mark$1); }
            }
        }
        if (!ok)
            { throw new SyntaxError("Unknown mark type: '" + marks[i] + "'"); }
    }
    return found;
}

function isTagRule(rule) { return rule.tag != null; }
function isStyleRule(rule) { return rule.style != null; }
/**
A DOM parser represents a strategy for parsing DOM content into a
ProseMirror document conforming to a given schema. Its behavior is
defined by an array of [rules](https://prosemirror.net/docs/ref/#model.ParseRule).
*/
var DOMParser = function DOMParser(
  /**
  The schema into which the parser parses.
  */
  schema, 
  /**
  The set of [parse rules](https://prosemirror.net/docs/ref/#model.ParseRule) that the parser
  uses, in order of precedence.
  */
  rules) {
      var this$1$1 = this;

      this.schema = schema;
      this.rules = rules;
      /**
      @internal
      */
      this.tags = [];
      /**
      @internal
      */
      this.styles = [];
      var matchedStyles = this.matchedStyles = [];
      rules.forEach(function (rule) {
          if (isTagRule(rule)) {
              this$1$1.tags.push(rule);
          }
          else if (isStyleRule(rule)) {
              var prop = /[^=]*/.exec(rule.style)[0];
              if (matchedStyles.indexOf(prop) < 0)
                  { matchedStyles.push(prop); }
              this$1$1.styles.push(rule);
          }
      });
      // Only normalize list elements when lists in the schema can't directly contain themselves
      this.normalizeLists = !this.tags.some(function (r) {
          if (!/^(ul|ol)\b/.test(r.tag) || !r.node)
              { return false; }
          var node = schema.nodes[r.node];
          return node.contentMatch.matchType(node);
      });
  };
  /**
  Parse a document from the content of a DOM node.
  */
  DOMParser.prototype.parse = function parse (dom, options) {
        if ( options === void 0 ) options = {};

      var context = new ParseContext$1(this, options, false);
      context.addAll(dom, Mark.none, options.from, options.to);
      return context.finish();
  };
  /**
  Parses the content of the given DOM node, like
  [`parse`](https://prosemirror.net/docs/ref/#model.DOMParser.parse), and takes the same set of
  options. But unlike that method, which produces a whole node,
  this one returns a slice that is open at the sides, meaning that
  the schema constraints aren't applied to the start of nodes to
  the left of the input and the end of nodes at the end.
  */
  DOMParser.prototype.parseSlice = function parseSlice (dom, options) {
        if ( options === void 0 ) options = {};

      var context = new ParseContext$1(this, options, true);
      context.addAll(dom, Mark.none, options.from, options.to);
      return Slice.maxOpen(context.finish());
  };
  /**
  @internal
  */
  DOMParser.prototype.matchTag = function matchTag (dom, context, after) {
      for (var i = after ? this.tags.indexOf(after) + 1 : 0; i < this.tags.length; i++) {
          var rule = this.tags[i];
          if (matches(dom, rule.tag) &&
              (rule.namespace === undefined || dom.namespaceURI == rule.namespace) &&
              (!rule.context || context.matchesContext(rule.context))) {
              if (rule.getAttrs) {
                  var result = rule.getAttrs(dom);
                  if (result === false)
                      { continue; }
                  rule.attrs = result || undefined;
              }
              return rule;
          }
      }
  };
  /**
  @internal
  */
  DOMParser.prototype.matchStyle = function matchStyle (prop, value, context, after) {
      for (var i = after ? this.styles.indexOf(after) + 1 : 0; i < this.styles.length; i++) {
          var rule = this.styles[i], style = rule.style;
          if (style.indexOf(prop) != 0 ||
              rule.context && !context.matchesContext(rule.context) ||
              // Test that the style string either precisely matches the prop,
              // or has an '=' sign after the prop, followed by the given
              // value.
              style.length > prop.length &&
                  (style.charCodeAt(prop.length) != 61 || style.slice(prop.length + 1) != value))
              { continue; }
          if (rule.getAttrs) {
              var result = rule.getAttrs(value);
              if (result === false)
                  { continue; }
              rule.attrs = result || undefined;
          }
          return rule;
      }
  };
  /**
  @internal
  */
  DOMParser.schemaRules = function schemaRules (schema) {
      var result = [];
      function insert(rule) {
          var priority = rule.priority == null ? 50 : rule.priority, i = 0;
          for (; i < result.length; i++) {
              var next = result[i], nextPriority = next.priority == null ? 50 : next.priority;
              if (nextPriority < priority)
                  { break; }
          }
          result.splice(i, 0, rule);
      }
      var loop = function ( name ) {
          var rules = schema.marks[name].spec.parseDOM;
          if (rules)
              { rules.forEach(function (rule) {
                  insert(rule = copy$1(rule));
                  if (!(rule.mark || rule.ignore || rule.clearMark))
                      { rule.mark = name; }
              }); }
      };

        for (var name in schema.marks) loop( name );
      var loop$1 = function ( name ) {
          var rules$1 = schema.nodes[name$1].spec.parseDOM;
          if (rules$1)
              { rules$1.forEach(function (rule) {
                  insert(rule = copy$1(rule));
                  if (!(rule.node || rule.ignore || rule.mark))
                      { rule.node = name$1; }
              }); }
      };

        for (var name$1 in schema.nodes) loop$1();
      return result;
  };
  /**
  Construct a DOM parser using the parsing rules listed in a
  schema's [node specs](https://prosemirror.net/docs/ref/#model.NodeSpec.parseDOM), reordered by
  [priority](https://prosemirror.net/docs/ref/#model.ParseRule.priority).
  */
  DOMParser.fromSchema = function fromSchema (schema) {
      return schema.cached.domParser ||
          (schema.cached.domParser = new DOMParser(schema, DOMParser.schemaRules(schema)));
  };
var blockTags = {
    address: true, article: true, aside: true, blockquote: true, canvas: true,
    dd: true, div: true, dl: true, fieldset: true, figcaption: true, figure: true,
    footer: true, form: true, h1: true, h2: true, h3: true, h4: true, h5: true,
    h6: true, header: true, hgroup: true, hr: true, li: true, noscript: true, ol: true,
    output: true, p: true, pre: true, section: true, table: true, tfoot: true, ul: true
};
var ignoreTags = {
    head: true, noscript: true, object: true, script: true, style: true, title: true
};
var listTags = { ol: true, ul: true };
// Using a bitfield for node context options
var OPT_PRESERVE_WS = 1, OPT_PRESERVE_WS_FULL = 2, OPT_OPEN_LEFT = 4;
function wsOptionsFor(type, preserveWhitespace, base) {
    if (preserveWhitespace != null)
        { return (preserveWhitespace ? OPT_PRESERVE_WS : 0) |
            (preserveWhitespace === "full" ? OPT_PRESERVE_WS_FULL : 0); }
    return type && type.whitespace == "pre" ? OPT_PRESERVE_WS | OPT_PRESERVE_WS_FULL : base & ~OPT_OPEN_LEFT;
}
var NodeContext = function NodeContext(type, attrs, marks, solid, match, options) {
      this.type = type;
      this.attrs = attrs;
      this.marks = marks;
      this.solid = solid;
      this.options = options;
      this.content = [];
      // Marks applied to the node's children
      this.activeMarks = Mark.none;
      this.match = match || (options & OPT_OPEN_LEFT ? null : type.contentMatch);
  };
  NodeContext.prototype.findWrapping = function findWrapping (node) {
      if (!this.match) {
          if (!this.type)
              { return []; }
          var fill = this.type.contentMatch.fillBefore(Fragment.from(node));
          if (fill) {
              this.match = this.type.contentMatch.matchFragment(fill);
          }
          else {
              var start = this.type.contentMatch, wrap;
              if (wrap = start.findWrapping(node.type)) {
                  this.match = start;
                  return wrap;
              }
              else {
                  return null;
              }
          }
      }
      return this.match.findWrapping(node.type);
  };
  NodeContext.prototype.finish = function finish (openEnd) {
      if (!(this.options & OPT_PRESERVE_WS)) { // Strip trailing whitespace
          var last = this.content[this.content.length - 1], m;
          if (last && last.isText && (m = /[ \t\r\n\u000c]+$/.exec(last.text))) {
              var text = last;
              if (last.text.length == m[0].length)
                  { this.content.pop(); }
              else
                  { this.content[this.content.length - 1] = text.withText(text.text.slice(0, text.text.length - m[0].length)); }
          }
      }
      var content = Fragment.from(this.content);
      if (!openEnd && this.match)
          { content = content.append(this.match.fillBefore(Fragment.empty, true)); }
      return this.type ? this.type.create(this.attrs, content, this.marks) : content;
  };
  NodeContext.prototype.inlineContext = function inlineContext (node) {
      if (this.type)
          { return this.type.inlineContent; }
      if (this.content.length)
          { return this.content[0].isInline; }
      return node.parentNode && !blockTags.hasOwnProperty(node.parentNode.nodeName.toLowerCase());
  };
var ParseContext$1 = function ParseContext(
  // The parser we are using.
  parser, 
  // The options passed to this parse.
  options, isOpen) {
      this.parser = parser;
      this.options = options;
      this.isOpen = isOpen;
      this.open = 0;
      var topNode = options.topNode, topContext;
      var topOptions = wsOptionsFor(null, options.preserveWhitespace, 0) | (isOpen ? OPT_OPEN_LEFT : 0);
      if (topNode)
          { topContext = new NodeContext(topNode.type, topNode.attrs, Mark.none, true, options.topMatch || topNode.type.contentMatch, topOptions); }
      else if (isOpen)
          { topContext = new NodeContext(null, null, Mark.none, true, null, topOptions); }
      else
          { topContext = new NodeContext(parser.schema.topNodeType, null, Mark.none, true, null, topOptions); }
      this.nodes = [topContext];
      this.find = options.findPositions;
      this.needsBlock = false;
  };

var prototypeAccessors$10$2 = { top: { configurable: true },currentPos: { configurable: true } };
  prototypeAccessors$10$2.top.get = function () {
      return this.nodes[this.open];
  };
  // Add a DOM node to the content. Text is inserted as text node,
  // otherwise, the node is passed to `addElement` or, if it has a
  // `style` attribute, `addElementWithStyles`.
  ParseContext$1.prototype.addDOM = function addDOM (dom, marks) {
      if (dom.nodeType == 3)
          { this.addTextNode(dom, marks); }
      else if (dom.nodeType == 1)
          { this.addElement(dom, marks); }
  };
  ParseContext$1.prototype.addTextNode = function addTextNode (dom, marks) {
      var value = dom.nodeValue;
      var top = this.top;
      if (top.options & OPT_PRESERVE_WS_FULL ||
          top.inlineContext(dom) ||
          /[^ \t\r\n\u000c]/.test(value)) {
          if (!(top.options & OPT_PRESERVE_WS)) {
              value = value.replace(/[ \t\r\n\u000c]+/g, " ");
              // If this starts with whitespace, and there is no node before it, or
              // a hard break, or a text node that ends with whitespace, strip the
              // leading space.
              if (/^[ \t\r\n\u000c]/.test(value) && this.open == this.nodes.length - 1) {
                  var nodeBefore = top.content[top.content.length - 1];
                  var domNodeBefore = dom.previousSibling;
                  if (!nodeBefore ||
                      (domNodeBefore && domNodeBefore.nodeName == 'BR') ||
                      (nodeBefore.isText && /[ \t\r\n\u000c]$/.test(nodeBefore.text)))
                      { value = value.slice(1); }
              }
          }
          else if (!(top.options & OPT_PRESERVE_WS_FULL)) {
              value = value.replace(/\r?\n|\r/g, " ");
          }
          else {
              value = value.replace(/\r\n?/g, "\n");
          }
          if (value)
              { this.insertNode(this.parser.schema.text(value), marks); }
          this.findInText(dom);
      }
      else {
          this.findInside(dom);
      }
  };
  // Try to find a handler for the given tag and use that to parse. If
  // none is found, the element's content nodes are added directly.
  ParseContext$1.prototype.addElement = function addElement (dom, marks, matchAfter) {
      var name = dom.nodeName.toLowerCase(), ruleID;
      if (listTags.hasOwnProperty(name) && this.parser.normalizeLists)
          { normalizeList(dom); }
      var rule = (this.options.ruleFromNode && this.options.ruleFromNode(dom)) ||
          (ruleID = this.parser.matchTag(dom, this, matchAfter));
      if (rule ? rule.ignore : ignoreTags.hasOwnProperty(name)) {
          this.findInside(dom);
          this.ignoreFallback(dom, marks);
      }
      else if (!rule || rule.skip || rule.closeParent) {
          if (rule && rule.closeParent)
              { this.open = Math.max(0, this.open - 1); }
          else if (rule && rule.skip.nodeType)
              { dom = rule.skip; }
          var sync, top = this.top, oldNeedsBlock = this.needsBlock;
          if (blockTags.hasOwnProperty(name)) {
              if (top.content.length && top.content[0].isInline && this.open) {
                  this.open--;
                  top = this.top;
              }
              sync = true;
              if (!top.type)
                  { this.needsBlock = true; }
          }
          else if (!dom.firstChild) {
              this.leafFallback(dom, marks);
              return;
          }
          var innerMarks = rule && rule.skip ? marks : this.readStyles(dom, marks);
          if (innerMarks)
              { this.addAll(dom, innerMarks); }
          if (sync)
              { this.sync(top); }
          this.needsBlock = oldNeedsBlock;
      }
      else {
          var innerMarks$1 = this.readStyles(dom, marks);
          if (innerMarks$1)
              { this.addElementByRule(dom, rule, innerMarks$1, rule.consuming === false ? ruleID : undefined); }
      }
  };
  // Called for leaf DOM nodes that would otherwise be ignored
  ParseContext$1.prototype.leafFallback = function leafFallback (dom, marks) {
      if (dom.nodeName == "BR" && this.top.type && this.top.type.inlineContent)
          { this.addTextNode(dom.ownerDocument.createTextNode("\n"), marks); }
  };
  // Called for ignored nodes
  ParseContext$1.prototype.ignoreFallback = function ignoreFallback (dom, marks) {
      // Ignored BR nodes should at least create an inline context
      if (dom.nodeName == "BR" && (!this.top.type || !this.top.type.inlineContent))
          { this.findPlace(this.parser.schema.text("-"), marks); }
  };
  // Run any style parser associated with the node's styles. Either
  // return an updated array of marks, or null to indicate some of the
  // styles had a rule with `ignore` set.
  ParseContext$1.prototype.readStyles = function readStyles (dom, marks) {
        var this$1$1 = this;

      var styles = dom.style;
      // Because many properties will only show up in 'normalized' form
      // in `style.item` (i.e. text-decoration becomes
      // text-decoration-line, text-decoration-color, etc), we directly
      // query the styles mentioned in our rules instead of iterating
      // over the items.
      if (styles && styles.length)
          { for (var i = 0; i < this.parser.matchedStyles.length; i++) {
              var name = this.parser.matchedStyles[i], value = styles.getPropertyValue(name);
              if (value)
                  { var loop = function ( after$1 ) {
                      var rule = this$1$1.parser.matchStyle(name, value, this$1$1, after$1);
                      if (!rule)
                          { return 'break'; }
                      if (rule.ignore)
                          { return { v: null }; }
                      if (rule.clearMark)
                          { marks = marks.filter(function (m) { return !rule.clearMark(m); }); }
                      else
                          { marks = marks.concat(this$1$1.parser.schema.marks[rule.mark].create(rule.attrs)); }
                      if (rule.consuming === false)
                          { after$1 = rule; }
                      else
                          { return 'break';
 }
                  };

                      for (var after = undefined;;) {
                        var returned = loop( after );

                        if ( returned === 'break' ) break;
                        if ( returned ) return returned.v;
                      } }
          } }
      return marks;
  };
  // Look up a handler for the given node. If none are found, return
  // false. Otherwise, apply it, use its return value to drive the way
  // the node's content is wrapped, and return true.
  ParseContext$1.prototype.addElementByRule = function addElementByRule (dom, rule, marks, continueAfter) {
        var this$1$1 = this;

      var sync, nodeType;
      if (rule.node) {
          nodeType = this.parser.schema.nodes[rule.node];
          if (!nodeType.isLeaf) {
              var inner = this.enter(nodeType, rule.attrs || null, marks, rule.preserveWhitespace);
              if (inner) {
                  sync = true;
                  marks = inner;
              }
          }
          else if (!this.insertNode(nodeType.create(rule.attrs), marks)) {
              this.leafFallback(dom, marks);
          }
      }
      else {
          var markType = this.parser.schema.marks[rule.mark];
          marks = marks.concat(markType.create(rule.attrs));
      }
      var startIn = this.top;
      if (nodeType && nodeType.isLeaf) {
          this.findInside(dom);
      }
      else if (continueAfter) {
          this.addElement(dom, marks, continueAfter);
      }
      else if (rule.getContent) {
          this.findInside(dom);
          rule.getContent(dom, this.parser.schema).forEach(function (node) { return this$1$1.insertNode(node, marks); });
      }
      else {
          var contentDOM = dom;
          if (typeof rule.contentElement == "string")
              { contentDOM = dom.querySelector(rule.contentElement); }
          else if (typeof rule.contentElement == "function")
              { contentDOM = rule.contentElement(dom); }
          else if (rule.contentElement)
              { contentDOM = rule.contentElement; }
          this.findAround(dom, contentDOM, true);
          this.addAll(contentDOM, marks);
      }
      if (sync && this.sync(startIn))
          { this.open--; }
  };
  // Add all child nodes between `startIndex` and `endIndex` (or the
  // whole node, if not given). If `sync` is passed, use it to
  // synchronize after every block element.
  ParseContext$1.prototype.addAll = function addAll (parent, marks, startIndex, endIndex) {
      var index = startIndex || 0;
      for (var dom = startIndex ? parent.childNodes[startIndex] : parent.firstChild, end = endIndex == null ? null : parent.childNodes[endIndex]; dom != end; dom = dom.nextSibling, ++index) {
          this.findAtPoint(parent, index);
          this.addDOM(dom, marks);
      }
      this.findAtPoint(parent, index);
  };
  // Try to find a way to fit the given node type into the current
  // context. May add intermediate wrappers and/or leave non-solid
  // nodes that we're in.
  ParseContext$1.prototype.findPlace = function findPlace (node, marks) {
      var route, sync;
      for (var depth = this.open; depth >= 0; depth--) {
          var cx = this.nodes[depth];
          var found = cx.findWrapping(node);
          if (found && (!route || route.length > found.length)) {
              route = found;
              sync = cx;
              if (!found.length)
                  { break; }
          }
          if (cx.solid)
              { break; }
      }
      if (!route)
          { return null; }
      this.sync(sync);
      for (var i = 0; i < route.length; i++)
          { marks = this.enterInner(route[i], null, marks, false); }
      return marks;
  };
  // Try to insert the given node, adjusting the context when needed.
  ParseContext$1.prototype.insertNode = function insertNode (node, marks) {
      if (node.isInline && this.needsBlock && !this.top.type) {
          var block = this.textblockFromContext();
          if (block)
              { marks = this.enterInner(block, null, marks); }
      }
      var innerMarks = this.findPlace(node, marks);
      if (innerMarks) {
          this.closeExtra();
          var top = this.top;
          if (top.match)
              { top.match = top.match.matchType(node.type); }
          var nodeMarks = Mark.none;
          for (var m of innerMarks.concat(node.marks))
              if (top.type ? top.type.allowsMarkType(m.type) : markMayApply(m.type, node.type))
                  { nodeMarks = m.addToSet(nodeMarks); }
          top.content.push(node.mark(nodeMarks));
          return true;
      }
      return false;
  };
  // Try to start a node of the given type, adjusting the context when
  // necessary.
  ParseContext$1.prototype.enter = function enter (type, attrs, marks, preserveWS) {
      var innerMarks = this.findPlace(type.create(attrs), marks);
      if (innerMarks)
          { innerMarks = this.enterInner(type, attrs, marks, true, preserveWS); }
      return innerMarks;
  };
  // Open a node of the given type
  ParseContext$1.prototype.enterInner = function enterInner (type, attrs, marks, solid, preserveWS) {
        if ( solid === void 0 ) solid = false;

      this.closeExtra();
      var top = this.top;
      top.match = top.match && top.match.matchType(type);
      var options = wsOptionsFor(type, preserveWS, top.options);
      if ((top.options & OPT_OPEN_LEFT) && top.content.length == 0)
          { options |= OPT_OPEN_LEFT; }
      var applyMarks = Mark.none;
      marks = marks.filter(function (m) {
          if (top.type ? top.type.allowsMarkType(m.type) : markMayApply(m.type, type)) {
              applyMarks = m.addToSet(applyMarks);
              return false;
          }
          return true;
      });
      this.nodes.push(new NodeContext(type, attrs, applyMarks, solid, null, options));
      this.open++;
      return marks;
  };
  // Make sure all nodes above this.open are finished and added to
  // their parents
  ParseContext$1.prototype.closeExtra = function closeExtra (openEnd) {
        if ( openEnd === void 0 ) openEnd = false;

      var i = this.nodes.length - 1;
      if (i > this.open) {
          for (; i > this.open; i--)
              { this.nodes[i - 1].content.push(this.nodes[i].finish(openEnd)); }
          this.nodes.length = this.open + 1;
      }
  };
  ParseContext$1.prototype.finish = function finish () {
      this.open = 0;
      this.closeExtra(this.isOpen);
      return this.nodes[0].finish(this.isOpen || this.options.topOpen);
  };
  ParseContext$1.prototype.sync = function sync (to) {
      for (var i = this.open; i >= 0; i--)
          { if (this.nodes[i] == to) {
              this.open = i;
              return true;
          } }
      return false;
  };
  prototypeAccessors$10$2.currentPos.get = function () {
      this.closeExtra();
      var pos = 0;
      for (var i = this.open; i >= 0; i--) {
          var content = this.nodes[i].content;
          for (var j = content.length - 1; j >= 0; j--)
              { pos += content[j].nodeSize; }
          if (i)
              { pos++; }
      }
      return pos;
  };
  ParseContext$1.prototype.findAtPoint = function findAtPoint (parent, offset) {
      if (this.find)
          { for (var i = 0; i < this.find.length; i++) {
              if (this.find[i].node == parent && this.find[i].offset == offset)
                  { this.find[i].pos = this.currentPos; }
          } }
  };
  ParseContext$1.prototype.findInside = function findInside (parent) {
      if (this.find)
          { for (var i = 0; i < this.find.length; i++) {
              if (this.find[i].pos == null && parent.nodeType == 1 && parent.contains(this.find[i].node))
                  { this.find[i].pos = this.currentPos; }
          } }
  };
  ParseContext$1.prototype.findAround = function findAround (parent, content, before) {
      if (parent != content && this.find)
          { for (var i = 0; i < this.find.length; i++) {
              if (this.find[i].pos == null && parent.nodeType == 1 && parent.contains(this.find[i].node)) {
                  var pos = content.compareDocumentPosition(this.find[i].node);
                  if (pos & (before ? 2 : 4))
                      { this.find[i].pos = this.currentPos; }
              }
          } }
  };
  ParseContext$1.prototype.findInText = function findInText (textNode) {
      if (this.find)
          { for (var i = 0; i < this.find.length; i++) {
              if (this.find[i].node == textNode)
                  { this.find[i].pos = this.currentPos - (textNode.nodeValue.length - this.find[i].offset); }
          } }
  };
  // Determines whether the given context string matches this context.
  ParseContext$1.prototype.matchesContext = function matchesContext (context) {
        var this$1$1 = this;

      if (context.indexOf("|") > -1)
          { return context.split(/\s*\|\s*/).some(this.matchesContext, this); }
      var parts = context.split("/");
      var option = this.options.context;
      var useRoot = !this.isOpen && (!option || option.parent.type == this.nodes[0].type);
      var minDepth = -(option ? option.depth + 1 : 0) + (useRoot ? 0 : 1);
      var match = function (i, depth) {
          for (; i >= 0; i--) {
              var part = parts[i];
              if (part == "") {
                  if (i == parts.length - 1 || i == 0)
                      { continue; }
                  for (; depth >= minDepth; depth--)
                      { if (match(i - 1, depth))
                          { return true; } }
                  return false;
              }
              else {
                  var next = depth > 0 || (depth == 0 && useRoot) ? this$1$1.nodes[depth].type
                      : option && depth >= minDepth ? option.node(depth - minDepth).type
                          : null;
                  if (!next || (next.name != part && next.groups.indexOf(part) == -1))
                      { return false; }
                  depth--;
              }
          }
          return true;
      };
      return match(parts.length - 1, this.open);
  };
  ParseContext$1.prototype.textblockFromContext = function textblockFromContext () {
      var $context = this.options.context;
      if ($context)
          { for (var d = $context.depth; d >= 0; d--) {
              var deflt = $context.node(d).contentMatchAt($context.indexAfter(d)).defaultType;
              if (deflt && deflt.isTextblock && deflt.defaultAttrs)
                  { return deflt; }
          } }
      for (var name in this.parser.schema.nodes) {
          var type = this.parser.schema.nodes[name];
          if (type.isTextblock && type.defaultAttrs)
              { return type; }
      }
  };

Object.defineProperties( ParseContext$1.prototype, prototypeAccessors$10$2 );
// Kludge to work around directly nested list nodes produced by some
// tools and allowed by browsers to mean that the nested list is
// actually part of the list item above it.
function normalizeList(dom) {
    for (var child = dom.firstChild, prevItem = null; child; child = child.nextSibling) {
        var name = child.nodeType == 1 ? child.nodeName.toLowerCase() : null;
        if (name && listTags.hasOwnProperty(name) && prevItem) {
            prevItem.appendChild(child);
            child = prevItem;
        }
        else if (name == "li") {
            prevItem = child;
        }
        else if (name) {
            prevItem = null;
        }
    }
}
// Apply a CSS selector.
function matches(dom, selector) {
    return (dom.matches || dom.msMatchesSelector || dom.webkitMatchesSelector || dom.mozMatchesSelector).call(dom, selector);
}
function copy$1(obj) {
    var copy = {};
    for (var prop in obj)
        { copy[prop] = obj[prop]; }
    return copy;
}
// Used when finding a mark at the top level of a fragment parse.
// Checks whether it would be reasonable to apply a given mark type to
// a given node, by looking at the way the mark occurs in the schema.
function markMayApply(markType, nodeType) {
    var nodes = nodeType.schema.nodes;
    var loop = function ( name ) {
        var parent = nodes[name];
        if (!parent.allowsMarkType(markType))
            { return; }
        var seen = [], scan = function (match) {
            seen.push(match);
            for (var i = 0; i < match.edgeCount; i++) {
                var ref = match.edge(i);
                var type = ref.type;
                var next = ref.next;
                if (type == nodeType)
                    { return true; }
                if (seen.indexOf(next) < 0 && scan(next))
                    { return true; }
            }
        };
        if (scan(parent.contentMatch))
            { return { v: true }; }
    };

    for (var name in nodes) {
      var returned = loop( name );

      if ( returned ) return returned.v;
    }
}

/**
A DOM serializer knows how to convert ProseMirror nodes and
marks of various types to DOM nodes.
*/
var DOMSerializer = function DOMSerializer(
  /**
  The node serialization functions.
  */
  nodes, 
  /**
  The mark serialization functions.
  */
  marks) {
      this.nodes = nodes;
      this.marks = marks;
  };
  /**
  Serialize the content of this fragment to a DOM fragment. When
  not in the browser, the `document` option, containing a DOM
  document, should be passed so that the serializer can create
  nodes.
  */
  DOMSerializer.prototype.serializeFragment = function serializeFragment (fragment, options, target) {
        var this$1$1 = this;
        if ( options === void 0 ) options = {};

      if (!target)
          { target = doc$3(options).createDocumentFragment(); }
      var top = target, active = [];
      fragment.forEach(function (node) {
          if (active.length || node.marks.length) {
              var keep = 0, rendered = 0;
              while (keep < active.length && rendered < node.marks.length) {
                  var next = node.marks[rendered];
                  if (!this$1$1.marks[next.type.name]) {
                      rendered++;
                      continue;
                  }
                  if (!next.eq(active[keep][0]) || next.type.spec.spanning === false)
                      { break; }
                  keep++;
                  rendered++;
              }
              while (keep < active.length)
                  { top = active.pop()[1]; }
              while (rendered < node.marks.length) {
                  var add = node.marks[rendered++];
                  var markDOM = this$1$1.serializeMark(add, node.isInline, options);
                  if (markDOM) {
                      active.push([add, top]);
                      top.appendChild(markDOM.dom);
                      top = markDOM.contentDOM || markDOM.dom;
                  }
              }
          }
          top.appendChild(this$1$1.serializeNodeInner(node, options));
      });
      return target;
  };
  /**
  @internal
  */
  DOMSerializer.prototype.serializeNodeInner = function serializeNodeInner (node, options) {
      var ref = renderSpec(doc$3(options), this.nodes[node.type.name](node), null, node.attrs);
        var dom = ref.dom;
        var contentDOM = ref.contentDOM;
      if (contentDOM) {
          if (node.isLeaf)
              { throw new RangeError("Content hole not allowed in a leaf node spec"); }
          this.serializeFragment(node.content, options, contentDOM);
      }
      return dom;
  };
  /**
  Serialize this node to a DOM node. This can be useful when you
  need to serialize a part of a document, as opposed to the whole
  document. To serialize a whole document, use
  [`serializeFragment`](https://prosemirror.net/docs/ref/#model.DOMSerializer.serializeFragment) on
  its [content](https://prosemirror.net/docs/ref/#model.Node.content).
  */
  DOMSerializer.prototype.serializeNode = function serializeNode (node, options) {
        if ( options === void 0 ) options = {};

      var dom = this.serializeNodeInner(node, options);
      for (var i = node.marks.length - 1; i >= 0; i--) {
          var wrap = this.serializeMark(node.marks[i], node.isInline, options);
          if (wrap) {
              (wrap.contentDOM || wrap.dom).appendChild(dom);
              dom = wrap.dom;
          }
      }
      return dom;
  };
  /**
  @internal
  */
  DOMSerializer.prototype.serializeMark = function serializeMark (mark, inline, options) {
        if ( options === void 0 ) options = {};

      var toDOM = this.marks[mark.type.name];
      return toDOM && renderSpec(doc$3(options), toDOM(mark, inline), null, mark.attrs);
  };
  DOMSerializer.renderSpec = function renderSpec$1 (doc, structure, xmlNS, blockArraysIn) {
        if ( xmlNS === void 0 ) xmlNS = null;

      return renderSpec(doc, structure, xmlNS, blockArraysIn);
  };
  /**
  Build a serializer using the [`toDOM`](https://prosemirror.net/docs/ref/#model.NodeSpec.toDOM)
  properties in a schema's node and mark specs.
  */
  DOMSerializer.fromSchema = function fromSchema (schema) {
      return schema.cached.domSerializer ||
          (schema.cached.domSerializer = new DOMSerializer(this.nodesFromSchema(schema), this.marksFromSchema(schema)));
  };
  /**
  Gather the serializers in a schema's node specs into an object.
  This can be useful as a base to build a custom serializer from.
  */
  DOMSerializer.nodesFromSchema = function nodesFromSchema (schema) {
      var result = gatherToDOM(schema.nodes);
      if (!result.text)
          { result.text = function (node) { return node.text; }; }
      return result;
  };
  /**
  Gather the serializers in a schema's mark specs into an object.
  */
  DOMSerializer.marksFromSchema = function marksFromSchema (schema) {
      return gatherToDOM(schema.marks);
  };
function gatherToDOM(obj) {
    var result = {};
    for (var name in obj) {
        var toDOM = obj[name].spec.toDOM;
        if (toDOM)
            { result[name] = toDOM; }
    }
    return result;
}
function doc$3(options) {
    return options.document || window.document;
}
var suspiciousAttributeCache = new WeakMap();
function suspiciousAttributes(attrs) {
    var value = suspiciousAttributeCache.get(attrs);
    if (value === undefined)
        { suspiciousAttributeCache.set(attrs, value = suspiciousAttributesInner(attrs)); }
    return value;
}
function suspiciousAttributesInner(attrs) {
    var result = null;
    function scan(value) {
        if (value && typeof value == "object") {
            if (Array.isArray(value)) {
                if (typeof value[0] == "string") {
                    if (!result)
                        { result = []; }
                    result.push(value);
                }
                else {
                    for (var i = 0; i < value.length; i++)
                        { scan(value[i]); }
                }
            }
            else {
                for (var prop in value)
                    { scan(value[prop]); }
            }
        }
    }
    scan(attrs);
    return result;
}
function renderSpec(doc, structure, xmlNS, blockArraysIn) {
    if (typeof structure == "string")
        { return { dom: doc.createTextNode(structure) }; }
    if (structure.nodeType != null)
        { return { dom: structure }; }
    if (structure.dom && structure.dom.nodeType != null)
        { return structure; }
    var tagName = structure[0], suspicious;
    if (typeof tagName != "string")
        { throw new RangeError("Invalid array passed to renderSpec"); }
    if (blockArraysIn && (suspicious = suspiciousAttributes(blockArraysIn)) &&
        suspicious.indexOf(structure) > -1)
        { throw new RangeError("Using an array from an attribute object as a DOM spec. This may be an attempted cross site scripting attack."); }
    var space = tagName.indexOf(" ");
    if (space > 0) {
        xmlNS = tagName.slice(0, space);
        tagName = tagName.slice(space + 1);
    }
    var contentDOM;
    var dom = (xmlNS ? doc.createElementNS(xmlNS, tagName) : doc.createElement(tagName));
    var attrs = structure[1], start = 1;
    if (attrs && typeof attrs == "object" && attrs.nodeType == null && !Array.isArray(attrs)) {
        start = 2;
        for (var name in attrs)
            { if (attrs[name] != null) {
                var space$1 = name.indexOf(" ");
                if (space$1 > 0)
                    { dom.setAttributeNS(name.slice(0, space$1), name.slice(space$1 + 1), attrs[name]); }
                else
                    { dom.setAttribute(name, attrs[name]); }
            } }
    }
    for (var i = start; i < structure.length; i++) {
        var child = structure[i];
        if (child === 0) {
            if (i < structure.length - 1 || i > start)
                { throw new RangeError("Content hole must be the only child of its parent node"); }
            return { dom: dom, contentDOM: dom };
        }
        else {
            var ref = renderSpec(doc, child, xmlNS, blockArraysIn);
            var inner = ref.dom;
            var innerContent = ref.contentDOM;
            dom.appendChild(inner);
            if (innerContent) {
                if (contentDOM)
                    { throw new RangeError("Multiple content holes"); }
                contentDOM = innerContent;
            }
        }
    }
    return { dom: dom, contentDOM: contentDOM };
}var model=/*#__PURE__*/Object.freeze({__proto__:null,ContentMatch:ContentMatch,DOMParser:DOMParser,DOMSerializer:DOMSerializer,Fragment:Fragment,Mark:Mark,MarkType:MarkType,Node:Node$1,NodeRange:NodeRange,NodeType:NodeType$2,ReplaceError:ReplaceError,ResolvedPos:ResolvedPos,Schema:Schema,Slice:Slice});// Recovery values encode a range index and an offset. They are
// represented as numbers, because tons of them will be created when
// mapping, for example, a large number of decorations. The number's
// lower 16 bits provide the index, the remaining bits the offset.
//
// Note: We intentionally don't use bit shift operators to en- and
// decode these, since those clip to 32 bits, which we might in rare
// cases want to overflow. A 64-bit float can represent 48-bit
// integers precisely.
var lower16 = 0xffff;
var factor16 = Math.pow(2, 16);
function makeRecover(index, offset) { return index + offset * factor16; }
function recoverIndex(value) { return value & lower16; }
function recoverOffset(value) { return (value - (value & lower16)) / factor16; }
var DEL_BEFORE = 1, DEL_AFTER = 2, DEL_ACROSS = 4, DEL_SIDE = 8;
/**
An object representing a mapped position with extra
information.
*/
var MapResult = function MapResult(
/**
The mapped version of the position.
*/
pos, 
/**
@internal
*/
delInfo, 
/**
@internal
*/
recover) {
    this.pos = pos;
    this.delInfo = delInfo;
    this.recover = recover;
};

var prototypeAccessors$b = { deleted: { configurable: true },deletedBefore: { configurable: true },deletedAfter: { configurable: true },deletedAcross: { configurable: true } };
/**
Tells you whether the position was deleted, that is, whether the
step removed the token on the side queried (via the `assoc`)
argument from the document.
*/
prototypeAccessors$b.deleted.get = function () { return (this.delInfo & DEL_SIDE) > 0; };
/**
Tells you whether the token before the mapped position was deleted.
*/
prototypeAccessors$b.deletedBefore.get = function () { return (this.delInfo & (DEL_BEFORE | DEL_ACROSS)) > 0; };
/**
True when the token after the mapped position was deleted.
*/
prototypeAccessors$b.deletedAfter.get = function () { return (this.delInfo & (DEL_AFTER | DEL_ACROSS)) > 0; };
/**
Tells whether any of the steps mapped through deletes across the
position (including both the token before and after the
position).
*/
prototypeAccessors$b.deletedAcross.get = function () { return (this.delInfo & DEL_ACROSS) > 0; };

Object.defineProperties( MapResult.prototype, prototypeAccessors$b );
/**
A map describing the deletions and insertions made by a step, which
can be used to find the correspondence between positions in the
pre-step version of a document and the same position in the
post-step version.
*/
var StepMap = function StepMap(
/**
@internal
*/
ranges, 
/**
@internal
*/
inverted) {
    if ( inverted === void 0 ) inverted = false;

    this.ranges = ranges;
    this.inverted = inverted;
    if (!ranges.length && StepMap.empty)
        { return StepMap.empty; }
};
/**
@internal
*/
StepMap.prototype.recover = function recover (value) {
    var diff = 0, index = recoverIndex(value);
    if (!this.inverted)
        { for (var i = 0; i < index; i++)
            { diff += this.ranges[i * 3 + 2] - this.ranges[i * 3 + 1]; } }
    return this.ranges[index * 3] + diff + recoverOffset(value);
};
StepMap.prototype.mapResult = function mapResult (pos, assoc) {
    if ( assoc === void 0 ) assoc = 1;
 return this._map(pos, assoc, false); };
StepMap.prototype.map = function map (pos, assoc) {
    if ( assoc === void 0 ) assoc = 1;
 return this._map(pos, assoc, true); };
/**
@internal
*/
StepMap.prototype._map = function _map (pos, assoc, simple) {
    var diff = 0, oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
    for (var i = 0; i < this.ranges.length; i += 3) {
        var start = this.ranges[i] - (this.inverted ? diff : 0);
        if (start > pos)
            { break; }
        var oldSize = this.ranges[i + oldIndex], newSize = this.ranges[i + newIndex], end = start + oldSize;
        if (pos <= end) {
            var side = !oldSize ? assoc : pos == start ? -1 : pos == end ? 1 : assoc;
            var result = start + diff + (side < 0 ? 0 : newSize);
            if (simple)
                { return result; }
            var recover = pos == (assoc < 0 ? start : end) ? null : makeRecover(i / 3, pos - start);
            var del = pos == start ? DEL_AFTER : pos == end ? DEL_BEFORE : DEL_ACROSS;
            if (assoc < 0 ? pos != start : pos != end)
                { del |= DEL_SIDE; }
            return new MapResult(result, del, recover);
        }
        diff += newSize - oldSize;
    }
    return simple ? pos + diff : new MapResult(pos + diff, 0, null);
};
/**
@internal
*/
StepMap.prototype.touches = function touches (pos, recover) {
    var diff = 0, index = recoverIndex(recover);
    var oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
    for (var i = 0; i < this.ranges.length; i += 3) {
        var start = this.ranges[i] - (this.inverted ? diff : 0);
        if (start > pos)
            { break; }
        var oldSize = this.ranges[i + oldIndex], end = start + oldSize;
        if (pos <= end && i == index * 3)
            { return true; }
        diff += this.ranges[i + newIndex] - oldSize;
    }
    return false;
};
/**
Calls the given function on each of the changed ranges included in
this map.
*/
StepMap.prototype.forEach = function forEach (f) {
    var oldIndex = this.inverted ? 2 : 1, newIndex = this.inverted ? 1 : 2;
    for (var i = 0, diff = 0; i < this.ranges.length; i += 3) {
        var start = this.ranges[i], oldStart = start - (this.inverted ? diff : 0), newStart = start + (this.inverted ? 0 : diff);
        var oldSize = this.ranges[i + oldIndex], newSize = this.ranges[i + newIndex];
        f(oldStart, oldStart + oldSize, newStart, newStart + newSize);
        diff += newSize - oldSize;
    }
};
/**
Create an inverted version of this map. The result can be used to
map positions in the post-step document to the pre-step document.
*/
StepMap.prototype.invert = function invert () {
    return new StepMap(this.ranges, !this.inverted);
};
/**
@internal
*/
StepMap.prototype.toString = function toString () {
    return (this.inverted ? "-" : "") + JSON.stringify(this.ranges);
};
/**
Create a map that moves all positions by offset `n` (which may be
negative). This can be useful when applying steps meant for a
sub-document to a larger document, or vice-versa.
*/
StepMap.offset = function offset (n) {
    return n == 0 ? StepMap.empty : new StepMap(n < 0 ? [0, -n, 0] : [0, 0, n]);
};
/**
A StepMap that contains no changed ranges.
*/
StepMap.empty = new StepMap([]);
/**
A mapping represents a pipeline of zero or more [step
maps](https://prosemirror.net/docs/ref/#transform.StepMap). It has special provisions for losslessly
handling mapping positions through a series of steps in which some
steps are inverted versions of earlier steps. (This comes up when
‘[rebasing](/docs/guide/#transform.rebasing)’ steps for
collaboration or history management.)
*/
var Mapping = function Mapping(
/**
The step maps in this mapping.
*/
maps, 
/**
@internal
*/
mirror, 
/**
The starting position in the `maps` array, used when `map` or
`mapResult` is called.
*/
from, 
/**
The end position in the `maps` array.
*/
to) {
    if ( maps === void 0 ) maps = [];
    if ( from === void 0 ) from = 0;
    if ( to === void 0 ) to = maps.length;

    this.maps = maps;
    this.mirror = mirror;
    this.from = from;
    this.to = to;
};
/**
Create a mapping that maps only through a part of this one.
*/
Mapping.prototype.slice = function slice (from, to) {
        if ( from === void 0 ) from = 0;
        if ( to === void 0 ) to = this.maps.length;

    return new Mapping(this.maps, this.mirror, from, to);
};
/**
@internal
*/
Mapping.prototype.copy = function copy () {
    return new Mapping(this.maps.slice(), this.mirror && this.mirror.slice(), this.from, this.to);
};
/**
Add a step map to the end of this mapping. If `mirrors` is
given, it should be the index of the step map that is the mirror
image of this one.
*/
Mapping.prototype.appendMap = function appendMap (map, mirrors) {
    this.to = this.maps.push(map);
    if (mirrors != null)
        { this.setMirror(this.maps.length - 1, mirrors); }
};
/**
Add all the step maps in a given mapping to this one (preserving
mirroring information).
*/
Mapping.prototype.appendMapping = function appendMapping (mapping) {
    for (var i = 0, startSize = this.maps.length; i < mapping.maps.length; i++) {
        var mirr = mapping.getMirror(i);
        this.appendMap(mapping.maps[i], mirr != null && mirr < i ? startSize + mirr : undefined);
    }
};
/**
Finds the offset of the step map that mirrors the map at the
given offset, in this mapping (as per the second argument to
`appendMap`).
*/
Mapping.prototype.getMirror = function getMirror (n) {
    if (this.mirror)
        { for (var i = 0; i < this.mirror.length; i++)
            { if (this.mirror[i] == n)
                { return this.mirror[i + (i % 2 ? -1 : 1)]; } } }
};
/**
@internal
*/
Mapping.prototype.setMirror = function setMirror (n, m) {
    if (!this.mirror)
        { this.mirror = []; }
    this.mirror.push(n, m);
};
/**
Append the inverse of the given mapping to this one.
*/
Mapping.prototype.appendMappingInverted = function appendMappingInverted (mapping) {
    for (var i = mapping.maps.length - 1, totalSize = this.maps.length + mapping.maps.length; i >= 0; i--) {
        var mirr = mapping.getMirror(i);
        this.appendMap(mapping.maps[i].invert(), mirr != null && mirr > i ? totalSize - mirr - 1 : undefined);
    }
};
/**
Create an inverted version of this mapping.
*/
Mapping.prototype.invert = function invert () {
    var inverse = new Mapping;
    inverse.appendMappingInverted(this);
    return inverse;
};
/**
Map a position through this mapping.
*/
Mapping.prototype.map = function map (pos, assoc) {
        if ( assoc === void 0 ) assoc = 1;

    if (this.mirror)
        { return this._map(pos, assoc, true); }
    for (var i = this.from; i < this.to; i++)
        { pos = this.maps[i].map(pos, assoc); }
    return pos;
};
/**
Map a position through this mapping, returning a mapping
result.
*/
Mapping.prototype.mapResult = function mapResult (pos, assoc) {
    if ( assoc === void 0 ) assoc = 1;
 return this._map(pos, assoc, false); };
/**
@internal
*/
Mapping.prototype._map = function _map (pos, assoc, simple) {
    var delInfo = 0;
    for (var i = this.from; i < this.to; i++) {
        var map = this.maps[i], result = map.mapResult(pos, assoc);
        if (result.recover != null) {
            var corr = this.getMirror(i);
            if (corr != null && corr > i && corr < this.to) {
                i = corr;
                pos = this.maps[corr].recover(result.recover);
                continue;
            }
        }
        delInfo |= result.delInfo;
        pos = result.pos;
    }
    return simple ? pos : new MapResult(pos, delInfo, null);
};

var stepsByID = Object.create(null);
/**
A step object represents an atomic change. It generally applies
only to the document it was created for, since the positions
stored in it will only make sense for that document.

New steps are defined by creating classes that extend `Step`,
overriding the `apply`, `invert`, `map`, `getMap` and `fromJSON`
methods, and registering your class with a unique
JSON-serialization identifier using
[`Step.jsonID`](https://prosemirror.net/docs/ref/#transform.Step^jsonID).
*/
var Step = function Step () {};

Step.prototype.getMap = function getMap () { return StepMap.empty; };
/**
Try to merge this step with another one, to be applied directly
after it. Returns the merged step when possible, null if the
steps can't be merged.
*/
Step.prototype.merge = function merge (other) { return null; };
/**
Deserialize a step from its JSON representation. Will call
through to the step class' own implementation of this method.
*/
Step.fromJSON = function fromJSON (schema, json) {
    if (!json || !json.stepType)
        { throw new RangeError("Invalid input for Step.fromJSON"); }
    var type = stepsByID[json.stepType];
    if (!type)
        { throw new RangeError(("No step type " + (json.stepType) + " defined")); }
    return type.fromJSON(schema, json);
};
/**
To be able to serialize steps to JSON, each step needs a string
ID to attach to its JSON representation. Use this method to
register an ID for your step classes. Try to pick something
that's unlikely to clash with steps from other modules.
*/
Step.jsonID = function jsonID (id, stepClass) {
    if (id in stepsByID)
        { throw new RangeError("Duplicate use of step JSON ID " + id); }
    stepsByID[id] = stepClass;
    stepClass.prototype.jsonID = id;
    return stepClass;
};
/**
The result of [applying](https://prosemirror.net/docs/ref/#transform.Step.apply) a step. Contains either a
new document or a failure value.
*/
var StepResult = function StepResult(
/**
The transformed document, if successful.
*/
doc, 
/**
The failure message, if unsuccessful.
*/
failed) {
    this.doc = doc;
    this.failed = failed;
};
/**
Create a successful step result.
*/
StepResult.ok = function ok (doc) { return new StepResult(doc, null); };
/**
Create a failed step result.
*/
StepResult.fail = function fail (message) { return new StepResult(null, message); };
/**
Call [`Node.replace`](https://prosemirror.net/docs/ref/#model.Node.replace) with the given
arguments. Create a successful result if it succeeds, and a
failed one if it throws a `ReplaceError`.
*/
StepResult.fromReplace = function fromReplace (doc, from, to, slice) {
    try {
        return StepResult.ok(doc.replace(from, to, slice));
    }
    catch (e) {
        if (e instanceof ReplaceError)
            { return StepResult.fail(e.message); }
        throw e;
    }
};

function mapFragment(fragment, f, parent) {
    var mapped = [];
    for (var i = 0; i < fragment.childCount; i++) {
        var child = fragment.child(i);
        if (child.content.size)
            { child = child.copy(mapFragment(child.content, f, child)); }
        if (child.isInline)
            { child = f(child, parent, i); }
        mapped.push(child);
    }
    return Fragment.fromArray(mapped);
}
/**
Add a mark to all inline content between two positions.
*/
var AddMarkStep = /*@__PURE__*/(function (Step) {
    function AddMarkStep(
    /**
    The start of the marked range.
    */
    from, 
    /**
    The end of the marked range.
    */
    to, 
    /**
    The mark to add.
    */
    mark) {
        Step.call(this);
        this.from = from;
        this.to = to;
        this.mark = mark;
    }

    if ( Step ) AddMarkStep.__proto__ = Step;
    AddMarkStep.prototype = Object.create( Step && Step.prototype );
    AddMarkStep.prototype.constructor = AddMarkStep;
    AddMarkStep.prototype.apply = function apply (doc) {
        var this$1$1 = this;

        var oldSlice = doc.slice(this.from, this.to), $from = doc.resolve(this.from);
        var parent = $from.node($from.sharedDepth(this.to));
        var slice = new Slice(mapFragment(oldSlice.content, function (node, parent) {
            if (!node.isAtom || !parent.type.allowsMarkType(this$1$1.mark.type))
                { return node; }
            return node.mark(this$1$1.mark.addToSet(node.marks));
        }, parent), oldSlice.openStart, oldSlice.openEnd);
        return StepResult.fromReplace(doc, this.from, this.to, slice);
    };
    AddMarkStep.prototype.invert = function invert () {
        return new RemoveMarkStep(this.from, this.to, this.mark);
    };
    AddMarkStep.prototype.map = function map (mapping) {
        var from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
        if (from.deleted && to.deleted || from.pos >= to.pos)
            { return null; }
        return new AddMarkStep(from.pos, to.pos, this.mark);
    };
    AddMarkStep.prototype.merge = function merge (other) {
        if (other instanceof AddMarkStep &&
            other.mark.eq(this.mark) &&
            this.from <= other.to && this.to >= other.from)
            { return new AddMarkStep(Math.min(this.from, other.from), Math.max(this.to, other.to), this.mark); }
        return null;
    };
    AddMarkStep.prototype.toJSON = function toJSON () {
        return { stepType: "addMark", mark: this.mark.toJSON(),
            from: this.from, to: this.to };
    };
    /**
    @internal
    */
    AddMarkStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.from != "number" || typeof json.to != "number")
            { throw new RangeError("Invalid input for AddMarkStep.fromJSON"); }
        return new AddMarkStep(json.from, json.to, schema.markFromJSON(json.mark));
    };

    return AddMarkStep;
}(Step));
Step.jsonID("addMark", AddMarkStep);
/**
Remove a mark from all inline content between two positions.
*/
var RemoveMarkStep = /*@__PURE__*/(function (Step) {
    function RemoveMarkStep(
    /**
    The start of the unmarked range.
    */
    from, 
    /**
    The end of the unmarked range.
    */
    to, 
    /**
    The mark to remove.
    */
    mark) {
        Step.call(this);
        this.from = from;
        this.to = to;
        this.mark = mark;
    }

    if ( Step ) RemoveMarkStep.__proto__ = Step;
    RemoveMarkStep.prototype = Object.create( Step && Step.prototype );
    RemoveMarkStep.prototype.constructor = RemoveMarkStep;
    RemoveMarkStep.prototype.apply = function apply (doc) {
        var this$1$1 = this;

        var oldSlice = doc.slice(this.from, this.to);
        var slice = new Slice(mapFragment(oldSlice.content, function (node) {
            return node.mark(this$1$1.mark.removeFromSet(node.marks));
        }, doc), oldSlice.openStart, oldSlice.openEnd);
        return StepResult.fromReplace(doc, this.from, this.to, slice);
    };
    RemoveMarkStep.prototype.invert = function invert () {
        return new AddMarkStep(this.from, this.to, this.mark);
    };
    RemoveMarkStep.prototype.map = function map (mapping) {
        var from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
        if (from.deleted && to.deleted || from.pos >= to.pos)
            { return null; }
        return new RemoveMarkStep(from.pos, to.pos, this.mark);
    };
    RemoveMarkStep.prototype.merge = function merge (other) {
        if (other instanceof RemoveMarkStep &&
            other.mark.eq(this.mark) &&
            this.from <= other.to && this.to >= other.from)
            { return new RemoveMarkStep(Math.min(this.from, other.from), Math.max(this.to, other.to), this.mark); }
        return null;
    };
    RemoveMarkStep.prototype.toJSON = function toJSON () {
        return { stepType: "removeMark", mark: this.mark.toJSON(),
            from: this.from, to: this.to };
    };
    /**
    @internal
    */
    RemoveMarkStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.from != "number" || typeof json.to != "number")
            { throw new RangeError("Invalid input for RemoveMarkStep.fromJSON"); }
        return new RemoveMarkStep(json.from, json.to, schema.markFromJSON(json.mark));
    };

    return RemoveMarkStep;
}(Step));
Step.jsonID("removeMark", RemoveMarkStep);
/**
Add a mark to a specific node.
*/
var AddNodeMarkStep = /*@__PURE__*/(function (Step) {
    function AddNodeMarkStep(
    /**
    The position of the target node.
    */
    pos, 
    /**
    The mark to add.
    */
    mark) {
        Step.call(this);
        this.pos = pos;
        this.mark = mark;
    }

    if ( Step ) AddNodeMarkStep.__proto__ = Step;
    AddNodeMarkStep.prototype = Object.create( Step && Step.prototype );
    AddNodeMarkStep.prototype.constructor = AddNodeMarkStep;
    AddNodeMarkStep.prototype.apply = function apply (doc) {
        var node = doc.nodeAt(this.pos);
        if (!node)
            { return StepResult.fail("No node at mark step's position"); }
        var updated = node.type.create(node.attrs, null, this.mark.addToSet(node.marks));
        return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
    };
    AddNodeMarkStep.prototype.invert = function invert (doc) {
        var node = doc.nodeAt(this.pos);
        if (node) {
            var newSet = this.mark.addToSet(node.marks);
            if (newSet.length == node.marks.length) {
                for (var i = 0; i < node.marks.length; i++)
                    { if (!node.marks[i].isInSet(newSet))
                        { return new AddNodeMarkStep(this.pos, node.marks[i]); } }
                return new AddNodeMarkStep(this.pos, this.mark);
            }
        }
        return new RemoveNodeMarkStep(this.pos, this.mark);
    };
    AddNodeMarkStep.prototype.map = function map (mapping) {
        var pos = mapping.mapResult(this.pos, 1);
        return pos.deletedAfter ? null : new AddNodeMarkStep(pos.pos, this.mark);
    };
    AddNodeMarkStep.prototype.toJSON = function toJSON () {
        return { stepType: "addNodeMark", pos: this.pos, mark: this.mark.toJSON() };
    };
    /**
    @internal
    */
    AddNodeMarkStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.pos != "number")
            { throw new RangeError("Invalid input for AddNodeMarkStep.fromJSON"); }
        return new AddNodeMarkStep(json.pos, schema.markFromJSON(json.mark));
    };

    return AddNodeMarkStep;
}(Step));
Step.jsonID("addNodeMark", AddNodeMarkStep);
/**
Remove a mark from a specific node.
*/
var RemoveNodeMarkStep = /*@__PURE__*/(function (Step) {
    function RemoveNodeMarkStep(
    /**
    The position of the target node.
    */
    pos, 
    /**
    The mark to remove.
    */
    mark) {
        Step.call(this);
        this.pos = pos;
        this.mark = mark;
    }

    if ( Step ) RemoveNodeMarkStep.__proto__ = Step;
    RemoveNodeMarkStep.prototype = Object.create( Step && Step.prototype );
    RemoveNodeMarkStep.prototype.constructor = RemoveNodeMarkStep;
    RemoveNodeMarkStep.prototype.apply = function apply (doc) {
        var node = doc.nodeAt(this.pos);
        if (!node)
            { return StepResult.fail("No node at mark step's position"); }
        var updated = node.type.create(node.attrs, null, this.mark.removeFromSet(node.marks));
        return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
    };
    RemoveNodeMarkStep.prototype.invert = function invert (doc) {
        var node = doc.nodeAt(this.pos);
        if (!node || !this.mark.isInSet(node.marks))
            { return this; }
        return new AddNodeMarkStep(this.pos, this.mark);
    };
    RemoveNodeMarkStep.prototype.map = function map (mapping) {
        var pos = mapping.mapResult(this.pos, 1);
        return pos.deletedAfter ? null : new RemoveNodeMarkStep(pos.pos, this.mark);
    };
    RemoveNodeMarkStep.prototype.toJSON = function toJSON () {
        return { stepType: "removeNodeMark", pos: this.pos, mark: this.mark.toJSON() };
    };
    /**
    @internal
    */
    RemoveNodeMarkStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.pos != "number")
            { throw new RangeError("Invalid input for RemoveNodeMarkStep.fromJSON"); }
        return new RemoveNodeMarkStep(json.pos, schema.markFromJSON(json.mark));
    };

    return RemoveNodeMarkStep;
}(Step));
Step.jsonID("removeNodeMark", RemoveNodeMarkStep);

/**
Replace a part of the document with a slice of new content.
*/
var ReplaceStep = /*@__PURE__*/(function (Step) {
    function ReplaceStep(
    /**
    The start position of the replaced range.
    */
    from, 
    /**
    The end position of the replaced range.
    */
    to, 
    /**
    The slice to insert.
    */
    slice, 
    /**
    @internal
    */
    structure) {
        if ( structure === void 0 ) structure = false;

        Step.call(this);
        this.from = from;
        this.to = to;
        this.slice = slice;
        this.structure = structure;
    }

    if ( Step ) ReplaceStep.__proto__ = Step;
    ReplaceStep.prototype = Object.create( Step && Step.prototype );
    ReplaceStep.prototype.constructor = ReplaceStep;
    ReplaceStep.prototype.apply = function apply (doc) {
        if (this.structure && contentBetween(doc, this.from, this.to))
            { return StepResult.fail("Structure replace would overwrite content"); }
        return StepResult.fromReplace(doc, this.from, this.to, this.slice);
    };
    ReplaceStep.prototype.getMap = function getMap () {
        return new StepMap([this.from, this.to - this.from, this.slice.size]);
    };
    ReplaceStep.prototype.invert = function invert (doc) {
        return new ReplaceStep(this.from, this.from + this.slice.size, doc.slice(this.from, this.to));
    };
    ReplaceStep.prototype.map = function map (mapping) {
        var from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
        if (from.deletedAcross && to.deletedAcross)
            { return null; }
        return new ReplaceStep(from.pos, Math.max(from.pos, to.pos), this.slice);
    };
    ReplaceStep.prototype.merge = function merge (other) {
        if (!(other instanceof ReplaceStep) || other.structure || this.structure)
            { return null; }
        if (this.from + this.slice.size == other.from && !this.slice.openEnd && !other.slice.openStart) {
            var slice = this.slice.size + other.slice.size == 0 ? Slice.empty
                : new Slice(this.slice.content.append(other.slice.content), this.slice.openStart, other.slice.openEnd);
            return new ReplaceStep(this.from, this.to + (other.to - other.from), slice, this.structure);
        }
        else if (other.to == this.from && !this.slice.openStart && !other.slice.openEnd) {
            var slice$1 = this.slice.size + other.slice.size == 0 ? Slice.empty
                : new Slice(other.slice.content.append(this.slice.content), other.slice.openStart, this.slice.openEnd);
            return new ReplaceStep(other.from, this.to, slice$1, this.structure);
        }
        else {
            return null;
        }
    };
    ReplaceStep.prototype.toJSON = function toJSON () {
        var json = { stepType: "replace", from: this.from, to: this.to };
        if (this.slice.size)
            { json.slice = this.slice.toJSON(); }
        if (this.structure)
            { json.structure = true; }
        return json;
    };
    /**
    @internal
    */
    ReplaceStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.from != "number" || typeof json.to != "number")
            { throw new RangeError("Invalid input for ReplaceStep.fromJSON"); }
        return new ReplaceStep(json.from, json.to, Slice.fromJSON(schema, json.slice), !!json.structure);
    };

    return ReplaceStep;
}(Step));
Step.jsonID("replace", ReplaceStep);
/**
Replace a part of the document with a slice of content, but
preserve a range of the replaced content by moving it into the
slice.
*/
var ReplaceAroundStep = /*@__PURE__*/(function (Step) {
    function ReplaceAroundStep(
    /**
    The start position of the replaced range.
    */
    from, 
    /**
    The end position of the replaced range.
    */
    to, 
    /**
    The start of preserved range.
    */
    gapFrom, 
    /**
    The end of preserved range.
    */
    gapTo, 
    /**
    The slice to insert.
    */
    slice, 
    /**
    The position in the slice where the preserved range should be
    inserted.
    */
    insert, 
    /**
    @internal
    */
    structure) {
        if ( structure === void 0 ) structure = false;

        Step.call(this);
        this.from = from;
        this.to = to;
        this.gapFrom = gapFrom;
        this.gapTo = gapTo;
        this.slice = slice;
        this.insert = insert;
        this.structure = structure;
    }

    if ( Step ) ReplaceAroundStep.__proto__ = Step;
    ReplaceAroundStep.prototype = Object.create( Step && Step.prototype );
    ReplaceAroundStep.prototype.constructor = ReplaceAroundStep;
    ReplaceAroundStep.prototype.apply = function apply (doc) {
        if (this.structure && (contentBetween(doc, this.from, this.gapFrom) ||
            contentBetween(doc, this.gapTo, this.to)))
            { return StepResult.fail("Structure gap-replace would overwrite content"); }
        var gap = doc.slice(this.gapFrom, this.gapTo);
        if (gap.openStart || gap.openEnd)
            { return StepResult.fail("Gap is not a flat range"); }
        var inserted = this.slice.insertAt(this.insert, gap.content);
        if (!inserted)
            { return StepResult.fail("Content does not fit in gap"); }
        return StepResult.fromReplace(doc, this.from, this.to, inserted);
    };
    ReplaceAroundStep.prototype.getMap = function getMap () {
        return new StepMap([this.from, this.gapFrom - this.from, this.insert,
            this.gapTo, this.to - this.gapTo, this.slice.size - this.insert]);
    };
    ReplaceAroundStep.prototype.invert = function invert (doc) {
        var gap = this.gapTo - this.gapFrom;
        return new ReplaceAroundStep(this.from, this.from + this.slice.size + gap, this.from + this.insert, this.from + this.insert + gap, doc.slice(this.from, this.to).removeBetween(this.gapFrom - this.from, this.gapTo - this.from), this.gapFrom - this.from, this.structure);
    };
    ReplaceAroundStep.prototype.map = function map (mapping) {
        var from = mapping.mapResult(this.from, 1), to = mapping.mapResult(this.to, -1);
        var gapFrom = this.from == this.gapFrom ? from.pos : mapping.map(this.gapFrom, -1);
        var gapTo = this.to == this.gapTo ? to.pos : mapping.map(this.gapTo, 1);
        if ((from.deletedAcross && to.deletedAcross) || gapFrom < from.pos || gapTo > to.pos)
            { return null; }
        return new ReplaceAroundStep(from.pos, to.pos, gapFrom, gapTo, this.slice, this.insert, this.structure);
    };
    ReplaceAroundStep.prototype.toJSON = function toJSON () {
        var json = { stepType: "replaceAround", from: this.from, to: this.to,
            gapFrom: this.gapFrom, gapTo: this.gapTo, insert: this.insert };
        if (this.slice.size)
            { json.slice = this.slice.toJSON(); }
        if (this.structure)
            { json.structure = true; }
        return json;
    };
    /**
    @internal
    */
    ReplaceAroundStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.from != "number" || typeof json.to != "number" ||
            typeof json.gapFrom != "number" || typeof json.gapTo != "number" || typeof json.insert != "number")
            { throw new RangeError("Invalid input for ReplaceAroundStep.fromJSON"); }
        return new ReplaceAroundStep(json.from, json.to, json.gapFrom, json.gapTo, Slice.fromJSON(schema, json.slice), json.insert, !!json.structure);
    };

    return ReplaceAroundStep;
}(Step));
Step.jsonID("replaceAround", ReplaceAroundStep);
function contentBetween(doc, from, to) {
    var $from = doc.resolve(from), dist = to - from, depth = $from.depth;
    while (dist > 0 && depth > 0 && $from.indexAfter(depth) == $from.node(depth).childCount) {
        depth--;
        dist--;
    }
    if (dist > 0) {
        var next = $from.node(depth).maybeChild($from.indexAfter(depth));
        while (dist > 0) {
            if (!next || next.isLeaf)
                { return true; }
            next = next.firstChild;
            dist--;
        }
    }
    return false;
}

function addMark(tr, from, to, mark) {
    var removed = [], added = [];
    var removing, adding;
    tr.doc.nodesBetween(from, to, function (node, pos, parent) {
        if (!node.isInline)
            { return; }
        var marks = node.marks;
        if (!mark.isInSet(marks) && parent.type.allowsMarkType(mark.type)) {
            var start = Math.max(pos, from), end = Math.min(pos + node.nodeSize, to);
            var newSet = mark.addToSet(marks);
            for (var i = 0; i < marks.length; i++) {
                if (!marks[i].isInSet(newSet)) {
                    if (removing && removing.to == start && removing.mark.eq(marks[i]))
                        { removing.to = end; }
                    else
                        { removed.push(removing = new RemoveMarkStep(start, end, marks[i])); }
                }
            }
            if (adding && adding.to == start)
                { adding.to = end; }
            else
                { added.push(adding = new AddMarkStep(start, end, mark)); }
        }
    });
    removed.forEach(function (s) { return tr.step(s); });
    added.forEach(function (s) { return tr.step(s); });
}
function removeMark$1(tr, from, to, mark) {
    var matched = [], step = 0;
    tr.doc.nodesBetween(from, to, function (node, pos) {
        if (!node.isInline)
            { return; }
        step++;
        var toRemove = null;
        if (mark instanceof MarkType) {
            var set = node.marks, found;
            while (found = mark.isInSet(set)) {
                (toRemove || (toRemove = [])).push(found);
                set = found.removeFromSet(set);
            }
        }
        else if (mark) {
            if (mark.isInSet(node.marks))
                { toRemove = [mark]; }
        }
        else {
            toRemove = node.marks;
        }
        if (toRemove && toRemove.length) {
            var end = Math.min(pos + node.nodeSize, to);
            for (var i = 0; i < toRemove.length; i++) {
                var style = toRemove[i], found$1 = (void 0);
                for (var j = 0; j < matched.length; j++) {
                    var m = matched[j];
                    if (m.step == step - 1 && style.eq(matched[j].style))
                        { found$1 = m; }
                }
                if (found$1) {
                    found$1.to = end;
                    found$1.step = step;
                }
                else {
                    matched.push({ style: style, from: Math.max(pos, from), to: end, step: step });
                }
            }
        }
    });
    matched.forEach(function (m) { return tr.step(new RemoveMarkStep(m.from, m.to, m.style)); });
}
function clearIncompatible(tr, pos, parentType, match, clearNewlines) {
    if ( match === void 0 ) match = parentType.contentMatch;
    if ( clearNewlines === void 0 ) clearNewlines = true;

    var node = tr.doc.nodeAt(pos);
    var replSteps = [], cur = pos + 1;
    for (var i = 0; i < node.childCount; i++) {
        var child = node.child(i), end = cur + child.nodeSize;
        var allowed = match.matchType(child.type);
        if (!allowed) {
            replSteps.push(new ReplaceStep(cur, end, Slice.empty));
        }
        else {
            match = allowed;
            for (var j = 0; j < child.marks.length; j++)
                { if (!parentType.allowsMarkType(child.marks[j].type))
                    { tr.step(new RemoveMarkStep(cur, end, child.marks[j])); } }
            if (clearNewlines && child.isText && parentType.whitespace != "pre") {
                var m = (void 0), newline = /\r?\n|\r/g, slice = (void 0);
                while (m = newline.exec(child.text)) {
                    if (!slice)
                        { slice = new Slice(Fragment.from(parentType.schema.text(" ", parentType.allowedMarks(child.marks))), 0, 0); }
                    replSteps.push(new ReplaceStep(cur + m.index, cur + m.index + m[0].length, slice));
                }
            }
        }
        cur = end;
    }
    if (!match.validEnd) {
        var fill = match.fillBefore(Fragment.empty, true);
        tr.replace(cur, cur, new Slice(fill, 0, 0));
    }
    for (var i$1 = replSteps.length - 1; i$1 >= 0; i$1--)
        { tr.step(replSteps[i$1]); }
}

function canCut(node, start, end) {
    return (start == 0 || node.canReplace(start, node.childCount)) &&
        (end == node.childCount || node.canReplace(0, end));
}
/**
Try to find a target depth to which the content in the given range
can be lifted. Will not go across
[isolating](https://prosemirror.net/docs/ref/#model.NodeSpec.isolating) parent nodes.
*/
function liftTarget(range) {
    var parent = range.parent;
    var content = parent.content.cutByIndex(range.startIndex, range.endIndex);
    for (var depth = range.depth;; --depth) {
        var node = range.$from.node(depth);
        var index = range.$from.index(depth), endIndex = range.$to.indexAfter(depth);
        if (depth < range.depth && node.canReplace(index, endIndex, content))
            { return depth; }
        if (depth == 0 || node.type.spec.isolating || !canCut(node, index, endIndex))
            { break; }
    }
    return null;
}
function lift$2(tr, range, target) {
    var $from = range.$from;
    var $to = range.$to;
    var depth = range.depth;
    var gapStart = $from.before(depth + 1), gapEnd = $to.after(depth + 1);
    var start = gapStart, end = gapEnd;
    var before = Fragment.empty, openStart = 0;
    for (var d = depth, splitting = false; d > target; d--)
        { if (splitting || $from.index(d) > 0) {
            splitting = true;
            before = Fragment.from($from.node(d).copy(before));
            openStart++;
        }
        else {
            start--;
        } }
    var after = Fragment.empty, openEnd = 0;
    for (var d$1 = depth, splitting$1 = false; d$1 > target; d$1--)
        { if (splitting$1 || $to.after(d$1 + 1) < $to.end(d$1)) {
            splitting$1 = true;
            after = Fragment.from($to.node(d$1).copy(after));
            openEnd++;
        }
        else {
            end++;
        } }
    tr.step(new ReplaceAroundStep(start, end, gapStart, gapEnd, new Slice(before.append(after), openStart, openEnd), before.size - openStart, true));
}
/**
Try to find a valid way to wrap the content in the given range in a
node of the given type. May introduce extra nodes around and inside
the wrapper node, if necessary. Returns null if no valid wrapping
could be found. When `innerRange` is given, that range's content is
used as the content to fit into the wrapping, instead of the
content of `range`.
*/
function findWrapping(range, nodeType, attrs, innerRange) {
    if ( attrs === void 0 ) attrs = null;
    if ( innerRange === void 0 ) innerRange = range;

    var around = findWrappingOutside(range, nodeType);
    var inner = around && findWrappingInside(innerRange, nodeType);
    if (!inner)
        { return null; }
    return around.map(withAttrs)
        .concat({ type: nodeType, attrs: attrs }).concat(inner.map(withAttrs));
}
function withAttrs(type) { return { type: type, attrs: null }; }
function findWrappingOutside(range, type) {
    var parent = range.parent;
    var startIndex = range.startIndex;
    var endIndex = range.endIndex;
    var around = parent.contentMatchAt(startIndex).findWrapping(type);
    if (!around)
        { return null; }
    var outer = around.length ? around[0] : type;
    return parent.canReplaceWith(startIndex, endIndex, outer) ? around : null;
}
function findWrappingInside(range, type) {
    var parent = range.parent;
    var startIndex = range.startIndex;
    var endIndex = range.endIndex;
    var inner = parent.child(startIndex);
    var inside = type.contentMatch.findWrapping(inner.type);
    if (!inside)
        { return null; }
    var lastType = inside.length ? inside[inside.length - 1] : type;
    var innerMatch = lastType.contentMatch;
    for (var i = startIndex; innerMatch && i < endIndex; i++)
        { innerMatch = innerMatch.matchType(parent.child(i).type); }
    if (!innerMatch || !innerMatch.validEnd)
        { return null; }
    return inside;
}
function wrap(tr, range, wrappers) {
    var content = Fragment.empty;
    for (var i = wrappers.length - 1; i >= 0; i--) {
        if (content.size) {
            var match = wrappers[i].type.contentMatch.matchFragment(content);
            if (!match || !match.validEnd)
                { throw new RangeError("Wrapper type given to Transform.wrap does not form valid content of its parent wrapper"); }
        }
        content = Fragment.from(wrappers[i].type.create(wrappers[i].attrs, content));
    }
    var start = range.start, end = range.end;
    tr.step(new ReplaceAroundStep(start, end, start, end, new Slice(content, 0, 0), wrappers.length, true));
}
function setBlockType$1(tr, from, to, type, attrs) {
    if (!type.isTextblock)
        { throw new RangeError("Type given to setBlockType should be a textblock"); }
    var mapFrom = tr.steps.length;
    tr.doc.nodesBetween(from, to, function (node, pos) {
        var attrsHere = typeof attrs == "function" ? attrs(node) : attrs;
        if (node.isTextblock && !node.hasMarkup(type, attrsHere) &&
            canChangeType(tr.doc, tr.mapping.slice(mapFrom).map(pos), type)) {
            var convertNewlines = null;
            if (type.schema.linebreakReplacement) {
                var pre = type.whitespace == "pre", supportLinebreak = !!type.contentMatch.matchType(type.schema.linebreakReplacement);
                if (pre && !supportLinebreak)
                    { convertNewlines = false; }
                else if (!pre && supportLinebreak)
                    { convertNewlines = true; }
            }
            // Ensure all markup that isn't allowed in the new node type is cleared
            if (convertNewlines === false)
                { replaceLinebreaks(tr, node, pos, mapFrom); }
            clearIncompatible(tr, tr.mapping.slice(mapFrom).map(pos, 1), type, undefined, convertNewlines === null);
            var mapping = tr.mapping.slice(mapFrom);
            var startM = mapping.map(pos, 1), endM = mapping.map(pos + node.nodeSize, 1);
            tr.step(new ReplaceAroundStep(startM, endM, startM + 1, endM - 1, new Slice(Fragment.from(type.create(attrsHere, null, node.marks)), 0, 0), 1, true));
            if (convertNewlines === true)
                { replaceNewlines(tr, node, pos, mapFrom); }
            return false;
        }
    });
}
function replaceNewlines(tr, node, pos, mapFrom) {
    node.forEach(function (child, offset) {
        if (child.isText) {
            var m, newline = /\r?\n|\r/g;
            while (m = newline.exec(child.text)) {
                var start = tr.mapping.slice(mapFrom).map(pos + 1 + offset + m.index);
                tr.replaceWith(start, start + 1, node.type.schema.linebreakReplacement.create());
            }
        }
    });
}
function replaceLinebreaks(tr, node, pos, mapFrom) {
    node.forEach(function (child, offset) {
        if (child.type == child.type.schema.linebreakReplacement) {
            var start = tr.mapping.slice(mapFrom).map(pos + 1 + offset);
            tr.replaceWith(start, start + 1, node.type.schema.text("\n"));
        }
    });
}
function canChangeType(doc, pos, type) {
    var $pos = doc.resolve(pos), index = $pos.index();
    return $pos.parent.canReplaceWith(index, index + 1, type);
}
/**
Change the type, attributes, and/or marks of the node at `pos`.
When `type` isn't given, the existing node type is preserved,
*/
function setNodeMarkup(tr, pos, type, attrs, marks) {
    var node = tr.doc.nodeAt(pos);
    if (!node)
        { throw new RangeError("No node at given position"); }
    if (!type)
        { type = node.type; }
    var newNode = type.create(attrs, null, marks || node.marks);
    if (node.isLeaf)
        { return tr.replaceWith(pos, pos + node.nodeSize, newNode); }
    if (!type.validContent(node.content))
        { throw new RangeError("Invalid content for node type " + type.name); }
    tr.step(new ReplaceAroundStep(pos, pos + node.nodeSize, pos + 1, pos + node.nodeSize - 1, new Slice(Fragment.from(newNode), 0, 0), 1, true));
}
/**
Check whether splitting at the given position is allowed.
*/
function canSplit(doc, pos, depth, typesAfter) {
    if ( depth === void 0 ) depth = 1;

    var $pos = doc.resolve(pos), base = $pos.depth - depth;
    var innerType = (typesAfter && typesAfter[typesAfter.length - 1]) || $pos.parent;
    if (base < 0 || $pos.parent.type.spec.isolating ||
        !$pos.parent.canReplace($pos.index(), $pos.parent.childCount) ||
        !innerType.type.validContent($pos.parent.content.cutByIndex($pos.index(), $pos.parent.childCount)))
        { return false; }
    for (var d = $pos.depth - 1, i = depth - 2; d > base; d--, i--) {
        var node = $pos.node(d), index$1 = $pos.index(d);
        if (node.type.spec.isolating)
            { return false; }
        var rest = node.content.cutByIndex(index$1, node.childCount);
        var overrideChild = typesAfter && typesAfter[i + 1];
        if (overrideChild)
            { rest = rest.replaceChild(0, overrideChild.type.create(overrideChild.attrs)); }
        var after = (typesAfter && typesAfter[i]) || node;
        if (!node.canReplace(index$1 + 1, node.childCount) || !after.type.validContent(rest))
            { return false; }
    }
    var index = $pos.indexAfter(base);
    var baseType = typesAfter && typesAfter[0];
    return $pos.node(base).canReplaceWith(index, index, baseType ? baseType.type : $pos.node(base + 1).type);
}
function split(tr, pos, depth, typesAfter) {
    if ( depth === void 0 ) depth = 1;

    var $pos = tr.doc.resolve(pos), before = Fragment.empty, after = Fragment.empty;
    for (var d = $pos.depth, e = $pos.depth - depth, i = depth - 1; d > e; d--, i--) {
        before = Fragment.from($pos.node(d).copy(before));
        var typeAfter = typesAfter && typesAfter[i];
        after = Fragment.from(typeAfter ? typeAfter.type.create(typeAfter.attrs, after) : $pos.node(d).copy(after));
    }
    tr.step(new ReplaceStep(pos, pos, new Slice(before.append(after), depth, depth), true));
}
/**
Test whether the blocks before and after a given position can be
joined.
*/
function canJoin(doc, pos) {
    var $pos = doc.resolve(pos), index = $pos.index();
    return joinable($pos.nodeBefore, $pos.nodeAfter) &&
        $pos.parent.canReplace(index, index + 1);
}
function joinable(a, b) {
    return !!(a && b && !a.isLeaf && a.canAppend(b));
}
/**
Find an ancestor of the given position that can be joined to the
block before (or after if `dir` is positive). Returns the joinable
point, if any.
*/
function joinPoint(doc, pos, dir) {
    if ( dir === void 0 ) dir = -1;

    var $pos = doc.resolve(pos);
    for (var d = $pos.depth;; d--) {
        var before = (void 0), after = (void 0), index = $pos.index(d);
        if (d == $pos.depth) {
            before = $pos.nodeBefore;
            after = $pos.nodeAfter;
        }
        else if (dir > 0) {
            before = $pos.node(d + 1);
            index++;
            after = $pos.node(d).maybeChild(index);
        }
        else {
            before = $pos.node(d).maybeChild(index - 1);
            after = $pos.node(d + 1);
        }
        if (before && !before.isTextblock && joinable(before, after) &&
            $pos.node(d).canReplace(index, index + 1))
            { return pos; }
        if (d == 0)
            { break; }
        pos = dir < 0 ? $pos.before(d) : $pos.after(d);
    }
}
function join(tr, pos, depth) {
    var step = new ReplaceStep(pos - depth, pos + depth, Slice.empty, true);
    tr.step(step);
}
/**
Try to find a point where a node of the given type can be inserted
near `pos`, by searching up the node hierarchy when `pos` itself
isn't a valid place but is at the start or end of a node. Return
null if no position was found.
*/
function insertPoint(doc, pos, nodeType) {
    var $pos = doc.resolve(pos);
    if ($pos.parent.canReplaceWith($pos.index(), $pos.index(), nodeType))
        { return pos; }
    if ($pos.parentOffset == 0)
        { for (var d = $pos.depth - 1; d >= 0; d--) {
            var index = $pos.index(d);
            if ($pos.node(d).canReplaceWith(index, index, nodeType))
                { return $pos.before(d + 1); }
            if (index > 0)
                { return null; }
        } }
    if ($pos.parentOffset == $pos.parent.content.size)
        { for (var d$1 = $pos.depth - 1; d$1 >= 0; d$1--) {
            var index$1 = $pos.indexAfter(d$1);
            if ($pos.node(d$1).canReplaceWith(index$1, index$1, nodeType))
                { return $pos.after(d$1 + 1); }
            if (index$1 < $pos.node(d$1).childCount)
                { return null; }
        } }
    return null;
}
/**
Finds a position at or around the given position where the given
slice can be inserted. Will look at parent nodes' nearest boundary
and try there, even if the original position wasn't directly at the
start or end of that node. Returns null when no position was found.
*/
function dropPoint(doc, pos, slice) {
    var $pos = doc.resolve(pos);
    if (!slice.content.size)
        { return pos; }
    var content = slice.content;
    for (var i = 0; i < slice.openStart; i++)
        { content = content.firstChild.content; }
    for (var pass = 1; pass <= (slice.openStart == 0 && slice.size ? 2 : 1); pass++) {
        for (var d = $pos.depth; d >= 0; d--) {
            var bias = d == $pos.depth ? 0 : $pos.pos <= ($pos.start(d + 1) + $pos.end(d + 1)) / 2 ? -1 : 1;
            var insertPos = $pos.index(d) + (bias > 0 ? 1 : 0);
            var parent = $pos.node(d), fits = false;
            if (pass == 1) {
                fits = parent.canReplace(insertPos, insertPos, content);
            }
            else {
                var wrapping = parent.contentMatchAt(insertPos).findWrapping(content.firstChild.type);
                fits = wrapping && parent.canReplaceWith(insertPos, insertPos, wrapping[0]);
            }
            if (fits)
                { return bias == 0 ? $pos.pos : bias < 0 ? $pos.before(d + 1) : $pos.after(d + 1); }
        }
    }
    return null;
}

/**
‘Fit’ a slice into a given position in the document, producing a
[step](https://prosemirror.net/docs/ref/#transform.Step) that inserts it. Will return null if
there's no meaningful way to insert the slice here, or inserting it
would be a no-op (an empty slice over an empty range).
*/
function replaceStep(doc, from, to, slice) {
    if ( to === void 0 ) to = from;
    if ( slice === void 0 ) slice = Slice.empty;

    if (from == to && !slice.size)
        { return null; }
    var $from = doc.resolve(from), $to = doc.resolve(to);
    // Optimization -- avoid work if it's obvious that it's not needed.
    if (fitsTrivially($from, $to, slice))
        { return new ReplaceStep(from, to, slice); }
    return new Fitter($from, $to, slice).fit();
}
function fitsTrivially($from, $to, slice) {
    return !slice.openStart && !slice.openEnd && $from.start() == $to.start() &&
        $from.parent.canReplace($from.index(), $to.index(), slice.content);
}
// Algorithm for 'placing' the elements of a slice into a gap:
//
// We consider the content of each node that is open to the left to be
// independently placeable. I.e. in <p("foo"), p("bar")>, when the
// paragraph on the left is open, "foo" can be placed (somewhere on
// the left side of the replacement gap) independently from p("bar").
//
// This class tracks the state of the placement progress in the
// following properties:
//
//  - `frontier` holds a stack of `{type, match}` objects that
//    represent the open side of the replacement. It starts at
//    `$from`, then moves forward as content is placed, and is finally
//    reconciled with `$to`.
//
//  - `unplaced` is a slice that represents the content that hasn't
//    been placed yet.
//
//  - `placed` is a fragment of placed content. Its open-start value
//    is implicit in `$from`, and its open-end value in `frontier`.
var Fitter = function Fitter($from, $to, unplaced) {
    this.$from = $from;
    this.$to = $to;
    this.unplaced = unplaced;
    this.frontier = [];
    this.placed = Fragment.empty;
    for (var i = 0; i <= $from.depth; i++) {
        var node = $from.node(i);
        this.frontier.push({
            type: node.type,
            match: node.contentMatchAt($from.indexAfter(i))
        });
    }
    for (var i$1 = $from.depth; i$1 > 0; i$1--)
        { this.placed = Fragment.from($from.node(i$1).copy(this.placed)); }
};

var prototypeAccessors$1$2 = { depth: { configurable: true } };
prototypeAccessors$1$2.depth.get = function () { return this.frontier.length - 1; };
Fitter.prototype.fit = function fit () {
    // As long as there's unplaced content, try to place some of it.
    // If that fails, either increase the open score of the unplaced
    // slice, or drop nodes from it, and then try again.
    while (this.unplaced.size) {
        var fit = this.findFittable();
        if (fit)
            { this.placeNodes(fit); }
        else
            { this.openMore() || this.dropNode(); }
    }
    // When there's inline content directly after the frontier _and_
    // directly after `this.$to`, we must generate a `ReplaceAround`
    // step that pulls that content into the node after the frontier.
    // That means the fitting must be done to the end of the textblock
    // node after `this.$to`, not `this.$to` itself.
    var moveInline = this.mustMoveInline(), placedSize = this.placed.size - this.depth - this.$from.depth;
    var $from = this.$from, $to = this.close(moveInline < 0 ? this.$to : $from.doc.resolve(moveInline));
    if (!$to)
        { return null; }
    // If closing to `$to` succeeded, create a step
    var content = this.placed, openStart = $from.depth, openEnd = $to.depth;
    while (openStart && openEnd && content.childCount == 1) { // Normalize by dropping open parent nodes
        content = content.firstChild.content;
        openStart--;
        openEnd--;
    }
    var slice = new Slice(content, openStart, openEnd);
    if (moveInline > -1)
        { return new ReplaceAroundStep($from.pos, moveInline, this.$to.pos, this.$to.end(), slice, placedSize); }
    if (slice.size || $from.pos != this.$to.pos) // Don't generate no-op steps
        { return new ReplaceStep($from.pos, $to.pos, slice); }
    return null;
};
// Find a position on the start spine of `this.unplaced` that has
// content that can be moved somewhere on the frontier. Returns two
// depths, one for the slice and one for the frontier.
Fitter.prototype.findFittable = function findFittable () {
    var startDepth = this.unplaced.openStart;
    for (var cur = this.unplaced.content, d = 0, openEnd = this.unplaced.openEnd; d < startDepth; d++) {
        var node = cur.firstChild;
        if (cur.childCount > 1)
            { openEnd = 0; }
        if (node.type.spec.isolating && openEnd <= d) {
            startDepth = d;
            break;
        }
        cur = node.content;
    }
    // Only try wrapping nodes (pass 2) after finding a place without
    // wrapping failed.
    for (var pass = 1; pass <= 2; pass++) {
        for (var sliceDepth = pass == 1 ? startDepth : this.unplaced.openStart; sliceDepth >= 0; sliceDepth--) {
            var fragment = (void 0), parent = null;
            if (sliceDepth) {
                parent = contentAt(this.unplaced.content, sliceDepth - 1).firstChild;
                fragment = parent.content;
            }
            else {
                fragment = this.unplaced.content;
            }
            var first = fragment.firstChild;
            for (var frontierDepth = this.depth; frontierDepth >= 0; frontierDepth--) {
                var ref = this.frontier[frontierDepth];
                    var type = ref.type;
                    var match = ref.match;
                    var wrap = (void 0), inject = null;
                // In pass 1, if the next node matches, or there is no next
                // node but the parents look compatible, we've found a
                // place.
                if (pass == 1 && (first ? match.matchType(first.type) || (inject = match.fillBefore(Fragment.from(first), false))
                    : parent && type.compatibleContent(parent.type)))
                    { return { sliceDepth: sliceDepth, frontierDepth: frontierDepth, parent: parent, inject: inject }; }
                // In pass 2, look for a set of wrapping nodes that make
                // `first` fit here.
                else if (pass == 2 && first && (wrap = match.findWrapping(first.type)))
                    { return { sliceDepth: sliceDepth, frontierDepth: frontierDepth, parent: parent, wrap: wrap }; }
                // Don't continue looking further up if the parent node
                // would fit here.
                if (parent && match.matchType(parent.type))
                    { break; }
            }
        }
    }
};
Fitter.prototype.openMore = function openMore () {
    var ref = this.unplaced;
        var content = ref.content;
        var openStart = ref.openStart;
        var openEnd = ref.openEnd;
    var inner = contentAt(content, openStart);
    if (!inner.childCount || inner.firstChild.isLeaf)
        { return false; }
    this.unplaced = new Slice(content, openStart + 1, Math.max(openEnd, inner.size + openStart >= content.size - openEnd ? openStart + 1 : 0));
    return true;
};
Fitter.prototype.dropNode = function dropNode () {
    var ref = this.unplaced;
        var content = ref.content;
        var openStart = ref.openStart;
        var openEnd = ref.openEnd;
    var inner = contentAt(content, openStart);
    if (inner.childCount <= 1 && openStart > 0) {
        var openAtEnd = content.size - openStart <= openStart + inner.size;
        this.unplaced = new Slice(dropFromFragment(content, openStart - 1, 1), openStart - 1, openAtEnd ? openStart - 1 : openEnd);
    }
    else {
        this.unplaced = new Slice(dropFromFragment(content, openStart, 1), openStart, openEnd);
    }
};
// Move content from the unplaced slice at `sliceDepth` to the
// frontier node at `frontierDepth`. Close that frontier node when
// applicable.
Fitter.prototype.placeNodes = function placeNodes (ref) {
        var sliceDepth = ref.sliceDepth;
        var frontierDepth = ref.frontierDepth;
        var parent = ref.parent;
        var inject = ref.inject;
        var wrap = ref.wrap;

    while (this.depth > frontierDepth)
        { this.closeFrontierNode(); }
    if (wrap)
        { for (var i = 0; i < wrap.length; i++)
            { this.openFrontierNode(wrap[i]); } }
    var slice = this.unplaced, fragment = parent ? parent.content : slice.content;
    var openStart = slice.openStart - sliceDepth;
    var taken = 0, add = [];
    var ref$1 = this.frontier[frontierDepth];
        var match = ref$1.match;
        var type = ref$1.type;
    if (inject) {
        for (var i$1 = 0; i$1 < inject.childCount; i$1++)
            { add.push(inject.child(i$1)); }
        match = match.matchFragment(inject);
    }
    // Computes the amount of (end) open nodes at the end of the
    // fragment. When 0, the parent is open, but no more. When
    // negative, nothing is open.
    var openEndCount = (fragment.size + sliceDepth) - (slice.content.size - slice.openEnd);
    // Scan over the fragment, fitting as many child nodes as
    // possible.
    while (taken < fragment.childCount) {
        var next = fragment.child(taken), matches = match.matchType(next.type);
        if (!matches)
            { break; }
        taken++;
        if (taken > 1 || openStart == 0 || next.content.size) { // Drop empty open nodes
            match = matches;
            add.push(closeNodeStart(next.mark(type.allowedMarks(next.marks)), taken == 1 ? openStart : 0, taken == fragment.childCount ? openEndCount : -1));
        }
    }
    var toEnd = taken == fragment.childCount;
    if (!toEnd)
        { openEndCount = -1; }
    this.placed = addToFragment(this.placed, frontierDepth, Fragment.from(add));
    this.frontier[frontierDepth].match = match;
    // If the parent types match, and the entire node was moved, and
    // it's not open, close this frontier node right away.
    if (toEnd && openEndCount < 0 && parent && parent.type == this.frontier[this.depth].type && this.frontier.length > 1)
        { this.closeFrontierNode(); }
    // Add new frontier nodes for any open nodes at the end.
    for (var i$2 = 0, cur = fragment; i$2 < openEndCount; i$2++) {
        var node = cur.lastChild;
        this.frontier.push({ type: node.type, match: node.contentMatchAt(node.childCount) });
        cur = node.content;
    }
    // Update `this.unplaced`. Drop the entire node from which we
    // placed it we got to its end, otherwise just drop the placed
    // nodes.
    this.unplaced = !toEnd ? new Slice(dropFromFragment(slice.content, sliceDepth, taken), slice.openStart, slice.openEnd)
        : sliceDepth == 0 ? Slice.empty
            : new Slice(dropFromFragment(slice.content, sliceDepth - 1, 1), sliceDepth - 1, openEndCount < 0 ? slice.openEnd : sliceDepth - 1);
};
Fitter.prototype.mustMoveInline = function mustMoveInline () {
    if (!this.$to.parent.isTextblock)
        { return -1; }
    var top = this.frontier[this.depth], level;
    if (!top.type.isTextblock || !contentAfterFits(this.$to, this.$to.depth, top.type, top.match, false) ||
        (this.$to.depth == this.depth && (level = this.findCloseLevel(this.$to)) && level.depth == this.depth))
        { return -1; }
    var ref = this.$to;
        var depth = ref.depth;
        var after = this.$to.after(depth);
    while (depth > 1 && after == this.$to.end(--depth))
        { ++after; }
    return after;
};
Fitter.prototype.findCloseLevel = function findCloseLevel ($to) {
    scan: for (var i = Math.min(this.depth, $to.depth); i >= 0; i--) {
        var ref = this.frontier[i];
            var match = ref.match;
            var type = ref.type;
        var dropInner = i < $to.depth && $to.end(i + 1) == $to.pos + ($to.depth - (i + 1));
        var fit = contentAfterFits($to, i, type, match, dropInner);
        if (!fit)
            { continue; }
        for (var d = i - 1; d >= 0; d--) {
            var ref$1 = this.frontier[d];
                var match$1 = ref$1.match;
                var type$1 = ref$1.type;
            var matches = contentAfterFits($to, d, type$1, match$1, true);
            if (!matches || matches.childCount)
                { continue scan; }
        }
        return { depth: i, fit: fit, move: dropInner ? $to.doc.resolve($to.after(i + 1)) : $to };
    }
};
Fitter.prototype.close = function close ($to) {
    var close = this.findCloseLevel($to);
    if (!close)
        { return null; }
    while (this.depth > close.depth)
        { this.closeFrontierNode(); }
    if (close.fit.childCount)
        { this.placed = addToFragment(this.placed, close.depth, close.fit); }
    $to = close.move;
    for (var d = close.depth + 1; d <= $to.depth; d++) {
        var node = $to.node(d), add = node.type.contentMatch.fillBefore(node.content, true, $to.index(d));
        this.openFrontierNode(node.type, node.attrs, add);
    }
    return $to;
};
Fitter.prototype.openFrontierNode = function openFrontierNode (type, attrs, content) {
        if ( attrs === void 0 ) attrs = null;

    var top = this.frontier[this.depth];
    top.match = top.match.matchType(type);
    this.placed = addToFragment(this.placed, this.depth, Fragment.from(type.create(attrs, content)));
    this.frontier.push({ type: type, match: type.contentMatch });
};
Fitter.prototype.closeFrontierNode = function closeFrontierNode () {
    var open = this.frontier.pop();
    var add = open.match.fillBefore(Fragment.empty, true);
    if (add.childCount)
        { this.placed = addToFragment(this.placed, this.frontier.length, add); }
};

Object.defineProperties( Fitter.prototype, prototypeAccessors$1$2 );
function dropFromFragment(fragment, depth, count) {
    if (depth == 0)
        { return fragment.cutByIndex(count, fragment.childCount); }
    return fragment.replaceChild(0, fragment.firstChild.copy(dropFromFragment(fragment.firstChild.content, depth - 1, count)));
}
function addToFragment(fragment, depth, content) {
    if (depth == 0)
        { return fragment.append(content); }
    return fragment.replaceChild(fragment.childCount - 1, fragment.lastChild.copy(addToFragment(fragment.lastChild.content, depth - 1, content)));
}
function contentAt(fragment, depth) {
    for (var i = 0; i < depth; i++)
        { fragment = fragment.firstChild.content; }
    return fragment;
}
function closeNodeStart(node, openStart, openEnd) {
    if (openStart <= 0)
        { return node; }
    var frag = node.content;
    if (openStart > 1)
        { frag = frag.replaceChild(0, closeNodeStart(frag.firstChild, openStart - 1, frag.childCount == 1 ? openEnd - 1 : 0)); }
    if (openStart > 0) {
        frag = node.type.contentMatch.fillBefore(frag).append(frag);
        if (openEnd <= 0)
            { frag = frag.append(node.type.contentMatch.matchFragment(frag).fillBefore(Fragment.empty, true)); }
    }
    return node.copy(frag);
}
function contentAfterFits($to, depth, type, match, open) {
    var node = $to.node(depth), index = open ? $to.indexAfter(depth) : $to.index(depth);
    if (index == node.childCount && !type.compatibleContent(node.type))
        { return null; }
    var fit = match.fillBefore(node.content, true, index);
    return fit && !invalidMarks(type, node.content, index) ? fit : null;
}
function invalidMarks(type, fragment, start) {
    for (var i = start; i < fragment.childCount; i++)
        { if (!type.allowsMarks(fragment.child(i).marks))
            { return true; } }
    return false;
}
function definesContent(type) {
    return type.spec.defining || type.spec.definingForContent;
}
function replaceRange$1(tr, from, to, slice) {
    if (!slice.size)
        { return tr.deleteRange(from, to); }
    var $from = tr.doc.resolve(from), $to = tr.doc.resolve(to);
    if (fitsTrivially($from, $to, slice))
        { return tr.step(new ReplaceStep(from, to, slice)); }
    var targetDepths = coveredDepths($from, tr.doc.resolve(to));
    // Can't replace the whole document, so remove 0 if it's present
    if (targetDepths[targetDepths.length - 1] == 0)
        { targetDepths.pop(); }
    // Negative numbers represent not expansion over the whole node at
    // that depth, but replacing from $from.before(-D) to $to.pos.
    var preferredTarget = -($from.depth + 1);
    targetDepths.unshift(preferredTarget);
    // This loop picks a preferred target depth, if one of the covering
    // depths is not outside of a defining node, and adds negative
    // depths for any depth that has $from at its start and does not
    // cross a defining node.
    for (var d = $from.depth, pos = $from.pos - 1; d > 0; d--, pos--) {
        var spec = $from.node(d).type.spec;
        if (spec.defining || spec.definingAsContext || spec.isolating)
            { break; }
        if (targetDepths.indexOf(d) > -1)
            { preferredTarget = d; }
        else if ($from.before(d) == pos)
            { targetDepths.splice(1, 0, -d); }
    }
    // Try to fit each possible depth of the slice into each possible
    // target depth, starting with the preferred depths.
    var preferredTargetIndex = targetDepths.indexOf(preferredTarget);
    var leftNodes = [], preferredDepth = slice.openStart;
    for (var content = slice.content, i = 0;; i++) {
        var node = content.firstChild;
        leftNodes.push(node);
        if (i == slice.openStart)
            { break; }
        content = node.content;
    }
    // Back up preferredDepth to cover defining textblocks directly
    // above it, possibly skipping a non-defining textblock.
    for (var d$1 = preferredDepth - 1; d$1 >= 0; d$1--) {
        var leftNode = leftNodes[d$1], def = definesContent(leftNode.type);
        if (def && !leftNode.sameMarkup($from.node(Math.abs(preferredTarget) - 1)))
            { preferredDepth = d$1; }
        else if (def || !leftNode.type.isTextblock)
            { break; }
    }
    for (var j = slice.openStart; j >= 0; j--) {
        var openDepth = (j + preferredDepth + 1) % (slice.openStart + 1);
        var insert = leftNodes[openDepth];
        if (!insert)
            { continue; }
        for (var i$1 = 0; i$1 < targetDepths.length; i$1++) {
            // Loop over possible expansion levels, starting with the
            // preferred one
            var targetDepth = targetDepths[(i$1 + preferredTargetIndex) % targetDepths.length], expand = true;
            if (targetDepth < 0) {
                expand = false;
                targetDepth = -targetDepth;
            }
            var parent = $from.node(targetDepth - 1), index = $from.index(targetDepth - 1);
            if (parent.canReplaceWith(index, index, insert.type, insert.marks))
                { return tr.replace($from.before(targetDepth), expand ? $to.after(targetDepth) : to, new Slice(closeFragment(slice.content, 0, slice.openStart, openDepth), openDepth, slice.openEnd)); }
        }
    }
    var startSteps = tr.steps.length;
    for (var i$2 = targetDepths.length - 1; i$2 >= 0; i$2--) {
        tr.replace(from, to, slice);
        if (tr.steps.length > startSteps)
            { break; }
        var depth = targetDepths[i$2];
        if (depth < 0)
            { continue; }
        from = $from.before(depth);
        to = $to.after(depth);
    }
}
function closeFragment(fragment, depth, oldOpen, newOpen, parent) {
    if (depth < oldOpen) {
        var first = fragment.firstChild;
        fragment = fragment.replaceChild(0, first.copy(closeFragment(first.content, depth + 1, oldOpen, newOpen, first)));
    }
    if (depth > newOpen) {
        var match = parent.contentMatchAt(0);
        var start = match.fillBefore(fragment).append(fragment);
        fragment = start.append(match.matchFragment(start).fillBefore(Fragment.empty, true));
    }
    return fragment;
}
function replaceRangeWith(tr, from, to, node) {
    if (!node.isInline && from == to && tr.doc.resolve(from).parent.content.size) {
        var point = insertPoint(tr.doc, from, node.type);
        if (point != null)
            { from = to = point; }
    }
    tr.replaceRange(from, to, new Slice(Fragment.from(node), 0, 0));
}
function deleteRange(tr, from, to) {
    var $from = tr.doc.resolve(from), $to = tr.doc.resolve(to);
    var covered = coveredDepths($from, $to);
    for (var i = 0; i < covered.length; i++) {
        var depth = covered[i], last = i == covered.length - 1;
        if ((last && depth == 0) || $from.node(depth).type.contentMatch.validEnd)
            { return tr.delete($from.start(depth), $to.end(depth)); }
        if (depth > 0 && (last || $from.node(depth - 1).canReplace($from.index(depth - 1), $to.indexAfter(depth - 1))))
            { return tr.delete($from.before(depth), $to.after(depth)); }
    }
    for (var d = 1; d <= $from.depth && d <= $to.depth; d++) {
        if (from - $from.start(d) == $from.depth - d && to > $from.end(d) && $to.end(d) - to != $to.depth - d)
            { return tr.delete($from.before(d), to); }
    }
    tr.delete(from, to);
}
// Returns an array of all depths for which $from - $to spans the
// whole content of the nodes at that depth.
function coveredDepths($from, $to) {
    var result = [], minDepth = Math.min($from.depth, $to.depth);
    for (var d = minDepth; d >= 0; d--) {
        var start = $from.start(d);
        if (start < $from.pos - ($from.depth - d) ||
            $to.end(d) > $to.pos + ($to.depth - d) ||
            $from.node(d).type.spec.isolating ||
            $to.node(d).type.spec.isolating)
            { break; }
        if (start == $to.start(d) ||
            (d == $from.depth && d == $to.depth && $from.parent.inlineContent && $to.parent.inlineContent &&
                d && $to.start(d - 1) == start - 1))
            { result.push(d); }
    }
    return result;
}

/**
Update an attribute in a specific node.
*/
var AttrStep = /*@__PURE__*/(function (Step) {
    function AttrStep(
    /**
    The position of the target node.
    */
    pos, 
    /**
    The attribute to set.
    */
    attr, 
    // The attribute's new value.
    value) {
        Step.call(this);
        this.pos = pos;
        this.attr = attr;
        this.value = value;
    }

    if ( Step ) AttrStep.__proto__ = Step;
    AttrStep.prototype = Object.create( Step && Step.prototype );
    AttrStep.prototype.constructor = AttrStep;
    AttrStep.prototype.apply = function apply (doc) {
        var node = doc.nodeAt(this.pos);
        if (!node)
            { return StepResult.fail("No node at attribute step's position"); }
        var attrs = Object.create(null);
        for (var name in node.attrs)
            { attrs[name] = node.attrs[name]; }
        attrs[this.attr] = this.value;
        var updated = node.type.create(attrs, null, node.marks);
        return StepResult.fromReplace(doc, this.pos, this.pos + 1, new Slice(Fragment.from(updated), 0, node.isLeaf ? 0 : 1));
    };
    AttrStep.prototype.getMap = function getMap () {
        return StepMap.empty;
    };
    AttrStep.prototype.invert = function invert (doc) {
        return new AttrStep(this.pos, this.attr, doc.nodeAt(this.pos).attrs[this.attr]);
    };
    AttrStep.prototype.map = function map (mapping) {
        var pos = mapping.mapResult(this.pos, 1);
        return pos.deletedAfter ? null : new AttrStep(pos.pos, this.attr, this.value);
    };
    AttrStep.prototype.toJSON = function toJSON () {
        return { stepType: "attr", pos: this.pos, attr: this.attr, value: this.value };
    };
    AttrStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.pos != "number" || typeof json.attr != "string")
            { throw new RangeError("Invalid input for AttrStep.fromJSON"); }
        return new AttrStep(json.pos, json.attr, json.value);
    };

    return AttrStep;
}(Step));
Step.jsonID("attr", AttrStep);
/**
Update an attribute in the doc node.
*/
var DocAttrStep = /*@__PURE__*/(function (Step) {
    function DocAttrStep(
    /**
    The attribute to set.
    */
    attr, 
    // The attribute's new value.
    value) {
        Step.call(this);
        this.attr = attr;
        this.value = value;
    }

    if ( Step ) DocAttrStep.__proto__ = Step;
    DocAttrStep.prototype = Object.create( Step && Step.prototype );
    DocAttrStep.prototype.constructor = DocAttrStep;
    DocAttrStep.prototype.apply = function apply (doc) {
        var attrs = Object.create(null);
        for (var name in doc.attrs)
            { attrs[name] = doc.attrs[name]; }
        attrs[this.attr] = this.value;
        var updated = doc.type.create(attrs, doc.content, doc.marks);
        return StepResult.ok(updated);
    };
    DocAttrStep.prototype.getMap = function getMap () {
        return StepMap.empty;
    };
    DocAttrStep.prototype.invert = function invert (doc) {
        return new DocAttrStep(this.attr, doc.attrs[this.attr]);
    };
    DocAttrStep.prototype.map = function map (mapping) {
        return this;
    };
    DocAttrStep.prototype.toJSON = function toJSON () {
        return { stepType: "docAttr", attr: this.attr, value: this.value };
    };
    DocAttrStep.fromJSON = function fromJSON (schema, json) {
        if (typeof json.attr != "string")
            { throw new RangeError("Invalid input for DocAttrStep.fromJSON"); }
        return new DocAttrStep(json.attr, json.value);
    };

    return DocAttrStep;
}(Step));
Step.jsonID("docAttr", DocAttrStep);

/**
@internal
*/
var TransformError = /*@__PURE__*/(function (Error) {
    function TransformError () {
        Error.apply(this, arguments);
    }if ( Error ) TransformError.__proto__ = Error;
    TransformError.prototype = Object.create( Error && Error.prototype );
    TransformError.prototype.constructor = TransformError;

    

    return TransformError;
}(Error));
TransformError = function TransformError(message) {
    var err = Error.call(this, message);
    err.__proto__ = TransformError.prototype;
    return err;
};
TransformError.prototype = Object.create(Error.prototype);
TransformError.prototype.constructor = TransformError;
TransformError.prototype.name = "TransformError";
/**
Abstraction to build up and track an array of
[steps](https://prosemirror.net/docs/ref/#transform.Step) representing a document transformation.

Most transforming methods return the `Transform` object itself, so
that they can be chained.
*/
var Transform = function Transform(
/**
The current document (the result of applying the steps in the
transform).
*/
doc) {
    this.doc = doc;
    /**
    The steps in this transform.
    */
    this.steps = [];
    /**
    The documents before each of the steps.
    */
    this.docs = [];
    /**
    A mapping with the maps for each of the steps in this transform.
    */
    this.mapping = new Mapping;
};

var prototypeAccessors$2$3 = { before: { configurable: true },docChanged: { configurable: true } };
/**
The starting document.
*/
prototypeAccessors$2$3.before.get = function () { return this.docs.length ? this.docs[0] : this.doc; };
/**
Apply a new step in this transform, saving the result. Throws an
error when the step fails.
*/
Transform.prototype.step = function step (step$1) {
    var result = this.maybeStep(step$1);
    if (result.failed)
        { throw new TransformError(result.failed); }
    return this;
};
/**
Try to apply a step in this transformation, ignoring it if it
fails. Returns the step result.
*/
Transform.prototype.maybeStep = function maybeStep (step) {
    var result = step.apply(this.doc);
    if (!result.failed)
        { this.addStep(step, result.doc); }
    return result;
};
/**
True when the document has been changed (when there are any
steps).
*/
prototypeAccessors$2$3.docChanged.get = function () {
    return this.steps.length > 0;
};
/**
@internal
*/
Transform.prototype.addStep = function addStep (step, doc) {
    this.docs.push(this.doc);
    this.steps.push(step);
    this.mapping.appendMap(step.getMap());
    this.doc = doc;
};
/**
Replace the part of the document between `from` and `to` with the
given `slice`.
*/
Transform.prototype.replace = function replace (from, to, slice) {
        if ( to === void 0 ) to = from;
        if ( slice === void 0 ) slice = Slice.empty;

    var step = replaceStep(this.doc, from, to, slice);
    if (step)
        { this.step(step); }
    return this;
};
/**
Replace the given range with the given content, which may be a
fragment, node, or array of nodes.
*/
Transform.prototype.replaceWith = function replaceWith (from, to, content) {
    return this.replace(from, to, new Slice(Fragment.from(content), 0, 0));
};
/**
Delete the content between the given positions.
*/
Transform.prototype.delete = function delete$1 (from, to) {
    return this.replace(from, to, Slice.empty);
};
/**
Insert the given content at the given position.
*/
Transform.prototype.insert = function insert (pos, content) {
    return this.replaceWith(pos, pos, content);
};
/**
Replace a range of the document with a given slice, using
`from`, `to`, and the slice's
[`openStart`](https://prosemirror.net/docs/ref/#model.Slice.openStart) property as hints, rather
than fixed start and end points. This method may grow the
replaced area or close open nodes in the slice in order to get a
fit that is more in line with WYSIWYG expectations, by dropping
fully covered parent nodes of the replaced region when they are
marked [non-defining as
context](https://prosemirror.net/docs/ref/#model.NodeSpec.definingAsContext), or including an
open parent node from the slice that _is_ marked as [defining
its content](https://prosemirror.net/docs/ref/#model.NodeSpec.definingForContent).
    
This is the method, for example, to handle paste. The similar
[`replace`](https://prosemirror.net/docs/ref/#transform.Transform.replace) method is a more
primitive tool which will _not_ move the start and end of its given
range, and is useful in situations where you need more precise
control over what happens.
*/
Transform.prototype.replaceRange = function replaceRange$1$1 (from, to, slice) {
    replaceRange$1(this, from, to, slice);
    return this;
};
/**
Replace the given range with a node, but use `from` and `to` as
hints, rather than precise positions. When from and to are the same
and are at the start or end of a parent node in which the given
node doesn't fit, this method may _move_ them out towards a parent
that does allow the given node to be placed. When the given range
completely covers a parent node, this method may completely replace
that parent node.
*/
Transform.prototype.replaceRangeWith = function replaceRangeWith$1 (from, to, node) {
    replaceRangeWith(this, from, to, node);
    return this;
};
/**
Delete the given range, expanding it to cover fully covered
parent nodes until a valid replace is found.
*/
Transform.prototype.deleteRange = function deleteRange$1 (from, to) {
    deleteRange(this, from, to);
    return this;
};
/**
Split the content in the given range off from its parent, if there
is sibling content before or after it, and move it up the tree to
the depth specified by `target`. You'll probably want to use
[`liftTarget`](https://prosemirror.net/docs/ref/#transform.liftTarget) to compute `target`, to make
sure the lift is valid.
*/
Transform.prototype.lift = function lift$1 (range, target) {
    lift$2(this, range, target);
    return this;
};
/**
Join the blocks around the given position. If depth is 2, their
last and first siblings are also joined, and so on.
*/
Transform.prototype.join = function join$1 (pos, depth) {
        if ( depth === void 0 ) depth = 1;

    join(this, pos, depth);
    return this;
};
/**
Wrap the given [range](https://prosemirror.net/docs/ref/#model.NodeRange) in the given set of wrappers.
The wrappers are assumed to be valid in this position, and should
probably be computed with [`findWrapping`](https://prosemirror.net/docs/ref/#transform.findWrapping).
*/
Transform.prototype.wrap = function wrap$1 (range, wrappers) {
    wrap(this, range, wrappers);
    return this;
};
/**
Set the type of all textblocks (partly) between `from` and `to` to
the given node type with the given attributes.
*/
Transform.prototype.setBlockType = function setBlockType$1$1 (from, to, type, attrs) {
        if ( to === void 0 ) to = from;
        if ( attrs === void 0 ) attrs = null;

    setBlockType$1(this, from, to, type, attrs);
    return this;
};
/**
Change the type, attributes, and/or marks of the node at `pos`.
When `type` isn't given, the existing node type is preserved,
*/
Transform.prototype.setNodeMarkup = function setNodeMarkup$1 (pos, type, attrs, marks) {
        if ( attrs === void 0 ) attrs = null;

    setNodeMarkup(this, pos, type, attrs, marks);
    return this;
};
/**
Set a single attribute on a given node to a new value.
The `pos` addresses the document content. Use `setDocAttribute`
to set attributes on the document itself.
*/
Transform.prototype.setNodeAttribute = function setNodeAttribute (pos, attr, value) {
    this.step(new AttrStep(pos, attr, value));
    return this;
};
/**
Set a single attribute on the document to a new value.
*/
Transform.prototype.setDocAttribute = function setDocAttribute (attr, value) {
    this.step(new DocAttrStep(attr, value));
    return this;
};
/**
Add a mark to the node at position `pos`.
*/
Transform.prototype.addNodeMark = function addNodeMark (pos, mark) {
    this.step(new AddNodeMarkStep(pos, mark));
    return this;
};
/**
Remove a mark (or a mark of the given type) from the node at
position `pos`.
*/
Transform.prototype.removeNodeMark = function removeNodeMark (pos, mark) {
    if (!(mark instanceof Mark)) {
        var node = this.doc.nodeAt(pos);
        if (!node)
            { throw new RangeError("No node at position " + pos); }
        mark = mark.isInSet(node.marks);
        if (!mark)
            { return this; }
    }
    this.step(new RemoveNodeMarkStep(pos, mark));
    return this;
};
/**
Split the node at the given position, and optionally, if `depth` is
greater than one, any number of nodes above that. By default, the
parts split off will inherit the node type of the original node.
This can be changed by passing an array of types and attributes to
use after the split.
*/
Transform.prototype.split = function split$1 (pos, depth, typesAfter) {
        if ( depth === void 0 ) depth = 1;

    split(this, pos, depth, typesAfter);
    return this;
};
/**
Add the given mark to the inline content between `from` and `to`.
*/
Transform.prototype.addMark = function addMark$1 (from, to, mark) {
    addMark(this, from, to, mark);
    return this;
};
/**
Remove marks from inline nodes between `from` and `to`. When
`mark` is a single mark, remove precisely that mark. When it is
a mark type, remove all marks of that type. When it is null,
remove all marks of any type.
*/
Transform.prototype.removeMark = function removeMark$1$1 (from, to, mark) {
    removeMark$1(this, from, to, mark);
    return this;
};
/**
Removes all marks and nodes from the content of the node at
`pos` that don't match the given new parent node type. Accepts
an optional starting [content match](https://prosemirror.net/docs/ref/#model.ContentMatch) as
third argument.
*/
Transform.prototype.clearIncompatible = function clearIncompatible$1 (pos, parentType, match) {
    clearIncompatible(this, pos, parentType, match);
    return this;
};

Object.defineProperties( Transform.prototype, prototypeAccessors$2$3 );var transform=/*#__PURE__*/Object.freeze({__proto__:null,AddMarkStep:AddMarkStep,AddNodeMarkStep:AddNodeMarkStep,AttrStep:AttrStep,DocAttrStep:DocAttrStep,MapResult:MapResult,Mapping:Mapping,RemoveMarkStep:RemoveMarkStep,RemoveNodeMarkStep:RemoveNodeMarkStep,ReplaceAroundStep:ReplaceAroundStep,ReplaceStep:ReplaceStep,Step:Step,StepMap:StepMap,StepResult:StepResult,Transform:Transform,get TransformError(){return TransformError},canJoin:canJoin,canSplit:canSplit,dropPoint:dropPoint,findWrapping:findWrapping,insertPoint:insertPoint,joinPoint:joinPoint,liftTarget:liftTarget,replaceStep:replaceStep});var classesById = Object.create(null);
/**
Superclass for editor selections. Every selection type should
extend this. Should not be instantiated directly.
*/
var Selection = function Selection(
/**
The resolved anchor of the selection (the side that stays in
place when the selection is modified).
*/
$anchor, 
/**
The resolved head of the selection (the side that moves when
the selection is modified).
*/
$head, ranges) {
    this.$anchor = $anchor;
    this.$head = $head;
    this.ranges = ranges || [new SelectionRange$1($anchor.min($head), $anchor.max($head))];
};

var prototypeAccessors$a = { anchor: { configurable: true },head: { configurable: true },from: { configurable: true },to: { configurable: true },$from: { configurable: true },$to: { configurable: true },empty: { configurable: true } };
/**
The selection's anchor, as an unresolved position.
*/
prototypeAccessors$a.anchor.get = function () { return this.$anchor.pos; };
/**
The selection's head.
*/
prototypeAccessors$a.head.get = function () { return this.$head.pos; };
/**
The lower bound of the selection's main range.
*/
prototypeAccessors$a.from.get = function () { return this.$from.pos; };
/**
The upper bound of the selection's main range.
*/
prototypeAccessors$a.to.get = function () { return this.$to.pos; };
/**
The resolved lower  bound of the selection's main range.
*/
prototypeAccessors$a.$from.get = function () {
    return this.ranges[0].$from;
};
/**
The resolved upper bound of the selection's main range.
*/
prototypeAccessors$a.$to.get = function () {
    return this.ranges[0].$to;
};
/**
Indicates whether the selection contains any content.
*/
prototypeAccessors$a.empty.get = function () {
    var ranges = this.ranges;
    for (var i = 0; i < ranges.length; i++)
        { if (ranges[i].$from.pos != ranges[i].$to.pos)
            { return false; } }
    return true;
};
/**
Get the content of this selection as a slice.
*/
Selection.prototype.content = function content () {
    return this.$from.doc.slice(this.from, this.to, true);
};
/**
Replace the selection with a slice or, if no slice is given,
delete the selection. Will append to the given transaction.
*/
Selection.prototype.replace = function replace (tr, content) {
        if ( content === void 0 ) content = Slice.empty;

    // Put the new selection at the position after the inserted
    // content. When that ended in an inline node, search backwards,
    // to get the position after that node. If not, search forward.
    var lastNode = content.content.lastChild, lastParent = null;
    for (var i = 0; i < content.openEnd; i++) {
        lastParent = lastNode;
        lastNode = lastNode.lastChild;
    }
    var mapFrom = tr.steps.length, ranges = this.ranges;
    for (var i$1 = 0; i$1 < ranges.length; i$1++) {
        var ref = ranges[i$1];
            var $from = ref.$from;
            var $to = ref.$to;
            var mapping = tr.mapping.slice(mapFrom);
        tr.replaceRange(mapping.map($from.pos), mapping.map($to.pos), i$1 ? Slice.empty : content);
        if (i$1 == 0)
            { selectionToInsertionEnd(tr, mapFrom, (lastNode ? lastNode.isInline : lastParent && lastParent.isTextblock) ? -1 : 1); }
    }
};
/**
Replace the selection with the given node, appending the changes
to the given transaction.
*/
Selection.prototype.replaceWith = function replaceWith (tr, node) {
    var mapFrom = tr.steps.length, ranges = this.ranges;
    for (var i = 0; i < ranges.length; i++) {
        var ref = ranges[i];
            var $from = ref.$from;
            var $to = ref.$to;
            var mapping = tr.mapping.slice(mapFrom);
        var from = mapping.map($from.pos), to = mapping.map($to.pos);
        if (i) {
            tr.deleteRange(from, to);
        }
        else {
            tr.replaceRangeWith(from, to, node);
            selectionToInsertionEnd(tr, mapFrom, node.isInline ? -1 : 1);
        }
    }
};
/**
Find a valid cursor or leaf node selection starting at the given
position and searching back if `dir` is negative, and forward if
positive. When `textOnly` is true, only consider cursor
selections. Will return null when no valid selection position is
found.
*/
Selection.findFrom = function findFrom ($pos, dir, textOnly) {
        if ( textOnly === void 0 ) textOnly = false;

    var inner = $pos.parent.inlineContent ? new TextSelection($pos)
        : findSelectionIn($pos.node(0), $pos.parent, $pos.pos, $pos.index(), dir, textOnly);
    if (inner)
        { return inner; }
    for (var depth = $pos.depth - 1; depth >= 0; depth--) {
        var found = dir < 0
            ? findSelectionIn($pos.node(0), $pos.node(depth), $pos.before(depth + 1), $pos.index(depth), dir, textOnly)
            : findSelectionIn($pos.node(0), $pos.node(depth), $pos.after(depth + 1), $pos.index(depth) + 1, dir, textOnly);
        if (found)
            { return found; }
    }
    return null;
};
/**
Find a valid cursor or leaf node selection near the given
position. Searches forward first by default, but if `bias` is
negative, it will search backwards first.
*/
Selection.near = function near ($pos, bias) {
        if ( bias === void 0 ) bias = 1;

    return this.findFrom($pos, bias) || this.findFrom($pos, -bias) || new AllSelection($pos.node(0));
};
/**
Find the cursor or leaf node selection closest to the start of
the given document. Will return an
[`AllSelection`](https://prosemirror.net/docs/ref/#state.AllSelection) if no valid position
exists.
*/
Selection.atStart = function atStart (doc) {
    return findSelectionIn(doc, doc, 0, 0, 1) || new AllSelection(doc);
};
/**
Find the cursor or leaf node selection closest to the end of the
given document.
*/
Selection.atEnd = function atEnd (doc) {
    return findSelectionIn(doc, doc, doc.content.size, doc.childCount, -1) || new AllSelection(doc);
};
/**
Deserialize the JSON representation of a selection. Must be
implemented for custom classes (as a static class method).
*/
Selection.fromJSON = function fromJSON (doc, json) {
    if (!json || !json.type)
        { throw new RangeError("Invalid input for Selection.fromJSON"); }
    var cls = classesById[json.type];
    if (!cls)
        { throw new RangeError(("No selection type " + (json.type) + " defined")); }
    return cls.fromJSON(doc, json);
};
/**
To be able to deserialize selections from JSON, custom selection
classes must register themselves with an ID string, so that they
can be disambiguated. Try to pick something that's unlikely to
clash with classes from other modules.
*/
Selection.jsonID = function jsonID (id, selectionClass) {
    if (id in classesById)
        { throw new RangeError("Duplicate use of selection JSON ID " + id); }
    classesById[id] = selectionClass;
    selectionClass.prototype.jsonID = id;
    return selectionClass;
};
/**
Get a [bookmark](https://prosemirror.net/docs/ref/#state.SelectionBookmark) for this selection,
which is a value that can be mapped without having access to a
current document, and later resolved to a real selection for a
given document again. (This is used mostly by the history to
track and restore old selections.) The default implementation of
this method just converts the selection to a text selection and
returns the bookmark for that.
*/
Selection.prototype.getBookmark = function getBookmark () {
    return TextSelection.between(this.$anchor, this.$head).getBookmark();
};

Object.defineProperties( Selection.prototype, prototypeAccessors$a );
Selection.prototype.visible = true;
/**
Represents a selected range in a document.
*/
var SelectionRange$1 = function SelectionRange(
/**
The lower bound of the range.
*/
$from, 
/**
The upper bound of the range.
*/
$to) {
    this.$from = $from;
    this.$to = $to;
};
var warnedAboutTextSelection = false;
function checkTextSelection($pos) {
    if (!warnedAboutTextSelection && !$pos.parent.inlineContent) {
        warnedAboutTextSelection = true;
        console["warn"]("TextSelection endpoint not pointing into a node with inline content (" + $pos.parent.type.name + ")");
    }
}
/**
A text selection represents a classical editor selection, with a
head (the moving side) and anchor (immobile side), both of which
point into textblock nodes. It can be empty (a regular cursor
position).
*/
var TextSelection = /*@__PURE__*/(function (Selection) {
    function TextSelection($anchor, $head) {
        if ( $head === void 0 ) $head = $anchor;

        checkTextSelection($anchor);
        checkTextSelection($head);
        Selection.call(this, $anchor, $head);
    }

    if ( Selection ) TextSelection.__proto__ = Selection;
    TextSelection.prototype = Object.create( Selection && Selection.prototype );
    TextSelection.prototype.constructor = TextSelection;

    var prototypeAccessors$1 = { $cursor: { configurable: true } };
    /**
    Returns a resolved position if this is a cursor selection (an
    empty text selection), and null otherwise.
    */
    prototypeAccessors$1.$cursor.get = function () { return this.$anchor.pos == this.$head.pos ? this.$head : null; };
    TextSelection.prototype.map = function map (doc, mapping) {
        var $head = doc.resolve(mapping.map(this.head));
        if (!$head.parent.inlineContent)
            { return Selection.near($head); }
        var $anchor = doc.resolve(mapping.map(this.anchor));
        return new TextSelection($anchor.parent.inlineContent ? $anchor : $head, $head);
    };
    TextSelection.prototype.replace = function replace (tr, content) {
        if ( content === void 0 ) content = Slice.empty;

        Selection.prototype.replace.call(this, tr, content);
        if (content == Slice.empty) {
            var marks = this.$from.marksAcross(this.$to);
            if (marks)
                { tr.ensureMarks(marks); }
        }
    };
    TextSelection.prototype.eq = function eq (other) {
        return other instanceof TextSelection && other.anchor == this.anchor && other.head == this.head;
    };
    TextSelection.prototype.getBookmark = function getBookmark () {
        return new TextBookmark(this.anchor, this.head);
    };
    TextSelection.prototype.toJSON = function toJSON () {
        return { type: "text", anchor: this.anchor, head: this.head };
    };
    /**
    @internal
    */
    TextSelection.fromJSON = function fromJSON (doc, json) {
        if (typeof json.anchor != "number" || typeof json.head != "number")
            { throw new RangeError("Invalid input for TextSelection.fromJSON"); }
        return new TextSelection(doc.resolve(json.anchor), doc.resolve(json.head));
    };
    /**
    Create a text selection from non-resolved positions.
    */
    TextSelection.create = function create (doc, anchor, head) {
        if ( head === void 0 ) head = anchor;

        var $anchor = doc.resolve(anchor);
        return new this($anchor, head == anchor ? $anchor : doc.resolve(head));
    };
    /**
    Return a text selection that spans the given positions or, if
    they aren't text positions, find a text selection near them.
    `bias` determines whether the method searches forward (default)
    or backwards (negative number) first. Will fall back to calling
    [`Selection.near`](https://prosemirror.net/docs/ref/#state.Selection^near) when the document
    doesn't contain a valid text position.
    */
    TextSelection.between = function between ($anchor, $head, bias) {
        var dPos = $anchor.pos - $head.pos;
        if (!bias || dPos)
            { bias = dPos >= 0 ? 1 : -1; }
        if (!$head.parent.inlineContent) {
            var found = Selection.findFrom($head, bias, true) || Selection.findFrom($head, -bias, true);
            if (found)
                { $head = found.$head; }
            else
                { return Selection.near($head, bias); }
        }
        if (!$anchor.parent.inlineContent) {
            if (dPos == 0) {
                $anchor = $head;
            }
            else {
                $anchor = (Selection.findFrom($anchor, -bias, true) || Selection.findFrom($anchor, bias, true)).$anchor;
                if (($anchor.pos < $head.pos) != (dPos < 0))
                    { $anchor = $head; }
            }
        }
        return new TextSelection($anchor, $head);
    };

    Object.defineProperties( TextSelection.prototype, prototypeAccessors$1 );

    return TextSelection;
}(Selection));
Selection.jsonID("text", TextSelection);
var TextBookmark = function TextBookmark(anchor, head) {
    this.anchor = anchor;
    this.head = head;
};
TextBookmark.prototype.map = function map (mapping) {
    return new TextBookmark(mapping.map(this.anchor), mapping.map(this.head));
};
TextBookmark.prototype.resolve = function resolve (doc) {
    return TextSelection.between(doc.resolve(this.anchor), doc.resolve(this.head));
};
/**
A node selection is a selection that points at a single node. All
nodes marked [selectable](https://prosemirror.net/docs/ref/#model.NodeSpec.selectable) can be the
target of a node selection. In such a selection, `from` and `to`
point directly before and after the selected node, `anchor` equals
`from`, and `head` equals `to`..
*/
var NodeSelection = /*@__PURE__*/(function (Selection) {
    function NodeSelection($pos) {
        var node = $pos.nodeAfter;
        var $end = $pos.node(0).resolve($pos.pos + node.nodeSize);
        Selection.call(this, $pos, $end);
        this.node = node;
    }

    if ( Selection ) NodeSelection.__proto__ = Selection;
    NodeSelection.prototype = Object.create( Selection && Selection.prototype );
    NodeSelection.prototype.constructor = NodeSelection;
    NodeSelection.prototype.map = function map (doc, mapping) {
        var ref = mapping.mapResult(this.anchor);
        var deleted = ref.deleted;
        var pos = ref.pos;
        var $pos = doc.resolve(pos);
        if (deleted)
            { return Selection.near($pos); }
        return new NodeSelection($pos);
    };
    NodeSelection.prototype.content = function content () {
        return new Slice(Fragment.from(this.node), 0, 0);
    };
    NodeSelection.prototype.eq = function eq (other) {
        return other instanceof NodeSelection && other.anchor == this.anchor;
    };
    NodeSelection.prototype.toJSON = function toJSON () {
        return { type: "node", anchor: this.anchor };
    };
    NodeSelection.prototype.getBookmark = function getBookmark () { return new NodeBookmark(this.anchor); };
    /**
    @internal
    */
    NodeSelection.fromJSON = function fromJSON (doc, json) {
        if (typeof json.anchor != "number")
            { throw new RangeError("Invalid input for NodeSelection.fromJSON"); }
        return new NodeSelection(doc.resolve(json.anchor));
    };
    /**
    Create a node selection from non-resolved positions.
    */
    NodeSelection.create = function create (doc, from) {
        return new NodeSelection(doc.resolve(from));
    };
    /**
    Determines whether the given node may be selected as a node
    selection.
    */
    NodeSelection.isSelectable = function isSelectable (node) {
        return !node.isText && node.type.spec.selectable !== false;
    };

    return NodeSelection;
}(Selection));
NodeSelection.prototype.visible = false;
Selection.jsonID("node", NodeSelection);
var NodeBookmark = function NodeBookmark(anchor) {
    this.anchor = anchor;
};
NodeBookmark.prototype.map = function map (mapping) {
    var ref = mapping.mapResult(this.anchor);
        var deleted = ref.deleted;
        var pos = ref.pos;
    return deleted ? new TextBookmark(pos, pos) : new NodeBookmark(pos);
};
NodeBookmark.prototype.resolve = function resolve (doc) {
    var $pos = doc.resolve(this.anchor), node = $pos.nodeAfter;
    if (node && NodeSelection.isSelectable(node))
        { return new NodeSelection($pos); }
    return Selection.near($pos);
};
/**
A selection type that represents selecting the whole document
(which can not necessarily be expressed with a text selection, when
there are for example leaf block nodes at the start or end of the
document).
*/
var AllSelection = /*@__PURE__*/(function (Selection) {
    function AllSelection(doc) {
        Selection.call(this, doc.resolve(0), doc.resolve(doc.content.size));
    }

    if ( Selection ) AllSelection.__proto__ = Selection;
    AllSelection.prototype = Object.create( Selection && Selection.prototype );
    AllSelection.prototype.constructor = AllSelection;
    AllSelection.prototype.replace = function replace (tr, content) {
        if ( content === void 0 ) content = Slice.empty;

        if (content == Slice.empty) {
            tr.delete(0, tr.doc.content.size);
            var sel = Selection.atStart(tr.doc);
            if (!sel.eq(tr.selection))
                { tr.setSelection(sel); }
        }
        else {
            Selection.prototype.replace.call(this, tr, content);
        }
    };
    AllSelection.prototype.toJSON = function toJSON () { return { type: "all" }; };
    /**
    @internal
    */
    AllSelection.fromJSON = function fromJSON (doc) { return new AllSelection(doc); };
    AllSelection.prototype.map = function map (doc) { return new AllSelection(doc); };
    AllSelection.prototype.eq = function eq (other) { return other instanceof AllSelection; };
    AllSelection.prototype.getBookmark = function getBookmark () { return AllBookmark; };

    return AllSelection;
}(Selection));
Selection.jsonID("all", AllSelection);
var AllBookmark = {
    map: function map() { return this; },
    resolve: function resolve(doc) { return new AllSelection(doc); }
};
// FIXME we'll need some awareness of text direction when scanning for selections
// Try to find a selection inside the given node. `pos` points at the
// position where the search starts. When `text` is true, only return
// text selections.
function findSelectionIn(doc, node, pos, index, dir, text) {
    if ( text === void 0 ) text = false;

    if (node.inlineContent)
        { return TextSelection.create(doc, pos); }
    for (var i = index - (dir > 0 ? 0 : 1); dir > 0 ? i < node.childCount : i >= 0; i += dir) {
        var child = node.child(i);
        if (!child.isAtom) {
            var inner = findSelectionIn(doc, child, pos + dir, dir < 0 ? child.childCount : 0, dir, text);
            if (inner)
                { return inner; }
        }
        else if (!text && NodeSelection.isSelectable(child)) {
            return NodeSelection.create(doc, pos - (dir < 0 ? child.nodeSize : 0));
        }
        pos += child.nodeSize * dir;
    }
    return null;
}
function selectionToInsertionEnd(tr, startLen, bias) {
    var last = tr.steps.length - 1;
    if (last < startLen)
        { return; }
    var step = tr.steps[last];
    if (!(step instanceof ReplaceStep || step instanceof ReplaceAroundStep))
        { return; }
    var map = tr.mapping.maps[last], end;
    map.forEach(function (_from, _to, _newFrom, newTo) { if (end == null)
        { end = newTo; } });
    tr.setSelection(Selection.near(tr.doc.resolve(end), bias));
}

var UPDATED_SEL = 1, UPDATED_MARKS = 2, UPDATED_SCROLL = 4;
/**
An editor state transaction, which can be applied to a state to
create an updated state. Use
[`EditorState.tr`](https://prosemirror.net/docs/ref/#state.EditorState.tr) to create an instance.

Transactions track changes to the document (they are a subclass of
[`Transform`](https://prosemirror.net/docs/ref/#transform.Transform)), but also other state changes,
like selection updates and adjustments of the set of [stored
marks](https://prosemirror.net/docs/ref/#state.EditorState.storedMarks). In addition, you can store
metadata properties in a transaction, which are extra pieces of
information that client code or plugins can use to describe what a
transaction represents, so that they can update their [own
state](https://prosemirror.net/docs/ref/#state.StateField) accordingly.

The [editor view](https://prosemirror.net/docs/ref/#view.EditorView) uses a few metadata
properties: it will attach a property `"pointer"` with the value
`true` to selection transactions directly caused by mouse or touch
input, a `"composition"` property holding an ID identifying the
composition that caused it to transactions caused by composed DOM
input, and a `"uiEvent"` property of that may be `"paste"`,
`"cut"`, or `"drop"`.
*/
var Transaction$1 = /*@__PURE__*/(function (Transform) {
    function Transaction(state) {
        Transform.call(this, state.doc);
        // The step count for which the current selection is valid.
        this.curSelectionFor = 0;
        // Bitfield to track which aspects of the state were updated by
        // this transaction.
        this.updated = 0;
        // Object used to store metadata properties for the transaction.
        this.meta = Object.create(null);
        this.time = Date.now();
        this.curSelection = state.selection;
        this.storedMarks = state.storedMarks;
    }

    if ( Transform ) Transaction.__proto__ = Transform;
    Transaction.prototype = Object.create( Transform && Transform.prototype );
    Transaction.prototype.constructor = Transaction;

    var prototypeAccessors$2 = { selection: { configurable: true },selectionSet: { configurable: true },storedMarksSet: { configurable: true },isGeneric: { configurable: true },scrolledIntoView: { configurable: true } };
    /**
    The transaction's current selection. This defaults to the editor
    selection [mapped](https://prosemirror.net/docs/ref/#state.Selection.map) through the steps in the
    transaction, but can be overwritten with
    [`setSelection`](https://prosemirror.net/docs/ref/#state.Transaction.setSelection).
    */
    prototypeAccessors$2.selection.get = function () {
        if (this.curSelectionFor < this.steps.length) {
            this.curSelection = this.curSelection.map(this.doc, this.mapping.slice(this.curSelectionFor));
            this.curSelectionFor = this.steps.length;
        }
        return this.curSelection;
    };
    /**
    Update the transaction's current selection. Will determine the
    selection that the editor gets when the transaction is applied.
    */
    Transaction.prototype.setSelection = function setSelection (selection) {
        if (selection.$from.doc != this.doc)
            { throw new RangeError("Selection passed to setSelection must point at the current document"); }
        this.curSelection = selection;
        this.curSelectionFor = this.steps.length;
        this.updated = (this.updated | UPDATED_SEL) & ~UPDATED_MARKS;
        this.storedMarks = null;
        return this;
    };
    /**
    Whether the selection was explicitly updated by this transaction.
    */
    prototypeAccessors$2.selectionSet.get = function () {
        return (this.updated & UPDATED_SEL) > 0;
    };
    /**
    Set the current stored marks.
    */
    Transaction.prototype.setStoredMarks = function setStoredMarks (marks) {
        this.storedMarks = marks;
        this.updated |= UPDATED_MARKS;
        return this;
    };
    /**
    Make sure the current stored marks or, if that is null, the marks
    at the selection, match the given set of marks. Does nothing if
    this is already the case.
    */
    Transaction.prototype.ensureMarks = function ensureMarks (marks) {
        if (!Mark.sameSet(this.storedMarks || this.selection.$from.marks(), marks))
            { this.setStoredMarks(marks); }
        return this;
    };
    /**
    Add a mark to the set of stored marks.
    */
    Transaction.prototype.addStoredMark = function addStoredMark (mark) {
        return this.ensureMarks(mark.addToSet(this.storedMarks || this.selection.$head.marks()));
    };
    /**
    Remove a mark or mark type from the set of stored marks.
    */
    Transaction.prototype.removeStoredMark = function removeStoredMark (mark) {
        return this.ensureMarks(mark.removeFromSet(this.storedMarks || this.selection.$head.marks()));
    };
    /**
    Whether the stored marks were explicitly set for this transaction.
    */
    prototypeAccessors$2.storedMarksSet.get = function () {
        return (this.updated & UPDATED_MARKS) > 0;
    };
    /**
    @internal
    */
    Transaction.prototype.addStep = function addStep (step, doc) {
        Transform.prototype.addStep.call(this, step, doc);
        this.updated = this.updated & ~UPDATED_MARKS;
        this.storedMarks = null;
    };
    /**
    Update the timestamp for the transaction.
    */
    Transaction.prototype.setTime = function setTime (time) {
        this.time = time;
        return this;
    };
    /**
    Replace the current selection with the given slice.
    */
    Transaction.prototype.replaceSelection = function replaceSelection (slice) {
        this.selection.replace(this, slice);
        return this;
    };
    /**
    Replace the selection with the given node. When `inheritMarks` is
    true and the content is inline, it inherits the marks from the
    place where it is inserted.
    */
    Transaction.prototype.replaceSelectionWith = function replaceSelectionWith (node, inheritMarks) {
        if ( inheritMarks === void 0 ) inheritMarks = true;

        var selection = this.selection;
        if (inheritMarks)
            { node = node.mark(this.storedMarks || (selection.empty ? selection.$from.marks() : (selection.$from.marksAcross(selection.$to) || Mark.none))); }
        selection.replaceWith(this, node);
        return this;
    };
    /**
    Delete the selection.
    */
    Transaction.prototype.deleteSelection = function deleteSelection () {
        this.selection.replace(this);
        return this;
    };
    /**
    Replace the given range, or the selection if no range is given,
    with a text node containing the given string.
    */
    Transaction.prototype.insertText = function insertText (text, from, to) {
        var schema = this.doc.type.schema;
        if (from == null) {
            if (!text)
                { return this.deleteSelection(); }
            return this.replaceSelectionWith(schema.text(text), true);
        }
        else {
            if (to == null)
                { to = from; }
            to = to == null ? from : to;
            if (!text)
                { return this.deleteRange(from, to); }
            var marks = this.storedMarks;
            if (!marks) {
                var $from = this.doc.resolve(from);
                marks = to == from ? $from.marks() : $from.marksAcross(this.doc.resolve(to));
            }
            this.replaceRangeWith(from, to, schema.text(text, marks));
            if (!this.selection.empty)
                { this.setSelection(Selection.near(this.selection.$to)); }
            return this;
        }
    };
    /**
    Store a metadata property in this transaction, keyed either by
    name or by plugin.
    */
    Transaction.prototype.setMeta = function setMeta (key, value) {
        this.meta[typeof key == "string" ? key : key.key] = value;
        return this;
    };
    /**
    Retrieve a metadata property for a given name or plugin.
    */
    Transaction.prototype.getMeta = function getMeta (key) {
        return this.meta[typeof key == "string" ? key : key.key];
    };
    /**
    Returns true if this transaction doesn't contain any metadata,
    and can thus safely be extended.
    */
    prototypeAccessors$2.isGeneric.get = function () {
        for (var _ in this.meta)
            { return false; }
        return true;
    };
    /**
    Indicate that the editor should scroll the selection into view
    when updated to the state produced by this transaction.
    */
    Transaction.prototype.scrollIntoView = function scrollIntoView () {
        this.updated |= UPDATED_SCROLL;
        return this;
    };
    /**
    True when this transaction has had `scrollIntoView` called on it.
    */
    prototypeAccessors$2.scrolledIntoView.get = function () {
        return (this.updated & UPDATED_SCROLL) > 0;
    };

    Object.defineProperties( Transaction.prototype, prototypeAccessors$2 );

    return Transaction;
}(Transform));

function bind(f, self) {
    return !self || !f ? f : f.bind(self);
}
var FieldDesc = function FieldDesc(name, desc, self) {
    this.name = name;
    this.init = bind(desc.init, self);
    this.apply = bind(desc.apply, self);
};
var baseFields = [
    new FieldDesc("doc", {
        init: function init(config) { return config.doc || config.schema.topNodeType.createAndFill(); },
        apply: function apply(tr) { return tr.doc; }
    }),
    new FieldDesc("selection", {
        init: function init(config, instance) { return config.selection || Selection.atStart(instance.doc); },
        apply: function apply(tr) { return tr.selection; }
    }),
    new FieldDesc("storedMarks", {
        init: function init(config) { return config.storedMarks || null; },
        apply: function apply(tr, _marks, _old, state) { return state.selection.$cursor ? tr.storedMarks : null; }
    }),
    new FieldDesc("scrollToSelection", {
        init: function init() { return 0; },
        apply: function apply(tr, prev) { return tr.scrolledIntoView ? prev + 1 : prev; }
    })
];
// Object wrapping the part of a state object that stays the same
// across transactions. Stored in the state's `config` property.
var Configuration$1 = function Configuration(schema, plugins) {
    var this$1$1 = this;

    this.schema = schema;
    this.plugins = [];
    this.pluginsByKey = Object.create(null);
    this.fields = baseFields.slice();
    if (plugins)
        { plugins.forEach(function (plugin) {
            if (this$1$1.pluginsByKey[plugin.key])
                { throw new RangeError("Adding different instances of a keyed plugin (" + plugin.key + ")"); }
            this$1$1.plugins.push(plugin);
            this$1$1.pluginsByKey[plugin.key] = plugin;
            if (plugin.spec.state)
                { this$1$1.fields.push(new FieldDesc(plugin.key, plugin.spec.state, plugin)); }
        }); }
};
/**
The state of a ProseMirror editor is represented by an object of
this type. A state is a persistent data structure—it isn't
updated, but rather a new state value is computed from an old one
using the [`apply`](https://prosemirror.net/docs/ref/#state.EditorState.apply) method.

A state holds a number of built-in fields, and plugins can
[define](https://prosemirror.net/docs/ref/#state.PluginSpec.state) additional fields.
*/
var EditorState$1 = function EditorState(
/**
@internal
*/
config) {
    this.config = config;
};

var prototypeAccessors$3$2 = { schema: { configurable: true },plugins: { configurable: true },tr: { configurable: true } };
/**
The schema of the state's document.
*/
prototypeAccessors$3$2.schema.get = function () {
    return this.config.schema;
};
/**
The plugins that are active in this state.
*/
prototypeAccessors$3$2.plugins.get = function () {
    return this.config.plugins;
};
/**
Apply the given transaction to produce a new state.
*/
EditorState$1.prototype.apply = function apply (tr) {
    return this.applyTransaction(tr).state;
};
/**
@internal
*/
EditorState$1.prototype.filterTransaction = function filterTransaction (tr, ignore) {
        if ( ignore === void 0 ) ignore = -1;

    for (var i = 0; i < this.config.plugins.length; i++)
        { if (i != ignore) {
            var plugin = this.config.plugins[i];
            if (plugin.spec.filterTransaction && !plugin.spec.filterTransaction.call(plugin, tr, this))
                { return false; }
        } }
    return true;
};
/**
Verbose variant of [`apply`](https://prosemirror.net/docs/ref/#state.EditorState.apply) that
returns the precise transactions that were applied (which might
be influenced by the [transaction
hooks](https://prosemirror.net/docs/ref/#state.PluginSpec.filterTransaction) of
plugins) along with the new state.
*/
EditorState$1.prototype.applyTransaction = function applyTransaction (rootTr) {
    if (!this.filterTransaction(rootTr))
        { return { state: this, transactions: [] }; }
    var trs = [rootTr], newState = this.applyInner(rootTr), seen = null;
    // This loop repeatedly gives plugins a chance to respond to
    // transactions as new transactions are added, making sure to only
    // pass the transactions the plugin did not see before.
    for (;;) {
        var haveNew = false;
        for (var i = 0; i < this.config.plugins.length; i++) {
            var plugin = this.config.plugins[i];
            if (plugin.spec.appendTransaction) {
                var n = seen ? seen[i].n : 0, oldState = seen ? seen[i].state : this;
                var tr = n < trs.length &&
                    plugin.spec.appendTransaction.call(plugin, n ? trs.slice(n) : trs, oldState, newState);
                if (tr && newState.filterTransaction(tr, i)) {
                    tr.setMeta("appendedTransaction", rootTr);
                    if (!seen) {
                        seen = [];
                        for (var j = 0; j < this.config.plugins.length; j++)
                            { seen.push(j < i ? { state: newState, n: trs.length } : { state: this, n: 0 }); }
                    }
                    trs.push(tr);
                    newState = newState.applyInner(tr);
                    haveNew = true;
                }
                if (seen)
                    { seen[i] = { state: newState, n: trs.length }; }
            }
        }
        if (!haveNew)
            { return { state: newState, transactions: trs }; }
    }
};
/**
@internal
*/
EditorState$1.prototype.applyInner = function applyInner (tr) {
    if (!tr.before.eq(this.doc))
        { throw new RangeError("Applying a mismatched transaction"); }
    var newInstance = new EditorState$1(this.config), fields = this.config.fields;
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i];
        newInstance[field.name] = field.apply(tr, this[field.name], this, newInstance);
    }
    return newInstance;
};
/**
Start a [transaction](https://prosemirror.net/docs/ref/#state.Transaction) from this state.
*/
prototypeAccessors$3$2.tr.get = function () { return new Transaction$1(this); };
/**
Create a new state.
*/
EditorState$1.create = function create (config) {
    var $config = new Configuration$1(config.doc ? config.doc.type.schema : config.schema, config.plugins);
    var instance = new EditorState$1($config);
    for (var i = 0; i < $config.fields.length; i++)
        { instance[$config.fields[i].name] = $config.fields[i].init(config, instance); }
    return instance;
};
/**
Create a new state based on this one, but with an adjusted set
of active plugins. State fields that exist in both sets of
plugins are kept unchanged. Those that no longer exist are
dropped, and those that are new are initialized using their
[`init`](https://prosemirror.net/docs/ref/#state.StateField.init) method, passing in the new
configuration object..
*/
EditorState$1.prototype.reconfigure = function reconfigure (config) {
    var $config = new Configuration$1(this.schema, config.plugins);
    var fields = $config.fields, instance = new EditorState$1($config);
    for (var i = 0; i < fields.length; i++) {
        var name = fields[i].name;
        instance[name] = this.hasOwnProperty(name) ? this[name] : fields[i].init(config, instance);
    }
    return instance;
};
/**
Serialize this state to JSON. If you want to serialize the state
of plugins, pass an object mapping property names to use in the
resulting JSON object to plugin objects. The argument may also be
a string or number, in which case it is ignored, to support the
way `JSON.stringify` calls `toString` methods.
*/
EditorState$1.prototype.toJSON = function toJSON (pluginFields) {
    var result = { doc: this.doc.toJSON(), selection: this.selection.toJSON() };
    if (this.storedMarks)
        { result.storedMarks = this.storedMarks.map(function (m) { return m.toJSON(); }); }
    if (pluginFields && typeof pluginFields == 'object')
        { for (var prop in pluginFields) {
            if (prop == "doc" || prop == "selection")
                { throw new RangeError("The JSON fields `doc` and `selection` are reserved"); }
            var plugin = pluginFields[prop], state = plugin.spec.state;
            if (state && state.toJSON)
                { result[prop] = state.toJSON.call(plugin, this[plugin.key]); }
        } }
    return result;
};
/**
Deserialize a JSON representation of a state. `config` should
have at least a `schema` field, and should contain array of
plugins to initialize the state with. `pluginFields` can be used
to deserialize the state of plugins, by associating plugin
instances with the property names they use in the JSON object.
*/
EditorState$1.fromJSON = function fromJSON (config, json, pluginFields) {
    if (!json)
        { throw new RangeError("Invalid input for EditorState.fromJSON"); }
    if (!config.schema)
        { throw new RangeError("Required config field 'schema' missing"); }
    var $config = new Configuration$1(config.schema, config.plugins);
    var instance = new EditorState$1($config);
    $config.fields.forEach(function (field) {
        if (field.name == "doc") {
            instance.doc = Node$1.fromJSON(config.schema, json.doc);
        }
        else if (field.name == "selection") {
            instance.selection = Selection.fromJSON(instance.doc, json.selection);
        }
        else if (field.name == "storedMarks") {
            if (json.storedMarks)
                { instance.storedMarks = json.storedMarks.map(config.schema.markFromJSON); }
        }
        else {
            if (pluginFields)
                { for (var prop in pluginFields) {
                    var plugin = pluginFields[prop], state = plugin.spec.state;
                    if (plugin.key == field.name && state && state.fromJSON &&
                        Object.prototype.hasOwnProperty.call(json, prop)) {
                        instance[field.name] = state.fromJSON.call(plugin, config, json[prop], instance);
                        return;
                    }
                } }
            instance[field.name] = field.init(config, instance);
        }
    });
    return instance;
};

Object.defineProperties( EditorState$1.prototype, prototypeAccessors$3$2 );

function bindProps(obj, self, target) {
    for (var prop in obj) {
        var val = obj[prop];
        if (val instanceof Function)
            { val = val.bind(self); }
        else if (prop == "handleDOMEvents")
            { val = bindProps(val, self, {}); }
        target[prop] = val;
    }
    return target;
}
/**
Plugins bundle functionality that can be added to an editor.
They are part of the [editor state](https://prosemirror.net/docs/ref/#state.EditorState) and
may influence that state and the view that contains it.
*/
var Plugin = function Plugin(
/**
The plugin's [spec object](https://prosemirror.net/docs/ref/#state.PluginSpec).
*/
spec) {
    this.spec = spec;
    /**
    The [props](https://prosemirror.net/docs/ref/#view.EditorProps) exported by this plugin.
    */
    this.props = {};
    if (spec.props)
        { bindProps(spec.props, this, this.props); }
    this.key = spec.key ? spec.key.key : createKey("plugin");
};
/**
Extract the plugin's state field from an editor state.
*/
Plugin.prototype.getState = function getState (state) { return state[this.key]; };
var keys = Object.create(null);
function createKey(name) {
    if (name in keys)
        { return name + "$" + ++keys[name]; }
    keys[name] = 0;
    return name + "$";
}
/**
A key is used to [tag](https://prosemirror.net/docs/ref/#state.PluginSpec.key) plugins in a way
that makes it possible to find them, given an editor state.
Assigning a key does mean only one plugin of that type can be
active in a state.
*/
var PluginKey = function PluginKey(name) {
if ( name === void 0 ) name = "key";
 this.key = createKey(name); };
/**
Get the active plugin with this key, if any, from an editor
state.
*/
PluginKey.prototype.get = function get (state) { return state.config.pluginsByKey[this.key]; };
/**
Get the plugin's state from an editor state.
*/
PluginKey.prototype.getState = function getState (state) { return state[this.key]; };var state=/*#__PURE__*/Object.freeze({__proto__:null,AllSelection:AllSelection,EditorState:EditorState$1,NodeSelection:NodeSelection,Plugin:Plugin,PluginKey:PluginKey,Selection:Selection,SelectionRange:SelectionRange$1,TextSelection:TextSelection,Transaction:Transaction$1});var domIndex$1 = function (node) {
    for (var index = 0;; index++) {
        node = node.previousSibling;
        if (!node)
            { return index; }
    }
};
var parentNode = function (node) {
    var parent = node.assignedSlot || node.parentNode;
    return parent && parent.nodeType == 11 ? parent.host : parent;
};
var reusedRange = null;
// Note that this will always return the same range, because DOM range
// objects are every expensive, and keep slowing down subsequent DOM
// updates, for some reason.
var textRange$1 = function (node, from, to) {
    var range = reusedRange || (reusedRange = document.createRange());
    range.setEnd(node, to == null ? node.nodeValue.length : to);
    range.setStart(node, from || 0);
    return range;
};
var clearReusedRange = function () {
    reusedRange = null;
};
// Scans forward and backward through DOM positions equivalent to the
// given one to see if the two are in the same place (i.e. after a
// text node vs at the end of that text node)
var isEquivalentPosition$1 = function (node, off, targetNode, targetOff) {
    return targetNode && (scanFor$1(node, off, targetNode, targetOff, -1) ||
        scanFor$1(node, off, targetNode, targetOff, 1));
};
var atomElements = /^(img|br|input|textarea|hr)$/i;
function scanFor$1(node, off, targetNode, targetOff, dir) {
    for (;;) {
        if (node == targetNode && off == targetOff)
            { return true; }
        if (off == (dir < 0 ? 0 : nodeSize$1(node))) {
            var parent = node.parentNode;
            if (!parent || parent.nodeType != 1 || hasBlockDesc(node) || atomElements.test(node.nodeName) ||
                node.contentEditable == "false")
                { return false; }
            off = domIndex$1(node) + (dir < 0 ? 0 : 1);
            node = parent;
        }
        else if (node.nodeType == 1) {
            node = node.childNodes[off + (dir < 0 ? -1 : 0)];
            if (node.contentEditable == "false")
                { return false; }
            off = dir < 0 ? nodeSize$1(node) : 0;
        }
        else {
            return false;
        }
    }
}
function nodeSize$1(node) {
    return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
}
function textNodeBefore$1(node, offset) {
    for (;;) {
        if (node.nodeType == 3 && offset)
            { return node; }
        if (node.nodeType == 1 && offset > 0) {
            if (node.contentEditable == "false")
                { return null; }
            node = node.childNodes[offset - 1];
            offset = nodeSize$1(node);
        }
        else if (node.parentNode && !hasBlockDesc(node)) {
            offset = domIndex$1(node);
            node = node.parentNode;
        }
        else {
            return null;
        }
    }
}
function textNodeAfter$1(node, offset) {
    for (;;) {
        if (node.nodeType == 3 && offset < node.nodeValue.length)
            { return node; }
        if (node.nodeType == 1 && offset < node.childNodes.length) {
            if (node.contentEditable == "false")
                { return null; }
            node = node.childNodes[offset];
            offset = 0;
        }
        else if (node.parentNode && !hasBlockDesc(node)) {
            offset = domIndex$1(node) + 1;
            node = node.parentNode;
        }
        else {
            return null;
        }
    }
}
function isOnEdge(node, offset, parent) {
    for (var atStart = offset == 0, atEnd = offset == nodeSize$1(node); atStart || atEnd;) {
        if (node == parent)
            { return true; }
        var index = domIndex$1(node);
        node = node.parentNode;
        if (!node)
            { return false; }
        atStart = atStart && index == 0;
        atEnd = atEnd && index == nodeSize$1(node);
    }
}
function hasBlockDesc(dom) {
    var desc;
    for (var cur = dom; cur; cur = cur.parentNode)
        { if (desc = cur.pmViewDesc)
            { break; } }
    return desc && desc.node && desc.node.isBlock && (desc.dom == dom || desc.contentDOM == dom);
}
// Work around Chrome issue https://bugs.chromium.org/p/chromium/issues/detail?id=447523
// (isCollapsed inappropriately returns true in shadow dom)
var selectionCollapsed = function (domSel) {
    return domSel.focusNode && isEquivalentPosition$1(domSel.focusNode, domSel.focusOffset, domSel.anchorNode, domSel.anchorOffset);
};
function keyEvent(keyCode, key) {
    var event = document.createEvent("Event");
    event.initEvent("keydown", true, true);
    event.keyCode = keyCode;
    event.key = event.code = key;
    return event;
}
function deepActiveElement(doc) {
    var elt = doc.activeElement;
    while (elt && elt.shadowRoot)
        { elt = elt.shadowRoot.activeElement; }
    return elt;
}
function caretFromPoint(doc, x, y) {
    if (doc.caretPositionFromPoint) {
        try { // Firefox throws for this call in hard-to-predict circumstances (#994)
            var pos = doc.caretPositionFromPoint(x, y);
            // Clip the offset, because Chrome will return a text offset
            // into <input> nodes, which can't be treated as a regular DOM
            // offset
            if (pos)
                { return { node: pos.offsetNode, offset: Math.min(nodeSize$1(pos.offsetNode), pos.offset) }; }
        }
        catch (_) { }
    }
    if (doc.caretRangeFromPoint) {
        var range = doc.caretRangeFromPoint(x, y);
        if (range)
            { return { node: range.startContainer, offset: Math.min(nodeSize$1(range.startContainer), range.startOffset) }; }
    }
}

var nav$1 = typeof navigator != "undefined" ? navigator : null;
var doc$2 = typeof document != "undefined" ? document : null;
var agent = (nav$1 && nav$1.userAgent) || "";
var ie_edge$1 = /Edge\/(\d+)/.exec(agent);
var ie_upto10$1 = /MSIE \d/.exec(agent);
var ie_11up$1 = /Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(agent);
var ie$2 = !!(ie_upto10$1 || ie_11up$1 || ie_edge$1);
var ie_version = ie_upto10$1 ? document.documentMode : ie_11up$1 ? +ie_11up$1[1] : ie_edge$1 ? +ie_edge$1[1] : 0;
var gecko$1 = !ie$2 && /gecko\/(\d+)/i.test(agent);
gecko$1 && +(/Firefox\/(\d+)/.exec(agent) || [0, 0])[1];
var _chrome = !ie$2 && /Chrome\/(\d+)/.exec(agent);
var chrome$1 = !!_chrome;
var chrome_version = _chrome ? +_chrome[1] : 0;
var safari$1 = !ie$2 && !!nav$1 && /Apple Computer/.test(nav$1.vendor);
// Is true for both iOS and iPadOS for convenience
var ios$1 = safari$1 && (/Mobile\/\w+/.test(agent) || !!nav$1 && nav$1.maxTouchPoints > 2);
var mac$4 = ios$1 || (nav$1 ? /Mac/.test(nav$1.platform) : false);
var windows = nav$1 ? /Win/.test(nav$1.platform) : false;
var android$1 = /Android \d/.test(agent);
var webkit$1 = !!doc$2 && "webkitFontSmoothing" in doc$2.documentElement.style;
var webkit_version = webkit$1 ? +(/\bAppleWebKit\/(\d+)/.exec(navigator.userAgent) || [0, 0])[1] : 0;

function windowRect$1(doc) {
    var vp = doc.defaultView && doc.defaultView.visualViewport;
    if (vp)
        { return {
            left: 0, right: vp.width,
            top: 0, bottom: vp.height
        }; }
    return { left: 0, right: doc.documentElement.clientWidth,
        top: 0, bottom: doc.documentElement.clientHeight };
}
function getSide(value, side) {
    return typeof value == "number" ? value : value[side];
}
function clientRect(node) {
    var rect = node.getBoundingClientRect();
    // Adjust for elements with style "transform: scale()"
    var scaleX = (rect.width / node.offsetWidth) || 1;
    var scaleY = (rect.height / node.offsetHeight) || 1;
    // Make sure scrollbar width isn't included in the rectangle
    return { left: rect.left, right: rect.left + node.clientWidth * scaleX,
        top: rect.top, bottom: rect.top + node.clientHeight * scaleY };
}
function scrollRectIntoView$1(view, rect, startDOM) {
    var scrollThreshold = view.someProp("scrollThreshold") || 0, scrollMargin = view.someProp("scrollMargin") || 5;
    var doc = view.dom.ownerDocument;
    for (var parent = startDOM || view.dom;; parent = parentNode(parent)) {
        if (!parent)
            { break; }
        if (parent.nodeType != 1)
            { continue; }
        var elt = parent;
        var atTop = elt == doc.body;
        var bounding = atTop ? windowRect$1(doc) : clientRect(elt);
        var moveX = 0, moveY = 0;
        if (rect.top < bounding.top + getSide(scrollThreshold, "top"))
            { moveY = -(bounding.top - rect.top + getSide(scrollMargin, "top")); }
        else if (rect.bottom > bounding.bottom - getSide(scrollThreshold, "bottom"))
            { moveY = rect.bottom - rect.top > bounding.bottom - bounding.top
                ? rect.top + getSide(scrollMargin, "top") - bounding.top
                : rect.bottom - bounding.bottom + getSide(scrollMargin, "bottom"); }
        if (rect.left < bounding.left + getSide(scrollThreshold, "left"))
            { moveX = -(bounding.left - rect.left + getSide(scrollMargin, "left")); }
        else if (rect.right > bounding.right - getSide(scrollThreshold, "right"))
            { moveX = rect.right - bounding.right + getSide(scrollMargin, "right"); }
        if (moveX || moveY) {
            if (atTop) {
                doc.defaultView.scrollBy(moveX, moveY);
            }
            else {
                var startX = elt.scrollLeft, startY = elt.scrollTop;
                if (moveY)
                    { elt.scrollTop += moveY; }
                if (moveX)
                    { elt.scrollLeft += moveX; }
                var dX = elt.scrollLeft - startX, dY = elt.scrollTop - startY;
                rect = { left: rect.left - dX, top: rect.top - dY, right: rect.right - dX, bottom: rect.bottom - dY };
            }
        }
        if (atTop || /^(fixed|sticky)$/.test(getComputedStyle(parent).position))
            { break; }
    }
}
// Store the scroll position of the editor's parent nodes, along with
// the top position of an element near the top of the editor, which
// will be used to make sure the visible viewport remains stable even
// when the size of the content above changes.
function storeScrollPos(view) {
    var rect = view.dom.getBoundingClientRect(), startY = Math.max(0, rect.top);
    var refDOM, refTop;
    for (var x = (rect.left + rect.right) / 2, y = startY + 1; y < Math.min(innerHeight, rect.bottom); y += 5) {
        var dom = view.root.elementFromPoint(x, y);
        if (!dom || dom == view.dom || !view.dom.contains(dom))
            { continue; }
        var localRect = dom.getBoundingClientRect();
        if (localRect.top >= startY - 20) {
            refDOM = dom;
            refTop = localRect.top;
            break;
        }
    }
    return { refDOM: refDOM, refTop: refTop, stack: scrollStack(view.dom) };
}
function scrollStack(dom) {
    var stack = [], doc = dom.ownerDocument;
    for (var cur = dom; cur; cur = parentNode(cur)) {
        stack.push({ dom: cur, top: cur.scrollTop, left: cur.scrollLeft });
        if (dom == doc)
            { break; }
    }
    return stack;
}
// Reset the scroll position of the editor's parent nodes to that what
// it was before, when storeScrollPos was called.
function resetScrollPos(ref) {
    var refDOM = ref.refDOM;
    var refTop = ref.refTop;
    var stack = ref.stack;

    var newRefTop = refDOM ? refDOM.getBoundingClientRect().top : 0;
    restoreScrollStack(stack, newRefTop == 0 ? 0 : newRefTop - refTop);
}
function restoreScrollStack(stack, dTop) {
    for (var i = 0; i < stack.length; i++) {
        var ref = stack[i];
        var dom = ref.dom;
        var top = ref.top;
        var left = ref.left;
        if (dom.scrollTop != top + dTop)
            { dom.scrollTop = top + dTop; }
        if (dom.scrollLeft != left)
            { dom.scrollLeft = left; }
    }
}
var preventScrollSupported$1 = null;
// Feature-detects support for .focus({preventScroll: true}), and uses
// a fallback kludge when not supported.
function focusPreventScroll$1(dom) {
    if (dom.setActive)
        { return dom.setActive(); } // in IE
    if (preventScrollSupported$1)
        { return dom.focus(preventScrollSupported$1); }
    var stored = scrollStack(dom);
    dom.focus(preventScrollSupported$1 == null ? {
        get preventScroll() {
            preventScrollSupported$1 = { preventScroll: true };
            return true;
        }
    } : undefined);
    if (!preventScrollSupported$1) {
        preventScrollSupported$1 = false;
        restoreScrollStack(stored, 0);
    }
}
function findOffsetInNode(node, coords) {
    var closest, dxClosest = 2e8, coordsClosest, offset = 0;
    var rowBot = coords.top, rowTop = coords.top;
    var firstBelow, coordsBelow;
    for (var child = node.firstChild, childIndex = 0; child; child = child.nextSibling, childIndex++) {
        var rects = (void 0);
        if (child.nodeType == 1)
            { rects = child.getClientRects(); }
        else if (child.nodeType == 3)
            { rects = textRange$1(child).getClientRects(); }
        else
            { continue; }
        for (var i = 0; i < rects.length; i++) {
            var rect = rects[i];
            if (rect.top <= rowBot && rect.bottom >= rowTop) {
                rowBot = Math.max(rect.bottom, rowBot);
                rowTop = Math.min(rect.top, rowTop);
                var dx = rect.left > coords.left ? rect.left - coords.left
                    : rect.right < coords.left ? coords.left - rect.right : 0;
                if (dx < dxClosest) {
                    closest = child;
                    dxClosest = dx;
                    coordsClosest = dx && closest.nodeType == 3 ? {
                        left: rect.right < coords.left ? rect.right : rect.left,
                        top: coords.top
                    } : coords;
                    if (child.nodeType == 1 && dx)
                        { offset = childIndex + (coords.left >= (rect.left + rect.right) / 2 ? 1 : 0); }
                    continue;
                }
            }
            else if (rect.top > coords.top && !firstBelow && rect.left <= coords.left && rect.right >= coords.left) {
                firstBelow = child;
                coordsBelow = { left: Math.max(rect.left, Math.min(rect.right, coords.left)), top: rect.top };
            }
            if (!closest && (coords.left >= rect.right && coords.top >= rect.top ||
                coords.left >= rect.left && coords.top >= rect.bottom))
                { offset = childIndex + 1; }
        }
    }
    if (!closest && firstBelow) {
        closest = firstBelow;
        coordsClosest = coordsBelow;
        dxClosest = 0;
    }
    if (closest && closest.nodeType == 3)
        { return findOffsetInText(closest, coordsClosest); }
    if (!closest || (dxClosest && closest.nodeType == 1))
        { return { node: node, offset: offset }; }
    return findOffsetInNode(closest, coordsClosest);
}
function findOffsetInText(node, coords) {
    var len = node.nodeValue.length;
    var range = document.createRange();
    for (var i = 0; i < len; i++) {
        range.setEnd(node, i + 1);
        range.setStart(node, i);
        var rect = singleRect(range, 1);
        if (rect.top == rect.bottom)
            { continue; }
        if (inRect(coords, rect))
            { return { node: node, offset: i + (coords.left >= (rect.left + rect.right) / 2 ? 1 : 0) }; }
    }
    return { node: node, offset: 0 };
}
function inRect(coords, rect) {
    return coords.left >= rect.left - 1 && coords.left <= rect.right + 1 &&
        coords.top >= rect.top - 1 && coords.top <= rect.bottom + 1;
}
function targetKludge(dom, coords) {
    var parent = dom.parentNode;
    if (parent && /^li$/i.test(parent.nodeName) && coords.left < dom.getBoundingClientRect().left)
        { return parent; }
    return dom;
}
function posFromElement(view, elt, coords) {
    var ref = findOffsetInNode(elt, coords);
    var node = ref.node;
    var offset = ref.offset;
    var bias = -1;
    if (node.nodeType == 1 && !node.firstChild) {
        var rect = node.getBoundingClientRect();
        bias = rect.left != rect.right && coords.left > (rect.left + rect.right) / 2 ? 1 : -1;
    }
    return view.docView.posFromDOM(node, offset, bias);
}
function posFromCaret(view, node, offset, coords) {
    // Browser (in caretPosition/RangeFromPoint) will agressively
    // normalize towards nearby inline nodes. Since we are interested in
    // positions between block nodes too, we first walk up the hierarchy
    // of nodes to see if there are block nodes that the coordinates
    // fall outside of. If so, we take the position before/after that
    // block. If not, we call `posFromDOM` on the raw node/offset.
    var outsideBlock = -1;
    for (var cur = node, sawBlock = false;;) {
        if (cur == view.dom)
            { break; }
        var desc = view.docView.nearestDesc(cur, true);
        if (!desc)
            { return null; }
        if (desc.dom.nodeType == 1 && (desc.node.isBlock && desc.parent || !desc.contentDOM)) {
            var rect = desc.dom.getBoundingClientRect();
            if (desc.node.isBlock && desc.parent) {
                // Only apply the horizontal test to the innermost block. Vertical for any parent.
                if (!sawBlock && rect.left > coords.left || rect.top > coords.top)
                    { outsideBlock = desc.posBefore; }
                else if (!sawBlock && rect.right < coords.left || rect.bottom < coords.top)
                    { outsideBlock = desc.posAfter; }
                sawBlock = true;
            }
            if (!desc.contentDOM && outsideBlock < 0 && !desc.node.isText) {
                // If we are inside a leaf, return the side of the leaf closer to the coords
                var before = desc.node.isBlock ? coords.top < (rect.top + rect.bottom) / 2
                    : coords.left < (rect.left + rect.right) / 2;
                return before ? desc.posBefore : desc.posAfter;
            }
        }
        cur = desc.dom.parentNode;
    }
    return outsideBlock > -1 ? outsideBlock : view.docView.posFromDOM(node, offset, -1);
}
function elementFromPoint(element, coords, box) {
    var len = element.childNodes.length;
    if (len && box.top < box.bottom) {
        for (var startI = Math.max(0, Math.min(len - 1, Math.floor(len * (coords.top - box.top) / (box.bottom - box.top)) - 2)), i = startI;;) {
            var child = element.childNodes[i];
            if (child.nodeType == 1) {
                var rects = child.getClientRects();
                for (var j = 0; j < rects.length; j++) {
                    var rect = rects[j];
                    if (inRect(coords, rect))
                        { return elementFromPoint(child, coords, rect); }
                }
            }
            if ((i = (i + 1) % len) == startI)
                { break; }
        }
    }
    return element;
}
// Given an x,y position on the editor, get the position in the document.
function posAtCoords$1(view, coords) {
    var assign;

    var doc = view.dom.ownerDocument, node, offset = 0;
    var caret = caretFromPoint(doc, coords.left, coords.top);
    if (caret)
        { ((assign = caret, node = assign.node, offset = assign.offset)); }
    var elt = (view.root.elementFromPoint ? view.root : doc)
        .elementFromPoint(coords.left, coords.top);
    var pos;
    if (!elt || !view.dom.contains(elt.nodeType != 1 ? elt.parentNode : elt)) {
        var box = view.dom.getBoundingClientRect();
        if (!inRect(coords, box))
            { return null; }
        elt = elementFromPoint(view.dom, coords, box);
        if (!elt)
            { return null; }
    }
    // Safari's caretRangeFromPoint returns nonsense when on a draggable element
    if (safari$1) {
        for (var p = elt; node && p; p = parentNode(p))
            { if (p.draggable)
                { node = undefined; } }
    }
    elt = targetKludge(elt, coords);
    if (node) {
        if (gecko$1 && node.nodeType == 1) {
            // Firefox will sometimes return offsets into <input> nodes, which
            // have no actual children, from caretPositionFromPoint (#953)
            offset = Math.min(offset, node.childNodes.length);
            // It'll also move the returned position before image nodes,
            // even if those are behind it.
            if (offset < node.childNodes.length) {
                var next = node.childNodes[offset], box$1;
                if (next.nodeName == "IMG" && (box$1 = next.getBoundingClientRect()).right <= coords.left &&
                    box$1.bottom > coords.top)
                    { offset++; }
            }
        }
        var prev;
        // When clicking above the right side of an uneditable node, Chrome will report a cursor position after that node.
        if (webkit$1 && offset && node.nodeType == 1 && (prev = node.childNodes[offset - 1]).nodeType == 1 &&
            prev.contentEditable == "false" && prev.getBoundingClientRect().top >= coords.top)
            { offset--; }
        // Suspiciously specific kludge to work around caret*FromPoint
        // never returning a position at the end of the document
        if (node == view.dom && offset == node.childNodes.length - 1 && node.lastChild.nodeType == 1 &&
            coords.top > node.lastChild.getBoundingClientRect().bottom)
            { pos = view.state.doc.content.size; }
        // Ignore positions directly after a BR, since caret*FromPoint
        // 'round up' positions that would be more accurately placed
        // before the BR node.
        else if (offset == 0 || node.nodeType != 1 || node.childNodes[offset - 1].nodeName != "BR")
            { pos = posFromCaret(view, node, offset, coords); }
    }
    if (pos == null)
        { pos = posFromElement(view, elt, coords); }
    var desc = view.docView.nearestDesc(elt, true);
    return { pos: pos, inside: desc ? desc.posAtStart - desc.border : -1 };
}
function nonZero(rect) {
    return rect.top < rect.bottom || rect.left < rect.right;
}
function singleRect(target, bias) {
    var rects = target.getClientRects();
    if (rects.length) {
        var first = rects[bias < 0 ? 0 : rects.length - 1];
        if (nonZero(first))
            { return first; }
    }
    return Array.prototype.find.call(rects, nonZero) || target.getBoundingClientRect();
}
var BIDI = /[\u0590-\u05f4\u0600-\u06ff\u0700-\u08ac]/;
// Given a position in the document model, get a bounding box of the
// character at that position, relative to the window.
function coordsAtPos(view, pos, side) {
    var ref = view.docView.domFromPos(pos, side < 0 ? -1 : 1);
    var node = ref.node;
    var offset = ref.offset;
    var atom = ref.atom;
    var supportEmptyRange = webkit$1 || gecko$1;
    if (node.nodeType == 3) {
        // These browsers support querying empty text ranges. Prefer that in
        // bidi context or when at the end of a node.
        if (supportEmptyRange && (BIDI.test(node.nodeValue) || (side < 0 ? !offset : offset == node.nodeValue.length))) {
            var rect = singleRect(textRange$1(node, offset, offset), side);
            // Firefox returns bad results (the position before the space)
            // when querying a position directly after line-broken
            // whitespace. Detect this situation and and kludge around it
            if (gecko$1 && offset && /\s/.test(node.nodeValue[offset - 1]) && offset < node.nodeValue.length) {
                var rectBefore = singleRect(textRange$1(node, offset - 1, offset - 1), -1);
                if (rectBefore.top == rect.top) {
                    var rectAfter = singleRect(textRange$1(node, offset, offset + 1), -1);
                    if (rectAfter.top != rect.top)
                        { return flattenV(rectAfter, rectAfter.left < rectBefore.left); }
                }
            }
            return rect;
        }
        else {
            var from = offset, to = offset, takeSide = side < 0 ? 1 : -1;
            if (side < 0 && !offset) {
                to++;
                takeSide = -1;
            }
            else if (side >= 0 && offset == node.nodeValue.length) {
                from--;
                takeSide = 1;
            }
            else if (side < 0) {
                from--;
            }
            else {
                to++;
            }
            return flattenV(singleRect(textRange$1(node, from, to), takeSide), takeSide < 0);
        }
    }
    var $dom = view.state.doc.resolve(pos - (atom || 0));
    // Return a horizontal line in block context
    if (!$dom.parent.inlineContent) {
        if (atom == null && offset && (side < 0 || offset == nodeSize$1(node))) {
            var before = node.childNodes[offset - 1];
            if (before.nodeType == 1)
                { return flattenH(before.getBoundingClientRect(), false); }
        }
        if (atom == null && offset < nodeSize$1(node)) {
            var after = node.childNodes[offset];
            if (after.nodeType == 1)
                { return flattenH(after.getBoundingClientRect(), true); }
        }
        return flattenH(node.getBoundingClientRect(), side >= 0);
    }
    // Inline, not in text node (this is not Bidi-safe)
    if (atom == null && offset && (side < 0 || offset == nodeSize$1(node))) {
        var before$1 = node.childNodes[offset - 1];
        var target = before$1.nodeType == 3 ? textRange$1(before$1, nodeSize$1(before$1) - (supportEmptyRange ? 0 : 1))
            // BR nodes tend to only return the rectangle before them.
            // Only use them if they are the last element in their parent
            : before$1.nodeType == 1 && (before$1.nodeName != "BR" || !before$1.nextSibling) ? before$1 : null;
        if (target)
            { return flattenV(singleRect(target, 1), false); }
    }
    if (atom == null && offset < nodeSize$1(node)) {
        var after$1 = node.childNodes[offset];
        while (after$1.pmViewDesc && after$1.pmViewDesc.ignoreForCoords)
            { after$1 = after$1.nextSibling; }
        var target$1 = !after$1 ? null : after$1.nodeType == 3 ? textRange$1(after$1, 0, (supportEmptyRange ? 0 : 1))
            : after$1.nodeType == 1 ? after$1 : null;
        if (target$1)
            { return flattenV(singleRect(target$1, -1), true); }
    }
    // All else failed, just try to get a rectangle for the target node
    return flattenV(singleRect(node.nodeType == 3 ? textRange$1(node) : node, -side), side >= 0);
}
function flattenV(rect, left) {
    if (rect.width == 0)
        { return rect; }
    var x = left ? rect.left : rect.right;
    return { top: rect.top, bottom: rect.bottom, left: x, right: x };
}
function flattenH(rect, top) {
    if (rect.height == 0)
        { return rect; }
    var y = top ? rect.top : rect.bottom;
    return { top: y, bottom: y, left: rect.left, right: rect.right };
}
function withFlushedState(view, state, f) {
    var viewState = view.state, active = view.root.activeElement;
    if (viewState != state)
        { view.updateState(state); }
    if (active != view.dom)
        { view.focus(); }
    try {
        return f();
    }
    finally {
        if (viewState != state)
            { view.updateState(viewState); }
        if (active != view.dom && active)
            { active.focus(); }
    }
}
// Whether vertical position motion in a given direction
// from a position would leave a text block.
function endOfTextblockVertical(view, state, dir) {
    var sel = state.selection;
    var $pos = dir == "up" ? sel.$from : sel.$to;
    return withFlushedState(view, state, function () {
        var ref = view.docView.domFromPos($pos.pos, dir == "up" ? -1 : 1);
        var dom = ref.node;
        for (;;) {
            var nearest = view.docView.nearestDesc(dom, true);
            if (!nearest)
                { break; }
            if (nearest.node.isBlock) {
                dom = nearest.contentDOM || nearest.dom;
                break;
            }
            dom = nearest.dom.parentNode;
        }
        var coords = coordsAtPos(view, $pos.pos, 1);
        for (var child = dom.firstChild; child; child = child.nextSibling) {
            var boxes = (void 0);
            if (child.nodeType == 1)
                { boxes = child.getClientRects(); }
            else if (child.nodeType == 3)
                { boxes = textRange$1(child, 0, child.nodeValue.length).getClientRects(); }
            else
                { continue; }
            for (var i = 0; i < boxes.length; i++) {
                var box = boxes[i];
                if (box.bottom > box.top + 1 &&
                    (dir == "up" ? coords.top - box.top > (box.bottom - coords.top) * 2
                        : box.bottom - coords.bottom > (coords.bottom - box.top) * 2))
                    { return false; }
            }
        }
        return true;
    });
}
var maybeRTL = /[\u0590-\u08ac]/;
function endOfTextblockHorizontal(view, state, dir) {
    var ref = state.selection;
    var $head = ref.$head;
    if (!$head.parent.isTextblock)
        { return false; }
    var offset = $head.parentOffset, atStart = !offset, atEnd = offset == $head.parent.content.size;
    var sel = view.domSelection();
    if (!sel)
        { return $head.pos == $head.start() || $head.pos == $head.end(); }
    // If the textblock is all LTR, or the browser doesn't support
    // Selection.modify (Edge), fall back to a primitive approach
    if (!maybeRTL.test($head.parent.textContent) || !sel.modify)
        { return dir == "left" || dir == "backward" ? atStart : atEnd; }
    return withFlushedState(view, state, function () {
        // This is a huge hack, but appears to be the best we can
        // currently do: use `Selection.modify` to move the selection by
        // one character, and see if that moves the cursor out of the
        // textblock (or doesn't move it at all, when at the start/end of
        // the document).
        var ref = view.domSelectionRange();
        var oldNode = ref.focusNode;
        var oldOff = ref.focusOffset;
        var anchorNode = ref.anchorNode;
        var anchorOffset = ref.anchorOffset;
        var oldBidiLevel = sel.caretBidiLevel // Only for Firefox
        ;
        sel.modify("move", dir, "character");
        var parentDOM = $head.depth ? view.docView.domAfterPos($head.before()) : view.dom;
        var ref$1 = view.domSelectionRange();
        var newNode = ref$1.focusNode;
        var newOff = ref$1.focusOffset;
        var result = newNode && !parentDOM.contains(newNode.nodeType == 1 ? newNode : newNode.parentNode) ||
            (oldNode == newNode && oldOff == newOff);
        // Restore the previous selection
        try {
            sel.collapse(anchorNode, anchorOffset);
            if (oldNode && (oldNode != anchorNode || oldOff != anchorOffset) && sel.extend)
                { sel.extend(oldNode, oldOff); }
        }
        catch (_) { }
        if (oldBidiLevel != null)
            { sel.caretBidiLevel = oldBidiLevel; }
        return result;
    });
}
var cachedState = null;
var cachedDir = null;
var cachedResult = false;
function endOfTextblock(view, state, dir) {
    if (cachedState == state && cachedDir == dir)
        { return cachedResult; }
    cachedState = state;
    cachedDir = dir;
    return cachedResult = dir == "up" || dir == "down"
        ? endOfTextblockVertical(view, state, dir)
        : endOfTextblockHorizontal(view, state, dir);
}

// View descriptions are data structures that describe the DOM that is
// used to represent the editor's content. They are used for:
//
// - Incremental redrawing when the document changes
//
// - Figuring out what part of the document a given DOM position
//   corresponds to
//
// - Wiring in custom implementations of the editing interface for a
//   given node
//
// They form a doubly-linked mutable tree, starting at `view.docView`.
var NOT_DIRTY = 0, CHILD_DIRTY = 1, CONTENT_DIRTY = 2, NODE_DIRTY = 3;
// Superclass for the various kinds of descriptions. Defines their
// basic structure and shared methods.
var ViewDesc = function ViewDesc(parent, children, dom, 
// This is the node that holds the child views. It may be null for
// descs that don't have children.
contentDOM) {
    this.parent = parent;
    this.children = children;
    this.dom = dom;
    this.contentDOM = contentDOM;
    this.dirty = NOT_DIRTY;
    // An expando property on the DOM node provides a link back to its
    // description.
    dom.pmViewDesc = this;
};

var prototypeAccessors$5 = { size: { configurable: true },border: { configurable: true },posBefore: { configurable: true },posAtStart: { configurable: true },posAfter: { configurable: true },posAtEnd: { configurable: true },contentLost: { configurable: true },domAtom: { configurable: true },ignoreForCoords: { configurable: true } };
// Used to check whether a given description corresponds to a
// widget/mark/node.
ViewDesc.prototype.matchesWidget = function matchesWidget (widget) { return false; };
ViewDesc.prototype.matchesMark = function matchesMark (mark) { return false; };
ViewDesc.prototype.matchesNode = function matchesNode (node, outerDeco, innerDeco) { return false; };
ViewDesc.prototype.matchesHack = function matchesHack (nodeName) { return false; };
// When parsing in-editor content (in domchange.js), we allow
// descriptions to determine the parse rules that should be used to
// parse them.
ViewDesc.prototype.parseRule = function parseRule () { return null; };
// Used by the editor's event handler to ignore events that come
// from certain descs.
ViewDesc.prototype.stopEvent = function stopEvent (event) { return false; };
// The size of the content represented by this desc.
prototypeAccessors$5.size.get = function () {
    var size = 0;
    for (var i = 0; i < this.children.length; i++)
        { size += this.children[i].size; }
    return size;
};
// For block nodes, this represents the space taken up by their
// start/end tokens.
prototypeAccessors$5.border.get = function () { return 0; };
ViewDesc.prototype.destroy = function destroy () {
    this.parent = undefined;
    if (this.dom.pmViewDesc == this)
        { this.dom.pmViewDesc = undefined; }
    for (var i = 0; i < this.children.length; i++)
        { this.children[i].destroy(); }
};
ViewDesc.prototype.posBeforeChild = function posBeforeChild (child) {
    for (var i = 0, pos = this.posAtStart;; i++) {
        var cur = this.children[i];
        if (cur == child)
            { return pos; }
        pos += cur.size;
    }
};
prototypeAccessors$5.posBefore.get = function () {
    return this.parent.posBeforeChild(this);
};
prototypeAccessors$5.posAtStart.get = function () {
    return this.parent ? this.parent.posBeforeChild(this) + this.border : 0;
};
prototypeAccessors$5.posAfter.get = function () {
    return this.posBefore + this.size;
};
prototypeAccessors$5.posAtEnd.get = function () {
    return this.posAtStart + this.size - 2 * this.border;
};
ViewDesc.prototype.localPosFromDOM = function localPosFromDOM (dom, offset, bias) {
    // If the DOM position is in the content, use the child desc after
    // it to figure out a position.
    if (this.contentDOM && this.contentDOM.contains(dom.nodeType == 1 ? dom : dom.parentNode)) {
        if (bias < 0) {
            var domBefore, desc;
            if (dom == this.contentDOM) {
                domBefore = dom.childNodes[offset - 1];
            }
            else {
                while (dom.parentNode != this.contentDOM)
                    { dom = dom.parentNode; }
                domBefore = dom.previousSibling;
            }
            while (domBefore && !((desc = domBefore.pmViewDesc) && desc.parent == this))
                { domBefore = domBefore.previousSibling; }
            return domBefore ? this.posBeforeChild(desc) + desc.size : this.posAtStart;
        }
        else {
            var domAfter, desc$1;
            if (dom == this.contentDOM) {
                domAfter = dom.childNodes[offset];
            }
            else {
                while (dom.parentNode != this.contentDOM)
                    { dom = dom.parentNode; }
                domAfter = dom.nextSibling;
            }
            while (domAfter && !((desc$1 = domAfter.pmViewDesc) && desc$1.parent == this))
                { domAfter = domAfter.nextSibling; }
            return domAfter ? this.posBeforeChild(desc$1) : this.posAtEnd;
        }
    }
    // Otherwise, use various heuristics, falling back on the bias
    // parameter, to determine whether to return the position at the
    // start or at the end of this view desc.
    var atEnd;
    if (dom == this.dom && this.contentDOM) {
        atEnd = offset > domIndex$1(this.contentDOM);
    }
    else if (this.contentDOM && this.contentDOM != this.dom && this.dom.contains(this.contentDOM)) {
        atEnd = dom.compareDocumentPosition(this.contentDOM) & 2;
    }
    else if (this.dom.firstChild) {
        if (offset == 0)
            { for (var search = dom;; search = search.parentNode) {
                if (search == this.dom) {
                    atEnd = false;
                    break;
                }
                if (search.previousSibling)
                    { break; }
            } }
        if (atEnd == null && offset == dom.childNodes.length)
            { for (var search$1 = dom;; search$1 = search$1.parentNode) {
                if (search$1 == this.dom) {
                    atEnd = true;
                    break;
                }
                if (search$1.nextSibling)
                    { break; }
            } }
    }
    return (atEnd == null ? bias > 0 : atEnd) ? this.posAtEnd : this.posAtStart;
};
ViewDesc.prototype.nearestDesc = function nearestDesc (dom, onlyNodes) {
        if ( onlyNodes === void 0 ) onlyNodes = false;

    for (var first = true, cur = dom; cur; cur = cur.parentNode) {
        var desc = this.getDesc(cur), nodeDOM = (void 0);
        if (desc && (!onlyNodes || desc.node)) {
            // If dom is outside of this desc's nodeDOM, don't count it.
            if (first && (nodeDOM = desc.nodeDOM) &&
                !(nodeDOM.nodeType == 1 ? nodeDOM.contains(dom.nodeType == 1 ? dom : dom.parentNode) : nodeDOM == dom))
                { first = false; }
            else
                { return desc; }
        }
    }
};
ViewDesc.prototype.getDesc = function getDesc (dom) {
    var desc = dom.pmViewDesc;
    for (var cur = desc; cur; cur = cur.parent)
        { if (cur == this)
            { return desc; } }
};
ViewDesc.prototype.posFromDOM = function posFromDOM (dom, offset, bias) {
    for (var scan = dom; scan; scan = scan.parentNode) {
        var desc = this.getDesc(scan);
        if (desc)
            { return desc.localPosFromDOM(dom, offset, bias); }
    }
    return -1;
};
// Find the desc for the node after the given pos, if any. (When a
// parent node overrode rendering, there might not be one.)
ViewDesc.prototype.descAt = function descAt (pos) {
    for (var i = 0, offset = 0; i < this.children.length; i++) {
        var child = this.children[i], end = offset + child.size;
        if (offset == pos && end != offset) {
            while (!child.border && child.children.length)
                { child = child.children[0]; }
            return child;
        }
        if (pos < end)
            { return child.descAt(pos - offset - child.border); }
        offset = end;
    }
};
ViewDesc.prototype.domFromPos = function domFromPos (pos, side) {
    if (!this.contentDOM)
        { return { node: this.dom, offset: 0, atom: pos + 1 }; }
    // First find the position in the child array
    var i = 0, offset = 0;
    for (var curPos = 0; i < this.children.length; i++) {
        var child = this.children[i], end = curPos + child.size;
        if (end > pos || child instanceof TrailingHackViewDesc) {
            offset = pos - curPos;
            break;
        }
        curPos = end;
    }
    // If this points into the middle of a child, call through
    if (offset)
        { return this.children[i].domFromPos(offset - this.children[i].border, side); }
    // Go back if there were any zero-length widgets with side >= 0 before this point
    for (var prev = (void 0); i && !(prev = this.children[i - 1]).size && prev instanceof WidgetViewDesc && prev.side >= 0; i--) { }
    // Scan towards the first useable node
    if (side <= 0) {
        var prev$1, enter = true;
        for (;; i--, enter = false) {
            prev$1 = i ? this.children[i - 1] : null;
            if (!prev$1 || prev$1.dom.parentNode == this.contentDOM)
                { break; }
        }
        if (prev$1 && side && enter && !prev$1.border && !prev$1.domAtom)
            { return prev$1.domFromPos(prev$1.size, side); }
        return { node: this.contentDOM, offset: prev$1 ? domIndex$1(prev$1.dom) + 1 : 0 };
    }
    else {
        var next, enter$1 = true;
        for (;; i++, enter$1 = false) {
            next = i < this.children.length ? this.children[i] : null;
            if (!next || next.dom.parentNode == this.contentDOM)
                { break; }
        }
        if (next && enter$1 && !next.border && !next.domAtom)
            { return next.domFromPos(0, side); }
        return { node: this.contentDOM, offset: next ? domIndex$1(next.dom) : this.contentDOM.childNodes.length };
    }
};
// Used to find a DOM range in a single parent for a given changed
// range.
ViewDesc.prototype.parseRange = function parseRange (from, to, base) {
        if ( base === void 0 ) base = 0;

    if (this.children.length == 0)
        { return { node: this.contentDOM, from: from, to: to, fromOffset: 0, toOffset: this.contentDOM.childNodes.length }; }
    var fromOffset = -1, toOffset = -1;
    for (var offset = base, i = 0;; i++) {
        var child = this.children[i], end = offset + child.size;
        if (fromOffset == -1 && from <= end) {
            var childBase = offset + child.border;
            // FIXME maybe descend mark views to parse a narrower range?
            if (from >= childBase && to <= end - child.border && child.node &&
                child.contentDOM && this.contentDOM.contains(child.contentDOM))
                { return child.parseRange(from, to, childBase); }
            from = offset;
            for (var j = i; j > 0; j--) {
                var prev = this.children[j - 1];
                if (prev.size && prev.dom.parentNode == this.contentDOM && !prev.emptyChildAt(1)) {
                    fromOffset = domIndex$1(prev.dom) + 1;
                    break;
                }
                from -= prev.size;
            }
            if (fromOffset == -1)
                { fromOffset = 0; }
        }
        if (fromOffset > -1 && (end > to || i == this.children.length - 1)) {
            to = end;
            for (var j$1 = i + 1; j$1 < this.children.length; j$1++) {
                var next = this.children[j$1];
                if (next.size && next.dom.parentNode == this.contentDOM && !next.emptyChildAt(-1)) {
                    toOffset = domIndex$1(next.dom);
                    break;
                }
                to += next.size;
            }
            if (toOffset == -1)
                { toOffset = this.contentDOM.childNodes.length; }
            break;
        }
        offset = end;
    }
    return { node: this.contentDOM, from: from, to: to, fromOffset: fromOffset, toOffset: toOffset };
};
ViewDesc.prototype.emptyChildAt = function emptyChildAt (side) {
    if (this.border || !this.contentDOM || !this.children.length)
        { return false; }
    var child = this.children[side < 0 ? 0 : this.children.length - 1];
    return child.size == 0 || child.emptyChildAt(side);
};
ViewDesc.prototype.domAfterPos = function domAfterPos (pos) {
    var ref = this.domFromPos(pos, 0);
        var node = ref.node;
        var offset = ref.offset;
    if (node.nodeType != 1 || offset == node.childNodes.length)
        { throw new RangeError("No node after pos " + pos); }
    return node.childNodes[offset];
};
// View descs are responsible for setting any selection that falls
// entirely inside of them, so that custom implementations can do
// custom things with the selection. Note that this falls apart when
// a selection starts in such a node and ends in another, in which
// case we just use whatever domFromPos produces as a best effort.
ViewDesc.prototype.setSelection = function setSelection (anchor, head, root, force) {
        if ( force === void 0 ) force = false;

    // If the selection falls entirely in a child, give it to that child
    var from = Math.min(anchor, head), to = Math.max(anchor, head);
    for (var i = 0, offset = 0; i < this.children.length; i++) {
        var child = this.children[i], end = offset + child.size;
        if (from > offset && to < end)
            { return child.setSelection(anchor - offset - child.border, head - offset - child.border, root, force); }
        offset = end;
    }
    var anchorDOM = this.domFromPos(anchor, anchor ? -1 : 1);
    var headDOM = head == anchor ? anchorDOM : this.domFromPos(head, head ? -1 : 1);
    var domSel = root.getSelection();
    var brKludge = false;
    // On Firefox, using Selection.collapse to put the cursor after a
    // BR node for some reason doesn't always work (#1073). On Safari,
    // the cursor sometimes inexplicable visually lags behind its
    // reported position in such situations (#1092).
    if ((gecko$1 || safari$1) && anchor == head) {
        var node = anchorDOM.node;
            var offset$1 = anchorDOM.offset;
        if (node.nodeType == 3) {
            brKludge = !!(offset$1 && node.nodeValue[offset$1 - 1] == "\n");
            // Issue #1128
            if (brKludge && offset$1 == node.nodeValue.length) {
                for (var scan = node, after = (void 0); scan; scan = scan.parentNode) {
                    if (after = scan.nextSibling) {
                        if (after.nodeName == "BR")
                            { anchorDOM = headDOM = { node: after.parentNode, offset: domIndex$1(after) + 1 }; }
                        break;
                    }
                    var desc = scan.pmViewDesc;
                    if (desc && desc.node && desc.node.isBlock)
                        { break; }
                }
            }
        }
        else {
            var prev = node.childNodes[offset$1 - 1];
            brKludge = prev && (prev.nodeName == "BR" || prev.contentEditable == "false");
        }
    }
    // Firefox can act strangely when the selection is in front of an
    // uneditable node. See #1163 and https://bugzilla.mozilla.org/show_bug.cgi?id=1709536
    if (gecko$1 && domSel.focusNode && domSel.focusNode != headDOM.node && domSel.focusNode.nodeType == 1) {
        var after$1 = domSel.focusNode.childNodes[domSel.focusOffset];
        if (after$1 && after$1.contentEditable == "false")
            { force = true; }
    }
    if (!(force || brKludge && safari$1) &&
        isEquivalentPosition$1(anchorDOM.node, anchorDOM.offset, domSel.anchorNode, domSel.anchorOffset) &&
        isEquivalentPosition$1(headDOM.node, headDOM.offset, domSel.focusNode, domSel.focusOffset))
        { return; }
    // Selection.extend can be used to create an 'inverted' selection
    // (one where the focus is before the anchor), but not all
    // browsers support it yet.
    var domSelExtended = false;
    if ((domSel.extend || anchor == head) && !brKludge) {
        domSel.collapse(anchorDOM.node, anchorDOM.offset);
        try {
            if (anchor != head)
                { domSel.extend(headDOM.node, headDOM.offset); }
            domSelExtended = true;
        }
        catch (_) {
            // In some cases with Chrome the selection is empty after calling
            // collapse, even when it should be valid. This appears to be a bug, but
            // it is difficult to isolate. If this happens fallback to the old path
            // without using extend.
            // Similarly, this could crash on Safari if the editor is hidden, and
            // there was no selection.
        }
    }
    if (!domSelExtended) {
        if (anchor > head) {
            var tmp = anchorDOM;
            anchorDOM = headDOM;
            headDOM = tmp;
        }
        var range = document.createRange();
        range.setEnd(headDOM.node, headDOM.offset);
        range.setStart(anchorDOM.node, anchorDOM.offset);
        domSel.removeAllRanges();
        domSel.addRange(range);
    }
};
ViewDesc.prototype.ignoreMutation = function ignoreMutation (mutation) {
    return !this.contentDOM && mutation.type != "selection";
};
prototypeAccessors$5.contentLost.get = function () {
    return this.contentDOM && this.contentDOM != this.dom && !this.dom.contains(this.contentDOM);
};
// Remove a subtree of the element tree that has been touched
// by a DOM change, so that the next update will redraw it.
ViewDesc.prototype.markDirty = function markDirty (from, to) {
    for (var offset = 0, i = 0; i < this.children.length; i++) {
        var child = this.children[i], end = offset + child.size;
        if (offset == end ? from <= end && to >= offset : from < end && to > offset) {
            var startInside = offset + child.border, endInside = end - child.border;
            if (from >= startInside && to <= endInside) {
                this.dirty = from == offset || to == end ? CONTENT_DIRTY : CHILD_DIRTY;
                if (from == startInside && to == endInside &&
                    (child.contentLost || child.dom.parentNode != this.contentDOM))
                    { child.dirty = NODE_DIRTY; }
                else
                    { child.markDirty(from - startInside, to - startInside); }
                return;
            }
            else {
                child.dirty = child.dom == child.contentDOM && child.dom.parentNode == this.contentDOM && !child.children.length
                    ? CONTENT_DIRTY : NODE_DIRTY;
            }
        }
        offset = end;
    }
    this.dirty = CONTENT_DIRTY;
};
ViewDesc.prototype.markParentsDirty = function markParentsDirty () {
    var level = 1;
    for (var node = this.parent; node; node = node.parent, level++) {
        var dirty = level == 1 ? CONTENT_DIRTY : CHILD_DIRTY;
        if (node.dirty < dirty)
            { node.dirty = dirty; }
    }
};
prototypeAccessors$5.domAtom.get = function () { return false; };
prototypeAccessors$5.ignoreForCoords.get = function () { return false; };
ViewDesc.prototype.isText = function isText (text) { return false; };

Object.defineProperties( ViewDesc.prototype, prototypeAccessors$5 );
// A widget desc represents a widget decoration, which is a DOM node
// drawn between the document nodes.
var WidgetViewDesc = /*@__PURE__*/(function (ViewDesc) {
    function WidgetViewDesc(parent, widget, view, pos) {
        var self, dom = widget.type.toDOM;
        if (typeof dom == "function")
            { dom = dom(view, function () {
                if (!self)
                    { return pos; }
                if (self.parent)
                    { return self.parent.posBeforeChild(self); }
            }); }
        if (!widget.type.spec.raw) {
            if (dom.nodeType != 1) {
                var wrap = document.createElement("span");
                wrap.appendChild(dom);
                dom = wrap;
            }
            dom.contentEditable = "false";
            dom.classList.add("ProseMirror-widget");
        }
        ViewDesc.call(this, parent, [], dom, null);
        this.widget = widget;
        this.widget = widget;
        self = this;
    }

    if ( ViewDesc ) WidgetViewDesc.__proto__ = ViewDesc;
    WidgetViewDesc.prototype = Object.create( ViewDesc && ViewDesc.prototype );
    WidgetViewDesc.prototype.constructor = WidgetViewDesc;

    var prototypeAccessors$1 = { domAtom: { configurable: true },side: { configurable: true } };
    WidgetViewDesc.prototype.matchesWidget = function matchesWidget (widget) {
        return this.dirty == NOT_DIRTY && widget.type.eq(this.widget.type);
    };
    WidgetViewDesc.prototype.parseRule = function parseRule () { return { ignore: true }; };
    WidgetViewDesc.prototype.stopEvent = function stopEvent (event) {
        var stop = this.widget.spec.stopEvent;
        return stop ? stop(event) : false;
    };
    WidgetViewDesc.prototype.ignoreMutation = function ignoreMutation (mutation) {
        return mutation.type != "selection" || this.widget.spec.ignoreSelection;
    };
    WidgetViewDesc.prototype.destroy = function destroy () {
        this.widget.type.destroy(this.dom);
        ViewDesc.prototype.destroy.call(this);
    };
    prototypeAccessors$1.domAtom.get = function () { return true; };
    prototypeAccessors$1.side.get = function () { return this.widget.type.side; };

    Object.defineProperties( WidgetViewDesc.prototype, prototypeAccessors$1 );

    return WidgetViewDesc;
}(ViewDesc));
var CompositionViewDesc = /*@__PURE__*/(function (ViewDesc) {
    function CompositionViewDesc(parent, dom, textDOM, text) {
        ViewDesc.call(this, parent, [], dom, null);
        this.textDOM = textDOM;
        this.text = text;
    }

    if ( ViewDesc ) CompositionViewDesc.__proto__ = ViewDesc;
    CompositionViewDesc.prototype = Object.create( ViewDesc && ViewDesc.prototype );
    CompositionViewDesc.prototype.constructor = CompositionViewDesc;

    var prototypeAccessors$2 = { size: { configurable: true } };
    prototypeAccessors$2.size.get = function () { return this.text.length; };
    CompositionViewDesc.prototype.localPosFromDOM = function localPosFromDOM (dom, offset) {
        if (dom != this.textDOM)
            { return this.posAtStart + (offset ? this.size : 0); }
        return this.posAtStart + offset;
    };
    CompositionViewDesc.prototype.domFromPos = function domFromPos (pos) {
        return { node: this.textDOM, offset: pos };
    };
    CompositionViewDesc.prototype.ignoreMutation = function ignoreMutation (mut) {
        return mut.type === 'characterData' && mut.target.nodeValue == mut.oldValue;
    };

    Object.defineProperties( CompositionViewDesc.prototype, prototypeAccessors$2 );

    return CompositionViewDesc;
}(ViewDesc));
// A mark desc represents a mark. May have multiple children,
// depending on how the mark is split. Note that marks are drawn using
// a fixed nesting order, for simplicity and predictability, so in
// some cases they will be split more often than would appear
// necessary.
var MarkViewDesc = /*@__PURE__*/(function (ViewDesc) {
    function MarkViewDesc(parent, mark, dom, contentDOM) {
        ViewDesc.call(this, parent, [], dom, contentDOM);
        this.mark = mark;
    }

    if ( ViewDesc ) MarkViewDesc.__proto__ = ViewDesc;
    MarkViewDesc.prototype = Object.create( ViewDesc && ViewDesc.prototype );
    MarkViewDesc.prototype.constructor = MarkViewDesc;
    MarkViewDesc.create = function create (parent, mark, inline, view) {
        var custom = view.nodeViews[mark.type.name];
        var spec = custom && custom(mark, view, inline);
        if (!spec || !spec.dom)
            { spec = DOMSerializer.renderSpec(document, mark.type.spec.toDOM(mark, inline), null, mark.attrs); }
        return new MarkViewDesc(parent, mark, spec.dom, spec.contentDOM || spec.dom);
    };
    MarkViewDesc.prototype.parseRule = function parseRule () {
        if ((this.dirty & NODE_DIRTY) || this.mark.type.spec.reparseInView)
            { return null; }
        return { mark: this.mark.type.name, attrs: this.mark.attrs, contentElement: this.contentDOM };
    };
    MarkViewDesc.prototype.matchesMark = function matchesMark (mark) { return this.dirty != NODE_DIRTY && this.mark.eq(mark); };
    MarkViewDesc.prototype.markDirty = function markDirty (from, to) {
        ViewDesc.prototype.markDirty.call(this, from, to);
        // Move dirty info to nearest node view
        if (this.dirty != NOT_DIRTY) {
            var parent = this.parent;
            while (!parent.node)
                { parent = parent.parent; }
            if (parent.dirty < this.dirty)
                { parent.dirty = this.dirty; }
            this.dirty = NOT_DIRTY;
        }
    };
    MarkViewDesc.prototype.slice = function slice (from, to, view) {
        var copy = MarkViewDesc.create(this.parent, this.mark, true, view);
        var nodes = this.children, size = this.size;
        if (to < size)
            { nodes = replaceNodes(nodes, to, size, view); }
        if (from > 0)
            { nodes = replaceNodes(nodes, 0, from, view); }
        for (var i = 0; i < nodes.length; i++)
            { nodes[i].parent = copy; }
        copy.children = nodes;
        return copy;
    };

    return MarkViewDesc;
}(ViewDesc));
// Node view descs are the main, most common type of view desc, and
// correspond to an actual node in the document. Unlike mark descs,
// they populate their child array themselves.
var NodeViewDesc = /*@__PURE__*/(function (ViewDesc) {
    function NodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM, nodeDOM, view, pos) {
        ViewDesc.call(this, parent, [], dom, contentDOM);
        this.node = node;
        this.outerDeco = outerDeco;
        this.innerDeco = innerDeco;
        this.nodeDOM = nodeDOM;
    }

    if ( ViewDesc ) NodeViewDesc.__proto__ = ViewDesc;
    NodeViewDesc.prototype = Object.create( ViewDesc && ViewDesc.prototype );
    NodeViewDesc.prototype.constructor = NodeViewDesc;

    var prototypeAccessors$3 = { size: { configurable: true },border: { configurable: true },domAtom: { configurable: true } };
    // By default, a node is rendered using the `toDOM` method from the
    // node type spec. But client code can use the `nodeViews` spec to
    // supply a custom node view, which can influence various aspects of
    // the way the node works.
    //
    // (Using subclassing for this was intentionally decided against,
    // since it'd require exposing a whole slew of finicky
    // implementation details to the user code that they probably will
    // never need.)
    NodeViewDesc.create = function create (parent, node, outerDeco, innerDeco, view, pos) {
        var assign;

        var custom = view.nodeViews[node.type.name], descObj;
        var spec = custom && custom(node, view, function () {
            // (This is a function that allows the custom view to find its
            // own position)
            if (!descObj)
                { return pos; }
            if (descObj.parent)
                { return descObj.parent.posBeforeChild(descObj); }
        }, outerDeco, innerDeco);
        var dom = spec && spec.dom, contentDOM = spec && spec.contentDOM;
        if (node.isText) {
            if (!dom)
                { dom = document.createTextNode(node.text); }
            else if (dom.nodeType != 3)
                { throw new RangeError("Text must be rendered as a DOM text node"); }
        }
        else if (!dom) {
            var spec$1 = DOMSerializer.renderSpec(document, node.type.spec.toDOM(node), null, node.attrs);
            ((assign = spec$1, dom = assign.dom, contentDOM = assign.contentDOM));
        }
        if (!contentDOM && !node.isText && dom.nodeName != "BR") { // Chrome gets confused by <br contenteditable=false>
            if (!dom.hasAttribute("contenteditable"))
                { dom.contentEditable = "false"; }
            if (node.type.spec.draggable)
                { dom.draggable = true; }
        }
        var nodeDOM = dom;
        dom = applyOuterDeco(dom, outerDeco, node);
        if (spec)
            { return descObj = new CustomNodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM || null, nodeDOM, spec, view, pos + 1); }
        else if (node.isText)
            { return new TextViewDesc(parent, node, outerDeco, innerDeco, dom, nodeDOM, view); }
        else
            { return new NodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM || null, nodeDOM); }
    };
    NodeViewDesc.prototype.parseRule = function parseRule () {
        var this$1$1 = this;

        // Experimental kludge to allow opt-in re-parsing of nodes
        if (this.node.type.spec.reparseInView)
            { return null; }
        // FIXME the assumption that this can always return the current
        // attrs means that if the user somehow manages to change the
        // attrs in the dom, that won't be picked up. Not entirely sure
        // whether this is a problem
        var rule = { node: this.node.type.name, attrs: this.node.attrs };
        if (this.node.type.whitespace == "pre")
            { rule.preserveWhitespace = "full"; }
        if (!this.contentDOM) {
            rule.getContent = function () { return this$1$1.node.content; };
        }
        else if (!this.contentLost) {
            rule.contentElement = this.contentDOM;
        }
        else {
            // Chrome likes to randomly recreate parent nodes when
            // backspacing things. When that happens, this tries to find the
            // new parent.
            for (var i = this.children.length - 1; i >= 0; i--) {
                var child = this.children[i];
                if (this.dom.contains(child.dom.parentNode)) {
                    rule.contentElement = child.dom.parentNode;
                    break;
                }
            }
            if (!rule.contentElement)
                { rule.getContent = function () { return Fragment.empty; }; }
        }
        return rule;
    };
    NodeViewDesc.prototype.matchesNode = function matchesNode (node, outerDeco, innerDeco) {
        return this.dirty == NOT_DIRTY && node.eq(this.node) &&
            sameOuterDeco(outerDeco, this.outerDeco) && innerDeco.eq(this.innerDeco);
    };
    prototypeAccessors$3.size.get = function () { return this.node.nodeSize; };
    prototypeAccessors$3.border.get = function () { return this.node.isLeaf ? 0 : 1; };
    // Syncs `this.children` to match `this.node.content` and the local
    // decorations, possibly introducing nesting for marks. Then, in a
    // separate step, syncs the DOM inside `this.contentDOM` to
    // `this.children`.
    NodeViewDesc.prototype.updateChildren = function updateChildren (view, pos) {
        var this$1$1 = this;

        var inline = this.node.inlineContent, off = pos;
        var composition = view.composing ? this.localCompositionInfo(view, pos) : null;
        var localComposition = composition && composition.pos > -1 ? composition : null;
        var compositionInChild = composition && composition.pos < 0;
        var updater = new ViewTreeUpdater(this, localComposition && localComposition.node, view);
        iterDeco(this.node, this.innerDeco, function (widget, i, insideNode) {
            if (widget.spec.marks)
                { updater.syncToMarks(widget.spec.marks, inline, view); }
            else if (widget.type.side >= 0 && !insideNode)
                { updater.syncToMarks(i == this$1$1.node.childCount ? Mark.none : this$1$1.node.child(i).marks, inline, view); }
            // If the next node is a desc matching this widget, reuse it,
            // otherwise insert the widget as a new view desc.
            updater.placeWidget(widget, view, off);
        }, function (child, outerDeco, innerDeco, i) {
            // Make sure the wrapping mark descs match the node's marks.
            updater.syncToMarks(child.marks, inline, view);
            // Try several strategies for drawing this node
            var compIndex;
            if (updater.findNodeMatch(child, outerDeco, innerDeco, i)) ;
            else if (compositionInChild && view.state.selection.from > off &&
                view.state.selection.to < off + child.nodeSize &&
                (compIndex = updater.findIndexWithChild(composition.node)) > -1 &&
                updater.updateNodeAt(child, outerDeco, innerDeco, compIndex, view)) ;
            else if (updater.updateNextNode(child, outerDeco, innerDeco, view, i, off)) ;
            else {
                // Add it as a new view
                updater.addNode(child, outerDeco, innerDeco, view, off);
            }
            off += child.nodeSize;
        });
        // Drop all remaining descs after the current position.
        updater.syncToMarks([], inline, view);
        if (this.node.isTextblock)
            { updater.addTextblockHacks(); }
        updater.destroyRest();
        // Sync the DOM if anything changed
        if (updater.changed || this.dirty == CONTENT_DIRTY) {
            // May have to protect focused DOM from being changed if a composition is active
            if (localComposition)
                { this.protectLocalComposition(view, localComposition); }
            renderDescs(this.contentDOM, this.children, view);
            if (ios$1)
                { iosHacks(this.dom); }
        }
    };
    NodeViewDesc.prototype.localCompositionInfo = function localCompositionInfo (view, pos) {
        // Only do something if both the selection and a focused text node
        // are inside of this node
        var ref = view.state.selection;
        var from = ref.from;
        var to = ref.to;
        if (!(view.state.selection instanceof TextSelection) || from < pos || to > pos + this.node.content.size)
            { return null; }
        var textNode = view.input.compositionNode;
        if (!textNode || !this.dom.contains(textNode.parentNode))
            { return null; }
        if (this.node.inlineContent) {
            // Find the text in the focused node in the node, stop if it's not
            // there (may have been modified through other means, in which
            // case it should overwritten)
            var text = textNode.nodeValue;
            var textPos = findTextInFragment(this.node.content, text, from - pos, to - pos);
            return textPos < 0 ? null : { node: textNode, pos: textPos, text: text };
        }
        else {
            return { node: textNode, pos: -1, text: "" };
        }
    };
    NodeViewDesc.prototype.protectLocalComposition = function protectLocalComposition (view, ref) {
        var node = ref.node;
        var pos = ref.pos;
        var text = ref.text;

        // The node is already part of a local view desc, leave it there
        if (this.getDesc(node))
            { return; }
        // Create a composition view for the orphaned nodes
        var topNode = node;
        for (;; topNode = topNode.parentNode) {
            if (topNode.parentNode == this.contentDOM)
                { break; }
            while (topNode.previousSibling)
                { topNode.parentNode.removeChild(topNode.previousSibling); }
            while (topNode.nextSibling)
                { topNode.parentNode.removeChild(topNode.nextSibling); }
            if (topNode.pmViewDesc)
                { topNode.pmViewDesc = undefined; }
        }
        var desc = new CompositionViewDesc(this, topNode, node, text);
        view.input.compositionNodes.push(desc);
        // Patch up this.children to contain the composition view
        this.children = replaceNodes(this.children, pos, pos + text.length, view, desc);
    };
    // If this desc must be updated to match the given node decoration,
    // do so and return true.
    NodeViewDesc.prototype.update = function update (node, outerDeco, innerDeco, view) {
        if (this.dirty == NODE_DIRTY ||
            !node.sameMarkup(this.node))
            { return false; }
        this.updateInner(node, outerDeco, innerDeco, view);
        return true;
    };
    NodeViewDesc.prototype.updateInner = function updateInner (node, outerDeco, innerDeco, view) {
        this.updateOuterDeco(outerDeco);
        this.node = node;
        this.innerDeco = innerDeco;
        if (this.contentDOM)
            { this.updateChildren(view, this.posAtStart); }
        this.dirty = NOT_DIRTY;
    };
    NodeViewDesc.prototype.updateOuterDeco = function updateOuterDeco (outerDeco) {
        if (sameOuterDeco(outerDeco, this.outerDeco))
            { return; }
        var needsWrap = this.nodeDOM.nodeType != 1;
        var oldDOM = this.dom;
        this.dom = patchOuterDeco(this.dom, this.nodeDOM, computeOuterDeco(this.outerDeco, this.node, needsWrap), computeOuterDeco(outerDeco, this.node, needsWrap));
        if (this.dom != oldDOM) {
            oldDOM.pmViewDesc = undefined;
            this.dom.pmViewDesc = this;
        }
        this.outerDeco = outerDeco;
    };
    // Mark this node as being the selected node.
    NodeViewDesc.prototype.selectNode = function selectNode () {
        if (this.nodeDOM.nodeType == 1)
            { this.nodeDOM.classList.add("ProseMirror-selectednode"); }
        if (this.contentDOM || !this.node.type.spec.draggable)
            { this.dom.draggable = true; }
    };
    // Remove selected node marking from this node.
    NodeViewDesc.prototype.deselectNode = function deselectNode () {
        if (this.nodeDOM.nodeType == 1) {
            this.nodeDOM.classList.remove("ProseMirror-selectednode");
            if (this.contentDOM || !this.node.type.spec.draggable)
                { this.dom.removeAttribute("draggable"); }
        }
    };
    prototypeAccessors$3.domAtom.get = function () { return this.node.isAtom; };

    Object.defineProperties( NodeViewDesc.prototype, prototypeAccessors$3 );

    return NodeViewDesc;
}(ViewDesc));
// Create a view desc for the top-level document node, to be exported
// and used by the view class.
function docViewDesc(doc, outerDeco, innerDeco, dom, view) {
    applyOuterDeco(dom, outerDeco, doc);
    var docView = new NodeViewDesc(undefined, doc, outerDeco, innerDeco, dom, dom, dom, view, 0);
    if (docView.contentDOM)
        { docView.updateChildren(view, 0); }
    return docView;
}
var TextViewDesc = /*@__PURE__*/(function (NodeViewDesc) {
    function TextViewDesc(parent, node, outerDeco, innerDeco, dom, nodeDOM, view) {
        NodeViewDesc.call(this, parent, node, outerDeco, innerDeco, dom, null, nodeDOM, view, 0);
    }

    if ( NodeViewDesc ) TextViewDesc.__proto__ = NodeViewDesc;
    TextViewDesc.prototype = Object.create( NodeViewDesc && NodeViewDesc.prototype );
    TextViewDesc.prototype.constructor = TextViewDesc;

    var prototypeAccessors$4 = { domAtom: { configurable: true } };
    TextViewDesc.prototype.parseRule = function parseRule () {
        var skip = this.nodeDOM.parentNode;
        while (skip && skip != this.dom && !skip.pmIsDeco)
            { skip = skip.parentNode; }
        return { skip: (skip || true) };
    };
    TextViewDesc.prototype.update = function update (node, outerDeco, innerDeco, view) {
        if (this.dirty == NODE_DIRTY || (this.dirty != NOT_DIRTY && !this.inParent()) ||
            !node.sameMarkup(this.node))
            { return false; }
        this.updateOuterDeco(outerDeco);
        if ((this.dirty != NOT_DIRTY || node.text != this.node.text) && node.text != this.nodeDOM.nodeValue) {
            this.nodeDOM.nodeValue = node.text;
            if (view.trackWrites == this.nodeDOM)
                { view.trackWrites = null; }
        }
        this.node = node;
        this.dirty = NOT_DIRTY;
        return true;
    };
    TextViewDesc.prototype.inParent = function inParent () {
        var parentDOM = this.parent.contentDOM;
        for (var n = this.nodeDOM; n; n = n.parentNode)
            { if (n == parentDOM)
                { return true; } }
        return false;
    };
    TextViewDesc.prototype.domFromPos = function domFromPos (pos) {
        return { node: this.nodeDOM, offset: pos };
    };
    TextViewDesc.prototype.localPosFromDOM = function localPosFromDOM (dom, offset, bias) {
        if (dom == this.nodeDOM)
            { return this.posAtStart + Math.min(offset, this.node.text.length); }
        return NodeViewDesc.prototype.localPosFromDOM.call(this, dom, offset, bias);
    };
    TextViewDesc.prototype.ignoreMutation = function ignoreMutation (mutation) {
        return mutation.type != "characterData" && mutation.type != "selection";
    };
    TextViewDesc.prototype.slice = function slice (from, to, view) {
        var node = this.node.cut(from, to), dom = document.createTextNode(node.text);
        return new TextViewDesc(this.parent, node, this.outerDeco, this.innerDeco, dom, dom, view);
    };
    TextViewDesc.prototype.markDirty = function markDirty (from, to) {
        NodeViewDesc.prototype.markDirty.call(this, from, to);
        if (this.dom != this.nodeDOM && (from == 0 || to == this.nodeDOM.nodeValue.length))
            { this.dirty = NODE_DIRTY; }
    };
    prototypeAccessors$4.domAtom.get = function () { return false; };
    TextViewDesc.prototype.isText = function isText (text) { return this.node.text == text; };

    Object.defineProperties( TextViewDesc.prototype, prototypeAccessors$4 );

    return TextViewDesc;
}(NodeViewDesc));
// A dummy desc used to tag trailing BR or IMG nodes created to work
// around contentEditable terribleness.
var TrailingHackViewDesc = /*@__PURE__*/(function (ViewDesc) {
    function TrailingHackViewDesc () {
        ViewDesc.apply(this, arguments);
    }

    if ( ViewDesc ) TrailingHackViewDesc.__proto__ = ViewDesc;
    TrailingHackViewDesc.prototype = Object.create( ViewDesc && ViewDesc.prototype );
    TrailingHackViewDesc.prototype.constructor = TrailingHackViewDesc;

    var prototypeAccessors$5 = { domAtom: { configurable: true },ignoreForCoords: { configurable: true } };

    TrailingHackViewDesc.prototype.parseRule = function parseRule () { return { ignore: true }; };
    TrailingHackViewDesc.prototype.matchesHack = function matchesHack (nodeName) { return this.dirty == NOT_DIRTY && this.dom.nodeName == nodeName; };
    prototypeAccessors$5.domAtom.get = function () { return true; };
    prototypeAccessors$5.ignoreForCoords.get = function () { return this.dom.nodeName == "IMG"; };

    Object.defineProperties( TrailingHackViewDesc.prototype, prototypeAccessors$5 );

    return TrailingHackViewDesc;
}(ViewDesc));
// A separate subclass is used for customized node views, so that the
// extra checks only have to be made for nodes that are actually
// customized.
var CustomNodeViewDesc = /*@__PURE__*/(function (NodeViewDesc) {
    function CustomNodeViewDesc(parent, node, outerDeco, innerDeco, dom, contentDOM, nodeDOM, spec, view, pos) {
        NodeViewDesc.call(this, parent, node, outerDeco, innerDeco, dom, contentDOM, nodeDOM, view, pos);
        this.spec = spec;
    }

    if ( NodeViewDesc ) CustomNodeViewDesc.__proto__ = NodeViewDesc;
    CustomNodeViewDesc.prototype = Object.create( NodeViewDesc && NodeViewDesc.prototype );
    CustomNodeViewDesc.prototype.constructor = CustomNodeViewDesc;
    // A custom `update` method gets to decide whether the update goes
    // through. If it does, and there's a `contentDOM` node, our logic
    // updates the children.
    CustomNodeViewDesc.prototype.update = function update (node, outerDeco, innerDeco, view) {
        if (this.dirty == NODE_DIRTY)
            { return false; }
        if (this.spec.update) {
            var result = this.spec.update(node, outerDeco, innerDeco);
            if (result)
                { this.updateInner(node, outerDeco, innerDeco, view); }
            return result;
        }
        else if (!this.contentDOM && !node.isLeaf) {
            return false;
        }
        else {
            return NodeViewDesc.prototype.update.call(this, node, outerDeco, innerDeco, view);
        }
    };
    CustomNodeViewDesc.prototype.selectNode = function selectNode () {
        this.spec.selectNode ? this.spec.selectNode() : NodeViewDesc.prototype.selectNode.call(this);
    };
    CustomNodeViewDesc.prototype.deselectNode = function deselectNode () {
        this.spec.deselectNode ? this.spec.deselectNode() : NodeViewDesc.prototype.deselectNode.call(this);
    };
    CustomNodeViewDesc.prototype.setSelection = function setSelection (anchor, head, root, force) {
        this.spec.setSelection ? this.spec.setSelection(anchor, head, root)
            : NodeViewDesc.prototype.setSelection.call(this, anchor, head, root, force);
    };
    CustomNodeViewDesc.prototype.destroy = function destroy () {
        if (this.spec.destroy)
            { this.spec.destroy(); }
        NodeViewDesc.prototype.destroy.call(this);
    };
    CustomNodeViewDesc.prototype.stopEvent = function stopEvent (event) {
        return this.spec.stopEvent ? this.spec.stopEvent(event) : false;
    };
    CustomNodeViewDesc.prototype.ignoreMutation = function ignoreMutation (mutation) {
        return this.spec.ignoreMutation ? this.spec.ignoreMutation(mutation) : NodeViewDesc.prototype.ignoreMutation.call(this, mutation);
    };

    return CustomNodeViewDesc;
}(NodeViewDesc));
// Sync the content of the given DOM node with the nodes associated
// with the given array of view descs, recursing into mark descs
// because this should sync the subtree for a whole node at a time.
function renderDescs(parentDOM, descs, view) {
    var dom = parentDOM.firstChild, written = false;
    for (var i = 0; i < descs.length; i++) {
        var desc = descs[i], childDOM = desc.dom;
        if (childDOM.parentNode == parentDOM) {
            while (childDOM != dom) {
                dom = rm(dom);
                written = true;
            }
            dom = dom.nextSibling;
        }
        else {
            written = true;
            parentDOM.insertBefore(childDOM, dom);
        }
        if (desc instanceof MarkViewDesc) {
            var pos = dom ? dom.previousSibling : parentDOM.lastChild;
            renderDescs(desc.contentDOM, desc.children, view);
            dom = pos ? pos.nextSibling : parentDOM.firstChild;
        }
    }
    while (dom) {
        dom = rm(dom);
        written = true;
    }
    if (written && view.trackWrites == parentDOM)
        { view.trackWrites = null; }
}
var OuterDecoLevel = function (nodeName) {
    if (nodeName)
        { this.nodeName = nodeName; }
};
OuterDecoLevel.prototype = Object.create(null);
var noDeco = [new OuterDecoLevel];
function computeOuterDeco(outerDeco, node, needsWrap) {
    if (outerDeco.length == 0)
        { return noDeco; }
    var top = needsWrap ? noDeco[0] : new OuterDecoLevel, result = [top];
    for (var i = 0; i < outerDeco.length; i++) {
        var attrs = outerDeco[i].type.attrs;
        if (!attrs)
            { continue; }
        if (attrs.nodeName)
            { result.push(top = new OuterDecoLevel(attrs.nodeName)); }
        for (var name in attrs) {
            var val = attrs[name];
            if (val == null)
                { continue; }
            if (needsWrap && result.length == 1)
                { result.push(top = new OuterDecoLevel(node.isInline ? "span" : "div")); }
            if (name == "class")
                { top.class = (top.class ? top.class + " " : "") + val; }
            else if (name == "style")
                { top.style = (top.style ? top.style + ";" : "") + val; }
            else if (name != "nodeName")
                { top[name] = val; }
        }
    }
    return result;
}
function patchOuterDeco(outerDOM, nodeDOM, prevComputed, curComputed) {
    // Shortcut for trivial case
    if (prevComputed == noDeco && curComputed == noDeco)
        { return nodeDOM; }
    var curDOM = nodeDOM;
    for (var i = 0; i < curComputed.length; i++) {
        var deco = curComputed[i], prev = prevComputed[i];
        if (i) {
            var parent = (void 0);
            if (prev && prev.nodeName == deco.nodeName && curDOM != outerDOM &&
                (parent = curDOM.parentNode) && parent.nodeName.toLowerCase() == deco.nodeName) {
                curDOM = parent;
            }
            else {
                parent = document.createElement(deco.nodeName);
                parent.pmIsDeco = true;
                parent.appendChild(curDOM);
                prev = noDeco[0];
                curDOM = parent;
            }
        }
        patchAttributes(curDOM, prev || noDeco[0], deco);
    }
    return curDOM;
}
function patchAttributes(dom, prev, cur) {
    for (var name in prev)
        { if (name != "class" && name != "style" && name != "nodeName" && !(name in cur))
            { dom.removeAttribute(name); } }
    for (var name$1 in cur)
        { if (name$1 != "class" && name$1 != "style" && name$1 != "nodeName" && cur[name$1] != prev[name$1])
            { dom.setAttribute(name$1, cur[name$1]); } }
    if (prev.class != cur.class) {
        var prevList = prev.class ? prev.class.split(" ").filter(Boolean) : [];
        var curList = cur.class ? cur.class.split(" ").filter(Boolean) : [];
        for (var i = 0; i < prevList.length; i++)
            { if (curList.indexOf(prevList[i]) == -1)
                { dom.classList.remove(prevList[i]); } }
        for (var i$1 = 0; i$1 < curList.length; i$1++)
            { if (prevList.indexOf(curList[i$1]) == -1)
                { dom.classList.add(curList[i$1]); } }
        if (dom.classList.length == 0)
            { dom.removeAttribute("class"); }
    }
    if (prev.style != cur.style) {
        if (prev.style) {
            var prop = /\s*([\w\-\xa1-\uffff]+)\s*:(?:"(?:\\.|[^"])*"|'(?:\\.|[^'])*'|\(.*?\)|[^;])*/g, m;
            while (m = prop.exec(prev.style))
                { dom.style.removeProperty(m[1]); }
        }
        if (cur.style)
            { dom.style.cssText += cur.style; }
    }
}
function applyOuterDeco(dom, deco, node) {
    return patchOuterDeco(dom, dom, noDeco, computeOuterDeco(deco, node, dom.nodeType != 1));
}
function sameOuterDeco(a, b) {
    if (a.length != b.length)
        { return false; }
    for (var i = 0; i < a.length; i++)
        { if (!a[i].type.eq(b[i].type))
            { return false; } }
    return true;
}
// Remove a DOM node and return its next sibling.
function rm(dom) {
    var next = dom.nextSibling;
    dom.parentNode.removeChild(dom);
    return next;
}
// Helper class for incrementally updating a tree of mark descs and
// the widget and node descs inside of them.
var ViewTreeUpdater = function ViewTreeUpdater(top, lock, view) {
    this.lock = lock;
    this.view = view;
    // Index into `this.top`'s child array, represents the current
    // update position.
    this.index = 0;
    // When entering a mark, the current top and index are pushed
    // onto this.
    this.stack = [];
    // Tracks whether anything was changed
    this.changed = false;
    this.top = top;
    this.preMatch = preMatch(top.node.content, top);
};
// Destroy and remove the children between the given indices in
// `this.top`.
ViewTreeUpdater.prototype.destroyBetween = function destroyBetween (start, end) {
    if (start == end)
        { return; }
    for (var i = start; i < end; i++)
        { this.top.children[i].destroy(); }
    this.top.children.splice(start, end - start);
    this.changed = true;
};
// Destroy all remaining children in `this.top`.
ViewTreeUpdater.prototype.destroyRest = function destroyRest () {
    this.destroyBetween(this.index, this.top.children.length);
};
// Sync the current stack of mark descs with the given array of
// marks, reusing existing mark descs when possible.
ViewTreeUpdater.prototype.syncToMarks = function syncToMarks (marks, inline, view) {
    var keep = 0, depth = this.stack.length >> 1;
    var maxKeep = Math.min(depth, marks.length);
    while (keep < maxKeep &&
        (keep == depth - 1 ? this.top : this.stack[(keep + 1) << 1])
            .matchesMark(marks[keep]) && marks[keep].type.spec.spanning !== false)
        { keep++; }
    while (keep < depth) {
        this.destroyRest();
        this.top.dirty = NOT_DIRTY;
        this.index = this.stack.pop();
        this.top = this.stack.pop();
        depth--;
    }
    while (depth < marks.length) {
        this.stack.push(this.top, this.index + 1);
        var found = -1;
        for (var i = this.index; i < Math.min(this.index + 3, this.top.children.length); i++) {
            var next = this.top.children[i];
            if (next.matchesMark(marks[depth]) && !this.isLocked(next.dom)) {
                found = i;
                break;
            }
        }
        if (found > -1) {
            if (found > this.index) {
                this.changed = true;
                this.destroyBetween(this.index, found);
            }
            this.top = this.top.children[this.index];
        }
        else {
            var markDesc = MarkViewDesc.create(this.top, marks[depth], inline, view);
            this.top.children.splice(this.index, 0, markDesc);
            this.top = markDesc;
            this.changed = true;
        }
        this.index = 0;
        depth++;
    }
};
// Try to find a node desc matching the given data. Skip over it and
// return true when successful.
ViewTreeUpdater.prototype.findNodeMatch = function findNodeMatch (node, outerDeco, innerDeco, index) {
    var found = -1, targetDesc;
    if (index >= this.preMatch.index &&
        (targetDesc = this.preMatch.matches[index - this.preMatch.index]).parent == this.top &&
        targetDesc.matchesNode(node, outerDeco, innerDeco)) {
        found = this.top.children.indexOf(targetDesc, this.index);
    }
    else {
        for (var i = this.index, e = Math.min(this.top.children.length, i + 5); i < e; i++) {
            var child = this.top.children[i];
            if (child.matchesNode(node, outerDeco, innerDeco) && !this.preMatch.matched.has(child)) {
                found = i;
                break;
            }
        }
    }
    if (found < 0)
        { return false; }
    this.destroyBetween(this.index, found);
    this.index++;
    return true;
};
ViewTreeUpdater.prototype.updateNodeAt = function updateNodeAt (node, outerDeco, innerDeco, index, view) {
    var child = this.top.children[index];
    if (child.dirty == NODE_DIRTY && child.dom == child.contentDOM)
        { child.dirty = CONTENT_DIRTY; }
    if (!child.update(node, outerDeco, innerDeco, view))
        { return false; }
    this.destroyBetween(this.index, index);
    this.index++;
    return true;
};
ViewTreeUpdater.prototype.findIndexWithChild = function findIndexWithChild (domNode) {
    for (;;) {
        var parent = domNode.parentNode;
        if (!parent)
            { return -1; }
        if (parent == this.top.contentDOM) {
            var desc = domNode.pmViewDesc;
            if (desc)
                { for (var i = this.index; i < this.top.children.length; i++) {
                    if (this.top.children[i] == desc)
                        { return i; }
                } }
            return -1;
        }
        domNode = parent;
    }
};
// Try to update the next node, if any, to the given data. Checks
// pre-matches to avoid overwriting nodes that could still be used.
ViewTreeUpdater.prototype.updateNextNode = function updateNextNode (node, outerDeco, innerDeco, view, index, pos) {
    for (var i = this.index; i < this.top.children.length; i++) {
        var next = this.top.children[i];
        if (next instanceof NodeViewDesc) {
            var preMatch = this.preMatch.matched.get(next);
            if (preMatch != null && preMatch != index)
                { return false; }
            var nextDOM = next.dom, updated = (void 0);
            // Can't update if nextDOM is or contains this.lock, except if
            // it's a text node whose content already matches the new text
            // and whose decorations match the new ones.
            var locked = this.isLocked(nextDOM) &&
                !(node.isText && next.node && next.node.isText && next.nodeDOM.nodeValue == node.text &&
                    next.dirty != NODE_DIRTY && sameOuterDeco(outerDeco, next.outerDeco));
            if (!locked && next.update(node, outerDeco, innerDeco, view)) {
                this.destroyBetween(this.index, i);
                if (next.dom != nextDOM)
                    { this.changed = true; }
                this.index++;
                return true;
            }
            else if (!locked && (updated = this.recreateWrapper(next, node, outerDeco, innerDeco, view, pos))) {
                this.destroyBetween(this.index, i);
                this.top.children[this.index] = updated;
                if (updated.contentDOM) {
                    updated.dirty = CONTENT_DIRTY;
                    updated.updateChildren(view, pos + 1);
                    updated.dirty = NOT_DIRTY;
                }
                this.changed = true;
                this.index++;
                return true;
            }
            break;
        }
    }
    return false;
};
// When a node with content is replaced by a different node with
// identical content, move over its children.
ViewTreeUpdater.prototype.recreateWrapper = function recreateWrapper (next, node, outerDeco, innerDeco, view, pos) {
    if (next.dirty || node.isAtom || !next.children.length ||
        !next.node.content.eq(node.content) ||
        !sameOuterDeco(outerDeco, next.outerDeco) || !innerDeco.eq(next.innerDeco))
        { return null; }
    var wrapper = NodeViewDesc.create(this.top, node, outerDeco, innerDeco, view, pos);
    if (wrapper.contentDOM) {
        wrapper.children = next.children;
        next.children = [];
        for (var ch of wrapper.children)
            ch.parent = wrapper;
    }
    next.destroy();
    return wrapper;
};
// Insert the node as a newly created node desc.
ViewTreeUpdater.prototype.addNode = function addNode (node, outerDeco, innerDeco, view, pos) {
    var desc = NodeViewDesc.create(this.top, node, outerDeco, innerDeco, view, pos);
    if (desc.contentDOM)
        { desc.updateChildren(view, pos + 1); }
    this.top.children.splice(this.index++, 0, desc);
    this.changed = true;
};
ViewTreeUpdater.prototype.placeWidget = function placeWidget (widget, view, pos) {
    var next = this.index < this.top.children.length ? this.top.children[this.index] : null;
    if (next && next.matchesWidget(widget) &&
        (widget == next.widget || !next.widget.type.toDOM.parentNode)) {
        this.index++;
    }
    else {
        var desc = new WidgetViewDesc(this.top, widget, view, pos);
        this.top.children.splice(this.index++, 0, desc);
        this.changed = true;
    }
};
// Make sure a textblock looks and behaves correctly in
// contentEditable.
ViewTreeUpdater.prototype.addTextblockHacks = function addTextblockHacks () {
    var lastChild = this.top.children[this.index - 1], parent = this.top;
    while (lastChild instanceof MarkViewDesc) {
        parent = lastChild;
        lastChild = parent.children[parent.children.length - 1];
    }
    if (!lastChild || // Empty textblock
        !(lastChild instanceof TextViewDesc) ||
        /\n$/.test(lastChild.node.text) ||
        (this.view.requiresGeckoHackNode && /\s$/.test(lastChild.node.text))) {
        // Avoid bugs in Safari's cursor drawing (#1165) and Chrome's mouse selection (#1152)
        if ((safari$1 || chrome$1) && lastChild && lastChild.dom.contentEditable == "false")
            { this.addHackNode("IMG", parent); }
        this.addHackNode("BR", this.top);
    }
};
ViewTreeUpdater.prototype.addHackNode = function addHackNode (nodeName, parent) {
    if (parent == this.top && this.index < parent.children.length && parent.children[this.index].matchesHack(nodeName)) {
        this.index++;
    }
    else {
        var dom = document.createElement(nodeName);
        if (nodeName == "IMG") {
            dom.className = "ProseMirror-separator";
            dom.alt = "";
        }
        if (nodeName == "BR")
            { dom.className = "ProseMirror-trailingBreak"; }
        var hack = new TrailingHackViewDesc(this.top, [], dom, null);
        if (parent != this.top)
            { parent.children.push(hack); }
        else
            { parent.children.splice(this.index++, 0, hack); }
        this.changed = true;
    }
};
ViewTreeUpdater.prototype.isLocked = function isLocked (node) {
    return this.lock && (node == this.lock || node.nodeType == 1 && node.contains(this.lock.parentNode));
};
// Iterate from the end of the fragment and array of descs to find
// directly matching ones, in order to avoid overeagerly reusing those
// for other nodes. Returns the fragment index of the first node that
// is part of the sequence of matched nodes at the end of the
// fragment.
function preMatch(frag, parentDesc) {
    var curDesc = parentDesc, descI = curDesc.children.length;
    var fI = frag.childCount, matched = new Map, matches = [];
    outer: while (fI > 0) {
        var desc = (void 0);
        for (;;) {
            if (descI) {
                var next = curDesc.children[descI - 1];
                if (next instanceof MarkViewDesc) {
                    curDesc = next;
                    descI = next.children.length;
                }
                else {
                    desc = next;
                    descI--;
                    break;
                }
            }
            else if (curDesc == parentDesc) {
                break outer;
            }
            else {
                // FIXME
                descI = curDesc.parent.children.indexOf(curDesc);
                curDesc = curDesc.parent;
            }
        }
        var node = desc.node;
        if (!node)
            { continue; }
        if (node != frag.child(fI - 1))
            { break; }
        --fI;
        matched.set(desc, fI);
        matches.push(desc);
    }
    return { index: fI, matched: matched, matches: matches.reverse() };
}
function compareSide(a, b) {
    return a.type.side - b.type.side;
}
// This function abstracts iterating over the nodes and decorations in
// a fragment. Calls `onNode` for each node, with its local and child
// decorations. Splits text nodes when there is a decoration starting
// or ending inside of them. Calls `onWidget` for each widget.
function iterDeco(parent, deco, onWidget, onNode) {
    var locals = deco.locals(parent), offset = 0;
    // Simple, cheap variant for when there are no local decorations
    if (locals.length == 0) {
        for (var i = 0; i < parent.childCount; i++) {
            var child = parent.child(i);
            onNode(child, locals, deco.forChild(offset, child), i);
            offset += child.nodeSize;
        }
        return;
    }
    var decoIndex = 0, active = [], restNode = null;
    for (var parentIndex = 0;;) {
        var widget = (void 0), widgets = (void 0);
        while (decoIndex < locals.length && locals[decoIndex].to == offset) {
            var next = locals[decoIndex++];
            if (next.widget) {
                if (!widget)
                    { widget = next; }
                else
                    { (widgets || (widgets = [widget])).push(next); }
            }
        }
        if (widget) {
            if (widgets) {
                widgets.sort(compareSide);
                for (var i$1 = 0; i$1 < widgets.length; i$1++)
                    { onWidget(widgets[i$1], parentIndex, !!restNode); }
            }
            else {
                onWidget(widget, parentIndex, !!restNode);
            }
        }
        var child$1 = (void 0), index = (void 0);
        if (restNode) {
            index = -1;
            child$1 = restNode;
            restNode = null;
        }
        else if (parentIndex < parent.childCount) {
            index = parentIndex;
            child$1 = parent.child(parentIndex++);
        }
        else {
            break;
        }
        for (var i$2 = 0; i$2 < active.length; i$2++)
            { if (active[i$2].to <= offset)
                { active.splice(i$2--, 1); } }
        while (decoIndex < locals.length && locals[decoIndex].from <= offset && locals[decoIndex].to > offset)
            { active.push(locals[decoIndex++]); }
        var end = offset + child$1.nodeSize;
        if (child$1.isText) {
            var cutAt = end;
            if (decoIndex < locals.length && locals[decoIndex].from < cutAt)
                { cutAt = locals[decoIndex].from; }
            for (var i$3 = 0; i$3 < active.length; i$3++)
                { if (active[i$3].to < cutAt)
                    { cutAt = active[i$3].to; } }
            if (cutAt < end) {
                restNode = child$1.cut(cutAt - offset);
                child$1 = child$1.cut(0, cutAt - offset);
                end = cutAt;
                index = -1;
            }
        }
        else {
            while (decoIndex < locals.length && locals[decoIndex].to < end)
                { decoIndex++; }
        }
        var outerDeco = child$1.isInline && !child$1.isLeaf ? active.filter(function (d) { return !d.inline; }) : active.slice();
        onNode(child$1, outerDeco, deco.forChild(offset, child$1), index);
        offset = end;
    }
}
// List markers in Mobile Safari will mysteriously disappear
// sometimes. This works around that.
function iosHacks(dom) {
    if (dom.nodeName == "UL" || dom.nodeName == "OL") {
        var oldCSS = dom.style.cssText;
        dom.style.cssText = oldCSS + "; list-style: square !important";
        window.getComputedStyle(dom).listStyle;
        dom.style.cssText = oldCSS;
    }
}
// Find a piece of text in an inline fragment, overlapping from-to
function findTextInFragment(frag, text, from, to) {
    for (var i = 0, pos = 0; i < frag.childCount && pos <= to;) {
        var child = frag.child(i++), childStart = pos;
        pos += child.nodeSize;
        if (!child.isText)
            { continue; }
        var str = child.text;
        while (i < frag.childCount) {
            var next = frag.child(i++);
            pos += next.nodeSize;
            if (!next.isText)
                { break; }
            str += next.text;
        }
        if (pos >= from) {
            if (pos >= to && str.slice(to - text.length - childStart, to - childStart) == text)
                { return to - text.length; }
            var found = childStart < to ? str.lastIndexOf(text, to - childStart - 1) : -1;
            if (found >= 0 && found + text.length + childStart >= from)
                { return childStart + found; }
            if (from == to && str.length >= (to + text.length) - childStart &&
                str.slice(to - childStart, to - childStart + text.length) == text)
                { return to; }
        }
    }
    return -1;
}
// Replace range from-to in an array of view descs with replacement
// (may be null to just delete). This goes very much against the grain
// of the rest of this code, which tends to create nodes with the
// right shape in one go, rather than messing with them after
// creation, but is necessary in the composition hack.
function replaceNodes(nodes, from, to, view, replacement) {
    var result = [];
    for (var i = 0, off = 0; i < nodes.length; i++) {
        var child = nodes[i], start = off, end = off += child.size;
        if (start >= to || end <= from) {
            result.push(child);
        }
        else {
            if (start < from)
                { result.push(child.slice(0, from - start, view)); }
            if (replacement) {
                result.push(replacement);
                replacement = undefined;
            }
            if (end > to)
                { result.push(child.slice(to - start, child.size, view)); }
        }
    }
    return result;
}

function selectionFromDOM(view, origin) {
    var assign;

    if ( origin === void 0 ) origin = null;
    var domSel = view.domSelectionRange(), doc = view.state.doc;
    if (!domSel.focusNode)
        { return null; }
    var nearestDesc = view.docView.nearestDesc(domSel.focusNode), inWidget = nearestDesc && nearestDesc.size == 0;
    var head = view.docView.posFromDOM(domSel.focusNode, domSel.focusOffset, 1);
    if (head < 0)
        { return null; }
    var $head = doc.resolve(head), anchor, selection;
    if (selectionCollapsed(domSel)) {
        anchor = head;
        while (nearestDesc && !nearestDesc.node)
            { nearestDesc = nearestDesc.parent; }
        var nearestDescNode = nearestDesc.node;
        if (nearestDesc && nearestDescNode.isAtom && NodeSelection.isSelectable(nearestDescNode) && nearestDesc.parent
            && !(nearestDescNode.isInline && isOnEdge(domSel.focusNode, domSel.focusOffset, nearestDesc.dom))) {
            var pos = nearestDesc.posBefore;
            selection = new NodeSelection(head == pos ? $head : doc.resolve(pos));
        }
    }
    else {
        if (domSel instanceof view.dom.ownerDocument.defaultView.Selection && domSel.rangeCount > 1) {
            var min = head, max = head;
            for (var i = 0; i < domSel.rangeCount; i++) {
                var range = domSel.getRangeAt(i);
                min = Math.min(min, view.docView.posFromDOM(range.startContainer, range.startOffset, 1));
                max = Math.max(max, view.docView.posFromDOM(range.endContainer, range.endOffset, -1));
            }
            if (min < 0)
                { return null; }
            (assign = max == view.state.selection.anchor ? [max, min] : [min, max], anchor = assign[0], head = assign[1]);
            $head = doc.resolve(head);
        }
        else {
            anchor = view.docView.posFromDOM(domSel.anchorNode, domSel.anchorOffset, 1);
        }
        if (anchor < 0)
            { return null; }
    }
    var $anchor = doc.resolve(anchor);
    if (!selection) {
        var bias = origin == "pointer" || (view.state.selection.head < $head.pos && !inWidget) ? 1 : -1;
        selection = selectionBetween(view, $anchor, $head, bias);
    }
    return selection;
}
function editorOwnsSelection(view) {
    return view.editable ? view.hasFocus() :
        hasSelection$1(view) && document.activeElement && document.activeElement.contains(view.dom);
}
function selectionToDOM(view, force) {
    if ( force === void 0 ) force = false;

    var sel = view.state.selection;
    syncNodeSelection(view, sel);
    if (!editorOwnsSelection(view))
        { return; }
    // The delayed drag selection causes issues with Cell Selections
    // in Safari. And the drag selection delay is to workarond issues
    // which only present in Chrome.
    if (!force && view.input.mouseDown && view.input.mouseDown.allowDefault && chrome$1) {
        var domSel = view.domSelectionRange(), curSel = view.domObserver.currentSelection;
        if (domSel.anchorNode && curSel.anchorNode &&
            isEquivalentPosition$1(domSel.anchorNode, domSel.anchorOffset, curSel.anchorNode, curSel.anchorOffset)) {
            view.input.mouseDown.delayedSelectionSync = true;
            view.domObserver.setCurSelection();
            return;
        }
    }
    view.domObserver.disconnectSelection();
    if (view.cursorWrapper) {
        selectCursorWrapper(view);
    }
    else {
        var anchor = sel.anchor;
        var head = sel.head;
        var resetEditableFrom, resetEditableTo;
        if (brokenSelectBetweenUneditable && !(sel instanceof TextSelection)) {
            if (!sel.$from.parent.inlineContent)
                { resetEditableFrom = temporarilyEditableNear(view, sel.from); }
            if (!sel.empty && !sel.$from.parent.inlineContent)
                { resetEditableTo = temporarilyEditableNear(view, sel.to); }
        }
        view.docView.setSelection(anchor, head, view.root, force);
        if (brokenSelectBetweenUneditable) {
            if (resetEditableFrom)
                { resetEditable(resetEditableFrom); }
            if (resetEditableTo)
                { resetEditable(resetEditableTo); }
        }
        if (sel.visible) {
            view.dom.classList.remove("ProseMirror-hideselection");
        }
        else {
            view.dom.classList.add("ProseMirror-hideselection");
            if ("onselectionchange" in document)
                { removeClassOnSelectionChange(view); }
        }
    }
    view.domObserver.setCurSelection();
    view.domObserver.connectSelection();
}
// Kludge to work around Webkit not allowing a selection to start/end
// between non-editable block nodes. We briefly make something
// editable, set the selection, then set it uneditable again.
var brokenSelectBetweenUneditable = safari$1 || chrome$1 && chrome_version < 63;
function temporarilyEditableNear(view, pos) {
    var ref = view.docView.domFromPos(pos, 0);
    var node = ref.node;
    var offset = ref.offset;
    var after = offset < node.childNodes.length ? node.childNodes[offset] : null;
    var before = offset ? node.childNodes[offset - 1] : null;
    if (safari$1 && after && after.contentEditable == "false")
        { return setEditable(after); }
    if ((!after || after.contentEditable == "false") &&
        (!before || before.contentEditable == "false")) {
        if (after)
            { return setEditable(after); }
        else if (before)
            { return setEditable(before); }
    }
}
function setEditable(element) {
    element.contentEditable = "true";
    if (safari$1 && element.draggable) {
        element.draggable = false;
        element.wasDraggable = true;
    }
    return element;
}
function resetEditable(element) {
    element.contentEditable = "false";
    if (element.wasDraggable) {
        element.draggable = true;
        element.wasDraggable = null;
    }
}
function removeClassOnSelectionChange(view) {
    var doc = view.dom.ownerDocument;
    doc.removeEventListener("selectionchange", view.input.hideSelectionGuard);
    var domSel = view.domSelectionRange();
    var node = domSel.anchorNode, offset = domSel.anchorOffset;
    doc.addEventListener("selectionchange", view.input.hideSelectionGuard = function () {
        if (domSel.anchorNode != node || domSel.anchorOffset != offset) {
            doc.removeEventListener("selectionchange", view.input.hideSelectionGuard);
            setTimeout(function () {
                if (!editorOwnsSelection(view) || view.state.selection.visible)
                    { view.dom.classList.remove("ProseMirror-hideselection"); }
            }, 20);
        }
    });
}
function selectCursorWrapper(view) {
    var domSel = view.domSelection(), range = document.createRange();
    if (!domSel)
        { return; }
    var node = view.cursorWrapper.dom, img = node.nodeName == "IMG";
    if (img)
        { range.setStart(node.parentNode, domIndex$1(node) + 1); }
    else
        { range.setStart(node, 0); }
    range.collapse(true);
    domSel.removeAllRanges();
    domSel.addRange(range);
    // Kludge to kill 'control selection' in IE11 when selecting an
    // invisible cursor wrapper, since that would result in those weird
    // resize handles and a selection that considers the absolutely
    // positioned wrapper, rather than the root editable node, the
    // focused element.
    if (!img && !view.state.selection.visible && ie$2 && ie_version <= 11) {
        node.disabled = true;
        node.disabled = false;
    }
}
function syncNodeSelection(view, sel) {
    if (sel instanceof NodeSelection) {
        var desc = view.docView.descAt(sel.from);
        if (desc != view.lastSelectedViewDesc) {
            clearNodeSelection(view);
            if (desc)
                { desc.selectNode(); }
            view.lastSelectedViewDesc = desc;
        }
    }
    else {
        clearNodeSelection(view);
    }
}
// Clear all DOM statefulness of the last node selection.
function clearNodeSelection(view) {
    if (view.lastSelectedViewDesc) {
        if (view.lastSelectedViewDesc.parent)
            { view.lastSelectedViewDesc.deselectNode(); }
        view.lastSelectedViewDesc = undefined;
    }
}
function selectionBetween(view, $anchor, $head, bias) {
    return view.someProp("createSelectionBetween", function (f) { return f(view, $anchor, $head); })
        || TextSelection.between($anchor, $head, bias);
}
function hasFocusAndSelection(view) {
    if (view.editable && !view.hasFocus())
        { return false; }
    return hasSelection$1(view);
}
function hasSelection$1(view) {
    var sel = view.domSelectionRange();
    if (!sel.anchorNode)
        { return false; }
    try {
        // Firefox will raise 'permission denied' errors when accessing
        // properties of `sel.anchorNode` when it's in a generated CSS
        // element.
        return view.dom.contains(sel.anchorNode.nodeType == 3 ? sel.anchorNode.parentNode : sel.anchorNode) &&
            (view.editable || view.dom.contains(sel.focusNode.nodeType == 3 ? sel.focusNode.parentNode : sel.focusNode));
    }
    catch (_) {
        return false;
    }
}
function anchorInRightPlace(view) {
    var anchorDOM = view.docView.domFromPos(view.state.selection.anchor, 0);
    var domSel = view.domSelectionRange();
    return isEquivalentPosition$1(anchorDOM.node, anchorDOM.offset, domSel.anchorNode, domSel.anchorOffset);
}

function moveSelectionBlock(state, dir) {
    var ref = state.selection;
    var $anchor = ref.$anchor;
    var $head = ref.$head;
    var $side = dir > 0 ? $anchor.max($head) : $anchor.min($head);
    var $start = !$side.parent.inlineContent ? $side : $side.depth ? state.doc.resolve(dir > 0 ? $side.after() : $side.before()) : null;
    return $start && Selection.findFrom($start, dir);
}
function apply(view, sel) {
    view.dispatch(view.state.tr.setSelection(sel).scrollIntoView());
    return true;
}
function selectHorizontally(view, dir, mods) {
    var sel = view.state.selection;
    if (sel instanceof TextSelection) {
        if (mods.indexOf("s") > -1) {
            var $head = sel.$head;
            var node = $head.textOffset ? null : dir < 0 ? $head.nodeBefore : $head.nodeAfter;
            if (!node || node.isText || !node.isLeaf)
                { return false; }
            var $newHead = view.state.doc.resolve($head.pos + node.nodeSize * (dir < 0 ? -1 : 1));
            return apply(view, new TextSelection(sel.$anchor, $newHead));
        }
        else if (!sel.empty) {
            return false;
        }
        else if (view.endOfTextblock(dir > 0 ? "forward" : "backward")) {
            var next = moveSelectionBlock(view.state, dir);
            if (next && (next instanceof NodeSelection))
                { return apply(view, next); }
            return false;
        }
        else if (!(mac$4 && mods.indexOf("m") > -1)) {
            var $head$1 = sel.$head, node$1 = $head$1.textOffset ? null : dir < 0 ? $head$1.nodeBefore : $head$1.nodeAfter, desc;
            if (!node$1 || node$1.isText)
                { return false; }
            var nodePos = dir < 0 ? $head$1.pos - node$1.nodeSize : $head$1.pos;
            if (!(node$1.isAtom || (desc = view.docView.descAt(nodePos)) && !desc.contentDOM))
                { return false; }
            if (NodeSelection.isSelectable(node$1)) {
                return apply(view, new NodeSelection(dir < 0 ? view.state.doc.resolve($head$1.pos - node$1.nodeSize) : $head$1));
            }
            else if (webkit$1) {
                // Chrome and Safari will introduce extra pointless cursor
                // positions around inline uneditable nodes, so we have to
                // take over and move the cursor past them (#937)
                return apply(view, new TextSelection(view.state.doc.resolve(dir < 0 ? nodePos : nodePos + node$1.nodeSize)));
            }
            else {
                return false;
            }
        }
    }
    else if (sel instanceof NodeSelection && sel.node.isInline) {
        return apply(view, new TextSelection(dir > 0 ? sel.$to : sel.$from));
    }
    else {
        var next$1 = moveSelectionBlock(view.state, dir);
        if (next$1)
            { return apply(view, next$1); }
        return false;
    }
}
function nodeLen(node) {
    return node.nodeType == 3 ? node.nodeValue.length : node.childNodes.length;
}
function isIgnorable(dom, dir) {
    var desc = dom.pmViewDesc;
    return desc && desc.size == 0 && (dir < 0 || dom.nextSibling || dom.nodeName != "BR");
}
function skipIgnoredNodes(view, dir) {
    return dir < 0 ? skipIgnoredNodesBefore(view) : skipIgnoredNodesAfter(view);
}
// Make sure the cursor isn't directly after one or more ignored
// nodes, which will confuse the browser's cursor motion logic.
function skipIgnoredNodesBefore(view) {
    var sel = view.domSelectionRange();
    var node = sel.focusNode, offset = sel.focusOffset;
    if (!node)
        { return; }
    var moveNode, moveOffset, force = false;
    // Gecko will do odd things when the selection is directly in front
    // of a non-editable node, so in that case, move it into the next
    // node if possible. Issue prosemirror/prosemirror#832.
    if (gecko$1 && node.nodeType == 1 && offset < nodeLen(node) && isIgnorable(node.childNodes[offset], -1))
        { force = true; }
    for (;;) {
        if (offset > 0) {
            if (node.nodeType != 1) {
                break;
            }
            else {
                var before = node.childNodes[offset - 1];
                if (isIgnorable(before, -1)) {
                    moveNode = node;
                    moveOffset = --offset;
                }
                else if (before.nodeType == 3) {
                    node = before;
                    offset = node.nodeValue.length;
                }
                else
                    { break; }
            }
        }
        else if (isBlockNode(node)) {
            break;
        }
        else {
            var prev = node.previousSibling;
            while (prev && isIgnorable(prev, -1)) {
                moveNode = node.parentNode;
                moveOffset = domIndex$1(prev);
                prev = prev.previousSibling;
            }
            if (!prev) {
                node = node.parentNode;
                if (node == view.dom)
                    { break; }
                offset = 0;
            }
            else {
                node = prev;
                offset = nodeLen(node);
            }
        }
    }
    if (force)
        { setSelFocus(view, node, offset); }
    else if (moveNode)
        { setSelFocus(view, moveNode, moveOffset); }
}
// Make sure the cursor isn't directly before one or more ignored
// nodes.
function skipIgnoredNodesAfter(view) {
    var sel = view.domSelectionRange();
    var node = sel.focusNode, offset = sel.focusOffset;
    if (!node)
        { return; }
    var len = nodeLen(node);
    var moveNode, moveOffset;
    for (;;) {
        if (offset < len) {
            if (node.nodeType != 1)
                { break; }
            var after = node.childNodes[offset];
            if (isIgnorable(after, 1)) {
                moveNode = node;
                moveOffset = ++offset;
            }
            else
                { break; }
        }
        else if (isBlockNode(node)) {
            break;
        }
        else {
            var next = node.nextSibling;
            while (next && isIgnorable(next, 1)) {
                moveNode = next.parentNode;
                moveOffset = domIndex$1(next) + 1;
                next = next.nextSibling;
            }
            if (!next) {
                node = node.parentNode;
                if (node == view.dom)
                    { break; }
                offset = len = 0;
            }
            else {
                node = next;
                offset = 0;
                len = nodeLen(node);
            }
        }
    }
    if (moveNode)
        { setSelFocus(view, moveNode, moveOffset); }
}
function isBlockNode(dom) {
    var desc = dom.pmViewDesc;
    return desc && desc.node && desc.node.isBlock;
}
function textNodeAfter$2(node, offset) {
    while (node && offset == node.childNodes.length && !hasBlockDesc(node)) {
        offset = domIndex$1(node) + 1;
        node = node.parentNode;
    }
    while (node && offset < node.childNodes.length) {
        var next = node.childNodes[offset];
        if (next.nodeType == 3)
            { return next; }
        if (next.nodeType == 1 && next.contentEditable == "false")
            { break; }
        node = next;
        offset = 0;
    }
}
function textNodeBefore$2(node, offset) {
    while (node && !offset && !hasBlockDesc(node)) {
        offset = domIndex$1(node);
        node = node.parentNode;
    }
    while (node && offset) {
        var next = node.childNodes[offset - 1];
        if (next.nodeType == 3)
            { return next; }
        if (next.nodeType == 1 && next.contentEditable == "false")
            { break; }
        node = next;
        offset = node.childNodes.length;
    }
}
function setSelFocus(view, node, offset) {
    if (node.nodeType != 3) {
        var before, after;
        if (after = textNodeAfter$2(node, offset)) {
            node = after;
            offset = 0;
        }
        else if (before = textNodeBefore$2(node, offset)) {
            node = before;
            offset = before.nodeValue.length;
        }
    }
    var sel = view.domSelection();
    if (!sel)
        { return; }
    if (selectionCollapsed(sel)) {
        var range = document.createRange();
        range.setEnd(node, offset);
        range.setStart(node, offset);
        sel.removeAllRanges();
        sel.addRange(range);
    }
    else if (sel.extend) {
        sel.extend(node, offset);
    }
    view.domObserver.setCurSelection();
    var state = view.state;
    // If no state update ends up happening, reset the selection.
    setTimeout(function () {
        if (view.state == state)
            { selectionToDOM(view); }
    }, 50);
}
function findDirection(view, pos) {
    var $pos = view.state.doc.resolve(pos);
    if (!(chrome$1 || windows) && $pos.parent.inlineContent) {
        var coords = view.coordsAtPos(pos);
        if (pos > $pos.start()) {
            var before = view.coordsAtPos(pos - 1);
            var mid = (before.top + before.bottom) / 2;
            if (mid > coords.top && mid < coords.bottom && Math.abs(before.left - coords.left) > 1)
                { return before.left < coords.left ? "ltr" : "rtl"; }
        }
        if (pos < $pos.end()) {
            var after = view.coordsAtPos(pos + 1);
            var mid$1 = (after.top + after.bottom) / 2;
            if (mid$1 > coords.top && mid$1 < coords.bottom && Math.abs(after.left - coords.left) > 1)
                { return after.left > coords.left ? "ltr" : "rtl"; }
        }
    }
    var computed = getComputedStyle(view.dom).direction;
    return computed == "rtl" ? "rtl" : "ltr";
}
// Check whether vertical selection motion would involve node
// selections. If so, apply it (if not, the result is left to the
// browser)
function selectVertically(view, dir, mods) {
    var sel = view.state.selection;
    if (sel instanceof TextSelection && !sel.empty || mods.indexOf("s") > -1)
        { return false; }
    if (mac$4 && mods.indexOf("m") > -1)
        { return false; }
    var $from = sel.$from;
    var $to = sel.$to;
    if (!$from.parent.inlineContent || view.endOfTextblock(dir < 0 ? "up" : "down")) {
        var next = moveSelectionBlock(view.state, dir);
        if (next && (next instanceof NodeSelection))
            { return apply(view, next); }
    }
    if (!$from.parent.inlineContent) {
        var side = dir < 0 ? $from : $to;
        var beyond = sel instanceof AllSelection ? Selection.near(side, dir) : Selection.findFrom(side, dir);
        return beyond ? apply(view, beyond) : false;
    }
    return false;
}
function stopNativeHorizontalDelete(view, dir) {
    if (!(view.state.selection instanceof TextSelection))
        { return true; }
    var ref = view.state.selection;
    var $head = ref.$head;
    var $anchor = ref.$anchor;
    var empty = ref.empty;
    if (!$head.sameParent($anchor))
        { return true; }
    if (!empty)
        { return false; }
    if (view.endOfTextblock(dir > 0 ? "forward" : "backward"))
        { return true; }
    var nextNode = !$head.textOffset && (dir < 0 ? $head.nodeBefore : $head.nodeAfter);
    if (nextNode && !nextNode.isText) {
        var tr = view.state.tr;
        if (dir < 0)
            { tr.delete($head.pos - nextNode.nodeSize, $head.pos); }
        else
            { tr.delete($head.pos, $head.pos + nextNode.nodeSize); }
        view.dispatch(tr);
        return true;
    }
    return false;
}
function switchEditable(view, node, state) {
    view.domObserver.stop();
    node.contentEditable = state;
    view.domObserver.start();
}
// Issue #867 / #1090 / https://bugs.chromium.org/p/chromium/issues/detail?id=903821
// In which Safari (and at some point in the past, Chrome) does really
// wrong things when the down arrow is pressed when the cursor is
// directly at the start of a textblock and has an uneditable node
// after it
function safariDownArrowBug(view) {
    if (!safari$1 || view.state.selection.$head.parentOffset > 0)
        { return false; }
    var ref = view.domSelectionRange();
    var focusNode = ref.focusNode;
    var focusOffset = ref.focusOffset;
    if (focusNode && focusNode.nodeType == 1 && focusOffset == 0 &&
        focusNode.firstChild && focusNode.firstChild.contentEditable == "false") {
        var child = focusNode.firstChild;
        switchEditable(view, child, "true");
        setTimeout(function () { return switchEditable(view, child, "false"); }, 20);
    }
    return false;
}
// A backdrop key mapping used to make sure we always suppress keys
// that have a dangerous default effect, even if the commands they are
// bound to return false, and to make sure that cursor-motion keys
// find a cursor (as opposed to a node selection) when pressed. For
// cursor-motion keys, the code in the handlers also takes care of
// block selections.
function getMods(event) {
    var result = "";
    if (event.ctrlKey)
        { result += "c"; }
    if (event.metaKey)
        { result += "m"; }
    if (event.altKey)
        { result += "a"; }
    if (event.shiftKey)
        { result += "s"; }
    return result;
}
function captureKeyDown(view, event) {
    var code = event.keyCode, mods = getMods(event);
    if (code == 8 || (mac$4 && code == 72 && mods == "c")) { // Backspace, Ctrl-h on Mac
        return stopNativeHorizontalDelete(view, -1) || skipIgnoredNodes(view, -1);
    }
    else if ((code == 46 && !event.shiftKey) || (mac$4 && code == 68 && mods == "c")) { // Delete, Ctrl-d on Mac
        return stopNativeHorizontalDelete(view, 1) || skipIgnoredNodes(view, 1);
    }
    else if (code == 13 || code == 27) { // Enter, Esc
        return true;
    }
    else if (code == 37 || (mac$4 && code == 66 && mods == "c")) { // Left arrow, Ctrl-b on Mac
        var dir = code == 37 ? (findDirection(view, view.state.selection.from) == "ltr" ? -1 : 1) : -1;
        return selectHorizontally(view, dir, mods) || skipIgnoredNodes(view, dir);
    }
    else if (code == 39 || (mac$4 && code == 70 && mods == "c")) { // Right arrow, Ctrl-f on Mac
        var dir$1 = code == 39 ? (findDirection(view, view.state.selection.from) == "ltr" ? 1 : -1) : 1;
        return selectHorizontally(view, dir$1, mods) || skipIgnoredNodes(view, dir$1);
    }
    else if (code == 38 || (mac$4 && code == 80 && mods == "c")) { // Up arrow, Ctrl-p on Mac
        return selectVertically(view, -1, mods) || skipIgnoredNodes(view, -1);
    }
    else if (code == 40 || (mac$4 && code == 78 && mods == "c")) { // Down arrow, Ctrl-n on Mac
        return safariDownArrowBug(view) || selectVertically(view, 1, mods) || skipIgnoredNodes(view, 1);
    }
    else if (mods == (mac$4 ? "m" : "c") &&
        (code == 66 || code == 73 || code == 89 || code == 90)) { // Mod-[biyz]
        return true;
    }
    return false;
}

function serializeForClipboard(view, slice) {
    view.someProp("transformCopied", function (f) { slice = f(slice, view); });
    var context = [];
    var content = slice.content;
    var openStart = slice.openStart;
    var openEnd = slice.openEnd;
    while (openStart > 1 && openEnd > 1 && content.childCount == 1 && content.firstChild.childCount == 1) {
        openStart--;
        openEnd--;
        var node = content.firstChild;
        context.push(node.type.name, node.attrs != node.type.defaultAttrs ? node.attrs : null);
        content = node.content;
    }
    var serializer = view.someProp("clipboardSerializer") || DOMSerializer.fromSchema(view.state.schema);
    var doc = detachedDoc(), wrap = doc.createElement("div");
    wrap.appendChild(serializer.serializeFragment(content, { document: doc }));
    var firstChild = wrap.firstChild, needsWrap, wrappers = 0;
    while (firstChild && firstChild.nodeType == 1 && (needsWrap = wrapMap[firstChild.nodeName.toLowerCase()])) {
        for (var i = needsWrap.length - 1; i >= 0; i--) {
            var wrapper = doc.createElement(needsWrap[i]);
            while (wrap.firstChild)
                { wrapper.appendChild(wrap.firstChild); }
            wrap.appendChild(wrapper);
            wrappers++;
        }
        firstChild = wrap.firstChild;
    }
    if (firstChild && firstChild.nodeType == 1)
        { firstChild.setAttribute("data-pm-slice", (openStart + " " + openEnd + (wrappers ? (" -" + wrappers) : "") + " " + (JSON.stringify(context)))); }
    var text = view.someProp("clipboardTextSerializer", function (f) { return f(slice, view); }) ||
        slice.content.textBetween(0, slice.content.size, "\n\n");
    return { dom: wrap, text: text, slice: slice };
}
// Read a slice of content from the clipboard (or drop data).
function parseFromClipboard(view, text, html, plainText, $context) {
    var inCode = $context.parent.type.spec.code;
    var dom, slice;
    if (!html && !text)
        { return null; }
    var asText = text && (plainText || inCode || !html);
    if (asText) {
        view.someProp("transformPastedText", function (f) { text = f(text, inCode || plainText, view); });
        if (inCode)
            { return text ? new Slice(Fragment.from(view.state.schema.text(text.replace(/\r\n?/g, "\n"))), 0, 0) : Slice.empty; }
        var parsed = view.someProp("clipboardTextParser", function (f) { return f(text, $context, plainText, view); });
        if (parsed) {
            slice = parsed;
        }
        else {
            var marks = $context.marks();
            var ref = view.state;
            var schema = ref.schema;
            var serializer = DOMSerializer.fromSchema(schema);
            dom = document.createElement("div");
            text.split(/(?:\r\n?|\n)+/).forEach(function (block) {
                var p = dom.appendChild(document.createElement("p"));
                if (block)
                    { p.appendChild(serializer.serializeNode(schema.text(block, marks))); }
            });
        }
    }
    else {
        view.someProp("transformPastedHTML", function (f) { html = f(html, view); });
        dom = readHTML(html);
        if (webkit$1)
            { restoreReplacedSpaces(dom); }
    }
    var contextNode = dom && dom.querySelector("[data-pm-slice]");
    var sliceData = contextNode && /^(\d+) (\d+)(?: -(\d+))? (.*)/.exec(contextNode.getAttribute("data-pm-slice") || "");
    if (sliceData && sliceData[3])
        { for (var i = +sliceData[3]; i > 0; i--) {
            var child = dom.firstChild;
            while (child && child.nodeType != 1)
                { child = child.nextSibling; }
            if (!child)
                { break; }
            dom = child;
        } }
    if (!slice) {
        var parser = view.someProp("clipboardParser") || view.someProp("domParser") || DOMParser.fromSchema(view.state.schema);
        slice = parser.parseSlice(dom, {
            preserveWhitespace: !!(asText || sliceData),
            context: $context,
            ruleFromNode: function ruleFromNode(dom) {
                if (dom.nodeName == "BR" && !dom.nextSibling &&
                    dom.parentNode && !inlineParents.test(dom.parentNode.nodeName))
                    { return { ignore: true }; }
                return null;
            }
        });
    }
    if (sliceData) {
        slice = addContext(closeSlice(slice, +sliceData[1], +sliceData[2]), sliceData[4]);
    }
    else { // HTML wasn't created by ProseMirror. Make sure top-level siblings are coherent
        slice = Slice.maxOpen(normalizeSiblings(slice.content, $context), true);
        if (slice.openStart || slice.openEnd) {
            var openStart = 0, openEnd = 0;
            for (var node = slice.content.firstChild; openStart < slice.openStart && !node.type.spec.isolating; openStart++, node = node.firstChild) { }
            for (var node$1 = slice.content.lastChild; openEnd < slice.openEnd && !node$1.type.spec.isolating; openEnd++, node$1 = node$1.lastChild) { }
            slice = closeSlice(slice, openStart, openEnd);
        }
    }
    view.someProp("transformPasted", function (f) { slice = f(slice, view); });
    return slice;
}
var inlineParents = /^(a|abbr|acronym|b|cite|code|del|em|i|ins|kbd|label|output|q|ruby|s|samp|span|strong|sub|sup|time|u|tt|var)$/i;
// Takes a slice parsed with parseSlice, which means there hasn't been
// any content-expression checking done on the top nodes, tries to
// find a parent node in the current context that might fit the nodes,
// and if successful, rebuilds the slice so that it fits into that parent.
//
// This addresses the problem that Transform.replace expects a
// coherent slice, and will fail to place a set of siblings that don't
// fit anywhere in the schema.
function normalizeSiblings(fragment, $context) {
    if (fragment.childCount < 2)
        { return fragment; }
    var loop = function ( d ) {
        var parent = $context.node(d);
        var match = parent.contentMatchAt($context.index(d));
        var lastWrap = (void 0), result = [];
        fragment.forEach(function (node) {
            if (!result)
                { return; }
            var wrap = match.findWrapping(node.type), inLast;
            if (!wrap)
                { return result = null; }
            if (inLast = result.length && lastWrap.length && addToSibling(wrap, lastWrap, node, result[result.length - 1], 0)) {
                result[result.length - 1] = inLast;
            }
            else {
                if (result.length)
                    { result[result.length - 1] = closeRight(result[result.length - 1], lastWrap.length); }
                var wrapped = withWrappers(node, wrap);
                result.push(wrapped);
                match = match.matchType(wrapped.type);
                lastWrap = wrap;
            }
        });
        if (result)
            { return { v: Fragment.from(result) }; }
    };

    for (var d = $context.depth; d >= 0; d--) {
        var returned = loop( d );

        if ( returned ) return returned.v;
    }
    return fragment;
}
function withWrappers(node, wrap, from) {
    if ( from === void 0 ) from = 0;

    for (var i = wrap.length - 1; i >= from; i--)
        { node = wrap[i].create(null, Fragment.from(node)); }
    return node;
}
// Used to group adjacent nodes wrapped in similar parents by
// normalizeSiblings into the same parent node
function addToSibling(wrap, lastWrap, node, sibling, depth) {
    if (depth < wrap.length && depth < lastWrap.length && wrap[depth] == lastWrap[depth]) {
        var inner = addToSibling(wrap, lastWrap, node, sibling.lastChild, depth + 1);
        if (inner)
            { return sibling.copy(sibling.content.replaceChild(sibling.childCount - 1, inner)); }
        var match = sibling.contentMatchAt(sibling.childCount);
        if (match.matchType(depth == wrap.length - 1 ? node.type : wrap[depth + 1]))
            { return sibling.copy(sibling.content.append(Fragment.from(withWrappers(node, wrap, depth + 1)))); }
    }
}
function closeRight(node, depth) {
    if (depth == 0)
        { return node; }
    var fragment = node.content.replaceChild(node.childCount - 1, closeRight(node.lastChild, depth - 1));
    var fill = node.contentMatchAt(node.childCount).fillBefore(Fragment.empty, true);
    return node.copy(fragment.append(fill));
}
function closeRange(fragment, side, from, to, depth, openEnd) {
    var node = side < 0 ? fragment.firstChild : fragment.lastChild, inner = node.content;
    if (fragment.childCount > 1)
        { openEnd = 0; }
    if (depth < to - 1)
        { inner = closeRange(inner, side, from, to, depth + 1, openEnd); }
    if (depth >= from)
        { inner = side < 0 ? node.contentMatchAt(0).fillBefore(inner, openEnd <= depth).append(inner)
            : inner.append(node.contentMatchAt(node.childCount).fillBefore(Fragment.empty, true)); }
    return fragment.replaceChild(side < 0 ? 0 : fragment.childCount - 1, node.copy(inner));
}
function closeSlice(slice, openStart, openEnd) {
    if (openStart < slice.openStart)
        { slice = new Slice(closeRange(slice.content, -1, openStart, slice.openStart, 0, slice.openEnd), openStart, slice.openEnd); }
    if (openEnd < slice.openEnd)
        { slice = new Slice(closeRange(slice.content, 1, openEnd, slice.openEnd, 0, 0), slice.openStart, openEnd); }
    return slice;
}
// Trick from jQuery -- some elements must be wrapped in other
// elements for innerHTML to work. I.e. if you do `div.innerHTML =
// "<td>..</td>"` the table cells are ignored.
var wrapMap = {
    thead: ["table"],
    tbody: ["table"],
    tfoot: ["table"],
    caption: ["table"],
    colgroup: ["table"],
    col: ["table", "colgroup"],
    tr: ["table", "tbody"],
    td: ["table", "tbody", "tr"],
    th: ["table", "tbody", "tr"]
};
var _detachedDoc = null;
function detachedDoc() {
    return _detachedDoc || (_detachedDoc = document.implementation.createHTMLDocument("title"));
}
function maybeWrapTrusted(html) {
    var trustedTypes = window.trustedTypes;
    if (!trustedTypes)
        { return html; }
    // With the require-trusted-types-for CSP, Chrome will block
    // innerHTML, even on a detached document. This wraps the string in
    // a way that makes the browser allow us to use its parser again.
    return trustedTypes.createPolicy("detachedDocument", { createHTML: function (s) { return s; } }).createHTML(html);
}
function readHTML(html) {
    var metas = /^(\s*<meta [^>]*>)*/.exec(html);
    if (metas)
        { html = html.slice(metas[0].length); }
    var elt = detachedDoc().createElement("div");
    var firstTag = /<([a-z][^>\s]+)/i.exec(html), wrap;
    if (wrap = firstTag && wrapMap[firstTag[1].toLowerCase()])
        { html = wrap.map(function (n) { return "<" + n + ">"; }).join("") + html + wrap.map(function (n) { return "</" + n + ">"; }).reverse().join(""); }
    elt.innerHTML = maybeWrapTrusted(html);
    if (wrap)
        { for (var i = 0; i < wrap.length; i++)
            { elt = elt.querySelector(wrap[i]) || elt; } }
    return elt;
}
// Webkit browsers do some hard-to-predict replacement of regular
// spaces with non-breaking spaces when putting content on the
// clipboard. This tries to convert such non-breaking spaces (which
// will be wrapped in a plain span on Chrome, a span with class
// Apple-converted-space on Safari) back to regular spaces.
function restoreReplacedSpaces(dom) {
    var nodes = dom.querySelectorAll(chrome$1 ? "span:not([class]):not([style])" : "span.Apple-converted-space");
    for (var i = 0; i < nodes.length; i++) {
        var node = nodes[i];
        if (node.childNodes.length == 1 && node.textContent == "\u00a0" && node.parentNode)
            { node.parentNode.replaceChild(dom.ownerDocument.createTextNode(" "), node); }
    }
}
function addContext(slice, context) {
    if (!slice.size)
        { return slice; }
    var schema = slice.content.firstChild.type.schema, array;
    try {
        array = JSON.parse(context);
    }
    catch (e) {
        return slice;
    }
    var content = slice.content;
    var openStart = slice.openStart;
    var openEnd = slice.openEnd;
    for (var i = array.length - 2; i >= 0; i -= 2) {
        var type = schema.nodes[array[i]];
        if (!type || type.hasRequiredAttrs())
            { break; }
        content = Fragment.from(type.create(array[i + 1], content));
        openStart++;
        openEnd++;
    }
    return new Slice(content, openStart, openEnd);
}

// A collection of DOM events that occur within the editor, and callback functions
// to invoke when the event fires.
var handlers$1 = {};
var editHandlers = {};
var passiveHandlers = { touchstart: true, touchmove: true };
var InputState$1 = function InputState() {
    this.shiftKey = false;
    this.mouseDown = null;
    this.lastKeyCode = null;
    this.lastKeyCodeTime = 0;
    this.lastClick = { time: 0, x: 0, y: 0, type: "" };
    this.lastSelectionOrigin = null;
    this.lastSelectionTime = 0;
    this.lastIOSEnter = 0;
    this.lastIOSEnterFallbackTimeout = -1;
    this.lastFocus = 0;
    this.lastTouch = 0;
    this.lastAndroidDelete = 0;
    this.composing = false;
    this.compositionNode = null;
    this.composingTimeout = -1;
    this.compositionNodes = [];
    this.compositionEndedAt = -2e8;
    this.compositionID = 1;
    // Set to a composition ID when there are pending changes at compositionend
    this.compositionPendingChanges = 0;
    this.domChangeCount = 0;
    this.eventHandlers = Object.create(null);
    this.hideSelectionGuard = null;
};
function initInput(view) {
    var loop = function ( event ) {
        var handler = handlers$1[event];
        view.dom.addEventListener(event, view.input.eventHandlers[event] = function (event) {
            if (eventBelongsToView(view, event) && !runCustomHandler(view, event) &&
                (view.editable || !(event.type in editHandlers)))
                { handler(view, event); }
        }, passiveHandlers[event] ? { passive: true } : undefined);
    };

    for (var event in handlers$1) loop( event );
    // On Safari, for reasons beyond my understanding, adding an input
    // event handler makes an issue where the composition vanishes when
    // you press enter go away.
    if (safari$1)
        { view.dom.addEventListener("input", function () { return null; }); }
    ensureListeners(view);
}
function setSelectionOrigin(view, origin) {
    view.input.lastSelectionOrigin = origin;
    view.input.lastSelectionTime = Date.now();
}
function destroyInput(view) {
    view.domObserver.stop();
    for (var type in view.input.eventHandlers)
        { view.dom.removeEventListener(type, view.input.eventHandlers[type]); }
    clearTimeout(view.input.composingTimeout);
    clearTimeout(view.input.lastIOSEnterFallbackTimeout);
}
function ensureListeners(view) {
    view.someProp("handleDOMEvents", function (currentHandlers) {
        for (var type in currentHandlers)
            { if (!view.input.eventHandlers[type])
                { view.dom.addEventListener(type, view.input.eventHandlers[type] = function (event) { return runCustomHandler(view, event); }); } }
    });
}
function runCustomHandler(view, event) {
    return view.someProp("handleDOMEvents", function (handlers) {
        var handler = handlers[event.type];
        return handler ? handler(view, event) || event.defaultPrevented : false;
    });
}
function eventBelongsToView(view, event) {
    if (!event.bubbles)
        { return true; }
    if (event.defaultPrevented)
        { return false; }
    for (var node = event.target; node != view.dom; node = node.parentNode)
        { if (!node || node.nodeType == 11 ||
            (node.pmViewDesc && node.pmViewDesc.stopEvent(event)))
            { return false; } }
    return true;
}
function dispatchEvent(view, event) {
    if (!runCustomHandler(view, event) && handlers$1[event.type] &&
        (view.editable || !(event.type in editHandlers)))
        { handlers$1[event.type](view, event); }
}
editHandlers.keydown = function (view, _event) {
    var event = _event;
    view.input.shiftKey = event.keyCode == 16 || event.shiftKey;
    if (inOrNearComposition(view, event))
        { return; }
    view.input.lastKeyCode = event.keyCode;
    view.input.lastKeyCodeTime = Date.now();
    // Suppress enter key events on Chrome Android, because those tend
    // to be part of a confused sequence of composition events fired,
    // and handling them eagerly tends to corrupt the input.
    if (android$1 && chrome$1 && event.keyCode == 13)
        { return; }
    if (view.domObserver.selectionChanged(view.domSelectionRange()))
        { view.domObserver.flush(); }
    else if (event.keyCode != 229)
        { view.domObserver.forceFlush(); }
    // On iOS, if we preventDefault enter key presses, the virtual
    // keyboard gets confused. So the hack here is to set a flag that
    // makes the DOM change code recognize that what just happens should
    // be replaced by whatever the Enter key handlers do.
    if (ios$1 && event.keyCode == 13 && !event.ctrlKey && !event.altKey && !event.metaKey) {
        var now = Date.now();
        view.input.lastIOSEnter = now;
        view.input.lastIOSEnterFallbackTimeout = setTimeout(function () {
            if (view.input.lastIOSEnter == now) {
                view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(13, "Enter")); });
                view.input.lastIOSEnter = 0;
            }
        }, 200);
    }
    else if (view.someProp("handleKeyDown", function (f) { return f(view, event); }) || captureKeyDown(view, event)) {
        event.preventDefault();
    }
    else {
        setSelectionOrigin(view, "key");
    }
};
editHandlers.keyup = function (view, event) {
    if (event.keyCode == 16)
        { view.input.shiftKey = false; }
};
editHandlers.keypress = function (view, _event) {
    var event = _event;
    if (inOrNearComposition(view, event) || !event.charCode ||
        event.ctrlKey && !event.altKey || mac$4 && event.metaKey)
        { return; }
    if (view.someProp("handleKeyPress", function (f) { return f(view, event); })) {
        event.preventDefault();
        return;
    }
    var sel = view.state.selection;
    if (!(sel instanceof TextSelection) || !sel.$from.sameParent(sel.$to)) {
        var text = String.fromCharCode(event.charCode);
        if (!/[\r\n]/.test(text) && !view.someProp("handleTextInput", function (f) { return f(view, sel.$from.pos, sel.$to.pos, text); }))
            { view.dispatch(view.state.tr.insertText(text).scrollIntoView()); }
        event.preventDefault();
    }
};
function eventCoords(event) { return { left: event.clientX, top: event.clientY }; }
function isNear(event, click) {
    var dx = click.x - event.clientX, dy = click.y - event.clientY;
    return dx * dx + dy * dy < 100;
}
function runHandlerOnContext(view, propName, pos, inside, event) {
    if (inside == -1)
        { return false; }
    var $pos = view.state.doc.resolve(inside);
    var loop = function ( i ) {
        if (view.someProp(propName, function (f) { return i > $pos.depth ? f(view, pos, $pos.nodeAfter, $pos.before(i), event, true)
            : f(view, pos, $pos.node(i), $pos.before(i), event, false); }))
            { return { v: true }; }
    };

    for (var i = $pos.depth + 1; i > 0; i--) {
        var returned = loop( i );

        if ( returned ) return returned.v;
    }
    return false;
}
function updateSelection(view, selection, origin) {
    if (!view.focused)
        { view.focus(); }
    if (view.state.selection.eq(selection))
        { return; }
    var tr = view.state.tr.setSelection(selection);
    if (origin == "pointer")
        { tr.setMeta("pointer", true); }
    view.dispatch(tr);
}
function selectClickedLeaf(view, inside) {
    if (inside == -1)
        { return false; }
    var $pos = view.state.doc.resolve(inside), node = $pos.nodeAfter;
    if (node && node.isAtom && NodeSelection.isSelectable(node)) {
        updateSelection(view, new NodeSelection($pos), "pointer");
        return true;
    }
    return false;
}
function selectClickedNode(view, inside) {
    if (inside == -1)
        { return false; }
    var sel = view.state.selection, selectedNode, selectAt;
    if (sel instanceof NodeSelection)
        { selectedNode = sel.node; }
    var $pos = view.state.doc.resolve(inside);
    for (var i = $pos.depth + 1; i > 0; i--) {
        var node = i > $pos.depth ? $pos.nodeAfter : $pos.node(i);
        if (NodeSelection.isSelectable(node)) {
            if (selectedNode && sel.$from.depth > 0 &&
                i >= sel.$from.depth && $pos.before(sel.$from.depth + 1) == sel.$from.pos)
                { selectAt = $pos.before(sel.$from.depth); }
            else
                { selectAt = $pos.before(i); }
            break;
        }
    }
    if (selectAt != null) {
        updateSelection(view, NodeSelection.create(view.state.doc, selectAt), "pointer");
        return true;
    }
    else {
        return false;
    }
}
function handleSingleClick(view, pos, inside, event, selectNode) {
    return runHandlerOnContext(view, "handleClickOn", pos, inside, event) ||
        view.someProp("handleClick", function (f) { return f(view, pos, event); }) ||
        (selectNode ? selectClickedNode(view, inside) : selectClickedLeaf(view, inside));
}
function handleDoubleClick(view, pos, inside, event) {
    return runHandlerOnContext(view, "handleDoubleClickOn", pos, inside, event) ||
        view.someProp("handleDoubleClick", function (f) { return f(view, pos, event); });
}
function handleTripleClick$1(view, pos, inside, event) {
    return runHandlerOnContext(view, "handleTripleClickOn", pos, inside, event) ||
        view.someProp("handleTripleClick", function (f) { return f(view, pos, event); }) ||
        defaultTripleClick(view, inside, event);
}
function defaultTripleClick(view, inside, event) {
    if (event.button != 0)
        { return false; }
    var doc = view.state.doc;
    if (inside == -1) {
        if (doc.inlineContent) {
            updateSelection(view, TextSelection.create(doc, 0, doc.content.size), "pointer");
            return true;
        }
        return false;
    }
    var $pos = doc.resolve(inside);
    for (var i = $pos.depth + 1; i > 0; i--) {
        var node = i > $pos.depth ? $pos.nodeAfter : $pos.node(i);
        var nodePos = $pos.before(i);
        if (node.inlineContent)
            { updateSelection(view, TextSelection.create(doc, nodePos + 1, nodePos + 1 + node.content.size), "pointer"); }
        else if (NodeSelection.isSelectable(node))
            { updateSelection(view, NodeSelection.create(doc, nodePos), "pointer"); }
        else
            { continue; }
        return true;
    }
}
function forceDOMFlush(view) {
    return endComposition(view);
}
var selectNodeModifier = mac$4 ? "metaKey" : "ctrlKey";
handlers$1.mousedown = function (view, _event) {
    var event = _event;
    view.input.shiftKey = event.shiftKey;
    var flushed = forceDOMFlush(view);
    var now = Date.now(), type = "singleClick";
    if (now - view.input.lastClick.time < 500 && isNear(event, view.input.lastClick) && !event[selectNodeModifier]) {
        if (view.input.lastClick.type == "singleClick")
            { type = "doubleClick"; }
        else if (view.input.lastClick.type == "doubleClick")
            { type = "tripleClick"; }
    }
    view.input.lastClick = { time: now, x: event.clientX, y: event.clientY, type: type };
    var pos = view.posAtCoords(eventCoords(event));
    if (!pos)
        { return; }
    if (type == "singleClick") {
        if (view.input.mouseDown)
            { view.input.mouseDown.done(); }
        view.input.mouseDown = new MouseDown(view, pos, event, !!flushed);
    }
    else if ((type == "doubleClick" ? handleDoubleClick : handleTripleClick$1)(view, pos.pos, pos.inside, event)) {
        event.preventDefault();
    }
    else {
        setSelectionOrigin(view, "pointer");
    }
};
var MouseDown = function MouseDown(view, pos, event, flushed) {
    var this$1$1 = this;

    this.view = view;
    this.pos = pos;
    this.event = event;
    this.flushed = flushed;
    this.delayedSelectionSync = false;
    this.mightDrag = null;
    this.startDoc = view.state.doc;
    this.selectNode = !!event[selectNodeModifier];
    this.allowDefault = event.shiftKey;
    var targetNode, targetPos;
    if (pos.inside > -1) {
        targetNode = view.state.doc.nodeAt(pos.inside);
        targetPos = pos.inside;
    }
    else {
        var $pos = view.state.doc.resolve(pos.pos);
        targetNode = $pos.parent;
        targetPos = $pos.depth ? $pos.before() : 0;
    }
    var target = flushed ? null : event.target;
    var targetDesc = target ? view.docView.nearestDesc(target, true) : null;
    this.target = targetDesc && targetDesc.dom.nodeType == 1 ? targetDesc.dom : null;
    var ref = view.state;
    var selection = ref.selection;
    if (event.button == 0 &&
        targetNode.type.spec.draggable && targetNode.type.spec.selectable !== false ||
        selection instanceof NodeSelection && selection.from <= targetPos && selection.to > targetPos)
        { this.mightDrag = {
            node: targetNode,
            pos: targetPos,
            addAttr: !!(this.target && !this.target.draggable),
            setUneditable: !!(this.target && gecko$1 && !this.target.hasAttribute("contentEditable"))
        }; }
    if (this.target && this.mightDrag && (this.mightDrag.addAttr || this.mightDrag.setUneditable)) {
        this.view.domObserver.stop();
        if (this.mightDrag.addAttr)
            { this.target.draggable = true; }
        if (this.mightDrag.setUneditable)
            { setTimeout(function () {
                if (this$1$1.view.input.mouseDown == this$1$1)
                    { this$1$1.target.setAttribute("contentEditable", "false"); }
            }, 20); }
        this.view.domObserver.start();
    }
    view.root.addEventListener("mouseup", this.up = this.up.bind(this));
    view.root.addEventListener("mousemove", this.move = this.move.bind(this));
    setSelectionOrigin(view, "pointer");
};
MouseDown.prototype.done = function done () {
        var this$1$1 = this;

    this.view.root.removeEventListener("mouseup", this.up);
    this.view.root.removeEventListener("mousemove", this.move);
    if (this.mightDrag && this.target) {
        this.view.domObserver.stop();
        if (this.mightDrag.addAttr)
            { this.target.removeAttribute("draggable"); }
        if (this.mightDrag.setUneditable)
            { this.target.removeAttribute("contentEditable"); }
        this.view.domObserver.start();
    }
    if (this.delayedSelectionSync)
        { setTimeout(function () { return selectionToDOM(this$1$1.view); }); }
    this.view.input.mouseDown = null;
};
MouseDown.prototype.up = function up (event) {
    this.done();
    if (!this.view.dom.contains(event.target))
        { return; }
    var pos = this.pos;
    if (this.view.state.doc != this.startDoc)
        { pos = this.view.posAtCoords(eventCoords(event)); }
    this.updateAllowDefault(event);
    if (this.allowDefault || !pos) {
        setSelectionOrigin(this.view, "pointer");
    }
    else if (handleSingleClick(this.view, pos.pos, pos.inside, event, this.selectNode)) {
        event.preventDefault();
    }
    else if (event.button == 0 &&
        (this.flushed ||
            // Safari ignores clicks on draggable elements
            (safari$1 && this.mightDrag && !this.mightDrag.node.isAtom) ||
            // Chrome will sometimes treat a node selection as a
            // cursor, but still report that the node is selected
            // when asked through getSelection. You'll then get a
            // situation where clicking at the point where that
            // (hidden) cursor is doesn't change the selection, and
            // thus doesn't get a reaction from ProseMirror. This
            // works around that.
            (chrome$1 && !this.view.state.selection.visible &&
                Math.min(Math.abs(pos.pos - this.view.state.selection.from), Math.abs(pos.pos - this.view.state.selection.to)) <= 2))) {
        updateSelection(this.view, Selection.near(this.view.state.doc.resolve(pos.pos)), "pointer");
        event.preventDefault();
    }
    else {
        setSelectionOrigin(this.view, "pointer");
    }
};
MouseDown.prototype.move = function move (event) {
    this.updateAllowDefault(event);
    setSelectionOrigin(this.view, "pointer");
    if (event.buttons == 0)
        { this.done(); }
};
MouseDown.prototype.updateAllowDefault = function updateAllowDefault (event) {
    if (!this.allowDefault && (Math.abs(this.event.x - event.clientX) > 4 ||
        Math.abs(this.event.y - event.clientY) > 4))
        { this.allowDefault = true; }
};
handlers$1.touchstart = function (view) {
    view.input.lastTouch = Date.now();
    forceDOMFlush(view);
    setSelectionOrigin(view, "pointer");
};
handlers$1.touchmove = function (view) {
    view.input.lastTouch = Date.now();
    setSelectionOrigin(view, "pointer");
};
handlers$1.contextmenu = function (view) { return forceDOMFlush(view); };
function inOrNearComposition(view, event) {
    if (view.composing)
        { return true; }
    // See https://www.stum.de/2016/06/24/handling-ime-events-in-javascript/.
    // On Japanese input method editors (IMEs), the Enter key is used to confirm character
    // selection. On Safari, when Enter is pressed, compositionend and keydown events are
    // emitted. The keydown event triggers newline insertion, which we don't want.
    // This method returns true if the keydown event should be ignored.
    // We only ignore it once, as pressing Enter a second time *should* insert a newline.
    // Furthermore, the keydown event timestamp must be close to the compositionEndedAt timestamp.
    // This guards against the case where compositionend is triggered without the keyboard
    // (e.g. character confirmation may be done with the mouse), and keydown is triggered
    // afterwards- we wouldn't want to ignore the keydown event in this case.
    if (safari$1 && Math.abs(event.timeStamp - view.input.compositionEndedAt) < 500) {
        view.input.compositionEndedAt = -2e8;
        return true;
    }
    return false;
}
// Drop active composition after 5 seconds of inactivity on Android
var timeoutComposition = android$1 ? 5000 : -1;
editHandlers.compositionstart = editHandlers.compositionupdate = function (view) {
    if (!view.composing) {
        view.domObserver.flush();
        var state = view.state;
        var $pos = state.selection.$to;
        if (state.selection instanceof TextSelection &&
            (state.storedMarks ||
                (!$pos.textOffset && $pos.parentOffset && $pos.nodeBefore.marks.some(function (m) { return m.type.spec.inclusive === false; })))) {
            // Need to wrap the cursor in mark nodes different from the ones in the DOM context
            view.markCursor = view.state.storedMarks || $pos.marks();
            endComposition(view, true);
            view.markCursor = null;
        }
        else {
            endComposition(view, !state.selection.empty);
            // In firefox, if the cursor is after but outside a marked node,
            // the inserted text won't inherit the marks. So this moves it
            // inside if necessary.
            if (gecko$1 && state.selection.empty && $pos.parentOffset && !$pos.textOffset && $pos.nodeBefore.marks.length) {
                var sel = view.domSelectionRange();
                for (var node = sel.focusNode, offset = sel.focusOffset; node && node.nodeType == 1 && offset != 0;) {
                    var before = offset < 0 ? node.lastChild : node.childNodes[offset - 1];
                    if (!before)
                        { break; }
                    if (before.nodeType == 3) {
                        var sel$1 = view.domSelection();
                        if (sel$1)
                            { sel$1.collapse(before, before.nodeValue.length); }
                        break;
                    }
                    else {
                        node = before;
                        offset = -1;
                    }
                }
            }
        }
        view.input.composing = true;
    }
    scheduleComposeEnd(view, timeoutComposition);
};
editHandlers.compositionend = function (view, event) {
    if (view.composing) {
        view.input.composing = false;
        view.input.compositionEndedAt = event.timeStamp;
        view.input.compositionPendingChanges = view.domObserver.pendingRecords().length ? view.input.compositionID : 0;
        view.input.compositionNode = null;
        if (view.input.compositionPendingChanges)
            { Promise.resolve().then(function () { return view.domObserver.flush(); }); }
        view.input.compositionID++;
        scheduleComposeEnd(view, 20);
    }
};
function scheduleComposeEnd(view, delay) {
    clearTimeout(view.input.composingTimeout);
    if (delay > -1)
        { view.input.composingTimeout = setTimeout(function () { return endComposition(view); }, delay); }
}
function clearComposition(view) {
    if (view.composing) {
        view.input.composing = false;
        view.input.compositionEndedAt = timestampFromCustomEvent();
    }
    while (view.input.compositionNodes.length > 0)
        { view.input.compositionNodes.pop().markParentsDirty(); }
}
function findCompositionNode$1(view) {
    var sel = view.domSelectionRange();
    if (!sel.focusNode)
        { return null; }
    var textBefore = textNodeBefore$1(sel.focusNode, sel.focusOffset);
    var textAfter = textNodeAfter$1(sel.focusNode, sel.focusOffset);
    if (textBefore && textAfter && textBefore != textAfter) {
        var descAfter = textAfter.pmViewDesc, lastChanged = view.domObserver.lastChangedTextNode;
        if (textBefore == lastChanged || textAfter == lastChanged)
            { return lastChanged; }
        if (!descAfter || !descAfter.isText(textAfter.nodeValue)) {
            return textAfter;
        }
        else if (view.input.compositionNode == textAfter) {
            var descBefore = textBefore.pmViewDesc;
            if (!(!descBefore || !descBefore.isText(textBefore.nodeValue)))
                { return textAfter; }
        }
    }
    return textBefore || textAfter;
}
function timestampFromCustomEvent() {
    var event = document.createEvent("Event");
    event.initEvent("event", true, true);
    return event.timeStamp;
}
/**
@internal
*/
function endComposition(view, restarting) {
    if ( restarting === void 0 ) restarting = false;

    if (android$1 && view.domObserver.flushingSoon >= 0)
        { return; }
    view.domObserver.forceFlush();
    clearComposition(view);
    if (restarting || view.docView && view.docView.dirty) {
        var sel = selectionFromDOM(view);
        if (sel && !sel.eq(view.state.selection))
            { view.dispatch(view.state.tr.setSelection(sel)); }
        else if ((view.markCursor || restarting) && !view.state.selection.empty)
            { view.dispatch(view.state.tr.deleteSelection()); }
        else
            { view.updateState(view.state); }
        return true;
    }
    return false;
}
function captureCopy$1(view, dom) {
    // The extra wrapper is somehow necessary on IE/Edge to prevent the
    // content from being mangled when it is put onto the clipboard
    if (!view.dom.parentNode)
        { return; }
    var wrap = view.dom.parentNode.appendChild(document.createElement("div"));
    wrap.appendChild(dom);
    wrap.style.cssText = "position: fixed; left: -10000px; top: 10px";
    var sel = getSelection(), range = document.createRange();
    range.selectNodeContents(dom);
    // Done because IE will fire a selectionchange moving the selection
    // to its start when removeAllRanges is called and the editor still
    // has focus (which will mess up the editor's selection state).
    view.dom.blur();
    sel.removeAllRanges();
    sel.addRange(range);
    setTimeout(function () {
        if (wrap.parentNode)
            { wrap.parentNode.removeChild(wrap); }
        view.focus();
    }, 50);
}
// This is very crude, but unfortunately both these browsers _pretend_
// that they have a clipboard API—all the objects and methods are
// there, they just don't work, and they are hard to test.
var brokenClipboardAPI$1 = (ie$2 && ie_version < 15) ||
    (ios$1 && webkit_version < 604);
handlers$1.copy = editHandlers.cut = function (view, _event) {
    var event = _event;
    var sel = view.state.selection, cut = event.type == "cut";
    if (sel.empty)
        { return; }
    // IE and Edge's clipboard interface is completely broken
    var data = brokenClipboardAPI$1 ? null : event.clipboardData;
    var slice = sel.content();
    var ref = serializeForClipboard(view, slice);
    var dom = ref.dom;
    var text = ref.text;
    if (data) {
        event.preventDefault();
        data.clearData();
        data.setData("text/html", dom.innerHTML);
        data.setData("text/plain", text);
    }
    else {
        captureCopy$1(view, dom);
    }
    if (cut)
        { view.dispatch(view.state.tr.deleteSelection().scrollIntoView().setMeta("uiEvent", "cut")); }
};
function sliceSingleNode(slice) {
    return slice.openStart == 0 && slice.openEnd == 0 && slice.content.childCount == 1 ? slice.content.firstChild : null;
}
function capturePaste$1(view, event) {
    if (!view.dom.parentNode)
        { return; }
    var plainText = view.input.shiftKey || view.state.selection.$from.parent.type.spec.code;
    var target = view.dom.parentNode.appendChild(document.createElement(plainText ? "textarea" : "div"));
    if (!plainText)
        { target.contentEditable = "true"; }
    target.style.cssText = "position: fixed; left: -10000px; top: 10px";
    target.focus();
    var plain = view.input.shiftKey && view.input.lastKeyCode != 45;
    setTimeout(function () {
        view.focus();
        if (target.parentNode)
            { target.parentNode.removeChild(target); }
        if (plainText)
            { doPaste$1(view, target.value, null, plain, event); }
        else
            { doPaste$1(view, target.textContent, target.innerHTML, plain, event); }
    }, 50);
}
function doPaste$1(view, text, html, preferPlain, event) {
    var slice = parseFromClipboard(view, text, html, preferPlain, view.state.selection.$from);
    if (view.someProp("handlePaste", function (f) { return f(view, event, slice || Slice.empty); }))
        { return true; }
    if (!slice)
        { return false; }
    var singleNode = sliceSingleNode(slice);
    var tr = singleNode
        ? view.state.tr.replaceSelectionWith(singleNode, preferPlain)
        : view.state.tr.replaceSelection(slice);
    view.dispatch(tr.scrollIntoView().setMeta("paste", true).setMeta("uiEvent", "paste"));
    return true;
}
function getText(clipboardData) {
    var text = clipboardData.getData("text/plain") || clipboardData.getData("Text");
    if (text)
        { return text; }
    var uris = clipboardData.getData("text/uri-list");
    return uris ? uris.replace(/\r?\n/g, " ") : "";
}
editHandlers.paste = function (view, _event) {
    var event = _event;
    // Handling paste from JavaScript during composition is very poorly
    // handled by browsers, so as a dodgy but preferable kludge, we just
    // let the browser do its native thing there, except on Android,
    // where the editor is almost always composing.
    if (view.composing && !android$1)
        { return; }
    var data = brokenClipboardAPI$1 ? null : event.clipboardData;
    var plain = view.input.shiftKey && view.input.lastKeyCode != 45;
    if (data && doPaste$1(view, getText(data), data.getData("text/html"), plain, event))
        { event.preventDefault(); }
    else
        { capturePaste$1(view, event); }
};
var Dragging = function Dragging(slice, move, node) {
    this.slice = slice;
    this.move = move;
    this.node = node;
};
var dragCopyModifier = mac$4 ? "altKey" : "ctrlKey";
handlers$1.dragstart = function (view, _event) {
    var event = _event;
    var mouseDown = view.input.mouseDown;
    if (mouseDown)
        { mouseDown.done(); }
    if (!event.dataTransfer)
        { return; }
    var sel = view.state.selection;
    var pos = sel.empty ? null : view.posAtCoords(eventCoords(event));
    var node;
    if (pos && pos.pos >= sel.from && pos.pos <= (sel instanceof NodeSelection ? sel.to - 1 : sel.to)) ;
    else if (mouseDown && mouseDown.mightDrag) {
        node = NodeSelection.create(view.state.doc, mouseDown.mightDrag.pos);
    }
    else if (event.target && event.target.nodeType == 1) {
        var desc = view.docView.nearestDesc(event.target, true);
        if (desc && desc.node.type.spec.draggable && desc != view.docView)
            { node = NodeSelection.create(view.state.doc, desc.posBefore); }
    }
    var draggedSlice = (node || view.state.selection).content();
    var ref = serializeForClipboard(view, draggedSlice);
    var dom = ref.dom;
    var text = ref.text;
    var slice = ref.slice;
    // Pre-120 Chrome versions clear files when calling `clearData` (#1472)
    if (!event.dataTransfer.files.length || !chrome$1 || chrome_version > 120)
        { event.dataTransfer.clearData(); }
    event.dataTransfer.setData(brokenClipboardAPI$1 ? "Text" : "text/html", dom.innerHTML);
    // See https://github.com/ProseMirror/prosemirror/issues/1156
    event.dataTransfer.effectAllowed = "copyMove";
    if (!brokenClipboardAPI$1)
        { event.dataTransfer.setData("text/plain", text); }
    view.dragging = new Dragging(slice, !event[dragCopyModifier], node);
};
handlers$1.dragend = function (view) {
    var dragging = view.dragging;
    window.setTimeout(function () {
        if (view.dragging == dragging)
            { view.dragging = null; }
    }, 50);
};
editHandlers.dragover = editHandlers.dragenter = function (_, e) { return e.preventDefault(); };
editHandlers.drop = function (view, _event) {
    var event = _event;
    var dragging = view.dragging;
    view.dragging = null;
    if (!event.dataTransfer)
        { return; }
    var eventPos = view.posAtCoords(eventCoords(event));
    if (!eventPos)
        { return; }
    var $mouse = view.state.doc.resolve(eventPos.pos);
    var slice = dragging && dragging.slice;
    if (slice) {
        view.someProp("transformPasted", function (f) { slice = f(slice, view); });
    }
    else {
        slice = parseFromClipboard(view, getText(event.dataTransfer), brokenClipboardAPI$1 ? null : event.dataTransfer.getData("text/html"), false, $mouse);
    }
    var move = !!(dragging && !event[dragCopyModifier]);
    if (view.someProp("handleDrop", function (f) { return f(view, event, slice || Slice.empty, move); })) {
        event.preventDefault();
        return;
    }
    if (!slice)
        { return; }
    event.preventDefault();
    var insertPos = slice ? dropPoint(view.state.doc, $mouse.pos, slice) : $mouse.pos;
    if (insertPos == null)
        { insertPos = $mouse.pos; }
    var tr = view.state.tr;
    if (move) {
        var node = dragging.node;
        if (node)
            { node.replace(tr); }
        else
            { tr.deleteSelection(); }
    }
    var pos = tr.mapping.map(insertPos);
    var isNode = slice.openStart == 0 && slice.openEnd == 0 && slice.content.childCount == 1;
    var beforeInsert = tr.doc;
    if (isNode)
        { tr.replaceRangeWith(pos, pos, slice.content.firstChild); }
    else
        { tr.replaceRange(pos, pos, slice); }
    if (tr.doc.eq(beforeInsert))
        { return; }
    var $pos = tr.doc.resolve(pos);
    if (isNode && NodeSelection.isSelectable(slice.content.firstChild) &&
        $pos.nodeAfter && $pos.nodeAfter.sameMarkup(slice.content.firstChild)) {
        tr.setSelection(new NodeSelection($pos));
    }
    else {
        var end = tr.mapping.map(insertPos);
        tr.mapping.maps[tr.mapping.maps.length - 1].forEach(function (_from, _to, _newFrom, newTo) { return end = newTo; });
        tr.setSelection(selectionBetween(view, $pos, tr.doc.resolve(end)));
    }
    view.focus();
    view.dispatch(tr.setMeta("uiEvent", "drop"));
};
handlers$1.focus = function (view) {
    view.input.lastFocus = Date.now();
    if (!view.focused) {
        view.domObserver.stop();
        view.dom.classList.add("ProseMirror-focused");
        view.domObserver.start();
        view.focused = true;
        setTimeout(function () {
            if (view.docView && view.hasFocus() && !view.domObserver.currentSelection.eq(view.domSelectionRange()))
                { selectionToDOM(view); }
        }, 20);
    }
};
handlers$1.blur = function (view, _event) {
    var event = _event;
    if (view.focused) {
        view.domObserver.stop();
        view.dom.classList.remove("ProseMirror-focused");
        view.domObserver.start();
        if (event.relatedTarget && view.dom.contains(event.relatedTarget))
            { view.domObserver.currentSelection.clear(); }
        view.focused = false;
    }
};
handlers$1.beforeinput = function (view, _event) {
    var event = _event;
    // We should probably do more with beforeinput events, but support
    // is so spotty that I'm still waiting to see where they are going.
    // Very specific hack to deal with backspace sometimes failing on
    // Chrome Android when after an uneditable node.
    if (chrome$1 && android$1 && event.inputType == "deleteContentBackward") {
        view.domObserver.flushSoon();
        var ref = view.input;
        var domChangeCount = ref.domChangeCount;
        setTimeout(function () {
            if (view.input.domChangeCount != domChangeCount)
                { return; } // Event already had some effect
            // This bug tends to close the virtual keyboard, so we refocus
            view.dom.blur();
            view.focus();
            if (view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(8, "Backspace")); }))
                { return; }
            var ref = view.state.selection;
            var $cursor = ref.$cursor;
            // Crude approximation of backspace behavior when no command handled it
            if ($cursor && $cursor.pos > 0)
                { view.dispatch(view.state.tr.delete($cursor.pos - 1, $cursor.pos).scrollIntoView()); }
        }, 50);
    }
};
// Make sure all handlers get registered
for (var prop$1 in editHandlers)
    { handlers$1[prop$1] = editHandlers[prop$1]; }

function compareObjs(a, b) {
    if (a == b)
        { return true; }
    for (var p in a)
        { if (a[p] !== b[p])
            { return false; } }
    for (var p$1 in b)
        { if (!(p$1 in a))
            { return false; } }
    return true;
}
var WidgetType$1 = function WidgetType(toDOM, spec) {
    this.toDOM = toDOM;
    this.spec = spec || noSpec;
    this.side = this.spec.side || 0;
};
WidgetType$1.prototype.map = function map (mapping, span, offset, oldOffset) {
    var ref = mapping.mapResult(span.from + oldOffset, this.side < 0 ? -1 : 1);
        var pos = ref.pos;
        var deleted = ref.deleted;
    return deleted ? null : new Decoration$1(pos - offset, pos - offset, this);
};
WidgetType$1.prototype.valid = function valid () { return true; };
WidgetType$1.prototype.eq = function eq (other) {
    return this == other ||
        (other instanceof WidgetType$1 &&
            (this.spec.key && this.spec.key == other.spec.key ||
                this.toDOM == other.toDOM && compareObjs(this.spec, other.spec)));
};
WidgetType$1.prototype.destroy = function destroy (node) {
    if (this.spec.destroy)
        { this.spec.destroy(node); }
};
var InlineType = function InlineType(attrs, spec) {
    this.attrs = attrs;
    this.spec = spec || noSpec;
};
InlineType.prototype.map = function map (mapping, span, offset, oldOffset) {
    var from = mapping.map(span.from + oldOffset, this.spec.inclusiveStart ? -1 : 1) - offset;
    var to = mapping.map(span.to + oldOffset, this.spec.inclusiveEnd ? 1 : -1) - offset;
    return from >= to ? null : new Decoration$1(from, to, this);
};
InlineType.prototype.valid = function valid (_, span) { return span.from < span.to; };
InlineType.prototype.eq = function eq (other) {
    return this == other ||
        (other instanceof InlineType && compareObjs(this.attrs, other.attrs) &&
            compareObjs(this.spec, other.spec));
};
InlineType.is = function is (span) { return span.type instanceof InlineType; };
InlineType.prototype.destroy = function destroy () { };
var NodeType$1 = function NodeType(attrs, spec) {
    this.attrs = attrs;
    this.spec = spec || noSpec;
};
NodeType$1.prototype.map = function map (mapping, span, offset, oldOffset) {
    var from = mapping.mapResult(span.from + oldOffset, 1);
    if (from.deleted)
        { return null; }
    var to = mapping.mapResult(span.to + oldOffset, -1);
    if (to.deleted || to.pos <= from.pos)
        { return null; }
    return new Decoration$1(from.pos - offset, to.pos - offset, this);
};
NodeType$1.prototype.valid = function valid (node, span) {
    var ref = node.content.findIndex(span.from);
        var index = ref.index;
        var offset = ref.offset;
        var child;
    return offset == span.from && !(child = node.child(index)).isText && offset + child.nodeSize == span.to;
};
NodeType$1.prototype.eq = function eq (other) {
    return this == other ||
        (other instanceof NodeType$1 && compareObjs(this.attrs, other.attrs) &&
            compareObjs(this.spec, other.spec));
};
NodeType$1.prototype.destroy = function destroy () { };
/**
Decoration objects can be provided to the view through the
[`decorations` prop](https://prosemirror.net/docs/ref/#view.EditorProps.decorations). They come in
several variants—see the static members of this class for details.
*/
var Decoration$1 = function Decoration(
/**
The start position of the decoration.
*/
from, 
/**
The end position. Will be the same as `from` for [widget
decorations](https://prosemirror.net/docs/ref/#view.Decoration^widget).
*/
to, 
/**
@internal
*/
type) {
    this.from = from;
    this.to = to;
    this.type = type;
};

var prototypeAccessors$6$1 = { spec: { configurable: true },inline: { configurable: true },widget: { configurable: true } };
/**
@internal
*/
Decoration$1.prototype.copy = function copy (from, to) {
    return new Decoration$1(from, to, this.type);
};
/**
@internal
*/
Decoration$1.prototype.eq = function eq (other, offset) {
        if ( offset === void 0 ) offset = 0;

    return this.type.eq(other.type) && this.from + offset == other.from && this.to + offset == other.to;
};
/**
@internal
*/
Decoration$1.prototype.map = function map (mapping, offset, oldOffset) {
    return this.type.map(mapping, this, offset, oldOffset);
};
/**
Creates a widget decoration, which is a DOM node that's shown in
the document at the given position. It is recommended that you
delay rendering the widget by passing a function that will be
called when the widget is actually drawn in a view, but you can
also directly pass a DOM node. `getPos` can be used to find the
widget's current document position.
*/
Decoration$1.widget = function widget (pos, toDOM, spec) {
    return new Decoration$1(pos, pos, new WidgetType$1(toDOM, spec));
};
/**
Creates an inline decoration, which adds the given attributes to
each inline node between `from` and `to`.
*/
Decoration$1.inline = function inline (from, to, attrs, spec) {
    return new Decoration$1(from, to, new InlineType(attrs, spec));
};
/**
Creates a node decoration. `from` and `to` should point precisely
before and after a node in the document. That node, and only that
node, will receive the given attributes.
*/
Decoration$1.node = function node (from, to, attrs, spec) {
    return new Decoration$1(from, to, new NodeType$1(attrs, spec));
};
/**
The spec provided when creating this decoration. Can be useful
if you've stored extra information in that object.
*/
prototypeAccessors$6$1.spec.get = function () { return this.type.spec; };
/**
@internal
*/
prototypeAccessors$6$1.inline.get = function () { return this.type instanceof InlineType; };
/**
@internal
*/
prototypeAccessors$6$1.widget.get = function () { return this.type instanceof WidgetType$1; };

Object.defineProperties( Decoration$1.prototype, prototypeAccessors$6$1 );
var none$2 = [], noSpec = {};
/**
A collection of [decorations](https://prosemirror.net/docs/ref/#view.Decoration), organized in such
a way that the drawing algorithm can efficiently use and compare
them. This is a persistent data structure—it is not modified,
updates create a new value.
*/
var DecorationSet = function DecorationSet(local, children) {
    this.local = local.length ? local : none$2;
    this.children = children.length ? children : none$2;
};
/**
Create a set of decorations, using the structure of the given
document. This will consume (modify) the `decorations` array, so
you must make a copy if you want need to preserve that.
*/
DecorationSet.create = function create (doc, decorations) {
    return decorations.length ? buildTree$1(decorations, doc, 0, noSpec) : empty$1;
};
/**
Find all decorations in this set which touch the given range
(including decorations that start or end directly at the
boundaries) and match the given predicate on their spec. When
`start` and `end` are omitted, all decorations in the set are
considered. When `predicate` isn't given, all decorations are
assumed to match.
*/
DecorationSet.prototype.find = function find (start, end, predicate) {
    var result = [];
    this.findInner(start == null ? 0 : start, end == null ? 1e9 : end, result, 0, predicate);
    return result;
};
DecorationSet.prototype.findInner = function findInner (start, end, result, offset, predicate) {
    for (var i = 0; i < this.local.length; i++) {
        var span = this.local[i];
        if (span.from <= end && span.to >= start && (!predicate || predicate(span.spec)))
            { result.push(span.copy(span.from + offset, span.to + offset)); }
    }
    for (var i$1 = 0; i$1 < this.children.length; i$1 += 3) {
        if (this.children[i$1] < end && this.children[i$1 + 1] > start) {
            var childOff = this.children[i$1] + 1;
            this.children[i$1 + 2].findInner(start - childOff, end - childOff, result, offset + childOff, predicate);
        }
    }
};
/**
Map the set of decorations in response to a change in the
document.
*/
DecorationSet.prototype.map = function map (mapping, doc, options) {
    if (this == empty$1 || mapping.maps.length == 0)
        { return this; }
    return this.mapInner(mapping, doc, 0, 0, options || noSpec);
};
/**
@internal
*/
DecorationSet.prototype.mapInner = function mapInner (mapping, node, offset, oldOffset, options) {
    var newLocal;
    for (var i = 0; i < this.local.length; i++) {
        var mapped = this.local[i].map(mapping, offset, oldOffset);
        if (mapped && mapped.type.valid(node, mapped))
            { (newLocal || (newLocal = [])).push(mapped); }
        else if (options.onRemove)
            { options.onRemove(this.local[i].spec); }
    }
    if (this.children.length)
        { return mapChildren(this.children, newLocal || [], mapping, node, offset, oldOffset, options); }
    else
        { return newLocal ? new DecorationSet(newLocal.sort(byPos), none$2) : empty$1; }
};
/**
Add the given array of decorations to the ones in the set,
producing a new set. Consumes the `decorations` array. Needs
access to the current document to create the appropriate tree
structure.
*/
DecorationSet.prototype.add = function add (doc, decorations) {
    if (!decorations.length)
        { return this; }
    if (this == empty$1)
        { return DecorationSet.create(doc, decorations); }
    return this.addInner(doc, decorations, 0);
};
DecorationSet.prototype.addInner = function addInner (doc, decorations, offset) {
        var this$1$1 = this;

    var children, childIndex = 0;
    doc.forEach(function (childNode, childOffset) {
        var baseOffset = childOffset + offset, found;
        if (!(found = takeSpansForNode(decorations, childNode, baseOffset)))
            { return; }
        if (!children)
            { children = this$1$1.children.slice(); }
        while (childIndex < children.length && children[childIndex] < childOffset)
            { childIndex += 3; }
        if (children[childIndex] == childOffset)
            { children[childIndex + 2] = children[childIndex + 2].addInner(childNode, found, baseOffset + 1); }
        else
            { children.splice(childIndex, 0, childOffset, childOffset + childNode.nodeSize, buildTree$1(found, childNode, baseOffset + 1, noSpec)); }
        childIndex += 3;
    });
    var local = moveSpans(childIndex ? withoutNulls(decorations) : decorations, -offset);
    for (var i = 0; i < local.length; i++)
        { if (!local[i].type.valid(doc, local[i]))
            { local.splice(i--, 1); } }
    return new DecorationSet(local.length ? this.local.concat(local).sort(byPos) : this.local, children || this.children);
};
/**
Create a new set that contains the decorations in this set, minus
the ones in the given array.
*/
DecorationSet.prototype.remove = function remove (decorations) {
    if (decorations.length == 0 || this == empty$1)
        { return this; }
    return this.removeInner(decorations, 0);
};
DecorationSet.prototype.removeInner = function removeInner (decorations, offset) {
    var children = this.children, local = this.local;
    for (var i = 0; i < children.length; i += 3) {
        var found = (void 0);
        var from = children[i] + offset, to = children[i + 1] + offset;
        for (var j = 0, span = (void 0); j < decorations.length; j++)
            { if (span = decorations[j]) {
                if (span.from > from && span.to < to) {
                    decorations[j] = null;
                    (found || (found = [])).push(span);
                }
            } }
        if (!found)
            { continue; }
        if (children == this.children)
            { children = this.children.slice(); }
        var removed = children[i + 2].removeInner(found, from + 1);
        if (removed != empty$1) {
            children[i + 2] = removed;
        }
        else {
            children.splice(i, 3);
            i -= 3;
        }
    }
    if (local.length)
        { for (var i$1 = 0, span$1 = (void 0); i$1 < decorations.length; i$1++)
            { if (span$1 = decorations[i$1]) {
                for (var j$1 = 0; j$1 < local.length; j$1++)
                    { if (local[j$1].eq(span$1, offset)) {
                        if (local == this.local)
                            { local = this.local.slice(); }
                        local.splice(j$1--, 1);
                    } }
            } } }
    if (children == this.children && local == this.local)
        { return this; }
    return local.length || children.length ? new DecorationSet(local, children) : empty$1;
};
DecorationSet.prototype.forChild = function forChild (offset, node) {
    if (this == empty$1)
        { return this; }
    if (node.isLeaf)
        { return DecorationSet.empty; }
    var child, local;
    for (var i = 0; i < this.children.length; i += 3)
        { if (this.children[i] >= offset) {
            if (this.children[i] == offset)
                { child = this.children[i + 2]; }
            break;
        } }
    var start = offset + 1, end = start + node.content.size;
    for (var i$1 = 0; i$1 < this.local.length; i$1++) {
        var dec = this.local[i$1];
        if (dec.from < end && dec.to > start && (dec.type instanceof InlineType)) {
            var from = Math.max(start, dec.from) - start, to = Math.min(end, dec.to) - start;
            if (from < to)
                { (local || (local = [])).push(dec.copy(from, to)); }
        }
    }
    if (local) {
        var localSet = new DecorationSet(local.sort(byPos), none$2);
        return child ? new DecorationGroup([localSet, child]) : localSet;
    }
    return child || empty$1;
};
/**
@internal
*/
DecorationSet.prototype.eq = function eq (other) {
    if (this == other)
        { return true; }
    if (!(other instanceof DecorationSet) ||
        this.local.length != other.local.length ||
        this.children.length != other.children.length)
        { return false; }
    for (var i = 0; i < this.local.length; i++)
        { if (!this.local[i].eq(other.local[i]))
            { return false; } }
    for (var i$1 = 0; i$1 < this.children.length; i$1 += 3)
        { if (this.children[i$1] != other.children[i$1] ||
            this.children[i$1 + 1] != other.children[i$1 + 1] ||
            !this.children[i$1 + 2].eq(other.children[i$1 + 2]))
            { return false; } }
    return true;
};
/**
@internal
*/
DecorationSet.prototype.locals = function locals (node) {
    return removeOverlap(this.localsInner(node));
};
/**
@internal
*/
DecorationSet.prototype.localsInner = function localsInner (node) {
    if (this == empty$1)
        { return none$2; }
    if (node.inlineContent || !this.local.some(InlineType.is))
        { return this.local; }
    var result = [];
    for (var i = 0; i < this.local.length; i++) {
        if (!(this.local[i].type instanceof InlineType))
            { result.push(this.local[i]); }
    }
    return result;
};
DecorationSet.prototype.forEachSet = function forEachSet (f) { f(this); };
/**
The empty set of decorations.
*/
DecorationSet.empty = new DecorationSet([], []);
/**
@internal
*/
DecorationSet.removeOverlap = removeOverlap;
var empty$1 = DecorationSet.empty;
// An abstraction that allows the code dealing with decorations to
// treat multiple DecorationSet objects as if it were a single object
// with (a subset of) the same interface.
var DecorationGroup = function DecorationGroup(members) {
    this.members = members;
};
DecorationGroup.prototype.map = function map (mapping, doc) {
    var mappedDecos = this.members.map(function (member) { return member.map(mapping, doc, noSpec); });
    return DecorationGroup.from(mappedDecos);
};
DecorationGroup.prototype.forChild = function forChild (offset, child) {
    if (child.isLeaf)
        { return DecorationSet.empty; }
    var found = [];
    for (var i = 0; i < this.members.length; i++) {
        var result = this.members[i].forChild(offset, child);
        if (result == empty$1)
            { continue; }
        if (result instanceof DecorationGroup)
            { found = found.concat(result.members); }
        else
            { found.push(result); }
    }
    return DecorationGroup.from(found);
};
DecorationGroup.prototype.eq = function eq (other) {
    if (!(other instanceof DecorationGroup) ||
        other.members.length != this.members.length)
        { return false; }
    for (var i = 0; i < this.members.length; i++)
        { if (!this.members[i].eq(other.members[i]))
            { return false; } }
    return true;
};
DecorationGroup.prototype.locals = function locals (node) {
    var result, sorted = true;
    for (var i = 0; i < this.members.length; i++) {
        var locals = this.members[i].localsInner(node);
        if (!locals.length)
            { continue; }
        if (!result) {
            result = locals;
        }
        else {
            if (sorted) {
                result = result.slice();
                sorted = false;
            }
            for (var j = 0; j < locals.length; j++)
                { result.push(locals[j]); }
        }
    }
    return result ? removeOverlap(sorted ? result : result.sort(byPos)) : none$2;
};
// Create a group for the given array of decoration sets, or return
// a single set when possible.
DecorationGroup.from = function from (members) {
    switch (members.length) {
        case 0: return empty$1;
        case 1: return members[0];
        default: return new DecorationGroup(members.every(function (m) { return m instanceof DecorationSet; }) ? members :
            members.reduce(function (r, m) { return r.concat(m instanceof DecorationSet ? m : m.members); }, []));
    }
};
DecorationGroup.prototype.forEachSet = function forEachSet (f) {
    for (var i = 0; i < this.members.length; i++)
        { this.members[i].forEachSet(f); }
};
function mapChildren(oldChildren, newLocal, mapping, node, offset, oldOffset, options) {
    var children = oldChildren.slice();
    // Mark the children that are directly touched by changes, and
    // move those that are after the changes.
    var loop = function ( i, baseOffset$1 ) {
        var moved = 0;
        mapping.maps[i].forEach(function (oldStart, oldEnd, newStart, newEnd) {
            var dSize = (newEnd - newStart) - (oldEnd - oldStart);
            for (var i = 0; i < children.length; i += 3) {
                var end = children[i + 1];
                if (end < 0 || oldStart > end + baseOffset$1 - moved)
                    { continue; }
                var start = children[i] + baseOffset$1 - moved;
                if (oldEnd >= start) {
                    children[i + 1] = oldStart <= start ? -2 : -1;
                }
                else if (oldStart >= baseOffset$1 && dSize) {
                    children[i] += dSize;
                    children[i + 1] += dSize;
                }
            }
            moved += dSize;
        });
        baseOffset$1 = mapping.maps[i].map(baseOffset$1, -1);

        baseOffset = baseOffset$1;
    };

    for (var i = 0, baseOffset = oldOffset; i < mapping.maps.length; i++) loop( i, baseOffset );
    // Find the child nodes that still correspond to a single node,
    // recursively call mapInner on them and update their positions.
    var mustRebuild = false;
    for (var i$1 = 0; i$1 < children.length; i$1 += 3)
        { if (children[i$1 + 1] < 0) { // Touched nodes
            if (children[i$1 + 1] == -2) {
                mustRebuild = true;
                children[i$1 + 1] = -1;
                continue;
            }
            var from = mapping.map(oldChildren[i$1] + oldOffset), fromLocal = from - offset;
            if (fromLocal < 0 || fromLocal >= node.content.size) {
                mustRebuild = true;
                continue;
            }
            // Must read oldChildren because children was tagged with -1
            var to = mapping.map(oldChildren[i$1 + 1] + oldOffset, -1), toLocal = to - offset;
            var ref = node.content.findIndex(fromLocal);
            var index = ref.index;
            var childOffset = ref.offset;
            var childNode = node.maybeChild(index);
            if (childNode && childOffset == fromLocal && childOffset + childNode.nodeSize == toLocal) {
                var mapped = children[i$1 + 2]
                    .mapInner(mapping, childNode, from + 1, oldChildren[i$1] + oldOffset + 1, options);
                if (mapped != empty$1) {
                    children[i$1] = fromLocal;
                    children[i$1 + 1] = toLocal;
                    children[i$1 + 2] = mapped;
                }
                else {
                    children[i$1 + 1] = -2;
                    mustRebuild = true;
                }
            }
            else {
                mustRebuild = true;
            }
        } }
    // Remaining children must be collected and rebuilt into the appropriate structure
    if (mustRebuild) {
        var decorations = mapAndGatherRemainingDecorations(children, oldChildren, newLocal, mapping, offset, oldOffset, options);
        var built = buildTree$1(decorations, node, 0, options);
        newLocal = built.local;
        for (var i$2 = 0; i$2 < children.length; i$2 += 3)
            { if (children[i$2 + 1] < 0) {
                children.splice(i$2, 3);
                i$2 -= 3;
            } }
        for (var i$3 = 0, j = 0; i$3 < built.children.length; i$3 += 3) {
            var from$1 = built.children[i$3];
            while (j < children.length && children[j] < from$1)
                { j += 3; }
            children.splice(j, 0, built.children[i$3], built.children[i$3 + 1], built.children[i$3 + 2]);
        }
    }
    return new DecorationSet(newLocal.sort(byPos), children);
}
function moveSpans(spans, offset) {
    if (!offset || !spans.length)
        { return spans; }
    var result = [];
    for (var i = 0; i < spans.length; i++) {
        var span = spans[i];
        result.push(new Decoration$1(span.from + offset, span.to + offset, span.type));
    }
    return result;
}
function mapAndGatherRemainingDecorations(children, oldChildren, decorations, mapping, offset, oldOffset, options) {
    // Gather all decorations from the remaining marked children
    function gather(set, oldOffset) {
        for (var i = 0; i < set.local.length; i++) {
            var mapped = set.local[i].map(mapping, offset, oldOffset);
            if (mapped)
                { decorations.push(mapped); }
            else if (options.onRemove)
                { options.onRemove(set.local[i].spec); }
        }
        for (var i$1 = 0; i$1 < set.children.length; i$1 += 3)
            { gather(set.children[i$1 + 2], set.children[i$1] + oldOffset + 1); }
    }
    for (var i = 0; i < children.length; i += 3)
        { if (children[i + 1] == -1)
            { gather(children[i + 2], oldChildren[i] + oldOffset + 1); } }
    return decorations;
}
function takeSpansForNode(spans, node, offset) {
    if (node.isLeaf)
        { return null; }
    var end = offset + node.nodeSize, found = null;
    for (var i = 0, span = (void 0); i < spans.length; i++) {
        if ((span = spans[i]) && span.from > offset && span.to < end) {
            (found || (found = [])).push(span);
            spans[i] = null;
        }
    }
    return found;
}
function withoutNulls(array) {
    var result = [];
    for (var i = 0; i < array.length; i++)
        { if (array[i] != null)
            { result.push(array[i]); } }
    return result;
}
// Build up a tree that corresponds to a set of decorations. `offset`
// is a base offset that should be subtracted from the `from` and `to`
// positions in the spans (so that we don't have to allocate new spans
// for recursive calls).
function buildTree$1(spans, node, offset, options) {
    var children = [], hasNulls = false;
    node.forEach(function (childNode, localStart) {
        var found = takeSpansForNode(spans, childNode, localStart + offset);
        if (found) {
            hasNulls = true;
            var subtree = buildTree$1(found, childNode, offset + localStart + 1, options);
            if (subtree != empty$1)
                { children.push(localStart, localStart + childNode.nodeSize, subtree); }
        }
    });
    var locals = moveSpans(hasNulls ? withoutNulls(spans) : spans, -offset).sort(byPos);
    for (var i = 0; i < locals.length; i++)
        { if (!locals[i].type.valid(node, locals[i])) {
            if (options.onRemove)
                { options.onRemove(locals[i].spec); }
            locals.splice(i--, 1);
        } }
    return locals.length || children.length ? new DecorationSet(locals, children) : empty$1;
}
// Used to sort decorations so that ones with a low start position
// come first, and within a set with the same start position, those
// with an smaller end position come first.
function byPos(a, b) {
    return a.from - b.from || a.to - b.to;
}
// Scan a sorted array of decorations for partially overlapping spans,
// and split those so that only fully overlapping spans are left (to
// make subsequent rendering easier). Will return the input array if
// no partially overlapping spans are found (the common case).
function removeOverlap(spans) {
    var working = spans;
    for (var i = 0; i < working.length - 1; i++) {
        var span = working[i];
        if (span.from != span.to)
            { for (var j = i + 1; j < working.length; j++) {
                var next = working[j];
                if (next.from == span.from) {
                    if (next.to != span.to) {
                        if (working == spans)
                            { working = spans.slice(); }
                        // Followed by a partially overlapping larger span. Split that
                        // span.
                        working[j] = next.copy(next.from, span.to);
                        insertAhead(working, j + 1, next.copy(span.to, next.to));
                    }
                    continue;
                }
                else {
                    if (next.from < span.to) {
                        if (working == spans)
                            { working = spans.slice(); }
                        // The end of this one overlaps with a subsequent span. Split
                        // this one.
                        working[i] = span.copy(span.from, next.from);
                        insertAhead(working, j, span.copy(next.from, span.to));
                    }
                    break;
                }
            } }
    }
    return working;
}
function insertAhead(array, i, deco) {
    while (i < array.length && byPos(deco, array[i]) > 0)
        { i++; }
    array.splice(i, 0, deco);
}
// Get the decorations associated with the current props of a view.
function viewDecorations(view) {
    var found = [];
    view.someProp("decorations", function (f) {
        var result = f(view.state);
        if (result && result != empty$1)
            { found.push(result); }
    });
    if (view.cursorWrapper)
        { found.push(DecorationSet.create(view.state.doc, [view.cursorWrapper.deco])); }
    return DecorationGroup.from(found);
}

var observeOptions$1 = {
    childList: true,
    characterData: true,
    characterDataOldValue: true,
    attributes: true,
    attributeOldValue: true,
    subtree: true
};
// IE11 has very broken mutation observers, so we also listen to DOMCharacterDataModified
var useCharData$1 = ie$2 && ie_version <= 11;
var SelectionState = function SelectionState() {
    this.anchorNode = null;
    this.anchorOffset = 0;
    this.focusNode = null;
    this.focusOffset = 0;
};
SelectionState.prototype.set = function set (sel) {
    this.anchorNode = sel.anchorNode;
    this.anchorOffset = sel.anchorOffset;
    this.focusNode = sel.focusNode;
    this.focusOffset = sel.focusOffset;
};
SelectionState.prototype.clear = function clear () {
    this.anchorNode = this.focusNode = null;
};
SelectionState.prototype.eq = function eq (sel) {
    return sel.anchorNode == this.anchorNode && sel.anchorOffset == this.anchorOffset &&
        sel.focusNode == this.focusNode && sel.focusOffset == this.focusOffset;
};
var DOMObserver$1 = function DOMObserver(view, handleDOMChange) {
    var this$1$1 = this;

    this.view = view;
    this.handleDOMChange = handleDOMChange;
    this.queue = [];
    this.flushingSoon = -1;
    this.observer = null;
    this.currentSelection = new SelectionState;
    this.onCharData = null;
    this.suppressingSelectionUpdates = false;
    this.lastChangedTextNode = null;
    this.observer = window.MutationObserver &&
        new window.MutationObserver(function (mutations) {
            for (var i = 0; i < mutations.length; i++)
                { this$1$1.queue.push(mutations[i]); }
            // IE11 will sometimes (on backspacing out a single character
            // text node after a BR node) call the observer callback
            // before actually updating the DOM, which will cause
            // ProseMirror to miss the change (see #930)
            if (ie$2 && ie_version <= 11 && mutations.some(function (m) { return m.type == "childList" && m.removedNodes.length ||
                m.type == "characterData" && m.oldValue.length > m.target.nodeValue.length; }))
                { this$1$1.flushSoon(); }
            else
                { this$1$1.flush(); }
        });
    if (useCharData$1) {
        this.onCharData = function (e) {
            this$1$1.queue.push({ target: e.target, type: "characterData", oldValue: e.prevValue });
            this$1$1.flushSoon();
        };
    }
    this.onSelectionChange = this.onSelectionChange.bind(this);
};
DOMObserver$1.prototype.flushSoon = function flushSoon () {
        var this$1$1 = this;

    if (this.flushingSoon < 0)
        { this.flushingSoon = window.setTimeout(function () { this$1$1.flushingSoon = -1; this$1$1.flush(); }, 20); }
};
DOMObserver$1.prototype.forceFlush = function forceFlush () {
    if (this.flushingSoon > -1) {
        window.clearTimeout(this.flushingSoon);
        this.flushingSoon = -1;
        this.flush();
    }
};
DOMObserver$1.prototype.start = function start () {
    if (this.observer) {
        this.observer.takeRecords();
        this.observer.observe(this.view.dom, observeOptions$1);
    }
    if (this.onCharData)
        { this.view.dom.addEventListener("DOMCharacterDataModified", this.onCharData); }
    this.connectSelection();
};
DOMObserver$1.prototype.stop = function stop () {
        var this$1$1 = this;

    if (this.observer) {
        var take = this.observer.takeRecords();
        if (take.length) {
            for (var i = 0; i < take.length; i++)
                { this.queue.push(take[i]); }
            window.setTimeout(function () { return this$1$1.flush(); }, 20);
        }
        this.observer.disconnect();
    }
    if (this.onCharData)
        { this.view.dom.removeEventListener("DOMCharacterDataModified", this.onCharData); }
    this.disconnectSelection();
};
DOMObserver$1.prototype.connectSelection = function connectSelection () {
    this.view.dom.ownerDocument.addEventListener("selectionchange", this.onSelectionChange);
};
DOMObserver$1.prototype.disconnectSelection = function disconnectSelection () {
    this.view.dom.ownerDocument.removeEventListener("selectionchange", this.onSelectionChange);
};
DOMObserver$1.prototype.suppressSelectionUpdates = function suppressSelectionUpdates () {
        var this$1$1 = this;

    this.suppressingSelectionUpdates = true;
    setTimeout(function () { return this$1$1.suppressingSelectionUpdates = false; }, 50);
};
DOMObserver$1.prototype.onSelectionChange = function onSelectionChange () {
    if (!hasFocusAndSelection(this.view))
        { return; }
    if (this.suppressingSelectionUpdates)
        { return selectionToDOM(this.view); }
    // Deletions on IE11 fire their events in the wrong order, giving
    // us a selection change event before the DOM changes are
    // reported.
    if (ie$2 && ie_version <= 11 && !this.view.state.selection.empty) {
        var sel = this.view.domSelectionRange();
        // Selection.isCollapsed isn't reliable on IE
        if (sel.focusNode && isEquivalentPosition$1(sel.focusNode, sel.focusOffset, sel.anchorNode, sel.anchorOffset))
            { return this.flushSoon(); }
    }
    this.flush();
};
DOMObserver$1.prototype.setCurSelection = function setCurSelection () {
    this.currentSelection.set(this.view.domSelectionRange());
};
DOMObserver$1.prototype.ignoreSelectionChange = function ignoreSelectionChange (sel) {
    if (!sel.focusNode)
        { return true; }
    var ancestors = new Set, container;
    for (var scan = sel.focusNode; scan; scan = parentNode(scan))
        { ancestors.add(scan); }
    for (var scan$1 = sel.anchorNode; scan$1; scan$1 = parentNode(scan$1))
        { if (ancestors.has(scan$1)) {
            container = scan$1;
            break;
        } }
    var desc = container && this.view.docView.nearestDesc(container);
    if (desc && desc.ignoreMutation({
        type: "selection",
        target: container.nodeType == 3 ? container.parentNode : container
    })) {
        this.setCurSelection();
        return true;
    }
};
DOMObserver$1.prototype.pendingRecords = function pendingRecords () {
    if (this.observer)
        { for (var mut of this.observer.takeRecords())
            this.queue.push(mut); }
    return this.queue;
};
DOMObserver$1.prototype.selectionChanged = function selectionChanged (sel) {
    return !this.suppressingSelectionUpdates && !this.currentSelection.eq(sel) && hasFocusAndSelection(this.view) && !this.ignoreSelectionChange(sel);
};
DOMObserver$1.prototype.flush = function flush () {
    var ref = this;
        var view = ref.view;
    if (!view.docView || this.flushingSoon > -1)
        { return; }
    var mutations = this.pendingRecords();
    if (mutations.length)
        { this.queue = []; }
    var sel = view.domSelectionRange(), newSel = this.selectionChanged(sel);
    var from = -1, to = -1, typeOver = false, added = [];
    if (view.editable) {
        for (var i = 0; i < mutations.length; i++) {
            var result = this.registerMutation(mutations[i], added);
            if (result) {
                from = from < 0 ? result.from : Math.min(result.from, from);
                to = to < 0 ? result.to : Math.max(result.to, to);
                if (result.typeOver)
                    { typeOver = true; }
            }
        }
    }
    if (gecko$1 && added.length) {
        var brs = added.filter(function (n) { return n.nodeName == "BR"; });
        if (brs.length == 2) {
            var a = brs[0];
                var b = brs[1];
            if (a.parentNode && a.parentNode.parentNode == b.parentNode)
                { b.remove(); }
            else
                { a.remove(); }
        }
        else {
            var ref$1 = this.currentSelection;
                var focusNode = ref$1.focusNode;
            for (var br of brs) {
                var parent = br.parentNode;
                if (parent && parent.nodeName == "LI" && (!focusNode || blockParent(view, focusNode) != parent))
                    { br.remove(); }
            }
        }
    }
    var readSel = null;
    // If it looks like the browser has reset the selection to the
    // start of the document after focus, restore the selection from
    // the state
    if (from < 0 && newSel && view.input.lastFocus > Date.now() - 200 &&
        Math.max(view.input.lastTouch, view.input.lastClick.time) < Date.now() - 300 &&
        selectionCollapsed(sel) && (readSel = selectionFromDOM(view)) &&
        readSel.eq(Selection.near(view.state.doc.resolve(0), 1))) {
        view.input.lastFocus = 0;
        selectionToDOM(view);
        this.currentSelection.set(sel);
        view.scrollToSelection();
    }
    else if (from > -1 || newSel) {
        if (from > -1) {
            view.docView.markDirty(from, to);
            checkCSS(view);
        }
        this.handleDOMChange(from, to, typeOver, added);
        if (view.docView && view.docView.dirty)
            { view.updateState(view.state); }
        else if (!this.currentSelection.eq(sel))
            { selectionToDOM(view); }
        this.currentSelection.set(sel);
    }
};
DOMObserver$1.prototype.registerMutation = function registerMutation (mut, added) {
    // Ignore mutations inside nodes that were already noted as inserted
    if (added.indexOf(mut.target) > -1)
        { return null; }
    var desc = this.view.docView.nearestDesc(mut.target);
    if (mut.type == "attributes" &&
        (desc == this.view.docView || mut.attributeName == "contenteditable" ||
            // Firefox sometimes fires spurious events for null/empty styles
            (mut.attributeName == "style" && !mut.oldValue && !mut.target.getAttribute("style"))))
        { return null; }
    if (!desc || desc.ignoreMutation(mut))
        { return null; }
    if (mut.type == "childList") {
        for (var i = 0; i < mut.addedNodes.length; i++) {
            var node = mut.addedNodes[i];
            added.push(node);
            if (node.nodeType == 3)
                { this.lastChangedTextNode = node; }
        }
        if (desc.contentDOM && desc.contentDOM != desc.dom && !desc.contentDOM.contains(mut.target))
            { return { from: desc.posBefore, to: desc.posAfter }; }
        var prev = mut.previousSibling, next = mut.nextSibling;
        if (ie$2 && ie_version <= 11 && mut.addedNodes.length) {
            // IE11 gives us incorrect next/prev siblings for some
            // insertions, so if there are added nodes, recompute those
            for (var i$1 = 0; i$1 < mut.addedNodes.length; i$1++) {
                var ref = mut.addedNodes[i$1];
                    var previousSibling = ref.previousSibling;
                    var nextSibling = ref.nextSibling;
                if (!previousSibling || Array.prototype.indexOf.call(mut.addedNodes, previousSibling) < 0)
                    { prev = previousSibling; }
                if (!nextSibling || Array.prototype.indexOf.call(mut.addedNodes, nextSibling) < 0)
                    { next = nextSibling; }
            }
        }
        var fromOffset = prev && prev.parentNode == mut.target
            ? domIndex$1(prev) + 1 : 0;
        var from = desc.localPosFromDOM(mut.target, fromOffset, -1);
        var toOffset = next && next.parentNode == mut.target
            ? domIndex$1(next) : mut.target.childNodes.length;
        var to = desc.localPosFromDOM(mut.target, toOffset, 1);
        return { from: from, to: to };
    }
    else if (mut.type == "attributes") {
        return { from: desc.posAtStart - desc.border, to: desc.posAtEnd + desc.border };
    }
    else { // "characterData"
        this.lastChangedTextNode = mut.target;
        return {
            from: desc.posAtStart,
            to: desc.posAtEnd,
            // An event was generated for a text change that didn't change
            // any text. Mark the dom change to fall back to assuming the
            // selection was typed over with an identical value if it can't
            // find another change.
            typeOver: mut.target.nodeValue == mut.oldValue
        };
    }
};
var cssChecked = new WeakMap();
var cssCheckWarned = false;
function checkCSS(view) {
    if (cssChecked.has(view))
        { return; }
    cssChecked.set(view, null);
    if (['normal', 'nowrap', 'pre-line'].indexOf(getComputedStyle(view.dom).whiteSpace) !== -1) {
        view.requiresGeckoHackNode = gecko$1;
        if (cssCheckWarned)
            { return; }
        console["warn"]("ProseMirror expects the CSS white-space property to be set, preferably to 'pre-wrap'. It is recommended to load style/prosemirror.css from the prosemirror-view package.");
        cssCheckWarned = true;
    }
}
function rangeToSelectionRange(view, range) {
    var assign;

    var anchorNode = range.startContainer, anchorOffset = range.startOffset;
    var focusNode = range.endContainer, focusOffset = range.endOffset;
    var currentAnchor = view.domAtPos(view.state.selection.anchor);
    // Since such a range doesn't distinguish between anchor and head,
    // use a heuristic that flips it around if its end matches the
    // current anchor.
    if (isEquivalentPosition$1(currentAnchor.node, currentAnchor.offset, focusNode, focusOffset))
        { (assign = [focusNode, focusOffset, anchorNode, anchorOffset], anchorNode = assign[0], anchorOffset = assign[1], focusNode = assign[2], focusOffset = assign[3]); }
    return { anchorNode: anchorNode, anchorOffset: anchorOffset, focusNode: focusNode, focusOffset: focusOffset };
}
// Used to work around a Safari Selection/shadow DOM bug
// Based on https://github.com/codemirror/dev/issues/414 fix
function safariShadowSelectionRange(view, selection) {
    if (selection.getComposedRanges) {
        var range = selection.getComposedRanges(view.root)[0];
        if (range)
            { return rangeToSelectionRange(view, range); }
    }
    var found;
    function read(event) {
        event.preventDefault();
        event.stopImmediatePropagation();
        found = event.getTargetRanges()[0];
    }
    // Because Safari (at least in 2018-2022) doesn't provide regular
    // access to the selection inside a shadowRoot, we have to perform a
    // ridiculous hack to get at it—using `execCommand` to trigger a
    // `beforeInput` event so that we can read the target range from the
    // event.
    view.dom.addEventListener("beforeinput", read, true);
    document.execCommand("indent");
    view.dom.removeEventListener("beforeinput", read, true);
    return found ? rangeToSelectionRange(view, found) : null;
}
function blockParent(view, node) {
    for (var p = node.parentNode; p && p != view.dom; p = p.parentNode) {
        var desc = view.docView.nearestDesc(p, true);
        if (desc && desc.node.isBlock)
            { return p; }
    }
    return null;
}

// Note that all referencing and parsing is done with the
// start-of-operation selection and document, since that's the one
// that the DOM represents. If any changes came in in the meantime,
// the modification is mapped over those before it is applied, in
// readDOMChange.
function parseBetween(view, from_, to_) {
    var ref = view.docView.parseRange(from_, to_);
    var parent = ref.node;
    var fromOffset = ref.fromOffset;
    var toOffset = ref.toOffset;
    var from = ref.from;
    var to = ref.to;
    var domSel = view.domSelectionRange();
    var find;
    var anchor = domSel.anchorNode;
    if (anchor && view.dom.contains(anchor.nodeType == 1 ? anchor : anchor.parentNode)) {
        find = [{ node: anchor, offset: domSel.anchorOffset }];
        if (!selectionCollapsed(domSel))
            { find.push({ node: domSel.focusNode, offset: domSel.focusOffset }); }
    }
    // Work around issue in Chrome where backspacing sometimes replaces
    // the deleted content with a random BR node (issues #799, #831)
    if (chrome$1 && view.input.lastKeyCode === 8) {
        for (var off = toOffset; off > fromOffset; off--) {
            var node = parent.childNodes[off - 1], desc = node.pmViewDesc;
            if (node.nodeName == "BR" && !desc) {
                toOffset = off;
                break;
            }
            if (!desc || desc.size)
                { break; }
        }
    }
    var startDoc = view.state.doc;
    var parser = view.someProp("domParser") || DOMParser.fromSchema(view.state.schema);
    var $from = startDoc.resolve(from);
    var sel = null, doc = parser.parse(parent, {
        topNode: $from.parent,
        topMatch: $from.parent.contentMatchAt($from.index()),
        topOpen: true,
        from: fromOffset,
        to: toOffset,
        preserveWhitespace: $from.parent.type.whitespace == "pre" ? "full" : true,
        findPositions: find,
        ruleFromNode: ruleFromNode,
        context: $from
    });
    if (find && find[0].pos != null) {
        var anchor$1 = find[0].pos, head = find[1] && find[1].pos;
        if (head == null)
            { head = anchor$1; }
        sel = { anchor: anchor$1 + from, head: head + from };
    }
    return { doc: doc, sel: sel, from: from, to: to };
}
function ruleFromNode(dom) {
    var desc = dom.pmViewDesc;
    if (desc) {
        return desc.parseRule();
    }
    else if (dom.nodeName == "BR" && dom.parentNode) {
        // Safari replaces the list item or table cell with a BR
        // directly in the list node (?!) if you delete the last
        // character in a list item or table cell (#708, #862)
        if (safari$1 && /^(ul|ol)$/i.test(dom.parentNode.nodeName)) {
            var skip = document.createElement("div");
            skip.appendChild(document.createElement("li"));
            return { skip: skip };
        }
        else if (dom.parentNode.lastChild == dom || safari$1 && /^(tr|table)$/i.test(dom.parentNode.nodeName)) {
            return { ignore: true };
        }
    }
    else if (dom.nodeName == "IMG" && dom.getAttribute("mark-placeholder")) {
        return { ignore: true };
    }
    return null;
}
var isInline = /^(a|abbr|acronym|b|bd[io]|big|br|button|cite|code|data(list)?|del|dfn|em|i|ins|kbd|label|map|mark|meter|output|q|ruby|s|samp|small|span|strong|su[bp]|time|u|tt|var)$/i;
function readDOMChange(view, from, to, typeOver, addedNodes) {
    var compositionID = view.input.compositionPendingChanges || (view.composing ? view.input.compositionID : 0);
    view.input.compositionPendingChanges = 0;
    if (from < 0) {
        var origin = view.input.lastSelectionTime > Date.now() - 50 ? view.input.lastSelectionOrigin : null;
        var newSel = selectionFromDOM(view, origin);
        if (newSel && !view.state.selection.eq(newSel)) {
            if (chrome$1 && android$1 &&
                view.input.lastKeyCode === 13 && Date.now() - 100 < view.input.lastKeyCodeTime &&
                view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(13, "Enter")); }))
                { return; }
            var tr$1 = view.state.tr.setSelection(newSel);
            if (origin == "pointer")
                { tr$1.setMeta("pointer", true); }
            else if (origin == "key")
                { tr$1.scrollIntoView(); }
            if (compositionID)
                { tr$1.setMeta("composition", compositionID); }
            view.dispatch(tr$1);
        }
        return;
    }
    var $before = view.state.doc.resolve(from);
    var shared = $before.sharedDepth(to);
    from = $before.before(shared + 1);
    to = view.state.doc.resolve(to).after(shared + 1);
    var sel = view.state.selection;
    var parse = parseBetween(view, from, to);
    var doc = view.state.doc, compare = doc.slice(parse.from, parse.to);
    var preferredPos, preferredSide;
    // Prefer anchoring to end when Backspace is pressed
    if (view.input.lastKeyCode === 8 && Date.now() - 100 < view.input.lastKeyCodeTime) {
        preferredPos = view.state.selection.to;
        preferredSide = "end";
    }
    else {
        preferredPos = view.state.selection.from;
        preferredSide = "start";
    }
    view.input.lastKeyCode = null;
    var change = findDiff$1(compare.content, parse.doc.content, parse.from, preferredPos, preferredSide);
    if (change)
        { view.input.domChangeCount++; }
    if ((ios$1 && view.input.lastIOSEnter > Date.now() - 225 || android$1) &&
        addedNodes.some(function (n) { return n.nodeType == 1 && !isInline.test(n.nodeName); }) &&
        (!change || change.endA >= change.endB) &&
        view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(13, "Enter")); })) {
        view.input.lastIOSEnter = 0;
        return;
    }
    if (!change) {
        if (typeOver && sel instanceof TextSelection && !sel.empty && sel.$head.sameParent(sel.$anchor) &&
            !view.composing && !(parse.sel && parse.sel.anchor != parse.sel.head)) {
            change = { start: sel.from, endA: sel.to, endB: sel.to };
        }
        else {
            if (parse.sel) {
                var sel$1 = resolveSelection(view, view.state.doc, parse.sel);
                if (sel$1 && !sel$1.eq(view.state.selection)) {
                    var tr$2 = view.state.tr.setSelection(sel$1);
                    if (compositionID)
                        { tr$2.setMeta("composition", compositionID); }
                    view.dispatch(tr$2);
                }
            }
            return;
        }
    }
    // Handle the case where overwriting a selection by typing matches
    // the start or end of the selected content, creating a change
    // that's smaller than what was actually overwritten.
    if (view.state.selection.from < view.state.selection.to &&
        change.start == change.endB &&
        view.state.selection instanceof TextSelection) {
        if (change.start > view.state.selection.from && change.start <= view.state.selection.from + 2 &&
            view.state.selection.from >= parse.from) {
            change.start = view.state.selection.from;
        }
        else if (change.endA < view.state.selection.to && change.endA >= view.state.selection.to - 2 &&
            view.state.selection.to <= parse.to) {
            change.endB += (view.state.selection.to - change.endA);
            change.endA = view.state.selection.to;
        }
    }
    // IE11 will insert a non-breaking space _ahead_ of the space after
    // the cursor space when adding a space before another space. When
    // that happened, adjust the change to cover the space instead.
    if (ie$2 && ie_version <= 11 && change.endB == change.start + 1 &&
        change.endA == change.start && change.start > parse.from &&
        parse.doc.textBetween(change.start - parse.from - 1, change.start - parse.from + 1) == " \u00a0") {
        change.start--;
        change.endA--;
        change.endB--;
    }
    var $from = parse.doc.resolveNoCache(change.start - parse.from);
    var $to = parse.doc.resolveNoCache(change.endB - parse.from);
    var $fromA = doc.resolve(change.start);
    var inlineChange = $from.sameParent($to) && $from.parent.inlineContent && $fromA.end() >= change.endA;
    var nextSel;
    // If this looks like the effect of pressing Enter (or was recorded
    // as being an iOS enter press), just dispatch an Enter key instead.
    if (((ios$1 && view.input.lastIOSEnter > Date.now() - 225 &&
        (!inlineChange || addedNodes.some(function (n) { return n.nodeName == "DIV" || n.nodeName == "P"; }))) ||
        (!inlineChange && $from.pos < parse.doc.content.size && !$from.sameParent($to) &&
            (nextSel = Selection.findFrom(parse.doc.resolve($from.pos + 1), 1, true)) &&
            nextSel.head == $to.pos)) &&
        view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(13, "Enter")); })) {
        view.input.lastIOSEnter = 0;
        return;
    }
    // Same for backspace
    if (view.state.selection.anchor > change.start &&
        looksLikeBackspace(doc, change.start, change.endA, $from, $to) &&
        view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(8, "Backspace")); })) {
        if (android$1 && chrome$1)
            { view.domObserver.suppressSelectionUpdates(); } // #820
        return;
    }
    // Chrome Android will occasionally, during composition, delete the
    // entire composition and then immediately insert it again. This is
    // used to detect that situation.
    if (chrome$1 && android$1 && change.endB == change.start)
        { view.input.lastAndroidDelete = Date.now(); }
    // This tries to detect Android virtual keyboard
    // enter-and-pick-suggestion action. That sometimes (see issue
    // #1059) first fires a DOM mutation, before moving the selection to
    // the newly created block. And then, because ProseMirror cleans up
    // the DOM selection, it gives up moving the selection entirely,
    // leaving the cursor in the wrong place. When that happens, we drop
    // the new paragraph from the initial change, and fire a simulated
    // enter key afterwards.
    if (android$1 && !inlineChange && $from.start() != $to.start() && $to.parentOffset == 0 && $from.depth == $to.depth &&
        parse.sel && parse.sel.anchor == parse.sel.head && parse.sel.head == change.endA) {
        change.endB -= 2;
        $to = parse.doc.resolveNoCache(change.endB - parse.from);
        setTimeout(function () {
            view.someProp("handleKeyDown", function (f) { return f(view, keyEvent(13, "Enter")); });
        }, 20);
    }
    var chFrom = change.start, chTo = change.endA;
    var tr, storedMarks, markChange;
    if (inlineChange) {
        if ($from.pos == $to.pos) { // Deletion
            // IE11 sometimes weirdly moves the DOM selection around after
            // backspacing out the first element in a textblock
            if (ie$2 && ie_version <= 11 && $from.parentOffset == 0) {
                view.domObserver.suppressSelectionUpdates();
                setTimeout(function () { return selectionToDOM(view); }, 20);
            }
            tr = view.state.tr.delete(chFrom, chTo);
            storedMarks = doc.resolve(change.start).marksAcross(doc.resolve(change.endA));
        }
        else if ( // Adding or removing a mark
        change.endA == change.endB &&
            (markChange = isMarkChange($from.parent.content.cut($from.parentOffset, $to.parentOffset), $fromA.parent.content.cut($fromA.parentOffset, change.endA - $fromA.start())))) {
            tr = view.state.tr;
            if (markChange.type == "add")
                { tr.addMark(chFrom, chTo, markChange.mark); }
            else
                { tr.removeMark(chFrom, chTo, markChange.mark); }
        }
        else if ($from.parent.child($from.index()).isText && $from.index() == $to.index() - ($to.textOffset ? 0 : 1)) {
            // Both positions in the same text node -- simply insert text
            var text = $from.parent.textBetween($from.parentOffset, $to.parentOffset);
            if (view.someProp("handleTextInput", function (f) { return f(view, chFrom, chTo, text); }))
                { return; }
            tr = view.state.tr.insertText(text, chFrom, chTo);
        }
    }
    if (!tr)
        { tr = view.state.tr.replace(chFrom, chTo, parse.doc.slice(change.start - parse.from, change.endB - parse.from)); }
    if (parse.sel) {
        var sel$2 = resolveSelection(view, tr.doc, parse.sel);
        // Chrome Android will sometimes, during composition, report the
        // selection in the wrong place. If it looks like that is
        // happening, don't update the selection.
        // Edge just doesn't move the cursor forward when you start typing
        // in an empty block or between br nodes.
        if (sel$2 && !(chrome$1 && android$1 && view.composing && sel$2.empty &&
            (change.start != change.endB || view.input.lastAndroidDelete < Date.now() - 100) &&
            (sel$2.head == chFrom || sel$2.head == tr.mapping.map(chTo) - 1) ||
            ie$2 && sel$2.empty && sel$2.head == chFrom))
            { tr.setSelection(sel$2); }
    }
    if (storedMarks)
        { tr.ensureMarks(storedMarks); }
    if (compositionID)
        { tr.setMeta("composition", compositionID); }
    view.dispatch(tr.scrollIntoView());
}
function resolveSelection(view, doc, parsedSel) {
    if (Math.max(parsedSel.anchor, parsedSel.head) > doc.content.size)
        { return null; }
    return selectionBetween(view, doc.resolve(parsedSel.anchor), doc.resolve(parsedSel.head));
}
// Given two same-length, non-empty fragments of inline content,
// determine whether the first could be created from the second by
// removing or adding a single mark type.
function isMarkChange(cur, prev) {
    var curMarks = cur.firstChild.marks, prevMarks = prev.firstChild.marks;
    var added = curMarks, removed = prevMarks, type, mark, update;
    for (var i = 0; i < prevMarks.length; i++)
        { added = prevMarks[i].removeFromSet(added); }
    for (var i$1 = 0; i$1 < curMarks.length; i$1++)
        { removed = curMarks[i$1].removeFromSet(removed); }
    if (added.length == 1 && removed.length == 0) {
        mark = added[0];
        type = "add";
        update = function (node) { return node.mark(mark.addToSet(node.marks)); };
    }
    else if (added.length == 0 && removed.length == 1) {
        mark = removed[0];
        type = "remove";
        update = function (node) { return node.mark(mark.removeFromSet(node.marks)); };
    }
    else {
        return null;
    }
    var updated = [];
    for (var i$2 = 0; i$2 < prev.childCount; i$2++)
        { updated.push(update(prev.child(i$2))); }
    if (Fragment.from(updated).eq(cur))
        { return { mark: mark, type: type }; }
}
function looksLikeBackspace(old, start, end, $newStart, $newEnd) {
    if ( // The content must have shrunk
    end - start <= $newEnd.pos - $newStart.pos ||
        // newEnd must point directly at or after the end of the block that newStart points into
        skipClosingAndOpening($newStart, true, false) < $newEnd.pos)
        { return false; }
    var $start = old.resolve(start);
    // Handle the case where, rather than joining blocks, the change just removed an entire block
    if (!$newStart.parent.isTextblock) {
        var after = $start.nodeAfter;
        return after != null && end == start + after.nodeSize;
    }
    // Start must be at the end of a block
    if ($start.parentOffset < $start.parent.content.size || !$start.parent.isTextblock)
        { return false; }
    var $next = old.resolve(skipClosingAndOpening($start, true, true));
    // The next textblock must start before end and end near it
    if (!$next.parent.isTextblock || $next.pos > end ||
        skipClosingAndOpening($next, true, false) < end)
        { return false; }
    // The fragments after the join point must match
    return $newStart.parent.content.cut($newStart.parentOffset).eq($next.parent.content);
}
function skipClosingAndOpening($pos, fromEnd, mayOpen) {
    var depth = $pos.depth, end = fromEnd ? $pos.end() : $pos.pos;
    while (depth > 0 && (fromEnd || $pos.indexAfter(depth) == $pos.node(depth).childCount)) {
        depth--;
        end++;
        fromEnd = false;
    }
    if (mayOpen) {
        var next = $pos.node(depth).maybeChild($pos.indexAfter(depth));
        while (next && !next.isLeaf) {
            next = next.firstChild;
            end++;
        }
    }
    return end;
}
function findDiff$1(a, b, pos, preferredPos, preferredSide) {
    var start = a.findDiffStart(b, pos);
    if (start == null)
        { return null; }
    var ref = a.findDiffEnd(b, pos + a.size, pos + b.size);
    var endA = ref.a;
    var endB = ref.b;
    if (preferredSide == "end") {
        var adjust = Math.max(0, start - Math.min(endA, endB));
        preferredPos -= endA + adjust - start;
    }
    if (endA < start && a.size < b.size) {
        var move = preferredPos <= start && preferredPos >= endA ? start - preferredPos : 0;
        start -= move;
        if (start && start < b.size && isSurrogatePair(b.textBetween(start - 1, start + 1)))
            { start += move ? 1 : -1; }
        endB = start + (endB - endA);
        endA = start;
    }
    else if (endB < start) {
        var move$1 = preferredPos <= start && preferredPos >= endB ? start - preferredPos : 0;
        start -= move$1;
        if (start && start < a.size && isSurrogatePair(a.textBetween(start - 1, start + 1)))
            { start += move$1 ? 1 : -1; }
        endA = start + (endA - endB);
        endB = start;
    }
    return { start: start, endA: endA, endB: endB };
}
function isSurrogatePair(str) {
    if (str.length != 2)
        { return false; }
    var a = str.charCodeAt(0), b = str.charCodeAt(1);
    return a >= 0xDC00 && a <= 0xDFFF && b >= 0xD800 && b <= 0xDBFF;
}

/**
@internal
*/
var __serializeForClipboard = serializeForClipboard;
/**
@internal
*/
var __parseFromClipboard = parseFromClipboard;
/**
@internal
*/
var __endComposition = endComposition;
/**
An editor view manages the DOM structure that represents an
editable document. Its state and behavior are determined by its
[props](https://prosemirror.net/docs/ref/#view.DirectEditorProps).
*/
var EditorView$1 = function EditorView(place, props) {
    var this$1$1 = this;

    this._root = null;
    /**
    @internal
    */
    this.focused = false;
    /**
    Kludge used to work around a Chrome bug @internal
    */
    this.trackWrites = null;
    this.mounted = false;
    /**
    @internal
    */
    this.markCursor = null;
    /**
    @internal
    */
    this.cursorWrapper = null;
    /**
    @internal
    */
    this.lastSelectedViewDesc = undefined;
    /**
    @internal
    */
    this.input = new InputState$1;
    this.prevDirectPlugins = [];
    this.pluginViews = [];
    /**
    Holds `true` when a hack node is needed in Firefox to prevent the
    [space is eaten issue](https://github.com/ProseMirror/prosemirror/issues/651)
    @internal
    */
    this.requiresGeckoHackNode = false;
    /**
    When editor content is being dragged, this object contains
    information about the dragged slice and whether it is being
    copied or moved. At any other time, it is null.
    */
    this.dragging = null;
    this._props = props;
    this.state = props.state;
    this.directPlugins = props.plugins || [];
    this.directPlugins.forEach(checkStateComponent);
    this.dispatch = this.dispatch.bind(this);
    this.dom = (place && place.mount) || document.createElement("div");
    if (place) {
        if (place.appendChild)
            { place.appendChild(this.dom); }
        else if (typeof place == "function")
            { place(this.dom); }
        else if (place.mount)
            { this.mounted = true; }
    }
    this.editable = getEditable(this);
    updateCursorWrapper(this);
    this.nodeViews = buildNodeViews(this);
    this.docView = docViewDesc(this.state.doc, computeDocDeco(this), viewDecorations(this), this.dom, this);
    this.domObserver = new DOMObserver$1(this, function (from, to, typeOver, added) { return readDOMChange(this$1$1, from, to, typeOver, added); });
    this.domObserver.start();
    initInput(this);
    this.updatePluginViews();
};

var prototypeAccessors$7$1 = { composing: { configurable: true },props: { configurable: true },root: { configurable: true },isDestroyed: { configurable: true } };
/**
Holds `true` when a
[composition](https://w3c.github.io/uievents/#events-compositionevents)
is active.
*/
prototypeAccessors$7$1.composing.get = function () { return this.input.composing; };
/**
The view's current [props](https://prosemirror.net/docs/ref/#view.EditorProps).
*/
prototypeAccessors$7$1.props.get = function () {
    if (this._props.state != this.state) {
        var prev = this._props;
        this._props = {};
        for (var name in prev)
            { this._props[name] = prev[name]; }
        this._props.state = this.state;
    }
    return this._props;
};
/**
Update the view's props. Will immediately cause an update to
the DOM.
*/
EditorView$1.prototype.update = function update (props) {
    if (props.handleDOMEvents != this._props.handleDOMEvents)
        { ensureListeners(this); }
    var prevProps = this._props;
    this._props = props;
    if (props.plugins) {
        props.plugins.forEach(checkStateComponent);
        this.directPlugins = props.plugins;
    }
    this.updateStateInner(props.state, prevProps);
};
/**
Update the view by updating existing props object with the object
given as argument. Equivalent to `view.update(Object.assign({},
view.props, props))`.
*/
EditorView$1.prototype.setProps = function setProps (props) {
    var updated = {};
    for (var name in this._props)
        { updated[name] = this._props[name]; }
    updated.state = this.state;
    for (var name$1 in props)
        { updated[name$1] = props[name$1]; }
    this.update(updated);
};
/**
Update the editor's `state` prop, without touching any of the
other props.
*/
EditorView$1.prototype.updateState = function updateState (state) {
    this.updateStateInner(state, this._props);
};
EditorView$1.prototype.updateStateInner = function updateStateInner (state, prevProps) {
    var _a;
    var prev = this.state, redraw = false, updateSel = false;
    // When stored marks are added, stop composition, so that they can
    // be displayed.
    if (state.storedMarks && this.composing) {
        clearComposition(this);
        updateSel = true;
    }
    this.state = state;
    var pluginsChanged = prev.plugins != state.plugins || this._props.plugins != prevProps.plugins;
    if (pluginsChanged || this._props.plugins != prevProps.plugins || this._props.nodeViews != prevProps.nodeViews) {
        var nodeViews = buildNodeViews(this);
        if (changedNodeViews(nodeViews, this.nodeViews)) {
            this.nodeViews = nodeViews;
            redraw = true;
        }
    }
    if (pluginsChanged || prevProps.handleDOMEvents != this._props.handleDOMEvents) {
        ensureListeners(this);
    }
    this.editable = getEditable(this);
    updateCursorWrapper(this);
    var innerDeco = viewDecorations(this), outerDeco = computeDocDeco(this);
    var scroll = prev.plugins != state.plugins && !prev.doc.eq(state.doc) ? "reset"
        : state.scrollToSelection > prev.scrollToSelection ? "to selection" : "preserve";
    var updateDoc = redraw || !this.docView.matchesNode(state.doc, outerDeco, innerDeco);
    if (updateDoc || !state.selection.eq(prev.selection))
        { updateSel = true; }
    var oldScrollPos = scroll == "preserve" && updateSel && this.dom.style.overflowAnchor == null && storeScrollPos(this);
    if (updateSel) {
        this.domObserver.stop();
        // Work around an issue in Chrome, IE, and Edge where changing
        // the DOM around an active selection puts it into a broken
        // state where the thing the user sees differs from the
        // selection reported by the Selection object (#710, #973,
        // #1011, #1013, #1035).
        var forceSelUpdate = updateDoc && (ie$2 || chrome$1) && !this.composing &&
            !prev.selection.empty && !state.selection.empty && selectionContextChanged(prev.selection, state.selection);
        if (updateDoc) {
            // If the node that the selection points into is written to,
            // Chrome sometimes starts misreporting the selection, so this
            // tracks that and forces a selection reset when our update
            // did write to the node.
            var chromeKludge = chrome$1 ? (this.trackWrites = this.domSelectionRange().focusNode) : null;
            if (this.composing)
                { this.input.compositionNode = findCompositionNode$1(this); }
            if (redraw || !this.docView.update(state.doc, outerDeco, innerDeco, this)) {
                this.docView.updateOuterDeco(outerDeco);
                this.docView.destroy();
                this.docView = docViewDesc(state.doc, outerDeco, innerDeco, this.dom, this);
            }
            if (chromeKludge && !this.trackWrites)
                { forceSelUpdate = true; }
        }
        // Work around for an issue where an update arriving right between
        // a DOM selection change and the "selectionchange" event for it
        // can cause a spurious DOM selection update, disrupting mouse
        // drag selection.
        if (forceSelUpdate ||
            !(this.input.mouseDown && this.domObserver.currentSelection.eq(this.domSelectionRange()) &&
                anchorInRightPlace(this))) {
            selectionToDOM(this, forceSelUpdate);
        }
        else {
            syncNodeSelection(this, state.selection);
            this.domObserver.setCurSelection();
        }
        this.domObserver.start();
    }
    this.updatePluginViews(prev);
    if (((_a = this.dragging) === null || _a === void 0 ? void 0 : _a.node) && !prev.doc.eq(state.doc))
        { this.updateDraggedNode(this.dragging, prev); }
    if (scroll == "reset") {
        this.dom.scrollTop = 0;
    }
    else if (scroll == "to selection") {
        this.scrollToSelection();
    }
    else if (oldScrollPos) {
        resetScrollPos(oldScrollPos);
    }
};
/**
@internal
*/
EditorView$1.prototype.scrollToSelection = function scrollToSelection () {
        var this$1$1 = this;

    var startDOM = this.domSelectionRange().focusNode;
    if (this.someProp("handleScrollToSelection", function (f) { return f(this$1$1); })) ;
    else if (this.state.selection instanceof NodeSelection) {
        var target = this.docView.domAfterPos(this.state.selection.from);
        if (target.nodeType == 1)
            { scrollRectIntoView$1(this, target.getBoundingClientRect(), startDOM); }
    }
    else {
        scrollRectIntoView$1(this, this.coordsAtPos(this.state.selection.head, 1), startDOM);
    }
};
EditorView$1.prototype.destroyPluginViews = function destroyPluginViews () {
    var view;
    while (view = this.pluginViews.pop())
        { if (view.destroy)
            { view.destroy(); } }
};
EditorView$1.prototype.updatePluginViews = function updatePluginViews (prevState) {
    if (!prevState || prevState.plugins != this.state.plugins || this.directPlugins != this.prevDirectPlugins) {
        this.prevDirectPlugins = this.directPlugins;
        this.destroyPluginViews();
        for (var i = 0; i < this.directPlugins.length; i++) {
            var plugin = this.directPlugins[i];
            if (plugin.spec.view)
                { this.pluginViews.push(plugin.spec.view(this)); }
        }
        for (var i$1 = 0; i$1 < this.state.plugins.length; i$1++) {
            var plugin$1 = this.state.plugins[i$1];
            if (plugin$1.spec.view)
                { this.pluginViews.push(plugin$1.spec.view(this)); }
        }
    }
    else {
        for (var i$2 = 0; i$2 < this.pluginViews.length; i$2++) {
            var pluginView = this.pluginViews[i$2];
            if (pluginView.update)
                { pluginView.update(this, prevState); }
        }
    }
};
EditorView$1.prototype.updateDraggedNode = function updateDraggedNode (dragging, prev) {
    var sel = dragging.node, found = -1;
    if (this.state.doc.nodeAt(sel.from) == sel.node) {
        found = sel.from;
    }
    else {
        var movedPos = sel.from + (this.state.doc.content.size - prev.doc.content.size);
        var moved = movedPos > 0 && this.state.doc.nodeAt(movedPos);
        if (moved == sel.node)
            { found = movedPos; }
    }
    this.dragging = new Dragging(dragging.slice, dragging.move, found < 0 ? undefined : NodeSelection.create(this.state.doc, found));
};
EditorView$1.prototype.someProp = function someProp (propName, f) {
    var prop = this._props && this._props[propName], value;
    if (prop != null && (value = f ? f(prop) : prop))
        { return value; }
    for (var i = 0; i < this.directPlugins.length; i++) {
        var prop$1 = this.directPlugins[i].props[propName];
        if (prop$1 != null && (value = f ? f(prop$1) : prop$1))
            { return value; }
    }
    var plugins = this.state.plugins;
    if (plugins)
        { for (var i$1 = 0; i$1 < plugins.length; i$1++) {
            var prop$2 = plugins[i$1].props[propName];
            if (prop$2 != null && (value = f ? f(prop$2) : prop$2))
                { return value; }
        } }
};
/**
Query whether the view has focus.
*/
EditorView$1.prototype.hasFocus = function hasFocus () {
    // Work around IE not handling focus correctly if resize handles are shown.
    // If the cursor is inside an element with resize handles, activeElement
    // will be that element instead of this.dom.
    if (ie$2) {
        // If activeElement is within this.dom, and there are no other elements
        // setting `contenteditable` to false in between, treat it as focused.
        var node = this.root.activeElement;
        if (node == this.dom)
            { return true; }
        if (!node || !this.dom.contains(node))
            { return false; }
        while (node && this.dom != node && this.dom.contains(node)) {
            if (node.contentEditable == 'false')
                { return false; }
            node = node.parentElement;
        }
        return true;
    }
    return this.root.activeElement == this.dom;
};
/**
Focus the editor.
*/
EditorView$1.prototype.focus = function focus () {
    this.domObserver.stop();
    if (this.editable)
        { focusPreventScroll$1(this.dom); }
    selectionToDOM(this);
    this.domObserver.start();
};
/**
Get the document root in which the editor exists. This will
usually be the top-level `document`, but might be a [shadow
DOM](https://developer.mozilla.org/en-US/docs/Web/Web_Components/Shadow_DOM)
root if the editor is inside one.
*/
prototypeAccessors$7$1.root.get = function () {
        var this$1$1 = this;

    var cached = this._root;
    if (cached == null)
        { var loop = function ( search ) {
            if (search.nodeType == 9 || (search.nodeType == 11 && search.host)) {
                if (!search.getSelection)
                    { Object.getPrototypeOf(search).getSelection = function () { return search.ownerDocument.getSelection(); }; }
                return { v: this$1$1._root = search };
            }
        };

                for (var search = this$1$1.dom.parentNode; search; search = search.parentNode) {
                    var returned = loop( search );

                    if ( returned ) return returned.v;
                } }
    return cached || document;
};
/**
When an existing editor view is moved to a new document or
shadow tree, call this to make it recompute its root.
*/
EditorView$1.prototype.updateRoot = function updateRoot () {
    this._root = null;
};
/**
Given a pair of viewport coordinates, return the document
position that corresponds to them. May return null if the given
coordinates aren't inside of the editor. When an object is
returned, its `pos` property is the position nearest to the
coordinates, and its `inside` property holds the position of the
inner node that the position falls inside of, or -1 if it is at
the top level, not in any node.
*/
EditorView$1.prototype.posAtCoords = function posAtCoords$1$1 (coords) {
    return posAtCoords$1(this, coords);
};
/**
Returns the viewport rectangle at a given document position.
`left` and `right` will be the same number, as this returns a
flat cursor-ish rectangle. If the position is between two things
that aren't directly adjacent, `side` determines which element
is used. When < 0, the element before the position is used,
otherwise the element after.
*/
EditorView$1.prototype.coordsAtPos = function coordsAtPos$1 (pos, side) {
        if ( side === void 0 ) side = 1;

    return coordsAtPos(this, pos, side);
};
/**
Find the DOM position that corresponds to the given document
position. When `side` is negative, find the position as close as
possible to the content before the position. When positive,
prefer positions close to the content after the position. When
zero, prefer as shallow a position as possible.
    
Note that you should **not** mutate the editor's internal DOM,
only inspect it (and even that is usually not necessary).
*/
EditorView$1.prototype.domAtPos = function domAtPos (pos, side) {
        if ( side === void 0 ) side = 0;

    return this.docView.domFromPos(pos, side);
};
/**
Find the DOM node that represents the document node after the
given position. May return `null` when the position doesn't point
in front of a node or if the node is inside an opaque node view.
    
This is intended to be able to call things like
`getBoundingClientRect` on that DOM node. Do **not** mutate the
editor DOM directly, or add styling this way, since that will be
immediately overriden by the editor as it redraws the node.
*/
EditorView$1.prototype.nodeDOM = function nodeDOM (pos) {
    var desc = this.docView.descAt(pos);
    return desc ? desc.nodeDOM : null;
};
/**
Find the document position that corresponds to a given DOM
position. (Whenever possible, it is preferable to inspect the
document structure directly, rather than poking around in the
DOM, but sometimes—for example when interpreting an event
target—you don't have a choice.)
    
The `bias` parameter can be used to influence which side of a DOM
node to use when the position is inside a leaf node.
*/
EditorView$1.prototype.posAtDOM = function posAtDOM (node, offset, bias) {
        if ( bias === void 0 ) bias = -1;

    var pos = this.docView.posFromDOM(node, offset, bias);
    if (pos == null)
        { throw new RangeError("DOM position not inside the editor"); }
    return pos;
};
/**
Find out whether the selection is at the end of a textblock when
moving in a given direction. When, for example, given `"left"`,
it will return true if moving left from the current cursor
position would leave that position's parent textblock. Will apply
to the view's current state by default, but it is possible to
pass a different state.
*/
EditorView$1.prototype.endOfTextblock = function endOfTextblock$1 (dir, state) {
    return endOfTextblock(this, state || this.state, dir);
};
/**
Run the editor's paste logic with the given HTML string. The
`event`, if given, will be passed to the
[`handlePaste`](https://prosemirror.net/docs/ref/#view.EditorProps.handlePaste) hook.
*/
EditorView$1.prototype.pasteHTML = function pasteHTML (html, event) {
    return doPaste$1(this, "", html, false, event || new ClipboardEvent("paste"));
};
/**
Run the editor's paste logic with the given plain-text input.
*/
EditorView$1.prototype.pasteText = function pasteText (text, event) {
    return doPaste$1(this, text, null, true, event || new ClipboardEvent("paste"));
};
/**
Removes the editor from the DOM and destroys all [node
views](https://prosemirror.net/docs/ref/#view.NodeView).
*/
EditorView$1.prototype.destroy = function destroy () {
    if (!this.docView)
        { return; }
    destroyInput(this);
    this.destroyPluginViews();
    if (this.mounted) {
        this.docView.update(this.state.doc, [], viewDecorations(this), this);
        this.dom.textContent = "";
    }
    else if (this.dom.parentNode) {
        this.dom.parentNode.removeChild(this.dom);
    }
    this.docView.destroy();
    this.docView = null;
    clearReusedRange();
};
/**
This is true when the view has been
[destroyed](https://prosemirror.net/docs/ref/#view.EditorView.destroy) (and thus should not be
used anymore).
*/
prototypeAccessors$7$1.isDestroyed.get = function () {
    return this.docView == null;
};
/**
Used for testing.
*/
EditorView$1.prototype.dispatchEvent = function dispatchEvent$1 (event) {
    return dispatchEvent(this, event);
};
/**
Dispatch a transaction. Will call
[`dispatchTransaction`](https://prosemirror.net/docs/ref/#view.DirectEditorProps.dispatchTransaction)
when given, and otherwise defaults to applying the transaction to
the current state and calling
[`updateState`](https://prosemirror.net/docs/ref/#view.EditorView.updateState) with the result.
This method is bound to the view instance, so that it can be
easily passed around.
*/
EditorView$1.prototype.dispatch = function dispatch (tr) {
    var dispatchTransaction = this._props.dispatchTransaction;
    if (dispatchTransaction)
        { dispatchTransaction.call(this, tr); }
    else
        { this.updateState(this.state.apply(tr)); }
};
/**
@internal
*/
EditorView$1.prototype.domSelectionRange = function domSelectionRange () {
    var sel = this.domSelection();
    if (!sel)
        { return { focusNode: null, focusOffset: 0, anchorNode: null, anchorOffset: 0 }; }
    return safari$1 && this.root.nodeType === 11 &&
        deepActiveElement(this.dom.ownerDocument) == this.dom && safariShadowSelectionRange(this, sel) || sel;
};
/**
@internal
*/
EditorView$1.prototype.domSelection = function domSelection () {
    return this.root.getSelection();
};

Object.defineProperties( EditorView$1.prototype, prototypeAccessors$7$1 );
function computeDocDeco(view) {
    var attrs = Object.create(null);
    attrs.class = "ProseMirror";
    attrs.contenteditable = String(view.editable);
    view.someProp("attributes", function (value) {
        if (typeof value == "function")
            { value = value(view.state); }
        if (value)
            { for (var attr in value) {
                if (attr == "class")
                    { attrs.class += " " + value[attr]; }
                else if (attr == "style")
                    { attrs.style = (attrs.style ? attrs.style + ";" : "") + value[attr]; }
                else if (!attrs[attr] && attr != "contenteditable" && attr != "nodeName")
                    { attrs[attr] = String(value[attr]); }
            } }
    });
    if (!attrs.translate)
        { attrs.translate = "no"; }
    return [Decoration$1.node(0, view.state.doc.content.size, attrs)];
}
function updateCursorWrapper(view) {
    if (view.markCursor) {
        var dom = document.createElement("img");
        dom.className = "ProseMirror-separator";
        dom.setAttribute("mark-placeholder", "true");
        dom.setAttribute("alt", "");
        view.cursorWrapper = { dom: dom, deco: Decoration$1.widget(view.state.selection.from, dom, { raw: true, marks: view.markCursor }) };
    }
    else {
        view.cursorWrapper = null;
    }
}
function getEditable(view) {
    return !view.someProp("editable", function (value) { return value(view.state) === false; });
}
function selectionContextChanged(sel1, sel2) {
    var depth = Math.min(sel1.$anchor.sharedDepth(sel1.head), sel2.$anchor.sharedDepth(sel2.head));
    return sel1.$anchor.start(depth) != sel2.$anchor.start(depth);
}
function buildNodeViews(view) {
    var result = Object.create(null);
    function add(obj) {
        for (var prop in obj)
            { if (!Object.prototype.hasOwnProperty.call(result, prop))
                { result[prop] = obj[prop]; } }
    }
    view.someProp("nodeViews", add);
    view.someProp("markViews", add);
    return result;
}
function changedNodeViews(a, b) {
    var nA = 0, nB = 0;
    for (var prop in a) {
        if (a[prop] != b[prop])
            { return true; }
        nA++;
    }
    for (var _ in b)
        { nB++; }
    return nA != nB;
}
function checkStateComponent(plugin) {
    if (plugin.spec.state || plugin.spec.filterTransaction || plugin.spec.appendTransaction)
        { throw new RangeError("Plugins passed directly to the view must not have a state component"); }
}var view=/*#__PURE__*/Object.freeze({__proto__:null,Decoration:Decoration$1,DecorationSet:DecorationSet,EditorView:EditorView$1,__endComposition:__endComposition,__parseFromClipboard:__parseFromClipboard,__serializeForClipboard:__serializeForClipboard});/**
Input rules are regular expressions describing a piece of text
that, when typed, causes something to happen. This might be
changing two dashes into an emdash, wrapping a paragraph starting
with `"> "` into a blockquote, or something entirely different.
*/
var InputRule = function InputRule(
/**
@internal
*/
match, handler, options) {
    if ( options === void 0 ) options = {};

    this.match = match;
    this.match = match;
    this.handler = typeof handler == "string" ? stringHandler(handler) : handler;
    this.undoable = options.undoable !== false;
    this.inCode = options.inCode || false;
};
function stringHandler(string) {
    return function (state, match, start, end) {
        var insert = string;
        if (match[1]) {
            var offset = match[0].lastIndexOf(match[1]);
            insert += match[0].slice(offset + match[1].length);
            start += offset;
            var cutOff = start - end;
            if (cutOff > 0) {
                insert = match[0].slice(offset - cutOff, offset) + insert;
                start = end;
            }
        }
        return state.tr.insertText(insert, start, end);
    };
}
var MAX_MATCH = 500;
/**
Create an input rules plugin. When enabled, it will cause text
input that matches any of the given rules to trigger the rule's
action.
*/
function inputRules(ref) {
    var rules = ref.rules;

    var plugin = new Plugin({
        state: {
            init: function init() { return null; },
            apply: function apply(tr, prev) {
                var stored = tr.getMeta(this);
                if (stored)
                    { return stored; }
                return tr.selectionSet || tr.docChanged ? null : prev;
            }
        },
        props: {
            handleTextInput: function handleTextInput(view, from, to, text) {
                return run(view, from, to, text, rules, plugin);
            },
            handleDOMEvents: {
                compositionend: function (view) {
                    setTimeout(function () {
                        var ref = view.state.selection;
                        var $cursor = ref.$cursor;
                        if ($cursor)
                            { run(view, $cursor.pos, $cursor.pos, "", rules, plugin); }
                    });
                }
            }
        },
        isInputRules: true
    });
    return plugin;
}
function run(view, from, to, text, rules, plugin) {
    if (view.composing)
        { return false; }
    var state = view.state, $from = state.doc.resolve(from);
    var textBefore = $from.parent.textBetween(Math.max(0, $from.parentOffset - MAX_MATCH), $from.parentOffset, null, "\ufffc") + text;
    for (var i = 0; i < rules.length; i++) {
        var rule = rules[i];
        if ($from.parent.type.spec.code) {
            if (!rule.inCode)
                { continue; }
        }
        else if (rule.inCode === "only") {
            continue;
        }
        var match = rule.match.exec(textBefore);
        var tr = match && rule.handler(state, match, from - (match[0].length - text.length), to);
        if (!tr)
            { continue; }
        if (rule.undoable)
            { tr.setMeta(plugin, { transform: tr, from: from, to: to, text: text }); }
        view.dispatch(tr);
        return true;
    }
    return false;
}
/**
This is a command that will undo an input rule, if applying such a
rule was the last thing that the user did.
*/
var undoInputRule = function (state, dispatch) {
    var plugins = state.plugins;
    for (var i = 0; i < plugins.length; i++) {
        var plugin = plugins[i], undoable = (void 0);
        if (plugin.spec.isInputRules && (undoable = plugin.getState(state))) {
            if (dispatch) {
                var tr = state.tr, toUndo = undoable.transform;
                for (var j = toUndo.steps.length - 1; j >= 0; j--)
                    { tr.step(toUndo.steps[j].invert(toUndo.docs[j])); }
                if (undoable.text) {
                    var marks = tr.doc.resolve(undoable.from).marks();
                    tr.replaceWith(undoable.from, undoable.to, state.schema.text(undoable.text, marks));
                }
                else {
                    tr.delete(undoable.from, undoable.to);
                }
                dispatch(tr);
            }
            return true;
        }
    }
    return false;
};

/**
Converts double dashes to an emdash.
*/
var emDash = new InputRule(/--$/, "—");
/**
Converts three dots to an ellipsis character.
*/
var ellipsis = new InputRule(/\.\.\.$/, "…");
/**
“Smart” opening double quotes.
*/
var openDoubleQuote = new InputRule(/(?:^|[\s\{\[\(\<'"\u2018\u201C])(")$/, "“");
/**
“Smart” closing double quotes.
*/
var closeDoubleQuote = new InputRule(/"$/, "”");
/**
“Smart” opening single quotes.
*/
var openSingleQuote = new InputRule(/(?:^|[\s\{\[\(\<'"\u2018\u201C])(')$/, "‘");
/**
“Smart” closing single quotes.
*/
var closeSingleQuote = new InputRule(/'$/, "’");
/**
Smart-quote related input rules.
*/
var smartQuotes = [openDoubleQuote, closeDoubleQuote, openSingleQuote, closeSingleQuote];

/**
Build an input rule for automatically wrapping a textblock when a
given string is typed. The `regexp` argument is
directly passed through to the `InputRule` constructor. You'll
probably want the regexp to start with `^`, so that the pattern can
only occur at the start of a textblock.

`nodeType` is the type of node to wrap in. If it needs attributes,
you can either pass them directly, or pass a function that will
compute them from the regular expression match.

By default, if there's a node with the same type above the newly
wrapped node, the rule will try to [join](https://prosemirror.net/docs/ref/#transform.Transform.join) those
two nodes. You can pass a join predicate, which takes a regular
expression match and the node before the wrapped node, and can
return a boolean to indicate whether a join should happen.
*/
function wrappingInputRule(regexp, nodeType, getAttrs, joinPredicate) {
    if ( getAttrs === void 0 ) getAttrs = null;

    return new InputRule(regexp, function (state, match, start, end) {
        var attrs = getAttrs instanceof Function ? getAttrs(match) : getAttrs;
        var tr = state.tr.delete(start, end);
        var $start = tr.doc.resolve(start), range = $start.blockRange(), wrapping = range && findWrapping(range, nodeType, attrs);
        if (!wrapping)
            { return null; }
        tr.wrap(range, wrapping);
        var before = tr.doc.resolve(start - 1).nodeBefore;
        if (before && before.type == nodeType && canJoin(tr.doc, start - 1) &&
            (!joinPredicate || joinPredicate(match, before)))
            { tr.join(start - 1); }
        return tr;
    });
}
/**
Build an input rule that changes the type of a textblock when the
matched text is typed into it. You'll usually want to start your
regexp with `^` to that it is only matched at the start of a
textblock. The optional `getAttrs` parameter can be used to compute
the new node's attributes, and works the same as in the
`wrappingInputRule` function.
*/
function textblockTypeInputRule(regexp, nodeType, getAttrs) {
    if ( getAttrs === void 0 ) getAttrs = null;

    return new InputRule(regexp, function (state, match, start, end) {
        var $start = state.doc.resolve(start);
        var attrs = getAttrs instanceof Function ? getAttrs(match) : getAttrs;
        if (!$start.node(-1).canReplaceWith($start.index(-1), $start.indexAfter(-1), nodeType))
            { return null; }
        return state.tr
            .delete(start, end)
            .setBlockType(start, start, nodeType, attrs);
    });
}var inputRules$1=/*#__PURE__*/Object.freeze({__proto__:null,InputRule:InputRule,closeDoubleQuote:closeDoubleQuote,closeSingleQuote:closeSingleQuote,ellipsis:ellipsis,emDash:emDash,inputRules:inputRules,openDoubleQuote:openDoubleQuote,openSingleQuote:openSingleQuote,smartQuotes:smartQuotes,textblockTypeInputRule:textblockTypeInputRule,undoInputRule:undoInputRule,wrappingInputRule:wrappingInputRule});/**
Delete the selection, if there is one.
*/
var deleteSelection = function (state, dispatch) {
    if (state.selection.empty)
        { return false; }
    if (dispatch)
        { dispatch(state.tr.deleteSelection().scrollIntoView()); }
    return true;
};
function atBlockStart(state, view) {
    var ref = state.selection;
    var $cursor = ref.$cursor;
    if (!$cursor || (view ? !view.endOfTextblock("backward", state)
        : $cursor.parentOffset > 0))
        { return null; }
    return $cursor;
}
/**
If the selection is empty and at the start of a textblock, try to
reduce the distance between that block and the one before it—if
there's a block directly before it that can be joined, join them.
If not, try to move the selected block closer to the next one in
the document structure by lifting it out of its parent or moving it
into a parent of the previous block. Will use the view for accurate
(bidi-aware) start-of-textblock detection if given.
*/
var joinBackward = function (state, dispatch, view) {
    var $cursor = atBlockStart(state, view);
    if (!$cursor)
        { return false; }
    var $cut = findCutBefore($cursor);
    // If there is no node before this, try to lift
    if (!$cut) {
        var range = $cursor.blockRange(), target = range && liftTarget(range);
        if (target == null)
            { return false; }
        if (dispatch)
            { dispatch(state.tr.lift(range, target).scrollIntoView()); }
        return true;
    }
    var before = $cut.nodeBefore;
    // Apply the joining algorithm
    if (deleteBarrier(state, $cut, dispatch, -1))
        { return true; }
    // If the node below has no content and the node above is
    // selectable, delete the node below and select the one above.
    if ($cursor.parent.content.size == 0 &&
        (textblockAt(before, "end") || NodeSelection.isSelectable(before))) {
        for (var depth = $cursor.depth;; depth--) {
            var delStep = replaceStep(state.doc, $cursor.before(depth), $cursor.after(depth), Slice.empty);
            if (delStep && delStep.slice.size < delStep.to - delStep.from) {
                if (dispatch) {
                    var tr = state.tr.step(delStep);
                    tr.setSelection(textblockAt(before, "end")
                        ? Selection.findFrom(tr.doc.resolve(tr.mapping.map($cut.pos, -1)), -1)
                        : NodeSelection.create(tr.doc, $cut.pos - before.nodeSize));
                    dispatch(tr.scrollIntoView());
                }
                return true;
            }
            if (depth == 1 || $cursor.node(depth - 1).childCount > 1)
                { break; }
        }
    }
    // If the node before is an atom, delete it
    if (before.isAtom && $cut.depth == $cursor.depth - 1) {
        if (dispatch)
            { dispatch(state.tr.delete($cut.pos - before.nodeSize, $cut.pos).scrollIntoView()); }
        return true;
    }
    return false;
};
/**
A more limited form of [`joinBackward`]($commands.joinBackward)
that only tries to join the current textblock to the one before
it, if the cursor is at the start of a textblock.
*/
var joinTextblockBackward = function (state, dispatch, view) {
    var $cursor = atBlockStart(state, view);
    if (!$cursor)
        { return false; }
    var $cut = findCutBefore($cursor);
    return $cut ? joinTextblocksAround(state, $cut, dispatch) : false;
};
/**
A more limited form of [`joinForward`]($commands.joinForward)
that only tries to join the current textblock to the one after
it, if the cursor is at the end of a textblock.
*/
var joinTextblockForward = function (state, dispatch, view) {
    var $cursor = atBlockEnd(state, view);
    if (!$cursor)
        { return false; }
    var $cut = findCutAfter($cursor);
    return $cut ? joinTextblocksAround(state, $cut, dispatch) : false;
};
function joinTextblocksAround(state, $cut, dispatch) {
    var before = $cut.nodeBefore, beforeText = before, beforePos = $cut.pos - 1;
    for (; !beforeText.isTextblock; beforePos--) {
        if (beforeText.type.spec.isolating)
            { return false; }
        var child = beforeText.lastChild;
        if (!child)
            { return false; }
        beforeText = child;
    }
    var after = $cut.nodeAfter, afterText = after, afterPos = $cut.pos + 1;
    for (; !afterText.isTextblock; afterPos++) {
        if (afterText.type.spec.isolating)
            { return false; }
        var child$1 = afterText.firstChild;
        if (!child$1)
            { return false; }
        afterText = child$1;
    }
    var step = replaceStep(state.doc, beforePos, afterPos, Slice.empty);
    if (!step || step.from != beforePos ||
        step instanceof ReplaceStep && step.slice.size >= afterPos - beforePos)
        { return false; }
    if (dispatch) {
        var tr = state.tr.step(step);
        tr.setSelection(TextSelection.create(tr.doc, beforePos));
        dispatch(tr.scrollIntoView());
    }
    return true;
}
function textblockAt(node, side, only) {
    if ( only === void 0 ) only = false;

    for (var scan = node; scan; scan = (side == "start" ? scan.firstChild : scan.lastChild)) {
        if (scan.isTextblock)
            { return true; }
        if (only && scan.childCount != 1)
            { return false; }
    }
    return false;
}
/**
When the selection is empty and at the start of a textblock, select
the node before that textblock, if possible. This is intended to be
bound to keys like backspace, after
[`joinBackward`](https://prosemirror.net/docs/ref/#commands.joinBackward) or other deleting
commands, as a fall-back behavior when the schema doesn't allow
deletion at the selected point.
*/
var selectNodeBackward = function (state, dispatch, view) {
    var ref = state.selection;
    var $head = ref.$head;
    var empty = ref.empty;
    var $cut = $head;
    if (!empty)
        { return false; }
    if ($head.parent.isTextblock) {
        if (view ? !view.endOfTextblock("backward", state) : $head.parentOffset > 0)
            { return false; }
        $cut = findCutBefore($head);
    }
    var node = $cut && $cut.nodeBefore;
    if (!node || !NodeSelection.isSelectable(node))
        { return false; }
    if (dispatch)
        { dispatch(state.tr.setSelection(NodeSelection.create(state.doc, $cut.pos - node.nodeSize)).scrollIntoView()); }
    return true;
};
function findCutBefore($pos) {
    if (!$pos.parent.type.spec.isolating)
        { for (var i = $pos.depth - 1; i >= 0; i--) {
            if ($pos.index(i) > 0)
                { return $pos.doc.resolve($pos.before(i + 1)); }
            if ($pos.node(i).type.spec.isolating)
                { break; }
        } }
    return null;
}
function atBlockEnd(state, view) {
    var ref = state.selection;
    var $cursor = ref.$cursor;
    if (!$cursor || (view ? !view.endOfTextblock("forward", state)
        : $cursor.parentOffset < $cursor.parent.content.size))
        { return null; }
    return $cursor;
}
/**
If the selection is empty and the cursor is at the end of a
textblock, try to reduce or remove the boundary between that block
and the one after it, either by joining them or by moving the other
block closer to this one in the tree structure. Will use the view
for accurate start-of-textblock detection if given.
*/
var joinForward = function (state, dispatch, view) {
    var $cursor = atBlockEnd(state, view);
    if (!$cursor)
        { return false; }
    var $cut = findCutAfter($cursor);
    // If there is no node after this, there's nothing to do
    if (!$cut)
        { return false; }
    var after = $cut.nodeAfter;
    // Try the joining algorithm
    if (deleteBarrier(state, $cut, dispatch, 1))
        { return true; }
    // If the node above has no content and the node below is
    // selectable, delete the node above and select the one below.
    if ($cursor.parent.content.size == 0 &&
        (textblockAt(after, "start") || NodeSelection.isSelectable(after))) {
        var delStep = replaceStep(state.doc, $cursor.before(), $cursor.after(), Slice.empty);
        if (delStep && delStep.slice.size < delStep.to - delStep.from) {
            if (dispatch) {
                var tr = state.tr.step(delStep);
                tr.setSelection(textblockAt(after, "start") ? Selection.findFrom(tr.doc.resolve(tr.mapping.map($cut.pos)), 1)
                    : NodeSelection.create(tr.doc, tr.mapping.map($cut.pos)));
                dispatch(tr.scrollIntoView());
            }
            return true;
        }
    }
    // If the next node is an atom, delete it
    if (after.isAtom && $cut.depth == $cursor.depth - 1) {
        if (dispatch)
            { dispatch(state.tr.delete($cut.pos, $cut.pos + after.nodeSize).scrollIntoView()); }
        return true;
    }
    return false;
};
/**
When the selection is empty and at the end of a textblock, select
the node coming after that textblock, if possible. This is intended
to be bound to keys like delete, after
[`joinForward`](https://prosemirror.net/docs/ref/#commands.joinForward) and similar deleting
commands, to provide a fall-back behavior when the schema doesn't
allow deletion at the selected point.
*/
var selectNodeForward = function (state, dispatch, view) {
    var ref = state.selection;
    var $head = ref.$head;
    var empty = ref.empty;
    var $cut = $head;
    if (!empty)
        { return false; }
    if ($head.parent.isTextblock) {
        if (view ? !view.endOfTextblock("forward", state) : $head.parentOffset < $head.parent.content.size)
            { return false; }
        $cut = findCutAfter($head);
    }
    var node = $cut && $cut.nodeAfter;
    if (!node || !NodeSelection.isSelectable(node))
        { return false; }
    if (dispatch)
        { dispatch(state.tr.setSelection(NodeSelection.create(state.doc, $cut.pos)).scrollIntoView()); }
    return true;
};
function findCutAfter($pos) {
    if (!$pos.parent.type.spec.isolating)
        { for (var i = $pos.depth - 1; i >= 0; i--) {
            var parent = $pos.node(i);
            if ($pos.index(i) + 1 < parent.childCount)
                { return $pos.doc.resolve($pos.after(i + 1)); }
            if (parent.type.spec.isolating)
                { break; }
        } }
    return null;
}
/**
Join the selected block or, if there is a text selection, the
closest ancestor block of the selection that can be joined, with
the sibling above it.
*/
var joinUp = function (state, dispatch) {
    var sel = state.selection, nodeSel = sel instanceof NodeSelection, point;
    if (nodeSel) {
        if (sel.node.isTextblock || !canJoin(state.doc, sel.from))
            { return false; }
        point = sel.from;
    }
    else {
        point = joinPoint(state.doc, sel.from, -1);
        if (point == null)
            { return false; }
    }
    if (dispatch) {
        var tr = state.tr.join(point);
        if (nodeSel)
            { tr.setSelection(NodeSelection.create(tr.doc, point - state.doc.resolve(point).nodeBefore.nodeSize)); }
        dispatch(tr.scrollIntoView());
    }
    return true;
};
/**
Join the selected block, or the closest ancestor of the selection
that can be joined, with the sibling after it.
*/
var joinDown = function (state, dispatch) {
    var sel = state.selection, point;
    if (sel instanceof NodeSelection) {
        if (sel.node.isTextblock || !canJoin(state.doc, sel.to))
            { return false; }
        point = sel.to;
    }
    else {
        point = joinPoint(state.doc, sel.to, 1);
        if (point == null)
            { return false; }
    }
    if (dispatch)
        { dispatch(state.tr.join(point).scrollIntoView()); }
    return true;
};
/**
Lift the selected block, or the closest ancestor block of the
selection that can be lifted, out of its parent node.
*/
var lift$1 = function (state, dispatch) {
    var ref = state.selection;
    var $from = ref.$from;
    var $to = ref.$to;
    var range = $from.blockRange($to), target = range && liftTarget(range);
    if (target == null)
        { return false; }
    if (dispatch)
        { dispatch(state.tr.lift(range, target).scrollIntoView()); }
    return true;
};
/**
If the selection is in a node whose type has a truthy
[`code`](https://prosemirror.net/docs/ref/#model.NodeSpec.code) property in its spec, replace the
selection with a newline character.
*/
var newlineInCode = function (state, dispatch) {
    var ref = state.selection;
    var $head = ref.$head;
    var $anchor = ref.$anchor;
    if (!$head.parent.type.spec.code || !$head.sameParent($anchor))
        { return false; }
    if (dispatch)
        { dispatch(state.tr.insertText("\n").scrollIntoView()); }
    return true;
};
function defaultBlockAt(match) {
    for (var i = 0; i < match.edgeCount; i++) {
        var ref = match.edge(i);
        var type = ref.type;
        if (type.isTextblock && !type.hasRequiredAttrs())
            { return type; }
    }
    return null;
}
/**
When the selection is in a node with a truthy
[`code`](https://prosemirror.net/docs/ref/#model.NodeSpec.code) property in its spec, create a
default block after the code block, and move the cursor there.
*/
var exitCode = function (state, dispatch) {
    var ref = state.selection;
    var $head = ref.$head;
    var $anchor = ref.$anchor;
    if (!$head.parent.type.spec.code || !$head.sameParent($anchor))
        { return false; }
    var above = $head.node(-1), after = $head.indexAfter(-1), type = defaultBlockAt(above.contentMatchAt(after));
    if (!type || !above.canReplaceWith(after, after, type))
        { return false; }
    if (dispatch) {
        var pos = $head.after(), tr = state.tr.replaceWith(pos, pos, type.createAndFill());
        tr.setSelection(Selection.near(tr.doc.resolve(pos), 1));
        dispatch(tr.scrollIntoView());
    }
    return true;
};
/**
If a block node is selected, create an empty paragraph before (if
it is its parent's first child) or after it.
*/
var createParagraphNear = function (state, dispatch) {
    var sel = state.selection;
    var $from = sel.$from;
    var $to = sel.$to;
    if (sel instanceof AllSelection || $from.parent.inlineContent || $to.parent.inlineContent)
        { return false; }
    var type = defaultBlockAt($to.parent.contentMatchAt($to.indexAfter()));
    if (!type || !type.isTextblock)
        { return false; }
    if (dispatch) {
        var side = (!$from.parentOffset && $to.index() < $to.parent.childCount ? $from : $to).pos;
        var tr = state.tr.insert(side, type.createAndFill());
        tr.setSelection(TextSelection.create(tr.doc, side + 1));
        dispatch(tr.scrollIntoView());
    }
    return true;
};
/**
If the cursor is in an empty textblock that can be lifted, lift the
block.
*/
var liftEmptyBlock = function (state, dispatch) {
    var ref = state.selection;
    var $cursor = ref.$cursor;
    if (!$cursor || $cursor.parent.content.size)
        { return false; }
    if ($cursor.depth > 1 && $cursor.after() != $cursor.end(-1)) {
        var before = $cursor.before();
        if (canSplit(state.doc, before)) {
            if (dispatch)
                { dispatch(state.tr.split(before).scrollIntoView()); }
            return true;
        }
    }
    var range = $cursor.blockRange(), target = range && liftTarget(range);
    if (target == null)
        { return false; }
    if (dispatch)
        { dispatch(state.tr.lift(range, target).scrollIntoView()); }
    return true;
};
/**
Create a variant of [`splitBlock`](https://prosemirror.net/docs/ref/#commands.splitBlock) that uses
a custom function to determine the type of the newly split off block.
*/
function splitBlockAs(splitNode) {
    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        if (state.selection instanceof NodeSelection && state.selection.node.isBlock) {
            if (!$from.parentOffset || !canSplit(state.doc, $from.pos))
                { return false; }
            if (dispatch)
                { dispatch(state.tr.split($from.pos).scrollIntoView()); }
            return true;
        }
        if (!$from.parent.isBlock)
            { return false; }
        if (dispatch) {
            var atEnd = $to.parentOffset == $to.parent.content.size;
            var tr = state.tr;
            if (state.selection instanceof TextSelection || state.selection instanceof AllSelection)
                { tr.deleteSelection(); }
            var deflt = $from.depth == 0 ? null : defaultBlockAt($from.node(-1).contentMatchAt($from.indexAfter(-1)));
            var splitType = splitNode && splitNode($to.parent, atEnd, $from);
            var types = splitType ? [splitType] : atEnd && deflt ? [{ type: deflt }] : undefined;
            var can = canSplit(tr.doc, tr.mapping.map($from.pos), 1, types);
            if (!types && !can && canSplit(tr.doc, tr.mapping.map($from.pos), 1, deflt ? [{ type: deflt }] : undefined)) {
                if (deflt)
                    { types = [{ type: deflt }]; }
                can = true;
            }
            if (can) {
                tr.split(tr.mapping.map($from.pos), 1, types);
                if (!atEnd && !$from.parentOffset && $from.parent.type != deflt) {
                    var first = tr.mapping.map($from.before()), $first = tr.doc.resolve(first);
                    if (deflt && $from.node(-1).canReplaceWith($first.index(), $first.index() + 1, deflt))
                        { tr.setNodeMarkup(tr.mapping.map($from.before()), deflt); }
                }
            }
            dispatch(tr.scrollIntoView());
        }
        return true;
    };
}
/**
Split the parent block of the selection. If the selection is a text
selection, also delete its content.
*/
var splitBlock = splitBlockAs();
/**
Acts like [`splitBlock`](https://prosemirror.net/docs/ref/#commands.splitBlock), but without
resetting the set of active marks at the cursor.
*/
var splitBlockKeepMarks = function (state, dispatch) {
    return splitBlock(state, dispatch && (function (tr) {
        var marks = state.storedMarks || (state.selection.$to.parentOffset && state.selection.$from.marks());
        if (marks)
            { tr.ensureMarks(marks); }
        dispatch(tr);
    }));
};
/**
Move the selection to the node wrapping the current selection, if
any. (Will not select the document node.)
*/
var selectParentNode = function (state, dispatch) {
    var ref = state.selection;
    var $from = ref.$from;
    var to = ref.to;
    var pos;
    var same = $from.sharedDepth(to);
    if (same == 0)
        { return false; }
    pos = $from.before(same);
    if (dispatch)
        { dispatch(state.tr.setSelection(NodeSelection.create(state.doc, pos))); }
    return true;
};
/**
Select the whole document.
*/
var selectAll$1 = function (state, dispatch) {
    if (dispatch)
        { dispatch(state.tr.setSelection(new AllSelection(state.doc))); }
    return true;
};
function joinMaybeClear(state, $pos, dispatch) {
    var before = $pos.nodeBefore, after = $pos.nodeAfter, index = $pos.index();
    if (!before || !after || !before.type.compatibleContent(after.type))
        { return false; }
    if (!before.content.size && $pos.parent.canReplace(index - 1, index)) {
        if (dispatch)
            { dispatch(state.tr.delete($pos.pos - before.nodeSize, $pos.pos).scrollIntoView()); }
        return true;
    }
    if (!$pos.parent.canReplace(index, index + 1) || !(after.isTextblock || canJoin(state.doc, $pos.pos)))
        { return false; }
    if (dispatch)
        { dispatch(state.tr
            .clearIncompatible($pos.pos, before.type, before.contentMatchAt(before.childCount))
            .join($pos.pos)
            .scrollIntoView()); }
    return true;
}
function deleteBarrier(state, $cut, dispatch, dir) {
    var before = $cut.nodeBefore, after = $cut.nodeAfter, conn, match;
    var isolated = before.type.spec.isolating || after.type.spec.isolating;
    if (!isolated && joinMaybeClear(state, $cut, dispatch))
        { return true; }
    var canDelAfter = !isolated && $cut.parent.canReplace($cut.index(), $cut.index() + 1);
    if (canDelAfter &&
        (conn = (match = before.contentMatchAt(before.childCount)).findWrapping(after.type)) &&
        match.matchType(conn[0] || after.type).validEnd) {
        if (dispatch) {
            var end = $cut.pos + after.nodeSize, wrap = Fragment.empty;
            for (var i = conn.length - 1; i >= 0; i--)
                { wrap = Fragment.from(conn[i].create(null, wrap)); }
            wrap = Fragment.from(before.copy(wrap));
            var tr = state.tr.step(new ReplaceAroundStep($cut.pos - 1, end, $cut.pos, end, new Slice(wrap, 1, 0), conn.length, true));
            var joinAt = end + 2 * conn.length;
            if (canJoin(tr.doc, joinAt))
                { tr.join(joinAt); }
            dispatch(tr.scrollIntoView());
        }
        return true;
    }
    var selAfter = after.type.spec.isolating || (dir > 0 && isolated) ? null : Selection.findFrom($cut, 1);
    var range = selAfter && selAfter.$from.blockRange(selAfter.$to), target = range && liftTarget(range);
    if (target != null && target >= $cut.depth) {
        if (dispatch)
            { dispatch(state.tr.lift(range, target).scrollIntoView()); }
        return true;
    }
    if (canDelAfter && textblockAt(after, "start", true) && textblockAt(before, "end")) {
        var at = before, wrap$1 = [];
        for (;;) {
            wrap$1.push(at);
            if (at.isTextblock)
                { break; }
            at = at.lastChild;
        }
        var afterText = after, afterDepth = 1;
        for (; !afterText.isTextblock; afterText = afterText.firstChild)
            { afterDepth++; }
        if (at.canReplace(at.childCount, at.childCount, afterText.content)) {
            if (dispatch) {
                var end$1 = Fragment.empty;
                for (var i$1 = wrap$1.length - 1; i$1 >= 0; i$1--)
                    { end$1 = Fragment.from(wrap$1[i$1].copy(end$1)); }
                var tr$1 = state.tr.step(new ReplaceAroundStep($cut.pos - wrap$1.length, $cut.pos + after.nodeSize, $cut.pos + afterDepth, $cut.pos + after.nodeSize - afterDepth, new Slice(end$1, wrap$1.length, 0), 0, true));
                dispatch(tr$1.scrollIntoView());
            }
            return true;
        }
    }
    return false;
}
function selectTextblockSide(side) {
    return function (state, dispatch) {
        var sel = state.selection, $pos = side < 0 ? sel.$from : sel.$to;
        var depth = $pos.depth;
        while ($pos.node(depth).isInline) {
            if (!depth)
                { return false; }
            depth--;
        }
        if (!$pos.node(depth).isTextblock)
            { return false; }
        if (dispatch)
            { dispatch(state.tr.setSelection(TextSelection.create(state.doc, side < 0 ? $pos.start(depth) : $pos.end(depth)))); }
        return true;
    };
}
/**
Moves the cursor to the start of current text block.
*/
var selectTextblockStart = selectTextblockSide(-1);
/**
Moves the cursor to the end of current text block.
*/
var selectTextblockEnd = selectTextblockSide(1);
// Parameterized commands
/**
Wrap the selection in a node of the given type with the given
attributes.
*/
function wrapIn(nodeType, attrs) {
    if ( attrs === void 0 ) attrs = null;

    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        var range = $from.blockRange($to), wrapping = range && findWrapping(range, nodeType, attrs);
        if (!wrapping)
            { return false; }
        if (dispatch)
            { dispatch(state.tr.wrap(range, wrapping).scrollIntoView()); }
        return true;
    };
}
/**
Returns a command that tries to set the selected textblocks to the
given node type with the given attributes.
*/
function setBlockType(nodeType, attrs) {
    if ( attrs === void 0 ) attrs = null;

    return function (state, dispatch) {
        var applicable = false;
        for (var i = 0; i < state.selection.ranges.length && !applicable; i++) {
            var ref = state.selection.ranges[i];
            var from = ref.$from.pos;
            var to = ref.$to.pos;
            state.doc.nodesBetween(from, to, function (node, pos) {
                if (applicable)
                    { return false; }
                if (!node.isTextblock || node.hasMarkup(nodeType, attrs))
                    { return; }
                if (node.type == nodeType) {
                    applicable = true;
                }
                else {
                    var $pos = state.doc.resolve(pos), index = $pos.index();
                    applicable = $pos.parent.canReplaceWith(index, index + 1, nodeType);
                }
            });
        }
        if (!applicable)
            { return false; }
        if (dispatch) {
            var tr = state.tr;
            for (var i$1 = 0; i$1 < state.selection.ranges.length; i$1++) {
                var ref$1 = state.selection.ranges[i$1];
                var from$1 = ref$1.$from.pos;
                var to$1 = ref$1.$to.pos;
                tr.setBlockType(from$1, to$1, nodeType, attrs);
            }
            dispatch(tr.scrollIntoView());
        }
        return true;
    };
}
function markApplies$1(doc, ranges, type, enterAtoms) {
    var loop = function ( i ) {
        var ref = ranges[i];
        var $from = ref.$from;
        var $to = ref.$to;
        var can = $from.depth == 0 ? doc.inlineContent && doc.type.allowsMarkType(type) : false;
        doc.nodesBetween($from.pos, $to.pos, function (node, pos) {
            if (can || !enterAtoms && node.isAtom && node.isInline && pos >= $from.pos && pos + node.nodeSize <= $to.pos)
                { return false; }
            can = node.inlineContent && node.type.allowsMarkType(type);
        });
        if (can)
            { return { v: true }; }
    };

    for (var i = 0; i < ranges.length; i++) {
        var returned = loop( i );

        if ( returned ) return returned.v;
    }
    return false;
}
function removeInlineAtoms(ranges) {
    var result = [];
    var loop = function ( i ) {
        var ref = ranges[i];
        var $from = ref.$from;
        var $to = ref.$to;
        $from.doc.nodesBetween($from.pos, $to.pos, function (node, pos) {
            if (node.isAtom && node.content.size && node.isInline && pos >= $from.pos && pos + node.nodeSize <= $to.pos) {
                if (pos + 1 > $from.pos)
                    { result.push(new SelectionRange$1($from, $from.doc.resolve(pos + 1))); }
                $from = $from.doc.resolve(pos + 1 + node.content.size);
                return false;
            }
        });
        if ($from.pos < $to.pos)
            { result.push(new SelectionRange$1($from, $to)); }
    };

    for (var i = 0; i < ranges.length; i++) loop( i );
    return result;
}
/**
Create a command function that toggles the given mark with the
given attributes. Will return `false` when the current selection
doesn't support that mark. This will remove the mark if any marks
of that type exist in the selection, or add it otherwise. If the
selection is empty, this applies to the [stored
marks](https://prosemirror.net/docs/ref/#state.EditorState.storedMarks) instead of a range of the
document.
*/
function toggleMark(markType, attrs, options) {
    if ( attrs === void 0 ) attrs = null;

    var removeWhenPresent = (options && options.removeWhenPresent) !== false;
    var enterAtoms = (options && options.enterInlineAtoms) !== false;
    return function (state, dispatch) {
        var ref = state.selection;
        var empty = ref.empty;
        var $cursor = ref.$cursor;
        var ranges = ref.ranges;
        if ((empty && !$cursor) || !markApplies$1(state.doc, ranges, markType, enterAtoms))
            { return false; }
        if (dispatch) {
            if ($cursor) {
                if (markType.isInSet(state.storedMarks || $cursor.marks()))
                    { dispatch(state.tr.removeStoredMark(markType)); }
                else
                    { dispatch(state.tr.addStoredMark(markType.create(attrs))); }
            }
            else {
                var add, tr = state.tr;
                if (!enterAtoms)
                    { ranges = removeInlineAtoms(ranges); }
                if (removeWhenPresent) {
                    add = !ranges.some(function (r) { return state.doc.rangeHasMark(r.$from.pos, r.$to.pos, markType); });
                }
                else {
                    add = !ranges.every(function (r) {
                        var missing = false;
                        tr.doc.nodesBetween(r.$from.pos, r.$to.pos, function (node, pos, parent) {
                            if (missing)
                                { return false; }
                            missing = !markType.isInSet(node.marks) && !!parent && parent.type.allowsMarkType(markType) &&
                                !(node.isText && /^\s*$/.test(node.textBetween(Math.max(0, r.$from.pos - pos), Math.min(node.nodeSize, r.$to.pos - pos))));
                        });
                        return !missing;
                    });
                }
                for (var i = 0; i < ranges.length; i++) {
                    var ref$1 = ranges[i];
                    var $from = ref$1.$from;
                    var $to = ref$1.$to;
                    if (!add) {
                        tr.removeMark($from.pos, $to.pos, markType);
                    }
                    else {
                        var from = $from.pos, to = $to.pos, start = $from.nodeAfter, end = $to.nodeBefore;
                        var spaceStart = start && start.isText ? /^\s*/.exec(start.text)[0].length : 0;
                        var spaceEnd = end && end.isText ? /\s*$/.exec(end.text)[0].length : 0;
                        if (from + spaceStart < to) {
                            from += spaceStart;
                            to -= spaceEnd;
                        }
                        tr.addMark(from, to, markType.create(attrs));
                    }
                }
                dispatch(tr.scrollIntoView());
            }
        }
        return true;
    };
}
function wrapDispatchForJoin(dispatch, isJoinable) {
    return function (tr) {
        if (!tr.isGeneric)
            { return dispatch(tr); }
        var ranges = [];
        for (var i = 0; i < tr.mapping.maps.length; i++) {
            var map = tr.mapping.maps[i];
            for (var j = 0; j < ranges.length; j++)
                { ranges[j] = map.map(ranges[j]); }
            map.forEach(function (_s, _e, from, to) { return ranges.push(from, to); });
        }
        // Figure out which joinable points exist inside those ranges,
        // by checking all node boundaries in their parent nodes.
        var joinable = [];
        for (var i$1 = 0; i$1 < ranges.length; i$1 += 2) {
            var from = ranges[i$1], to = ranges[i$1 + 1];
            var $from = tr.doc.resolve(from), depth = $from.sharedDepth(to), parent = $from.node(depth);
            for (var index = $from.indexAfter(depth), pos = $from.after(depth + 1); pos <= to; ++index) {
                var after = parent.maybeChild(index);
                if (!after)
                    { break; }
                if (index && joinable.indexOf(pos) == -1) {
                    var before = parent.child(index - 1);
                    if (before.type == after.type && isJoinable(before, after))
                        { joinable.push(pos); }
                }
                pos += after.nodeSize;
            }
        }
        // Join the joinable points
        joinable.sort(function (a, b) { return a - b; });
        for (var i$2 = joinable.length - 1; i$2 >= 0; i$2--) {
            if (canJoin(tr.doc, joinable[i$2]))
                { tr.join(joinable[i$2]); }
        }
        dispatch(tr);
    };
}
/**
Wrap a command so that, when it produces a transform that causes
two joinable nodes to end up next to each other, those are joined.
Nodes are considered joinable when they are of the same type and
when the `isJoinable` predicate returns true for them or, if an
array of strings was passed, if their node type name is in that
array.
*/
function autoJoin(command, isJoinable) {
    var canJoin = Array.isArray(isJoinable) ? function (node) { return isJoinable.indexOf(node.type.name) > -1; }
        : isJoinable;
    return function (state, dispatch, view) { return command(state, dispatch && wrapDispatchForJoin(dispatch, canJoin), view); };
}
/**
Combine a number of command functions into a single function (which
calls them one by one until one returns true).
*/
function chainCommands() {
    var commands = [], len = arguments.length;
    while ( len-- ) commands[ len ] = arguments[ len ];

    return function (state, dispatch, view) {
        for (var i = 0; i < commands.length; i++)
            { if (commands[i](state, dispatch, view))
                { return true; } }
        return false;
    };
}
var backspace = chainCommands(deleteSelection, joinBackward, selectNodeBackward);
var del = chainCommands(deleteSelection, joinForward, selectNodeForward);
/**
A basic keymap containing bindings not specific to any schema.
Binds the following keys (when multiple commands are listed, they
are chained with [`chainCommands`](https://prosemirror.net/docs/ref/#commands.chainCommands)):

* **Enter** to `newlineInCode`, `createParagraphNear`, `liftEmptyBlock`, `splitBlock`
* **Mod-Enter** to `exitCode`
* **Backspace** and **Mod-Backspace** to `deleteSelection`, `joinBackward`, `selectNodeBackward`
* **Delete** and **Mod-Delete** to `deleteSelection`, `joinForward`, `selectNodeForward`
* **Mod-Delete** to `deleteSelection`, `joinForward`, `selectNodeForward`
* **Mod-a** to `selectAll`
*/
var pcBaseKeymap = {
    "Enter": chainCommands(newlineInCode, createParagraphNear, liftEmptyBlock, splitBlock),
    "Mod-Enter": exitCode,
    "Backspace": backspace,
    "Mod-Backspace": backspace,
    "Shift-Backspace": backspace,
    "Delete": del,
    "Mod-Delete": del,
    "Mod-a": selectAll$1
};
/**
A copy of `pcBaseKeymap` that also binds **Ctrl-h** like Backspace,
**Ctrl-d** like Delete, **Alt-Backspace** like Ctrl-Backspace, and
**Ctrl-Alt-Backspace**, **Alt-Delete**, and **Alt-d** like
Ctrl-Delete.
*/
var macBaseKeymap = {
    "Ctrl-h": pcBaseKeymap["Backspace"],
    "Alt-Backspace": pcBaseKeymap["Mod-Backspace"],
    "Ctrl-d": pcBaseKeymap["Delete"],
    "Ctrl-Alt-Backspace": pcBaseKeymap["Mod-Delete"],
    "Alt-Delete": pcBaseKeymap["Mod-Delete"],
    "Alt-d": pcBaseKeymap["Mod-Delete"],
    "Ctrl-a": selectTextblockStart,
    "Ctrl-e": selectTextblockEnd
};
for (var key$3 in pcBaseKeymap)
    { macBaseKeymap[key$3] = pcBaseKeymap[key$3]; }
var mac$3 = typeof navigator != "undefined" ? /Mac|iP(hone|[oa]d)/.test(navigator.platform)
    // @ts-ignore
    : typeof os != "undefined" && os.platform ? os.platform() == "darwin" : false;
/**
Depending on the detected platform, this will hold
[`pcBasekeymap`](https://prosemirror.net/docs/ref/#commands.pcBaseKeymap) or
[`macBaseKeymap`](https://prosemirror.net/docs/ref/#commands.macBaseKeymap).
*/
var baseKeymap = mac$3 ? macBaseKeymap : pcBaseKeymap;var commands=/*#__PURE__*/Object.freeze({__proto__:null,autoJoin:autoJoin,baseKeymap:baseKeymap,chainCommands:chainCommands,createParagraphNear:createParagraphNear,deleteSelection:deleteSelection,exitCode:exitCode,joinBackward:joinBackward,joinDown:joinDown,joinForward:joinForward,joinTextblockBackward:joinTextblockBackward,joinTextblockForward:joinTextblockForward,joinUp:joinUp,lift:lift$1,liftEmptyBlock:liftEmptyBlock,macBaseKeymap:macBaseKeymap,newlineInCode:newlineInCode,pcBaseKeymap:pcBaseKeymap,selectAll:selectAll$1,selectNodeBackward:selectNodeBackward,selectNodeForward:selectNodeForward,selectParentNode:selectParentNode,selectTextblockEnd:selectTextblockEnd,selectTextblockStart:selectTextblockStart,setBlockType:setBlockType,splitBlock:splitBlock,splitBlockAs:splitBlockAs,splitBlockKeepMarks:splitBlockKeepMarks,toggleMark:toggleMark,wrapIn:wrapIn});var GOOD_LEAF_SIZE = 200;

// :: class<T> A rope sequence is a persistent sequence data structure
// that supports appending, prepending, and slicing without doing a
// full copy. It is represented as a mostly-balanced tree.
var RopeSequence = function RopeSequence () {};

RopeSequence.prototype.append = function append (other) {
  if (!other.length) { return this }
  other = RopeSequence.from(other);

  return (!this.length && other) ||
    (other.length < GOOD_LEAF_SIZE && this.leafAppend(other)) ||
    (this.length < GOOD_LEAF_SIZE && other.leafPrepend(this)) ||
    this.appendInner(other)
};

// :: (union<[T], RopeSequence<T>>) → RopeSequence<T>
// Prepend an array or other rope to this one, returning a new rope.
RopeSequence.prototype.prepend = function prepend (other) {
  if (!other.length) { return this }
  return RopeSequence.from(other).append(this)
};

RopeSequence.prototype.appendInner = function appendInner (other) {
  return new Append(this, other)
};

// :: (?number, ?number) → RopeSequence<T>
// Create a rope repesenting a sub-sequence of this rope.
RopeSequence.prototype.slice = function slice (from, to) {
    if ( from === void 0 ) { from = 0; }
    if ( to === void 0 ) { to = this.length; }

  if (from >= to) { return RopeSequence.empty }
  return this.sliceInner(Math.max(0, from), Math.min(this.length, to))
};

// :: (number) → T
// Retrieve the element at the given position from this rope.
RopeSequence.prototype.get = function get (i) {
  if (i < 0 || i >= this.length) { return undefined }
  return this.getInner(i)
};

// :: ((element: T, index: number) → ?bool, ?number, ?number)
// Call the given function for each element between the given
// indices. This tends to be more efficient than looping over the
// indices and calling `get`, because it doesn't have to descend the
// tree for every element.
RopeSequence.prototype.forEach = function forEach (f, from, to) {
    if ( from === void 0 ) { from = 0; }
    if ( to === void 0 ) { to = this.length; }

  if (from <= to)
    { this.forEachInner(f, from, to, 0); }
  else
    { this.forEachInvertedInner(f, from, to, 0); }
};

// :: ((element: T, index: number) → U, ?number, ?number) → [U]
// Map the given functions over the elements of the rope, producing
// a flat array.
RopeSequence.prototype.map = function map (f, from, to) {
    if ( from === void 0 ) { from = 0; }
    if ( to === void 0 ) { to = this.length; }

  var result = [];
  this.forEach(function (elt, i) { return result.push(f(elt, i)); }, from, to);
  return result
};

// :: (?union<[T], RopeSequence<T>>) → RopeSequence<T>
// Create a rope representing the given array, or return the rope
// itself if a rope was given.
RopeSequence.from = function from (values) {
  if (values instanceof RopeSequence) { return values }
  return values && values.length ? new Leaf(values) : RopeSequence.empty
};

var Leaf = /*@__PURE__*/(function (RopeSequence) {
  function Leaf(values) {
    RopeSequence.call(this);
    this.values = values;
  }

  if ( RopeSequence ) { Leaf.__proto__ = RopeSequence; }
  Leaf.prototype = Object.create( RopeSequence && RopeSequence.prototype );
  Leaf.prototype.constructor = Leaf;

  var prototypeAccessors = { length: { configurable: true },depth: { configurable: true } };

  Leaf.prototype.flatten = function flatten () {
    return this.values
  };

  Leaf.prototype.sliceInner = function sliceInner (from, to) {
    if (from == 0 && to == this.length) { return this }
    return new Leaf(this.values.slice(from, to))
  };

  Leaf.prototype.getInner = function getInner (i) {
    return this.values[i]
  };

  Leaf.prototype.forEachInner = function forEachInner (f, from, to, start) {
    for (var i = from; i < to; i++)
      { if (f(this.values[i], start + i) === false) { return false } }
  };

  Leaf.prototype.forEachInvertedInner = function forEachInvertedInner (f, from, to, start) {
    for (var i = from - 1; i >= to; i--)
      { if (f(this.values[i], start + i) === false) { return false } }
  };

  Leaf.prototype.leafAppend = function leafAppend (other) {
    if (this.length + other.length <= GOOD_LEAF_SIZE)
      { return new Leaf(this.values.concat(other.flatten())) }
  };

  Leaf.prototype.leafPrepend = function leafPrepend (other) {
    if (this.length + other.length <= GOOD_LEAF_SIZE)
      { return new Leaf(other.flatten().concat(this.values)) }
  };

  prototypeAccessors.length.get = function () { return this.values.length };

  prototypeAccessors.depth.get = function () { return 0 };

  Object.defineProperties( Leaf.prototype, prototypeAccessors );

  return Leaf;
}(RopeSequence));

// :: RopeSequence
// The empty rope sequence.
RopeSequence.empty = new Leaf([]);

var Append = /*@__PURE__*/(function (RopeSequence) {
  function Append(left, right) {
    RopeSequence.call(this);
    this.left = left;
    this.right = right;
    this.length = left.length + right.length;
    this.depth = Math.max(left.depth, right.depth) + 1;
  }

  if ( RopeSequence ) { Append.__proto__ = RopeSequence; }
  Append.prototype = Object.create( RopeSequence && RopeSequence.prototype );
  Append.prototype.constructor = Append;

  Append.prototype.flatten = function flatten () {
    return this.left.flatten().concat(this.right.flatten())
  };

  Append.prototype.getInner = function getInner (i) {
    return i < this.left.length ? this.left.get(i) : this.right.get(i - this.left.length)
  };

  Append.prototype.forEachInner = function forEachInner (f, from, to, start) {
    var leftLen = this.left.length;
    if (from < leftLen &&
        this.left.forEachInner(f, from, Math.min(to, leftLen), start) === false)
      { return false }
    if (to > leftLen &&
        this.right.forEachInner(f, Math.max(from - leftLen, 0), Math.min(this.length, to) - leftLen, start + leftLen) === false)
      { return false }
  };

  Append.prototype.forEachInvertedInner = function forEachInvertedInner (f, from, to, start) {
    var leftLen = this.left.length;
    if (from > leftLen &&
        this.right.forEachInvertedInner(f, from - leftLen, Math.max(to, leftLen) - leftLen, start + leftLen) === false)
      { return false }
    if (to < leftLen &&
        this.left.forEachInvertedInner(f, Math.min(from, leftLen), to, start) === false)
      { return false }
  };

  Append.prototype.sliceInner = function sliceInner (from, to) {
    if (from == 0 && to == this.length) { return this }
    var leftLen = this.left.length;
    if (to <= leftLen) { return this.left.slice(from, to) }
    if (from >= leftLen) { return this.right.slice(from - leftLen, to - leftLen) }
    return this.left.slice(from, leftLen).append(this.right.slice(0, to - leftLen))
  };

  Append.prototype.leafAppend = function leafAppend (other) {
    var inner = this.right.leafAppend(other);
    if (inner) { return new Append(this.left, inner) }
  };

  Append.prototype.leafPrepend = function leafPrepend (other) {
    var inner = this.left.leafPrepend(other);
    if (inner) { return new Append(inner, this.right) }
  };

  Append.prototype.appendInner = function appendInner (other) {
    if (this.left.depth >= Math.max(this.right.depth, other.depth) + 1)
      { return new Append(this.left, new Append(this.right, other)) }
    return new Append(this, other)
  };

  return Append;
}(RopeSequence));// ProseMirror's history isn't simply a way to roll back to a previous
// state, because ProseMirror supports applying changes without adding
// them to the history (for example during collaboration).
//
// To this end, each 'Branch' (one for the undo history and one for
// the redo history) keeps an array of 'Items', which can optionally
// hold a step (an actual undoable change), and always hold a position
// map (which is needed to move changes below them to apply to the
// current document).
//
// An item that has both a step and a selection bookmark is the start
// of an 'event' — a group of changes that will be undone or redone at
// once. (It stores only the bookmark, since that way we don't have to
// provide a document until the selection is actually applied, which
// is useful when compressing.)
// Used to schedule history compression
var max_empty_items = 500;
var Branch = function Branch(items, eventCount) {
    this.items = items;
    this.eventCount = eventCount;
};
// Pop the latest event off the branch's history and apply it
// to a document transform.
Branch.prototype.popEvent = function popEvent (state, preserveItems) {
        var this$1$1 = this;

    if (this.eventCount == 0)
        { return null; }
    var end = this.items.length;
    for (;; end--) {
        var next = this.items.get(end - 1);
        if (next.selection) {
            --end;
            break;
        }
    }
    var remap, mapFrom;
    if (preserveItems) {
        remap = this.remapping(end, this.items.length);
        mapFrom = remap.maps.length;
    }
    var transform = state.tr;
    var selection, remaining;
    var addAfter = [], addBefore = [];
    this.items.forEach(function (item, i) {
        if (!item.step) {
            if (!remap) {
                remap = this$1$1.remapping(end, i + 1);
                mapFrom = remap.maps.length;
            }
            mapFrom--;
            addBefore.push(item);
            return;
        }
        if (remap) {
            addBefore.push(new Item(item.map));
            var step = item.step.map(remap.slice(mapFrom)), map;
            if (step && transform.maybeStep(step).doc) {
                map = transform.mapping.maps[transform.mapping.maps.length - 1];
                addAfter.push(new Item(map, undefined, undefined, addAfter.length + addBefore.length));
            }
            mapFrom--;
            if (map)
                { remap.appendMap(map, mapFrom); }
        }
        else {
            transform.maybeStep(item.step);
        }
        if (item.selection) {
            selection = remap ? item.selection.map(remap.slice(mapFrom)) : item.selection;
            remaining = new Branch(this$1$1.items.slice(0, end).append(addBefore.reverse().concat(addAfter)), this$1$1.eventCount - 1);
            return false;
        }
    }, this.items.length, 0);
    return { remaining: remaining, transform: transform, selection: selection };
};
// Create a new branch with the given transform added.
Branch.prototype.addTransform = function addTransform (transform, selection, histOptions, preserveItems) {
    var newItems = [], eventCount = this.eventCount;
    var oldItems = this.items, lastItem = !preserveItems && oldItems.length ? oldItems.get(oldItems.length - 1) : null;
    for (var i = 0; i < transform.steps.length; i++) {
        var step = transform.steps[i].invert(transform.docs[i]);
        var item = new Item(transform.mapping.maps[i], step, selection), merged = (void 0);
        if (merged = lastItem && lastItem.merge(item)) {
            item = merged;
            if (i)
                { newItems.pop(); }
            else
                { oldItems = oldItems.slice(0, oldItems.length - 1); }
        }
        newItems.push(item);
        if (selection) {
            eventCount++;
            selection = undefined;
        }
        if (!preserveItems)
            { lastItem = item; }
    }
    var overflow = eventCount - histOptions.depth;
    if (overflow > DEPTH_OVERFLOW) {
        oldItems = cutOffEvents(oldItems, overflow);
        eventCount -= overflow;
    }
    return new Branch(oldItems.append(newItems), eventCount);
};
Branch.prototype.remapping = function remapping (from, to) {
    var maps = new Mapping;
    this.items.forEach(function (item, i) {
        var mirrorPos = item.mirrorOffset != null && i - item.mirrorOffset >= from
            ? maps.maps.length - item.mirrorOffset : undefined;
        maps.appendMap(item.map, mirrorPos);
    }, from, to);
    return maps;
};
Branch.prototype.addMaps = function addMaps (array) {
    if (this.eventCount == 0)
        { return this; }
    return new Branch(this.items.append(array.map(function (map) { return new Item(map); })), this.eventCount);
};
// When the collab module receives remote changes, the history has
// to know about those, so that it can adjust the steps that were
// rebased on top of the remote changes, and include the position
// maps for the remote changes in its array of items.
Branch.prototype.rebased = function rebased (rebasedTransform, rebasedCount) {
    if (!this.eventCount)
        { return this; }
    var rebasedItems = [], start = Math.max(0, this.items.length - rebasedCount);
    var mapping = rebasedTransform.mapping;
    var newUntil = rebasedTransform.steps.length;
    var eventCount = this.eventCount;
    this.items.forEach(function (item) { if (item.selection)
        { eventCount--; } }, start);
    var iRebased = rebasedCount;
    this.items.forEach(function (item) {
        var pos = mapping.getMirror(--iRebased);
        if (pos == null)
            { return; }
        newUntil = Math.min(newUntil, pos);
        var map = mapping.maps[pos];
        if (item.step) {
            var step = rebasedTransform.steps[pos].invert(rebasedTransform.docs[pos]);
            var selection = item.selection && item.selection.map(mapping.slice(iRebased + 1, pos));
            if (selection)
                { eventCount++; }
            rebasedItems.push(new Item(map, step, selection));
        }
        else {
            rebasedItems.push(new Item(map));
        }
    }, start);
    var newMaps = [];
    for (var i = rebasedCount; i < newUntil; i++)
        { newMaps.push(new Item(mapping.maps[i])); }
    var items = this.items.slice(0, start).append(newMaps).append(rebasedItems);
    var branch = new Branch(items, eventCount);
    if (branch.emptyItemCount() > max_empty_items)
        { branch = branch.compress(this.items.length - rebasedItems.length); }
    return branch;
};
Branch.prototype.emptyItemCount = function emptyItemCount () {
    var count = 0;
    this.items.forEach(function (item) { if (!item.step)
        { count++; } });
    return count;
};
// Compressing a branch means rewriting it to push the air (map-only
// items) out. During collaboration, these naturally accumulate
// because each remote change adds one. The `upto` argument is used
// to ensure that only the items below a given level are compressed,
// because `rebased` relies on a clean, untouched set of items in
// order to associate old items with rebased steps.
Branch.prototype.compress = function compress (upto) {
        if ( upto === void 0 ) upto = this.items.length;

    var remap = this.remapping(0, upto), mapFrom = remap.maps.length;
    var items = [], events = 0;
    this.items.forEach(function (item, i) {
        if (i >= upto) {
            items.push(item);
            if (item.selection)
                { events++; }
        }
        else if (item.step) {
            var step = item.step.map(remap.slice(mapFrom)), map = step && step.getMap();
            mapFrom--;
            if (map)
                { remap.appendMap(map, mapFrom); }
            if (step) {
                var selection = item.selection && item.selection.map(remap.slice(mapFrom));
                if (selection)
                    { events++; }
                var newItem = new Item(map.invert(), step, selection), merged, last = items.length - 1;
                if (merged = items.length && items[last].merge(newItem))
                    { items[last] = merged; }
                else
                    { items.push(newItem); }
            }
        }
        else if (item.map) {
            mapFrom--;
        }
    }, this.items.length, 0);
    return new Branch(RopeSequence.from(items.reverse()), events);
};
Branch.empty = new Branch(RopeSequence.empty, 0);
function cutOffEvents(items, n) {
    var cutPoint;
    items.forEach(function (item, i) {
        if (item.selection && (n-- == 0)) {
            cutPoint = i;
            return false;
        }
    });
    return items.slice(cutPoint);
}
var Item = function Item(
// The (forward) step map for this item.
map, 
// The inverted step
step, 
// If this is non-null, this item is the start of a group, and
// this selection is the starting selection for the group (the one
// that was active before the first step was applied)
selection, 
// If this item is the inverse of a previous mapping on the stack,
// this points at the inverse's offset
mirrorOffset) {
    this.map = map;
    this.step = step;
    this.selection = selection;
    this.mirrorOffset = mirrorOffset;
};
Item.prototype.merge = function merge (other) {
    if (this.step && other.step && !other.selection) {
        var step = other.step.merge(this.step);
        if (step)
            { return new Item(step.getMap().invert(), step, this.selection); }
    }
};
// The value of the state field that tracks undo/redo history for that
// state. Will be stored in the plugin state when the history plugin
// is active.
var HistoryState = function HistoryState(done, undone, prevRanges, prevTime, prevComposition) {
    this.done = done;
    this.undone = undone;
    this.prevRanges = prevRanges;
    this.prevTime = prevTime;
    this.prevComposition = prevComposition;
};
var DEPTH_OVERFLOW = 20;
// Record a transformation in undo history.
function applyTransaction(history, state, tr, options) {
    var historyTr = tr.getMeta(historyKey), rebased;
    if (historyTr)
        { return historyTr.historyState; }
    if (tr.getMeta(closeHistoryKey))
        { history = new HistoryState(history.done, history.undone, null, 0, -1); }
    var appended = tr.getMeta("appendedTransaction");
    if (tr.steps.length == 0) {
        return history;
    }
    else if (appended && appended.getMeta(historyKey)) {
        if (appended.getMeta(historyKey).redo)
            { return new HistoryState(history.done.addTransform(tr, undefined, options, mustPreserveItems(state)), history.undone, rangesFor(tr.mapping.maps), history.prevTime, history.prevComposition); }
        else
            { return new HistoryState(history.done, history.undone.addTransform(tr, undefined, options, mustPreserveItems(state)), null, history.prevTime, history.prevComposition); }
    }
    else if (tr.getMeta("addToHistory") !== false && !(appended && appended.getMeta("addToHistory") === false)) {
        // Group transforms that occur in quick succession into one event.
        var composition = tr.getMeta("composition");
        var newGroup = history.prevTime == 0 ||
            (!appended && history.prevComposition != composition &&
                (history.prevTime < (tr.time || 0) - options.newGroupDelay || !isAdjacentTo(tr, history.prevRanges)));
        var prevRanges = appended ? mapRanges(history.prevRanges, tr.mapping) : rangesFor(tr.mapping.maps);
        return new HistoryState(history.done.addTransform(tr, newGroup ? state.selection.getBookmark() : undefined, options, mustPreserveItems(state)), Branch.empty, prevRanges, tr.time, composition == null ? history.prevComposition : composition);
    }
    else if (rebased = tr.getMeta("rebased")) {
        // Used by the collab module to tell the history that some of its
        // content has been rebased.
        return new HistoryState(history.done.rebased(tr, rebased), history.undone.rebased(tr, rebased), mapRanges(history.prevRanges, tr.mapping), history.prevTime, history.prevComposition);
    }
    else {
        return new HistoryState(history.done.addMaps(tr.mapping.maps), history.undone.addMaps(tr.mapping.maps), mapRanges(history.prevRanges, tr.mapping), history.prevTime, history.prevComposition);
    }
}
function isAdjacentTo(transform, prevRanges) {
    if (!prevRanges)
        { return false; }
    if (!transform.docChanged)
        { return true; }
    var adjacent = false;
    transform.mapping.maps[0].forEach(function (start, end) {
        for (var i = 0; i < prevRanges.length; i += 2)
            { if (start <= prevRanges[i + 1] && end >= prevRanges[i])
                { adjacent = true; } }
    });
    return adjacent;
}
function rangesFor(maps) {
    var result = [];
    for (var i = maps.length - 1; i >= 0 && result.length == 0; i--)
        { maps[i].forEach(function (_from, _to, from, to) { return result.push(from, to); }); }
    return result;
}
function mapRanges(ranges, mapping) {
    if (!ranges)
        { return null; }
    var result = [];
    for (var i = 0; i < ranges.length; i += 2) {
        var from = mapping.map(ranges[i], 1), to = mapping.map(ranges[i + 1], -1);
        if (from <= to)
            { result.push(from, to); }
    }
    return result;
}
// Apply the latest event from one branch to the document and shift the event
// onto the other branch.
function histTransaction(history, state, redo) {
    var preserveItems = mustPreserveItems(state);
    var histOptions = historyKey.get(state).spec.config;
    var pop = (redo ? history.undone : history.done).popEvent(state, preserveItems);
    if (!pop)
        { return null; }
    var selection = pop.selection.resolve(pop.transform.doc);
    var added = (redo ? history.done : history.undone).addTransform(pop.transform, state.selection.getBookmark(), histOptions, preserveItems);
    var newHist = new HistoryState(redo ? added : pop.remaining, redo ? pop.remaining : added, null, 0, -1);
    return pop.transform.setSelection(selection).setMeta(historyKey, { redo: redo, historyState: newHist });
}
var cachedPreserveItems = false, cachedPreserveItemsPlugins = null;
// Check whether any plugin in the given state has a
// `historyPreserveItems` property in its spec, in which case we must
// preserve steps exactly as they came in, so that they can be
// rebased.
function mustPreserveItems(state) {
    var plugins = state.plugins;
    if (cachedPreserveItemsPlugins != plugins) {
        cachedPreserveItems = false;
        cachedPreserveItemsPlugins = plugins;
        for (var i = 0; i < plugins.length; i++)
            { if (plugins[i].spec.historyPreserveItems) {
                cachedPreserveItems = true;
                break;
            } }
    }
    return cachedPreserveItems;
}
/**
Set a flag on the given transaction that will prevent further steps
from being appended to an existing history event (so that they
require a separate undo command to undo).
*/
function closeHistory(tr) {
    return tr.setMeta(closeHistoryKey, true);
}
var historyKey = new PluginKey("history");
var closeHistoryKey = new PluginKey("closeHistory");
/**
Returns a plugin that enables the undo history for an editor. The
plugin will track undo and redo stacks, which can be used with the
[`undo`](https://prosemirror.net/docs/ref/#history.undo) and [`redo`](https://prosemirror.net/docs/ref/#history.redo) commands.

You can set an `"addToHistory"` [metadata
property](https://prosemirror.net/docs/ref/#state.Transaction.setMeta) of `false` on a transaction
to prevent it from being rolled back by undo.
*/
function history(config) {
    if ( config === void 0 ) config = {};

    config = { depth: config.depth || 100,
        newGroupDelay: config.newGroupDelay || 500 };
    return new Plugin({
        key: historyKey,
        state: {
            init: function init() {
                return new HistoryState(Branch.empty, Branch.empty, null, 0, -1);
            },
            apply: function apply(tr, hist, state) {
                return applyTransaction(hist, state, tr, config);
            }
        },
        config: config,
        props: {
            handleDOMEvents: {
                beforeinput: function beforeinput(view, e) {
                    var inputType = e.inputType;
                    var command = inputType == "historyUndo" ? undo : inputType == "historyRedo" ? redo : null;
                    if (!command)
                        { return false; }
                    e.preventDefault();
                    return command(view.state, view.dispatch);
                }
            }
        }
    });
}
function buildCommand(redo, scroll) {
    return function (state, dispatch) {
        var hist = historyKey.getState(state);
        if (!hist || (redo ? hist.undone : hist.done).eventCount == 0)
            { return false; }
        if (dispatch) {
            var tr = histTransaction(hist, state, redo);
            if (tr)
                { dispatch(scroll ? tr.scrollIntoView() : tr); }
        }
        return true;
    };
}
/**
A command function that undoes the last change, if any.
*/
var undo = buildCommand(false, true);
/**
A command function that redoes the last undone change, if any.
*/
var redo = buildCommand(true, true);
/**
A command function that undoes the last change. Don't scroll the
selection into view.
*/
var undoNoScroll = buildCommand(false, false);
/**
A command function that redoes the last undone change. Don't
scroll the selection into view.
*/
var redoNoScroll = buildCommand(true, false);
/**
The amount of undoable events available in a given state.
*/
function undoDepth(state) {
    var hist = historyKey.getState(state);
    return hist ? hist.done.eventCount : 0;
}
/**
The amount of redoable events available in a given editor state.
*/
function redoDepth(state) {
    var hist = historyKey.getState(state);
    return hist ? hist.undone.eventCount : 0;
}var history$1=/*#__PURE__*/Object.freeze({__proto__:null,closeHistory:closeHistory,history:history,redo:redo,redoDepth:redoDepth,redoNoScroll:redoNoScroll,undo:undo,undoDepth:undoDepth,undoNoScroll:undoNoScroll});var base$2 = {
  8: "Backspace",
  9: "Tab",
  10: "Enter",
  12: "NumLock",
  13: "Enter",
  16: "Shift",
  17: "Control",
  18: "Alt",
  20: "CapsLock",
  27: "Escape",
  32: " ",
  33: "PageUp",
  34: "PageDown",
  35: "End",
  36: "Home",
  37: "ArrowLeft",
  38: "ArrowUp",
  39: "ArrowRight",
  40: "ArrowDown",
  44: "PrintScreen",
  45: "Insert",
  46: "Delete",
  59: ";",
  61: "=",
  91: "Meta",
  92: "Meta",
  106: "*",
  107: "+",
  108: ",",
  109: "-",
  110: ".",
  111: "/",
  144: "NumLock",
  145: "ScrollLock",
  160: "Shift",
  161: "Shift",
  162: "Control",
  163: "Control",
  164: "Alt",
  165: "Alt",
  173: "-",
  186: ";",
  187: "=",
  188: ",",
  189: "-",
  190: ".",
  191: "/",
  192: "`",
  219: "[",
  220: "\\",
  221: "]",
  222: "'"
};

var shift = {
  48: ")",
  49: "!",
  50: "@",
  51: "#",
  52: "$",
  53: "%",
  54: "^",
  55: "&",
  56: "*",
  57: "(",
  59: ":",
  61: "+",
  173: "_",
  186: ":",
  187: "+",
  188: "<",
  189: "_",
  190: ">",
  191: "?",
  192: "~",
  219: "{",
  220: "|",
  221: "}",
  222: "\""
};

var mac$2 = typeof navigator != "undefined" && /Mac/.test(navigator.platform);
var ie$1 = typeof navigator != "undefined" && /MSIE \d|Trident\/(?:[7-9]|\d{2,})\..*rv:(\d+)/.exec(navigator.userAgent);

// Fill in the digit keys
for (var i$4 = 0; i$4 < 10; i$4++) { base$2[48 + i$4] = base$2[96 + i$4] = String(i$4); }

// The function keys
for (var i$4 = 1; i$4 <= 24; i$4++) { base$2[i$4 + 111] = "F" + i$4; }

// And the alphabetic keys
for (var i$4 = 65; i$4 <= 90; i$4++) {
  base$2[i$4] = String.fromCharCode(i$4 + 32);
  shift[i$4] = String.fromCharCode(i$4);
}

// For each code that doesn't have a shift-equivalent, copy the base name
for (var code$3 in base$2) { if (!shift.hasOwnProperty(code$3)) { shift[code$3] = base$2[code$3]; } }

function keyName(event) {
  // On macOS, keys held with Shift and Cmd don't reflect the effect of Shift in `.key`.
  // On IE, shift effect is never included in `.key`.
  var ignoreKey = mac$2 && event.metaKey && event.shiftKey && !event.ctrlKey && !event.altKey ||
      ie$1 && event.shiftKey && event.key && event.key.length == 1 ||
      event.key == "Unidentified";
  var name = (!ignoreKey && event.key) ||
    (event.shiftKey ? shift : base$2)[event.keyCode] ||
    event.key || "Unidentified";
  // Edge sometimes produces wrong names (Issue #3)
  if (name == "Esc") { name = "Escape"; }
  if (name == "Del") { name = "Delete"; }
  // https://developer.microsoft.com/en-us/microsoft-edge/platform/issues/8860571/
  if (name == "Left") { name = "ArrowLeft"; }
  if (name == "Up") { name = "ArrowUp"; }
  if (name == "Right") { name = "ArrowRight"; }
  if (name == "Down") { name = "ArrowDown"; }
  return name
}var mac$1 = typeof navigator != "undefined" ? /Mac|iP(hone|[oa]d)/.test(navigator.platform) : false;
function normalizeKeyName$1(name) {
    var parts = name.split(/-(?!$)/), result = parts[parts.length - 1];
    if (result == "Space")
        { result = " "; }
    var alt, ctrl, shift, meta;
    for (var i = 0; i < parts.length - 1; i++) {
        var mod = parts[i];
        if (/^(cmd|meta|m)$/i.test(mod))
            { meta = true; }
        else if (/^a(lt)?$/i.test(mod))
            { alt = true; }
        else if (/^(c|ctrl|control)$/i.test(mod))
            { ctrl = true; }
        else if (/^s(hift)?$/i.test(mod))
            { shift = true; }
        else if (/^mod$/i.test(mod)) {
            if (mac$1)
                { meta = true; }
            else
                { ctrl = true; }
        }
        else
            { throw new Error("Unrecognized modifier name: " + mod); }
    }
    if (alt)
        { result = "Alt-" + result; }
    if (ctrl)
        { result = "Ctrl-" + result; }
    if (meta)
        { result = "Meta-" + result; }
    if (shift)
        { result = "Shift-" + result; }
    return result;
}
function normalize$2(map) {
    var copy = Object.create(null);
    for (var prop in map)
        { copy[normalizeKeyName$1(prop)] = map[prop]; }
    return copy;
}
function modifiers$1(name, event, shift) {
    if ( shift === void 0 ) shift = true;

    if (event.altKey)
        { name = "Alt-" + name; }
    if (event.ctrlKey)
        { name = "Ctrl-" + name; }
    if (event.metaKey)
        { name = "Meta-" + name; }
    if (shift && event.shiftKey)
        { name = "Shift-" + name; }
    return name;
}
/**
Create a keymap plugin for the given set of bindings.

Bindings should map key names to [command](https://prosemirror.net/docs/ref/#commands)-style
functions, which will be called with `(EditorState, dispatch,
EditorView)` arguments, and should return true when they've handled
the key. Note that the view argument isn't part of the command
protocol, but can be used as an escape hatch if a binding needs to
directly interact with the UI.

Key names may be strings like `"Shift-Ctrl-Enter"`—a key
identifier prefixed with zero or more modifiers. Key identifiers
are based on the strings that can appear in
[`KeyEvent.key`](https:developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key).
Use lowercase letters to refer to letter keys (or uppercase letters
if you want shift to be held). You may use `"Space"` as an alias
for the `" "` name.

Modifiers can be given in any order. `Shift-` (or `s-`), `Alt-` (or
`a-`), `Ctrl-` (or `c-` or `Control-`) and `Cmd-` (or `m-` or
`Meta-`) are recognized. For characters that are created by holding
shift, the `Shift-` prefix is implied, and should not be added
explicitly.

You can use `Mod-` as a shorthand for `Cmd-` on Mac and `Ctrl-` on
other platforms.

You can add multiple keymap plugins to an editor. The order in
which they appear determines their precedence (the ones early in
the array get to dispatch first).
*/
function keymap$5(bindings) {
    return new Plugin({ props: { handleKeyDown: keydownHandler(bindings) } });
}
/**
Given a set of bindings (using the same format as
[`keymap`](https://prosemirror.net/docs/ref/#keymap.keymap)), return a [keydown
handler](https://prosemirror.net/docs/ref/#view.EditorProps.handleKeyDown) that handles them.
*/
function keydownHandler(bindings) {
    var map = normalize$2(bindings);
    return function (view, event) {
        var name = keyName(event), baseName, direct = map[modifiers$1(name, event)];
        if (direct && direct(view.state, view.dispatch, view))
            { return true; }
        // A character key
        if (name.length == 1 && name != " ") {
            if (event.shiftKey) {
                // In case the name was already modified by shift, try looking
                // it up without its shift modifier
                var noShift = map[modifiers$1(name, event, false)];
                if (noShift && noShift(view.state, view.dispatch, view))
                    { return true; }
            }
            if ((event.shiftKey || event.altKey || event.metaKey || name.charCodeAt(0) > 127) &&
                (baseName = base$2[event.keyCode]) && baseName != name) {
                // Try falling back to the keyCode when there's a modifier
                // active or the character produced isn't ASCII, and our table
                // produces a different name from the the keyCode. See #668,
                // #1060
                var fromCode = map[modifiers$1(baseName, event)];
                if (fromCode && fromCode(view.state, view.dispatch, view))
                    { return true; }
            }
        }
        return false;
    };
}var keymap$6=/*#__PURE__*/Object.freeze({__proto__:null,keydownHandler:keydownHandler,keymap:keymap$5});function crelt() {
  var arguments$1 = arguments;

  var elt = arguments[0];
  if (typeof elt == "string") { elt = document.createElement(elt); }
  var i = 1, next = arguments[1];
  if (next && typeof next == "object" && next.nodeType == null && !Array.isArray(next)) {
    for (var name in next) { if (Object.prototype.hasOwnProperty.call(next, name)) {
      var value = next[name];
      if (typeof value == "string") { elt.setAttribute(name, value); }
      else if (value != null) { elt[name] = value; }
    } }
    i++;
  }
  for (; i < arguments.length; i++) { add(elt, arguments$1[i]); }
  return elt
}

function add(elt, child) {
  if (typeof child == "string") {
    elt.appendChild(document.createTextNode(child));
  } else if (child == null) ; else if (child.nodeType != null) {
    elt.appendChild(child);
  } else if (Array.isArray(child)) {
    for (var i = 0; i < child.length; i++) { add(elt, child[i]); }
  } else {
    throw new RangeError("Unsupported child node: " + child)
  }
}var SVG$1 = "http://www.w3.org/2000/svg";
var XLINK$1 = "http://www.w3.org/1999/xlink";
var prefix$2$1 = "ProseMirror-icon";
function hashPath(path) {
    var hash = 0;
    for (var i = 0; i < path.length; i++)
        { hash = (((hash << 5) - hash) + path.charCodeAt(i)) | 0; }
    return hash;
}
function getIcon$1(root, icon) {
    var doc = (root.nodeType == 9 ? root : root.ownerDocument) || document;
    var node = doc.createElement("div");
    node.className = prefix$2$1;
    if (icon.path) {
        var path = icon.path;
        var width = icon.width;
        var height = icon.height;
        var name = "pm-icon-" + hashPath(path).toString(16);
        if (!doc.getElementById(name))
            { buildSVG$1(root, name, icon); }
        var svg = node.appendChild(doc.createElementNS(SVG$1, "svg"));
        svg.style.width = (width / height) + "em";
        var use = svg.appendChild(doc.createElementNS(SVG$1, "use"));
        use.setAttributeNS(XLINK$1, "href", /([^#]*)/.exec(doc.location.toString())[1] + "#" + name);
    }
    else if (icon.dom) {
        node.appendChild(icon.dom.cloneNode(true));
    }
    else {
        var text = icon.text;
        var css = icon.css;
        node.appendChild(doc.createElement("span")).textContent = text || '';
        if (css)
            { node.firstChild.style.cssText = css; }
    }
    return node;
}
function buildSVG$1(root, name, data) {
    var ref = root.nodeType == 9 ? [root, root.body] : [root.ownerDocument || document, root];
    var doc = ref[0];
    var top = ref[1];
    var collection = doc.getElementById(prefix$2$1 + "-collection");
    if (!collection) {
        collection = doc.createElementNS(SVG$1, "svg");
        collection.id = prefix$2$1 + "-collection";
        collection.style.display = "none";
        top.insertBefore(collection, top.firstChild);
    }
    var sym = doc.createElementNS(SVG$1, "symbol");
    sym.id = name;
    sym.setAttribute("viewBox", "0 0 " + data.width + " " + data.height);
    var path = sym.appendChild(doc.createElementNS(SVG$1, "path"));
    path.setAttribute("d", data.path);
    collection.appendChild(sym);
}

var prefix$1$1 = "ProseMirror-menu";
/**
An icon or label that, when clicked, executes a command.
*/
var MenuItem$1 = function MenuItem(
/**
The spec used to create this item.
*/
spec) {
    this.spec = spec;
};
/**
Renders the icon according to its [display
spec](https://prosemirror.net/docs/ref/#menu.MenuItemSpec.display), and adds an event handler which
executes the command when the representation is clicked.
*/
MenuItem$1.prototype.render = function render (view) {
    var spec = this.spec;
    var dom = spec.render ? spec.render(view)
        : spec.icon ? getIcon$1(view.root, spec.icon)
            : spec.label ? crelt("div", null, translate$2(view, spec.label))
                : null;
    if (!dom)
        { throw new RangeError("MenuItem without icon or label property"); }
    if (spec.title) {
        var title = (typeof spec.title === "function" ? spec.title(view.state) : spec.title);
        dom.setAttribute("title", translate$2(view, title));
    }
    if (spec.class)
        { dom.classList.add(spec.class); }
    if (spec.css)
        { dom.style.cssText += spec.css; }
    dom.addEventListener("mousedown", function (e) {
        e.preventDefault();
        if (!dom.classList.contains(prefix$1$1 + "-disabled"))
            { spec.run(view.state, view.dispatch, view, e); }
    });
    function update(state) {
        if (spec.select) {
            var selected = spec.select(state);
            dom.style.display = selected ? "" : "none";
            if (!selected)
                { return false; }
        }
        var enabled = true;
        if (spec.enable) {
            enabled = spec.enable(state) || false;
            setClass$1(dom, prefix$1$1 + "-disabled", !enabled);
        }
        if (spec.active) {
            var active = enabled && spec.active(state) || false;
            setClass$1(dom, prefix$1$1 + "-active", active);
        }
        return true;
    }
    return { dom: dom, update: update };
};
function translate$2(view, text) {
    return view._props.translate ? view._props.translate(text) : text;
}
var lastMenuEvent$1 = { time: 0, node: null };
function markMenuEvent$1(e) {
    lastMenuEvent$1.time = Date.now();
    lastMenuEvent$1.node = e.target;
}
function isMenuEvent$1(wrapper) {
    return Date.now() - 100 < lastMenuEvent$1.time &&
        lastMenuEvent$1.node && wrapper.contains(lastMenuEvent$1.node);
}
/**
A drop-down menu, displayed as a label with a downwards-pointing
triangle to the right of it.
*/
var Dropdown$1 = function Dropdown(content, 
/**
@internal
*/
options) {
    if ( options === void 0 ) options = {};

    this.options = options;
    this.options = options || {};
    this.content = Array.isArray(content) ? content : [content];
};
/**
Render the dropdown menu and sub-items.
*/
Dropdown$1.prototype.render = function render (view) {
        var this$1$1 = this;

    var content = renderDropdownItems(this.content, view);
    var win = view.dom.ownerDocument.defaultView || window;
    var label = crelt("div", { class: prefix$1$1 + "-dropdown " + (this.options.class || ""),
        style: this.options.css }, translate$2(view, this.options.label || ""));
    if (this.options.title)
        { label.setAttribute("title", translate$2(view, this.options.title)); }
    var wrap = crelt("div", { class: prefix$1$1 + "-dropdown-wrap" }, label);
    var open = null;
    var listeningOnClose = null;
    var close = function () {
        if (open && open.close()) {
            open = null;
            win.removeEventListener("mousedown", listeningOnClose);
        }
    };
    label.addEventListener("mousedown", function (e) {
        e.preventDefault();
        markMenuEvent$1(e);
        if (open) {
            close();
        }
        else {
            open = this$1$1.expand(wrap, content.dom);
            win.addEventListener("mousedown", listeningOnClose = function () {
                if (!isMenuEvent$1(wrap))
                    { close(); }
            });
        }
    });
    function update(state) {
        var inner = content.update(state);
        wrap.style.display = inner ? "" : "none";
        return inner;
    }
    return { dom: wrap, update: update };
};
/**
@internal
*/
Dropdown$1.prototype.expand = function expand (dom, items) {
    var menuDOM = crelt("div", { class: prefix$1$1 + "-dropdown-menu " + (this.options.class || "") }, items);
    var done = false;
    function close() {
        if (done)
            { return false; }
        done = true;
        dom.removeChild(menuDOM);
        return true;
    }
    dom.appendChild(menuDOM);
    return { close: close, node: menuDOM };
};
function renderDropdownItems(items, view) {
    var rendered = [], updates = [];
    for (var i = 0; i < items.length; i++) {
        var ref = items[i].render(view);
        var dom = ref.dom;
        var update = ref.update;
        rendered.push(crelt("div", { class: prefix$1$1 + "-dropdown-item" }, dom));
        updates.push(update);
    }
    return { dom: rendered, update: combineUpdates(updates, rendered) };
}
function combineUpdates(updates, nodes) {
    return function (state) {
        var something = false;
        for (var i = 0; i < updates.length; i++) {
            var up = updates[i](state);
            nodes[i].style.display = up ? "" : "none";
            if (up)
                { something = true; }
        }
        return something;
    };
}
/**
Represents a submenu wrapping a group of elements that start
hidden and expand to the right when hovered over or tapped.
*/
var DropdownSubmenu$1 = function DropdownSubmenu(content, 
/**
@internal
*/
options) {
    if ( options === void 0 ) options = {};

    this.options = options;
    this.content = Array.isArray(content) ? content : [content];
};
/**
Renders the submenu.
*/
DropdownSubmenu$1.prototype.render = function render (view) {
    var items = renderDropdownItems(this.content, view);
    var win = view.dom.ownerDocument.defaultView || window;
    var label = crelt("div", { class: prefix$1$1 + "-submenu-label" }, translate$2(view, this.options.label || ""));
    var wrap = crelt("div", { class: prefix$1$1 + "-submenu-wrap" }, label, crelt("div", { class: prefix$1$1 + "-submenu" }, items.dom));
    var listeningOnClose = null;
    label.addEventListener("mousedown", function (e) {
        e.preventDefault();
        markMenuEvent$1(e);
        setClass$1(wrap, prefix$1$1 + "-submenu-wrap-active", false);
        if (!listeningOnClose)
            { win.addEventListener("mousedown", listeningOnClose = function () {
                if (!isMenuEvent$1(wrap)) {
                    wrap.classList.remove(prefix$1$1 + "-submenu-wrap-active");
                    win.removeEventListener("mousedown", listeningOnClose);
                    listeningOnClose = null;
                }
            }); }
    });
    function update(state) {
        var inner = items.update(state);
        wrap.style.display = inner ? "" : "none";
        return inner;
    }
    return { dom: wrap, update: update };
};
/**
Render the given, possibly nested, array of menu elements into a
document fragment, placing separators between them (and ensuring no
superfluous separators appear when some of the groups turn out to
be empty).
*/
function renderGrouped(view, content) {
    var result = document.createDocumentFragment();
    var updates = [], separators = [];
    for (var i = 0; i < content.length; i++) {
        var items = content[i], localUpdates = [], localNodes = [];
        for (var j = 0; j < items.length; j++) {
            var ref = items[j].render(view);
            var dom = ref.dom;
            var update$1 = ref.update;
            var span = crelt("span", { class: prefix$1$1 + "item" }, dom);
            result.appendChild(span);
            localNodes.push(span);
            localUpdates.push(update$1);
        }
        if (localUpdates.length) {
            updates.push(combineUpdates(localUpdates, localNodes));
            if (i < content.length - 1)
                { separators.push(result.appendChild(separator())); }
        }
    }
    function update(state) {
        var something = false, needSep = false;
        for (var i = 0; i < updates.length; i++) {
            var hasContent = updates[i](state);
            if (i)
                { separators[i - 1].style.display = needSep && hasContent ? "" : "none"; }
            needSep = hasContent;
            if (hasContent)
                { something = true; }
        }
        return something;
    }
    return { dom: result, update: update };
}
function separator() {
    return crelt("span", { class: prefix$1$1 + "separator" });
}
/**
A set of basic editor-related icons. Contains the properties
`join`, `lift`, `selectParentNode`, `undo`, `redo`, `strong`, `em`,
`code`, `link`, `bulletList`, `orderedList`, and `blockquote`, each
holding an object that can be used as the `icon` option to
`MenuItem`.
*/
var icons$1 = {
    join: {
        width: 800, height: 900,
        path: "M0 75h800v125h-800z M0 825h800v-125h-800z M250 400h100v-100h100v100h100v100h-100v100h-100v-100h-100z"
    },
    lift: {
        width: 1024, height: 1024,
        path: "M219 310v329q0 7-5 12t-12 5q-8 0-13-5l-164-164q-5-5-5-13t5-13l164-164q5-5 13-5 7 0 12 5t5 12zM1024 749v109q0 7-5 12t-12 5h-987q-7 0-12-5t-5-12v-109q0-7 5-12t12-5h987q7 0 12 5t5 12zM1024 530v109q0 7-5 12t-12 5h-621q-7 0-12-5t-5-12v-109q0-7 5-12t12-5h621q7 0 12 5t5 12zM1024 310v109q0 7-5 12t-12 5h-621q-7 0-12-5t-5-12v-109q0-7 5-12t12-5h621q7 0 12 5t5 12zM1024 91v109q0 7-5 12t-12 5h-987q-7 0-12-5t-5-12v-109q0-7 5-12t12-5h987q7 0 12 5t5 12z"
    },
    selectParentNode: { text: "\u2b1a", css: "font-weight: bold" },
    undo: {
        width: 1024, height: 1024,
        path: "M761 1024c113-206 132-520-313-509v253l-384-384 384-384v248c534-13 594 472 313 775z"
    },
    redo: {
        width: 1024, height: 1024,
        path: "M576 248v-248l384 384-384 384v-253c-446-10-427 303-313 509-280-303-221-789 313-775z"
    },
    strong: {
        width: 805, height: 1024,
        path: "M317 869q42 18 80 18 214 0 214-191 0-65-23-102-15-25-35-42t-38-26-46-14-48-6-54-1q-41 0-57 5 0 30-0 90t-0 90q0 4-0 38t-0 55 2 47 6 38zM309 442q24 4 62 4 46 0 81-7t62-25 42-51 14-81q0-40-16-70t-45-46-61-24-70-8q-28 0-74 7 0 28 2 86t2 86q0 15-0 45t-0 45q0 26 0 39zM0 950l1-53q8-2 48-9t60-15q4-6 7-15t4-19 3-18 1-21 0-19v-37q0-561-12-585-2-4-12-8t-25-6-28-4-27-2-17-1l-2-47q56-1 194-6t213-5q13 0 39 0t38 0q40 0 78 7t73 24 61 40 42 59 16 78q0 29-9 54t-22 41-36 32-41 25-48 22q88 20 146 76t58 141q0 57-20 102t-53 74-78 48-93 27-100 8q-25 0-75-1t-75-1q-60 0-175 6t-132 6z"
    },
    em: {
        width: 585, height: 1024,
        path: "M0 949l9-48q3-1 46-12t63-21q16-20 23-57 0-4 35-165t65-310 29-169v-14q-13-7-31-10t-39-4-33-3l10-58q18 1 68 3t85 4 68 1q27 0 56-1t69-4 56-3q-2 22-10 50-17 5-58 16t-62 19q-4 10-8 24t-5 22-4 26-3 24q-15 84-50 239t-44 203q-1 5-7 33t-11 51-9 47-3 32l0 10q9 2 105 17-1 25-9 56-6 0-18 0t-18 0q-16 0-49-5t-49-5q-78-1-117-1-29 0-81 5t-69 6z"
    },
    code: {
        width: 896, height: 1024,
        path: "M608 192l-96 96 224 224-224 224 96 96 288-320-288-320zM288 192l-288 320 288 320 96-96-224-224 224-224-96-96z"
    },
    link: {
        width: 951, height: 1024,
        path: "M832 694q0-22-16-38l-118-118q-16-16-38-16-24 0-41 18 1 1 10 10t12 12 8 10 7 14 2 15q0 22-16 38t-38 16q-8 0-15-2t-14-7-10-8-12-12-10-10q-18 17-18 41 0 22 16 38l117 118q15 15 38 15 22 0 38-14l84-83q16-16 16-38zM430 292q0-22-16-38l-117-118q-16-16-38-16-22 0-38 15l-84 83q-16 16-16 38 0 22 16 38l118 118q15 15 38 15 24 0 41-17-1-1-10-10t-12-12-8-10-7-14-2-15q0-22 16-38t38-16q8 0 15 2t14 7 10 8 12 12 10 10q18-17 18-41zM941 694q0 68-48 116l-84 83q-47 47-116 47-69 0-116-48l-117-118q-47-47-47-116 0-70 50-119l-50-50q-49 50-118 50-68 0-116-48l-118-118q-48-48-48-116t48-116l84-83q47-47 116-47 69 0 116 48l117 118q47 47 47 116 0 70-50 119l50 50q49-50 118-50 68 0 116 48l118 118q48 48 48 116z"
    },
    bulletList: {
        width: 768, height: 896,
        path: "M0 512h128v-128h-128v128zM0 256h128v-128h-128v128zM0 768h128v-128h-128v128zM256 512h512v-128h-512v128zM256 256h512v-128h-512v128zM256 768h512v-128h-512v128z"
    },
    orderedList: {
        width: 768, height: 896,
        path: "M320 512h448v-128h-448v128zM320 768h448v-128h-448v128zM320 128v128h448v-128h-448zM79 384h78v-256h-36l-85 23v50l43-2v185zM189 590c0-36-12-78-96-78-33 0-64 6-83 16l1 66c21-10 42-15 67-15s32 11 32 28c0 26-30 58-110 112v50h192v-67l-91 2c49-30 87-66 87-113l1-1z"
    },
    blockquote: {
        width: 640, height: 896,
        path: "M0 448v256h256v-256h-128c0 0 0-128 128-128v-128c0 0-256 0-256 256zM640 320v-128c0 0-256 0-256 256v256h256v-256h-128c0 0 0-128 128-128z"
    }
};
/**
Menu item for the `joinUp` command.
*/
var joinUpItem$1 = new MenuItem$1({
    title: "Join with above block",
    run: joinUp,
    select: function (state) { return joinUp(state); },
    icon: icons$1.join
});
/**
Menu item for the `lift` command.
*/
var liftItem$1 = new MenuItem$1({
    title: "Lift out of enclosing block",
    run: lift$1,
    select: function (state) { return lift$1(state); },
    icon: icons$1.lift
});
/**
Menu item for the `selectParentNode` command.
*/
var selectParentNodeItem$1 = new MenuItem$1({
    title: "Select parent node",
    run: selectParentNode,
    select: function (state) { return selectParentNode(state); },
    icon: icons$1.selectParentNode
});
/**
Menu item for the `undo` command.
*/
var undoItem$1 = new MenuItem$1({
    title: "Undo last change",
    run: undo,
    enable: function (state) { return undo(state); },
    icon: icons$1.undo
});
/**
Menu item for the `redo` command.
*/
var redoItem$1 = new MenuItem$1({
    title: "Redo last undone change",
    run: redo,
    enable: function (state) { return redo(state); },
    icon: icons$1.redo
});
/**
Build a menu item for wrapping the selection in a given node type.
Adds `run` and `select` properties to the ones present in
`options`. `options.attrs` may be an object that provides
attributes for the wrapping node.
*/
function wrapItem$1(nodeType, options) {
    var passedOptions = {
        run: function run(state, dispatch) {
            return wrapIn(nodeType, options.attrs)(state, dispatch);
        },
        select: function select(state) {
            return wrapIn(nodeType, options.attrs)(state);
        }
    };
    for (var prop in options)
        { passedOptions[prop] = options[prop]; }
    return new MenuItem$1(passedOptions);
}
/**
Build a menu item for changing the type of the textblock around the
selection to the given type. Provides `run`, `active`, and `select`
properties. Others must be given in `options`. `options.attrs` may
be an object to provide the attributes for the textblock node.
*/
function blockTypeItem$1(nodeType, options) {
    var command = setBlockType(nodeType, options.attrs);
    var passedOptions = {
        run: command,
        enable: function enable(state) { return command(state); },
        active: function active(state) {
            var ref = state.selection;
            var $from = ref.$from;
            var to = ref.to;
            var node = ref.node;
            if (node)
                { return node.hasMarkup(nodeType, options.attrs); }
            return to <= $from.end() && $from.parent.hasMarkup(nodeType, options.attrs);
        }
    };
    for (var prop in options)
        { passedOptions[prop] = options[prop]; }
    return new MenuItem$1(passedOptions);
}
// Work around classList.toggle being broken in IE11
function setClass$1(dom, cls, on) {
    if (on)
        { dom.classList.add(cls); }
    else
        { dom.classList.remove(cls); }
}

var prefix$3 = "ProseMirror-menubar";
function isIOS$1() {
    if (typeof navigator == "undefined")
        { return false; }
    var agent = navigator.userAgent;
    return !/Edge\/\d/.test(agent) && /AppleWebKit/.test(agent) && /Mobile\/\w+/.test(agent);
}
/**
A plugin that will place a menu bar above the editor. Note that
this involves wrapping the editor in an additional `<div>`.
*/
function menuBar$1(options) {
    return new Plugin({
        view: function view(editorView) { return new MenuBarView$1(editorView, options); }
    });
}
var MenuBarView$1 = function MenuBarView(editorView, options) {
    var this$1$1 = this;

    this.editorView = editorView;
    this.options = options;
    this.spacer = null;
    this.maxHeight = 0;
    this.widthForMaxHeight = 0;
    this.floating = false;
    this.scrollHandler = null;
    this.wrapper = crelt("div", { class: prefix$3 + "-wrapper" });
    this.menu = this.wrapper.appendChild(crelt("div", { class: prefix$3 }));
    this.menu.className = prefix$3;
    if (editorView.dom.parentNode)
        { editorView.dom.parentNode.replaceChild(this.wrapper, editorView.dom); }
    this.wrapper.appendChild(editorView.dom);
    var ref = renderGrouped(this.editorView, this.options.content);
    var dom = ref.dom;
    var update = ref.update;
    this.contentUpdate = update;
    this.menu.appendChild(dom);
    this.update();
    if (options.floating && !isIOS$1()) {
        this.updateFloat();
        var potentialScrollers = getAllWrapping(this.wrapper);
        this.scrollHandler = function (e) {
            var root = this$1$1.editorView.root;
            if (!(root.body || root).contains(this$1$1.wrapper))
                { potentialScrollers.forEach(function (el) { return el.removeEventListener("scroll", this$1$1.scrollHandler); }); }
            else
                { this$1$1.updateFloat(e.target.getBoundingClientRect ? e.target : undefined); }
        };
        potentialScrollers.forEach(function (el) { return el.addEventListener('scroll', this$1$1.scrollHandler); });
    }
};
MenuBarView$1.prototype.update = function update () {
    this.contentUpdate(this.editorView.state);
    if (this.floating) {
        this.updateScrollCursor();
    }
    else {
        if (this.menu.offsetWidth != this.widthForMaxHeight) {
            this.widthForMaxHeight = this.menu.offsetWidth;
            this.maxHeight = 0;
        }
        if (this.menu.offsetHeight > this.maxHeight) {
            this.maxHeight = this.menu.offsetHeight;
            this.menu.style.minHeight = this.maxHeight + "px";
        }
    }
};
MenuBarView$1.prototype.updateScrollCursor = function updateScrollCursor () {
    var selection = this.editorView.root.getSelection();
    if (!selection.focusNode)
        { return; }
    var rects = selection.getRangeAt(0).getClientRects();
    var selRect = rects[selectionIsInverted$1(selection) ? 0 : rects.length - 1];
    if (!selRect)
        { return; }
    var menuRect = this.menu.getBoundingClientRect();
    if (selRect.top < menuRect.bottom && selRect.bottom > menuRect.top) {
        var scrollable = findWrappingScrollable$1(this.wrapper);
        if (scrollable)
            { scrollable.scrollTop -= (menuRect.bottom - selRect.top); }
    }
};
MenuBarView$1.prototype.updateFloat = function updateFloat (scrollAncestor) {
    var parent = this.wrapper, editorRect = parent.getBoundingClientRect(), top = scrollAncestor ? Math.max(0, scrollAncestor.getBoundingClientRect().top) : 0;
    if (this.floating) {
        if (editorRect.top >= top || editorRect.bottom < this.menu.offsetHeight + 10) {
            this.floating = false;
            this.menu.style.position = this.menu.style.left = this.menu.style.top = this.menu.style.width = "";
            this.menu.style.display = "";
            this.spacer.parentNode.removeChild(this.spacer);
            this.spacer = null;
        }
        else {
            var border = (parent.offsetWidth - parent.clientWidth) / 2;
            this.menu.style.left = (editorRect.left + border) + "px";
            this.menu.style.display = editorRect.top > (this.editorView.dom.ownerDocument.defaultView || window).innerHeight
                ? "none" : "";
            if (scrollAncestor)
                { this.menu.style.top = top + "px"; }
        }
    }
    else {
        if (editorRect.top < top && editorRect.bottom >= this.menu.offsetHeight + 10) {
            this.floating = true;
            var menuRect = this.menu.getBoundingClientRect();
            this.menu.style.left = menuRect.left + "px";
            this.menu.style.width = menuRect.width + "px";
            if (scrollAncestor)
                { this.menu.style.top = top + "px"; }
            this.menu.style.position = "fixed";
            this.spacer = crelt("div", { class: prefix$3 + "-spacer", style: ("height: " + (menuRect.height) + "px") });
            parent.insertBefore(this.spacer, this.menu);
        }
    }
};
MenuBarView$1.prototype.destroy = function destroy () {
    if (this.wrapper.parentNode)
        { this.wrapper.parentNode.replaceChild(this.editorView.dom, this.wrapper); }
};
// Not precise, but close enough
function selectionIsInverted$1(selection) {
    if (selection.anchorNode == selection.focusNode)
        { return selection.anchorOffset > selection.focusOffset; }
    return selection.anchorNode.compareDocumentPosition(selection.focusNode) == Node.DOCUMENT_POSITION_FOLLOWING;
}
function findWrappingScrollable$1(node) {
    for (var cur = node.parentNode; cur; cur = cur.parentNode)
        { if (cur.scrollHeight > cur.clientHeight)
            { return cur; } }
}
function getAllWrapping(node) {
    var res = [node.ownerDocument.defaultView || window];
    for (var cur = node.parentNode; cur; cur = cur.parentNode)
        { res.push(cur); }
    return res;
}var pmmenu=/*#__PURE__*/Object.freeze({__proto__:null,Dropdown:Dropdown$1,DropdownSubmenu:DropdownSubmenu$1,MenuItem:MenuItem$1,blockTypeItem:blockTypeItem$1,icons:icons$1,joinUpItem:joinUpItem$1,liftItem:liftItem$1,menuBar:menuBar$1,redoItem:redoItem$1,renderGrouped:renderGrouped,selectParentNodeItem:selectParentNodeItem$1,undoItem:undoItem$1,wrapItem:wrapItem$1});// src/index.ts

// src/tablemap.ts
var readFromCache;
var addToCache;
if (typeof WeakMap != "undefined") {
  var cache$1 = /* @__PURE__ */ new WeakMap();
  readFromCache = function (key) { return cache$1.get(key); };
  addToCache = function (key, value) {
    cache$1.set(key, value);
    return value;
  };
} else {
  var cache$1$1 = [];
  var cacheSize = 10;
  var cachePos = 0;
  readFromCache = function (key) {
    for (var i = 0; i < cache$1$1.length; i += 2)
      { if (cache$1$1[i] == key)
        { return cache$1$1[i + 1]; } }
  };
  addToCache = function (key, value) {
    if (cachePos == cacheSize)
      { cachePos = 0; }
    cache$1$1[cachePos++] = key;
    return cache$1$1[cachePos++] = value;
  };
}
var TableMap = /*@__PURE__*/(function () {
  function TableMap(width, height, map, problems) {
    this.width = width;
    this.height = height;
    this.map = map;
    this.problems = problems;
  }
  // Find the dimensions of the cell at the given position.
  TableMap.prototype.findCell = function findCell (pos) {
    for (var i = 0; i < this.map.length; i++) {
      var curPos = this.map[i];
      if (curPos != pos)
        { continue; }
      var left = i % this.width;
      var top = i / this.width | 0;
      var right = left + 1;
      var bottom = top + 1;
      for (var j = 1; right < this.width && this.map[i + j] == curPos; j++) {
        right++;
      }
      for (var j$1 = 1; bottom < this.height && this.map[i + this.width * j$1] == curPos; j$1++) {
        bottom++;
      }
      return { left: left, top: top, right: right, bottom: bottom };
    }
    throw new RangeError(("No cell with offset " + pos + " found"));
  };
  // Find the left side of the cell at the given position.
  TableMap.prototype.colCount = function colCount (pos) {
    for (var i = 0; i < this.map.length; i++) {
      if (this.map[i] == pos) {
        return i % this.width;
      }
    }
    throw new RangeError(("No cell with offset " + pos + " found"));
  };
  // Find the next cell in the given direction, starting from the cell
  // at `pos`, if any.
  TableMap.prototype.nextCell = function nextCell (pos, axis, dir) {
    var ref = this.findCell(pos);
    var left = ref.left;
    var right = ref.right;
    var top = ref.top;
    var bottom = ref.bottom;
    if (axis == "horiz") {
      if (dir < 0 ? left == 0 : right == this.width)
        { return null; }
      return this.map[top * this.width + (dir < 0 ? left - 1 : right)];
    } else {
      if (dir < 0 ? top == 0 : bottom == this.height)
        { return null; }
      return this.map[left + this.width * (dir < 0 ? top - 1 : bottom)];
    }
  };
  // Get the rectangle spanning the two given cells.
  TableMap.prototype.rectBetween = function rectBetween (a, b) {
    var ref = this.findCell(a);
    var leftA = ref.left;
    var rightA = ref.right;
    var topA = ref.top;
    var bottomA = ref.bottom;
    var ref$1 = this.findCell(b);
    var leftB = ref$1.left;
    var rightB = ref$1.right;
    var topB = ref$1.top;
    var bottomB = ref$1.bottom;
    return {
      left: Math.min(leftA, leftB),
      top: Math.min(topA, topB),
      right: Math.max(rightA, rightB),
      bottom: Math.max(bottomA, bottomB)
    };
  };
  // Return the position of all cells that have the top left corner in
  // the given rectangle.
  TableMap.prototype.cellsInRect = function cellsInRect (rect) {
    var result = [];
    var seen = {};
    for (var row = rect.top; row < rect.bottom; row++) {
      for (var col = rect.left; col < rect.right; col++) {
        var index = row * this.width + col;
        var pos = this.map[index];
        if (seen[pos])
          { continue; }
        seen[pos] = true;
        if (col == rect.left && col && this.map[index - 1] == pos || row == rect.top && row && this.map[index - this.width] == pos) {
          continue;
        }
        result.push(pos);
      }
    }
    return result;
  };
  // Return the position at which the cell at the given row and column
  // starts, or would start, if a cell started there.
  TableMap.prototype.positionAt = function positionAt (row, col, table) {
    for (var i = 0, rowStart = 0; ; i++) {
      var rowEnd = rowStart + table.child(i).nodeSize;
      if (i == row) {
        var index = col + row * this.width;
        var rowEndIndex = (row + 1) * this.width;
        while (index < rowEndIndex && this.map[index] < rowStart)
          { index++; }
        return index == rowEndIndex ? rowEnd - 1 : this.map[index];
      }
      rowStart = rowEnd;
    }
  };
  // Find the table map for the given table node.
  TableMap.get = function get (table) {
    return readFromCache(table) || addToCache(table, computeMap(table));
  };

  return TableMap;
}());
function computeMap(table) {
  if (table.type.spec.tableRole != "table")
    { throw new RangeError("Not a table node: " + table.type.name); }
  var width = findWidth(table), height = table.childCount;
  var map = [];
  var mapPos = 0;
  var problems = null;
  var colWidths = [];
  for (var i = 0, e = width * height; i < e; i++)
    { map[i] = 0; }
  for (var row = 0, pos = 0; row < height; row++) {
    var rowNode = table.child(row);
    pos++;
    for (var i$1 = 0; ; i$1++) {
      while (mapPos < map.length && map[mapPos] != 0)
        { mapPos++; }
      if (i$1 == rowNode.childCount)
        { break; }
      var cellNode = rowNode.child(i$1);
      var ref = cellNode.attrs;
      var colspan = ref.colspan;
      var rowspan = ref.rowspan;
      var colwidth = ref.colwidth;
      for (var h = 0; h < rowspan; h++) {
        if (h + row >= height) {
          (problems || (problems = [])).push({
            type: "overlong_rowspan",
            pos: pos,
            n: rowspan - h
          });
          break;
        }
        var start = mapPos + h * width;
        for (var w = 0; w < colspan; w++) {
          if (map[start + w] == 0)
            { map[start + w] = pos; }
          else
            { (problems || (problems = [])).push({
              type: "collision",
              row: row,
              pos: pos,
              n: colspan - w
            }); }
          var colW = colwidth && colwidth[w];
          if (colW) {
            var widthIndex = (start + w) % width * 2, prev = colWidths[widthIndex];
            if (prev == null || prev != colW && colWidths[widthIndex + 1] == 1) {
              colWidths[widthIndex] = colW;
              colWidths[widthIndex + 1] = 1;
            } else if (prev == colW) {
              colWidths[widthIndex + 1]++;
            }
          }
        }
      }
      mapPos += colspan;
      pos += cellNode.nodeSize;
    }
    var expectedPos = (row + 1) * width;
    var missing = 0;
    while (mapPos < expectedPos)
      { if (map[mapPos++] == 0)
        { missing++; } }
    if (missing)
      { (problems || (problems = [])).push({ type: "missing", row: row, n: missing }); }
    pos++;
  }
  var tableMap = new TableMap(width, height, map, problems);
  var badWidths = false;
  for (var i$2 = 0; !badWidths && i$2 < colWidths.length; i$2 += 2)
    { if (colWidths[i$2] != null && colWidths[i$2 + 1] < height)
      { badWidths = true; } }
  if (badWidths)
    { findBadColWidths(tableMap, colWidths, table); }
  return tableMap;
}
function findWidth(table) {
  var width = -1;
  var hasRowSpan = false;
  for (var row = 0; row < table.childCount; row++) {
    var rowNode = table.child(row);
    var rowWidth = 0;
    if (hasRowSpan)
      { for (var j = 0; j < row; j++) {
        var prevRow = table.child(j);
        for (var i = 0; i < prevRow.childCount; i++) {
          var cell = prevRow.child(i);
          if (j + cell.attrs.rowspan > row)
            { rowWidth += cell.attrs.colspan; }
        }
      } }
    for (var i$1 = 0; i$1 < rowNode.childCount; i$1++) {
      var cell$1 = rowNode.child(i$1);
      rowWidth += cell$1.attrs.colspan;
      if (cell$1.attrs.rowspan > 1)
        { hasRowSpan = true; }
    }
    if (width == -1)
      { width = rowWidth; }
    else if (width != rowWidth)
      { width = Math.max(width, rowWidth); }
  }
  return width;
}
function findBadColWidths(map, colWidths, table) {
  if (!map.problems)
    { map.problems = []; }
  var seen = {};
  for (var i = 0; i < map.map.length; i++) {
    var pos = map.map[i];
    if (seen[pos])
      { continue; }
    seen[pos] = true;
    var node = table.nodeAt(pos);
    if (!node) {
      throw new RangeError(("No cell with offset " + pos + " found"));
    }
    var updated = null;
    var attrs = node.attrs;
    for (var j = 0; j < attrs.colspan; j++) {
      var col = (i + j) % map.width;
      var colWidth = colWidths[col * 2];
      if (colWidth != null && (!attrs.colwidth || attrs.colwidth[j] != colWidth))
        { (updated || (updated = freshColWidth(attrs)))[j] = colWidth; }
    }
    if (updated)
      { map.problems.unshift({
        type: "colwidth mismatch",
        pos: pos,
        colwidth: updated
      }); }
  }
}
function freshColWidth(attrs) {
  if (attrs.colwidth)
    { return attrs.colwidth.slice(); }
  var result = [];
  for (var i = 0; i < attrs.colspan; i++)
    { result.push(0); }
  return result;
}

// src/schema.ts
function getCellAttrs(dom, extraAttrs) {
  if (typeof dom === "string") {
    return {};
  }
  var widthAttr = dom.getAttribute("data-colwidth");
  var widths = widthAttr && /^\d+(,\d+)*$/.test(widthAttr) ? widthAttr.split(",").map(function (s) { return Number(s); }) : null;
  var colspan = Number(dom.getAttribute("colspan") || 1);
  var result = {
    colspan: colspan,
    rowspan: Number(dom.getAttribute("rowspan") || 1),
    colwidth: widths && widths.length == colspan ? widths : null
  };
  for (var prop in extraAttrs) {
    var getter = extraAttrs[prop].getFromDOM;
    var value = getter && getter(dom);
    if (value != null) {
      result[prop] = value;
    }
  }
  return result;
}
function setCellAttrs(node, extraAttrs) {
  var attrs = {};
  if (node.attrs.colspan != 1)
    { attrs.colspan = node.attrs.colspan; }
  if (node.attrs.rowspan != 1)
    { attrs.rowspan = node.attrs.rowspan; }
  if (node.attrs.colwidth)
    { attrs["data-colwidth"] = node.attrs.colwidth.join(","); }
  for (var prop in extraAttrs) {
    var setter = extraAttrs[prop].setDOMAttr;
    if (setter)
      { setter(node.attrs[prop], attrs); }
  }
  return attrs;
}
function tableNodes(options) {
  var extraAttrs = options.cellAttributes || {};
  var cellAttrs = {
    colspan: { default: 1 },
    rowspan: { default: 1 },
    colwidth: { default: null }
  };
  for (var prop in extraAttrs)
    { cellAttrs[prop] = { default: extraAttrs[prop].default }; }
  return {
    table: {
      content: "table_row+",
      tableRole: "table",
      isolating: true,
      group: options.tableGroup,
      parseDOM: [{ tag: "table" }],
      toDOM: function toDOM() {
        return ["table", ["tbody", 0]];
      }
    },
    table_row: {
      content: "(table_cell | table_header)*",
      tableRole: "row",
      parseDOM: [{ tag: "tr" }],
      toDOM: function toDOM() {
        return ["tr", 0];
      }
    },
    table_cell: {
      content: options.cellContent,
      attrs: cellAttrs,
      tableRole: "cell",
      isolating: true,
      parseDOM: [
        { tag: "td", getAttrs: function (dom) { return getCellAttrs(dom, extraAttrs); } }
      ],
      toDOM: function toDOM(node) {
        return ["td", setCellAttrs(node, extraAttrs), 0];
      }
    },
    table_header: {
      content: options.cellContent,
      attrs: cellAttrs,
      tableRole: "header_cell",
      isolating: true,
      parseDOM: [
        { tag: "th", getAttrs: function (dom) { return getCellAttrs(dom, extraAttrs); } }
      ],
      toDOM: function toDOM(node) {
        return ["th", setCellAttrs(node, extraAttrs), 0];
      }
    }
  };
}
function tableNodeTypes(schema) {
  var result = schema.cached.tableNodeTypes;
  if (!result) {
    result = schema.cached.tableNodeTypes = {};
    for (var name in schema.nodes) {
      var type = schema.nodes[name], role = type.spec.tableRole;
      if (role)
        { result[role] = type; }
    }
  }
  return result;
}

// src/util.ts
var tableEditingKey = new PluginKey("selectingCells");
function cellAround($pos) {
  for (var d = $pos.depth - 1; d > 0; d--)
    { if ($pos.node(d).type.spec.tableRole == "row")
      { return $pos.node(0).resolve($pos.before(d + 1)); } }
  return null;
}
function isInTable(state) {
  var $head = state.selection.$head;
  for (var d = $head.depth; d > 0; d--)
    { if ($head.node(d).type.spec.tableRole == "row")
      { return true; } }
  return false;
}
function selectionCell(state) {
  var sel = state.selection;
  if ("$anchorCell" in sel && sel.$anchorCell) {
    return sel.$anchorCell.pos > sel.$headCell.pos ? sel.$anchorCell : sel.$headCell;
  } else if ("node" in sel && sel.node && sel.node.type.spec.tableRole == "cell") {
    return sel.$anchor;
  }
  var $cell = cellAround(sel.$head) || cellNear(sel.$head);
  if ($cell) {
    return $cell;
  }
  throw new RangeError(("No cell found around position " + (sel.head)));
}
function cellNear($pos) {
  for (var after = $pos.nodeAfter, pos = $pos.pos; after; after = after.firstChild, pos++) {
    var role = after.type.spec.tableRole;
    if (role == "cell" || role == "header_cell")
      { return $pos.doc.resolve(pos); }
  }
  for (var before = $pos.nodeBefore, pos$1 = $pos.pos; before; before = before.lastChild, pos$1--) {
    var role$1 = before.type.spec.tableRole;
    if (role$1 == "cell" || role$1 == "header_cell")
      { return $pos.doc.resolve(pos$1 - before.nodeSize); }
  }
}
function pointsAtCell($pos) {
  return $pos.parent.type.spec.tableRole == "row" && !!$pos.nodeAfter;
}
function moveCellForward($pos) {
  return $pos.node(0).resolve($pos.pos + $pos.nodeAfter.nodeSize);
}
function inSameTable($cellA, $cellB) {
  return $cellA.depth == $cellB.depth && $cellA.pos >= $cellB.start(-1) && $cellA.pos <= $cellB.end(-1);
}
function nextCell($pos, axis, dir) {
  var table = $pos.node(-1);
  var map = TableMap.get(table);
  var tableStart = $pos.start(-1);
  var moved = map.nextCell($pos.pos - tableStart, axis, dir);
  return moved == null ? null : $pos.node(0).resolve(tableStart + moved);
}
function removeColSpan(attrs, pos, n) {
  if ( n === void 0 ) n = 1;

  var result = Object.assign({}, attrs, {colspan: attrs.colspan - n});
  if (result.colwidth) {
    result.colwidth = result.colwidth.slice();
    result.colwidth.splice(pos, n);
    if (!result.colwidth.some(function (w) { return w > 0; }))
      { result.colwidth = null; }
  }
  return result;
}
function addColSpan(attrs, pos, n) {
  if ( n === void 0 ) n = 1;

  var result = Object.assign({}, attrs, {colspan: attrs.colspan + n});
  if (result.colwidth) {
    result.colwidth = result.colwidth.slice();
    for (var i = 0; i < n; i++)
      { result.colwidth.splice(pos, 0, 0); }
  }
  return result;
}
function columnIsHeader(map, table, col) {
  var headerCell = tableNodeTypes(table.type.schema).header_cell;
  for (var row = 0; row < map.height; row++)
    { if (table.nodeAt(map.map[col + row * map.width]).type != headerCell)
      { return false; } }
  return true;
}

// src/cellselection.ts
var CellSelection = /*@__PURE__*/(function (Selection) {
  function _CellSelection($anchorCell, $headCell) {
    if ( $headCell === void 0 ) $headCell = $anchorCell;

    var table = $anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = $anchorCell.start(-1);
    var rect = map.rectBetween(
      $anchorCell.pos - tableStart,
      $headCell.pos - tableStart
    );
    var doc = $anchorCell.node(0);
    var cells = map.cellsInRect(rect).filter(function (p) { return p != $headCell.pos - tableStart; });
    cells.unshift($headCell.pos - tableStart);
    var ranges = cells.map(function (pos) {
      var cell = table.nodeAt(pos);
      if (!cell) {
        throw RangeError(("No cell with offset " + pos + " found"));
      }
      var from = tableStart + pos + 1;
      return new SelectionRange$1(
        doc.resolve(from),
        doc.resolve(from + cell.content.size)
      );
    });
    Selection.call(this, ranges[0].$from, ranges[0].$to, ranges);
    this.$anchorCell = $anchorCell;
    this.$headCell = $headCell;
  }

  if ( Selection ) _CellSelection.__proto__ = Selection;
  _CellSelection.prototype = Object.create( Selection && Selection.prototype );
  _CellSelection.prototype.constructor = _CellSelection;
  _CellSelection.prototype.map = function map (doc, mapping) {
    var $anchorCell = doc.resolve(mapping.map(this.$anchorCell.pos));
    var $headCell = doc.resolve(mapping.map(this.$headCell.pos));
    if (pointsAtCell($anchorCell) && pointsAtCell($headCell) && inSameTable($anchorCell, $headCell)) {
      var tableChanged = this.$anchorCell.node(-1) != $anchorCell.node(-1);
      if (tableChanged && this.isRowSelection())
        { return _CellSelection.rowSelection($anchorCell, $headCell); }
      else if (tableChanged && this.isColSelection())
        { return _CellSelection.colSelection($anchorCell, $headCell); }
      else
        { return new _CellSelection($anchorCell, $headCell); }
    }
    return TextSelection.between($anchorCell, $headCell);
  };
  // Returns a rectangular slice of table rows containing the selected
  // cells.
  _CellSelection.prototype.content = function content () {
    var table = this.$anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = this.$anchorCell.start(-1);
    var rect = map.rectBetween(
      this.$anchorCell.pos - tableStart,
      this.$headCell.pos - tableStart
    );
    var seen = {};
    var rows = [];
    for (var row = rect.top; row < rect.bottom; row++) {
      var rowContent = [];
      for (var index = row * map.width + rect.left, col = rect.left; col < rect.right; col++, index++) {
        var pos = map.map[index];
        if (seen[pos])
          { continue; }
        seen[pos] = true;
        var cellRect = map.findCell(pos);
        var cell = table.nodeAt(pos);
        if (!cell) {
          throw RangeError(("No cell with offset " + pos + " found"));
        }
        var extraLeft = rect.left - cellRect.left;
        var extraRight = cellRect.right - rect.right;
        if (extraLeft > 0 || extraRight > 0) {
          var attrs = cell.attrs;
          if (extraLeft > 0) {
            attrs = removeColSpan(attrs, 0, extraLeft);
          }
          if (extraRight > 0) {
            attrs = removeColSpan(
              attrs,
              attrs.colspan - extraRight,
              extraRight
            );
          }
          if (cellRect.left < rect.left) {
            cell = cell.type.createAndFill(attrs);
            if (!cell) {
              throw RangeError(
                ("Could not create cell with attrs " + (JSON.stringify(attrs)))
              );
            }
          } else {
            cell = cell.type.create(attrs, cell.content);
          }
        }
        if (cellRect.top < rect.top || cellRect.bottom > rect.bottom) {
          var attrs$1 = Object.assign({}, cell.attrs,
            {rowspan: Math.min(cellRect.bottom, rect.bottom) - Math.max(cellRect.top, rect.top)});
          if (cellRect.top < rect.top) {
            cell = cell.type.createAndFill(attrs$1);
          } else {
            cell = cell.type.create(attrs$1, cell.content);
          }
        }
        rowContent.push(cell);
      }
      rows.push(table.child(row).copy(Fragment.from(rowContent)));
    }
    var fragment = this.isColSelection() && this.isRowSelection() ? table : rows;
    return new Slice(Fragment.from(fragment), 1, 1);
  };
  _CellSelection.prototype.replace = function replace (tr, content) {
    if ( content === void 0 ) content = Slice.empty;

    var mapFrom = tr.steps.length, ranges = this.ranges;
    for (var i = 0; i < ranges.length; i++) {
      var ref = ranges[i];
      var $from = ref.$from;
      var $to = ref.$to;
      var mapping = tr.mapping.slice(mapFrom);
      tr.replace(
        mapping.map($from.pos),
        mapping.map($to.pos),
        i ? Slice.empty : content
      );
    }
    var sel = Selection.findFrom(
      tr.doc.resolve(tr.mapping.slice(mapFrom).map(this.to)),
      -1
    );
    if (sel)
      { tr.setSelection(sel); }
  };
  _CellSelection.prototype.replaceWith = function replaceWith (tr, node) {
    this.replace(tr, new Slice(Fragment.from(node), 0, 0));
  };
  _CellSelection.prototype.forEachCell = function forEachCell (f) {
    var table = this.$anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = this.$anchorCell.start(-1);
    var cells = map.cellsInRect(
      map.rectBetween(
        this.$anchorCell.pos - tableStart,
        this.$headCell.pos - tableStart
      )
    );
    for (var i = 0; i < cells.length; i++) {
      f(table.nodeAt(cells[i]), tableStart + cells[i]);
    }
  };
  // True if this selection goes all the way from the top to the
  // bottom of the table.
  _CellSelection.prototype.isColSelection = function isColSelection () {
    var anchorTop = this.$anchorCell.index(-1);
    var headTop = this.$headCell.index(-1);
    if (Math.min(anchorTop, headTop) > 0)
      { return false; }
    var anchorBottom = anchorTop + this.$anchorCell.nodeAfter.attrs.rowspan;
    var headBottom = headTop + this.$headCell.nodeAfter.attrs.rowspan;
    return Math.max(anchorBottom, headBottom) == this.$headCell.node(-1).childCount;
  };
  // Returns the smallest column selection that covers the given anchor
  // and head cell.
  _CellSelection.colSelection = function colSelection ($anchorCell, $headCell) {
    if ( $headCell === void 0 ) $headCell = $anchorCell;

    var table = $anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = $anchorCell.start(-1);
    var anchorRect = map.findCell($anchorCell.pos - tableStart);
    var headRect = map.findCell($headCell.pos - tableStart);
    var doc = $anchorCell.node(0);
    if (anchorRect.top <= headRect.top) {
      if (anchorRect.top > 0)
        { $anchorCell = doc.resolve(tableStart + map.map[anchorRect.left]); }
      if (headRect.bottom < map.height)
        { $headCell = doc.resolve(
          tableStart + map.map[map.width * (map.height - 1) + headRect.right - 1]
        ); }
    } else {
      if (headRect.top > 0)
        { $headCell = doc.resolve(tableStart + map.map[headRect.left]); }
      if (anchorRect.bottom < map.height)
        { $anchorCell = doc.resolve(
          tableStart + map.map[map.width * (map.height - 1) + anchorRect.right - 1]
        ); }
    }
    return new _CellSelection($anchorCell, $headCell);
  };
  // True if this selection goes all the way from the left to the
  // right of the table.
  _CellSelection.prototype.isRowSelection = function isRowSelection () {
    var table = this.$anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = this.$anchorCell.start(-1);
    var anchorLeft = map.colCount(this.$anchorCell.pos - tableStart);
    var headLeft = map.colCount(this.$headCell.pos - tableStart);
    if (Math.min(anchorLeft, headLeft) > 0)
      { return false; }
    var anchorRight = anchorLeft + this.$anchorCell.nodeAfter.attrs.colspan;
    var headRight = headLeft + this.$headCell.nodeAfter.attrs.colspan;
    return Math.max(anchorRight, headRight) == map.width;
  };
  _CellSelection.prototype.eq = function eq (other) {
    return other instanceof _CellSelection && other.$anchorCell.pos == this.$anchorCell.pos && other.$headCell.pos == this.$headCell.pos;
  };
  // Returns the smallest row selection that covers the given anchor
  // and head cell.
  _CellSelection.rowSelection = function rowSelection ($anchorCell, $headCell) {
    if ( $headCell === void 0 ) $headCell = $anchorCell;

    var table = $anchorCell.node(-1);
    var map = TableMap.get(table);
    var tableStart = $anchorCell.start(-1);
    var anchorRect = map.findCell($anchorCell.pos - tableStart);
    var headRect = map.findCell($headCell.pos - tableStart);
    var doc = $anchorCell.node(0);
    if (anchorRect.left <= headRect.left) {
      if (anchorRect.left > 0)
        { $anchorCell = doc.resolve(
          tableStart + map.map[anchorRect.top * map.width]
        ); }
      if (headRect.right < map.width)
        { $headCell = doc.resolve(
          tableStart + map.map[map.width * (headRect.top + 1) - 1]
        ); }
    } else {
      if (headRect.left > 0)
        { $headCell = doc.resolve(tableStart + map.map[headRect.top * map.width]); }
      if (anchorRect.right < map.width)
        { $anchorCell = doc.resolve(
          tableStart + map.map[map.width * (anchorRect.top + 1) - 1]
        ); }
    }
    return new _CellSelection($anchorCell, $headCell);
  };
  _CellSelection.prototype.toJSON = function toJSON () {
    return {
      type: "cell",
      anchor: this.$anchorCell.pos,
      head: this.$headCell.pos
    };
  };
  _CellSelection.fromJSON = function fromJSON (doc, json) {
    return new _CellSelection(doc.resolve(json.anchor), doc.resolve(json.head));
  };
  _CellSelection.create = function create (doc, anchorCell, headCell) {
    if ( headCell === void 0 ) headCell = anchorCell;

    return new _CellSelection(doc.resolve(anchorCell), doc.resolve(headCell));
  };
  _CellSelection.prototype.getBookmark = function getBookmark () {
    return new CellBookmark(this.$anchorCell.pos, this.$headCell.pos);
  };

  return _CellSelection;
}(Selection));
CellSelection.prototype.visible = false;
Selection.jsonID("cell", CellSelection);
var CellBookmark = /*@__PURE__*/(function () {
  function _CellBookmark(anchor, head) {
    this.anchor = anchor;
    this.head = head;
  }
  _CellBookmark.prototype.map = function map (mapping) {
    return new _CellBookmark(mapping.map(this.anchor), mapping.map(this.head));
  };
  _CellBookmark.prototype.resolve = function resolve (doc) {
    var $anchorCell = doc.resolve(this.anchor), $headCell = doc.resolve(this.head);
    if ($anchorCell.parent.type.spec.tableRole == "row" && $headCell.parent.type.spec.tableRole == "row" && $anchorCell.index() < $anchorCell.parent.childCount && $headCell.index() < $headCell.parent.childCount && inSameTable($anchorCell, $headCell))
      { return new CellSelection($anchorCell, $headCell); }
    else
      { return Selection.near($headCell, 1); }
  };

  return _CellBookmark;
}());
function drawCellSelection(state) {
  if (!(state.selection instanceof CellSelection))
    { return null; }
  var cells = [];
  state.selection.forEachCell(function (node, pos) {
    cells.push(
      Decoration$1.node(pos, pos + node.nodeSize, { class: "selectedCell" })
    );
  });
  return DecorationSet.create(state.doc, cells);
}
function isCellBoundarySelection(ref) {
  var $from = ref.$from;
  var $to = ref.$to;

  if ($from.pos == $to.pos || $from.pos < $from.pos - 6)
    { return false; }
  var afterFrom = $from.pos;
  var beforeTo = $to.pos;
  var depth = $from.depth;
  for (; depth >= 0; depth--, afterFrom++)
    { if ($from.after(depth + 1) < $from.end(depth))
      { break; } }
  for (var d = $to.depth; d >= 0; d--, beforeTo--)
    { if ($to.before(d + 1) > $to.start(d))
      { break; } }
  return afterFrom == beforeTo && /row|table/.test($from.node(depth).type.spec.tableRole);
}
function isTextSelectionAcrossCells(ref) {
  var $from = ref.$from;
  var $to = ref.$to;

  var fromCellBoundaryNode;
  var toCellBoundaryNode;
  for (var i = $from.depth; i > 0; i--) {
    var node = $from.node(i);
    if (node.type.spec.tableRole === "cell" || node.type.spec.tableRole === "header_cell") {
      fromCellBoundaryNode = node;
      break;
    }
  }
  for (var i$1 = $to.depth; i$1 > 0; i$1--) {
    var node$1 = $to.node(i$1);
    if (node$1.type.spec.tableRole === "cell" || node$1.type.spec.tableRole === "header_cell") {
      toCellBoundaryNode = node$1;
      break;
    }
  }
  return fromCellBoundaryNode !== toCellBoundaryNode && $to.parentOffset === 0;
}
function normalizeSelection(state, tr, allowTableNodeSelection) {
  var sel = (tr || state).selection;
  var doc = (tr || state).doc;
  var normalize;
  var role;
  if (sel instanceof NodeSelection && (role = sel.node.type.spec.tableRole)) {
    if (role == "cell" || role == "header_cell") {
      normalize = CellSelection.create(doc, sel.from);
    } else if (role == "row") {
      var $cell = doc.resolve(sel.from + 1);
      normalize = CellSelection.rowSelection($cell, $cell);
    } else if (!allowTableNodeSelection) {
      var map = TableMap.get(sel.node);
      var start = sel.from + 1;
      var lastCell = start + map.map[map.width * map.height - 1];
      normalize = CellSelection.create(doc, start + 1, lastCell);
    }
  } else if (sel instanceof TextSelection && isCellBoundarySelection(sel)) {
    normalize = TextSelection.create(doc, sel.from);
  } else if (sel instanceof TextSelection && isTextSelectionAcrossCells(sel)) {
    normalize = TextSelection.create(doc, sel.$from.start(), sel.$from.end());
  }
  if (normalize)
    { (tr || (tr = state.tr)).setSelection(normalize); }
  return tr;
}
var fixTablesKey = new PluginKey("fix-tables");
function changedDescendants(old, cur, offset, f) {
  var oldSize = old.childCount, curSize = cur.childCount;
  outer:
    for (var i = 0, j = 0; i < curSize; i++) {
      var child = cur.child(i);
      for (var scan = j, e = Math.min(oldSize, i + 3); scan < e; scan++) {
        if (old.child(scan) == child) {
          j = scan + 1;
          offset += child.nodeSize;
          continue outer;
        }
      }
      f(child, offset);
      if (j < oldSize && old.child(j).sameMarkup(child))
        { changedDescendants(old.child(j), child, offset + 1, f); }
      else
        { child.nodesBetween(0, child.content.size, f, offset + 1); }
      offset += child.nodeSize;
    }
}
function fixTables(state, oldState) {
  var tr;
  var check = function (node, pos) {
    if (node.type.spec.tableRole == "table")
      { tr = fixTable(state, node, pos, tr); }
  };
  if (!oldState)
    { state.doc.descendants(check); }
  else if (oldState.doc != state.doc)
    { changedDescendants(oldState.doc, state.doc, 0, check); }
  return tr;
}
function fixTable(state, table, tablePos, tr) {
  var map = TableMap.get(table);
  if (!map.problems)
    { return tr; }
  if (!tr)
    { tr = state.tr; }
  var mustAdd = [];
  for (var i = 0; i < map.height; i++)
    { mustAdd.push(0); }
  for (var i$1 = 0; i$1 < map.problems.length; i$1++) {
    var prob = map.problems[i$1];
    if (prob.type == "collision") {
      var cell = table.nodeAt(prob.pos);
      if (!cell)
        { continue; }
      var attrs = cell.attrs;
      for (var j = 0; j < attrs.rowspan; j++)
        { mustAdd[prob.row + j] += prob.n; }
      tr.setNodeMarkup(
        tr.mapping.map(tablePos + 1 + prob.pos),
        null,
        removeColSpan(attrs, attrs.colspan - prob.n, prob.n)
      );
    } else if (prob.type == "missing") {
      mustAdd[prob.row] += prob.n;
    } else if (prob.type == "overlong_rowspan") {
      var cell$1 = table.nodeAt(prob.pos);
      if (!cell$1)
        { continue; }
      tr.setNodeMarkup(tr.mapping.map(tablePos + 1 + prob.pos), null, Object.assign({}, cell$1.attrs,
        {rowspan: cell$1.attrs.rowspan - prob.n}));
    } else if (prob.type == "colwidth mismatch") {
      var cell$2 = table.nodeAt(prob.pos);
      if (!cell$2)
        { continue; }
      tr.setNodeMarkup(tr.mapping.map(tablePos + 1 + prob.pos), null, Object.assign({}, cell$2.attrs,
        {colwidth: prob.colwidth}));
    }
  }
  var first, last;
  for (var i$2 = 0; i$2 < mustAdd.length; i$2++)
    { if (mustAdd[i$2]) {
      if (first == null)
        { first = i$2; }
      last = i$2;
    } }
  for (var i$3 = 0, pos = tablePos + 1; i$3 < map.height; i$3++) {
    var row = table.child(i$3);
    var end = pos + row.nodeSize;
    var add = mustAdd[i$3];
    if (add > 0) {
      var role = "cell";
      if (row.firstChild) {
        role = row.firstChild.type.spec.tableRole;
      }
      var nodes = [];
      for (var j$1 = 0; j$1 < add; j$1++) {
        var node = tableNodeTypes(state.schema)[role].createAndFill();
        if (node)
          { nodes.push(node); }
      }
      var side = (i$3 == 0 || first == i$3 - 1) && last == i$3 ? pos + 1 : end - 1;
      tr.insert(tr.mapping.map(side), nodes);
    }
    pos = end;
  }
  return tr.setMeta(fixTablesKey, { fixTables: true });
}
function selectedRect(state) {
  var sel = state.selection;
  var $pos = selectionCell(state);
  var table = $pos.node(-1);
  var tableStart = $pos.start(-1);
  var map = TableMap.get(table);
  var rect = sel instanceof CellSelection ? map.rectBetween(
    sel.$anchorCell.pos - tableStart,
    sel.$headCell.pos - tableStart
  ) : map.findCell($pos.pos - tableStart);
  return Object.assign({}, rect, {tableStart: tableStart, map: map, table: table});
}
function addColumn(tr, ref, col) {
  var map = ref.map;
  var tableStart = ref.tableStart;
  var table = ref.table;

  var refColumn = col > 0 ? -1 : 0;
  if (columnIsHeader(map, table, col + refColumn)) {
    refColumn = col == 0 || col == map.width ? null : 0;
  }
  for (var row = 0; row < map.height; row++) {
    var index = row * map.width + col;
    if (col > 0 && col < map.width && map.map[index - 1] == map.map[index]) {
      var pos = map.map[index];
      var cell = table.nodeAt(pos);
      tr.setNodeMarkup(
        tr.mapping.map(tableStart + pos),
        null,
        addColSpan(cell.attrs, col - map.colCount(pos))
      );
      row += cell.attrs.rowspan - 1;
    } else {
      var type = refColumn == null ? tableNodeTypes(table.type.schema).cell : table.nodeAt(map.map[index + refColumn]).type;
      var pos$1 = map.positionAt(row, col, table);
      tr.insert(tr.mapping.map(tableStart + pos$1), type.createAndFill());
    }
  }
  return tr;
}
function addColumnBefore(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state);
    dispatch(addColumn(state.tr, rect, rect.left));
  }
  return true;
}
function addColumnAfter(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state);
    dispatch(addColumn(state.tr, rect, rect.right));
  }
  return true;
}
function removeColumn(tr, ref, col) {
  var map = ref.map;
  var table = ref.table;
  var tableStart = ref.tableStart;

  var mapStart = tr.mapping.maps.length;
  for (var row = 0; row < map.height; ) {
    var index = row * map.width + col;
    var pos = map.map[index];
    var cell = table.nodeAt(pos);
    var attrs = cell.attrs;
    if (col > 0 && map.map[index - 1] == pos || col < map.width - 1 && map.map[index + 1] == pos) {
      tr.setNodeMarkup(
        tr.mapping.slice(mapStart).map(tableStart + pos),
        null,
        removeColSpan(attrs, col - map.colCount(pos))
      );
    } else {
      var start = tr.mapping.slice(mapStart).map(tableStart + pos);
      tr.delete(start, start + cell.nodeSize);
    }
    row += attrs.rowspan;
  }
}
function deleteColumn(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state);
    var tr = state.tr;
    if (rect.left == 0 && rect.right == rect.map.width)
      { return false; }
    for (var i = rect.right - 1; ; i--) {
      removeColumn(tr, rect, i);
      if (i == rect.left)
        { break; }
      var table = rect.tableStart ? tr.doc.nodeAt(rect.tableStart - 1) : tr.doc;
      if (!table) {
        throw RangeError("No table found");
      }
      rect.table = table;
      rect.map = TableMap.get(table);
    }
    dispatch(tr);
  }
  return true;
}
function rowIsHeader(map, table, row) {
  var _a;
  var headerCell = tableNodeTypes(table.type.schema).header_cell;
  for (var col = 0; col < map.width; col++)
    { if (((_a = table.nodeAt(map.map[col + row * map.width])) == null ? void 0 : _a.type) != headerCell)
      { return false; } }
  return true;
}
function addRow(tr, ref, row) {
  var map = ref.map;
  var tableStart = ref.tableStart;
  var table = ref.table;

  var _a;
  var rowPos = tableStart;
  for (var i = 0; i < row; i++)
    { rowPos += table.child(i).nodeSize; }
  var cells = [];
  var refRow = row > 0 ? -1 : 0;
  if (rowIsHeader(map, table, row + refRow))
    { refRow = row == 0 || row == map.height ? null : 0; }
  for (var col = 0, index = map.width * row; col < map.width; col++, index++) {
    if (row > 0 && row < map.height && map.map[index] == map.map[index - map.width]) {
      var pos = map.map[index];
      var attrs = table.nodeAt(pos).attrs;
      tr.setNodeMarkup(tableStart + pos, null, Object.assign({}, attrs,
        {rowspan: attrs.rowspan + 1}));
      col += attrs.colspan - 1;
    } else {
      var type = refRow == null ? tableNodeTypes(table.type.schema).cell : (_a = table.nodeAt(map.map[index + refRow * map.width])) == null ? void 0 : _a.type;
      var node = type == null ? void 0 : type.createAndFill();
      if (node)
        { cells.push(node); }
    }
  }
  tr.insert(rowPos, tableNodeTypes(table.type.schema).row.create(null, cells));
  return tr;
}
function addRowBefore(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state);
    dispatch(addRow(state.tr, rect, rect.top));
  }
  return true;
}
function addRowAfter(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state);
    dispatch(addRow(state.tr, rect, rect.bottom));
  }
  return true;
}
function removeRow(tr, ref, row) {
  var map = ref.map;
  var table = ref.table;
  var tableStart = ref.tableStart;

  var rowPos = 0;
  for (var i = 0; i < row; i++)
    { rowPos += table.child(i).nodeSize; }
  var nextRow = rowPos + table.child(row).nodeSize;
  var mapFrom = tr.mapping.maps.length;
  tr.delete(rowPos + tableStart, nextRow + tableStart);
  var seen = /* @__PURE__ */ new Set();
  for (var col = 0, index = row * map.width; col < map.width; col++, index++) {
    var pos = map.map[index];
    if (seen.has(pos))
      { continue; }
    seen.add(pos);
    if (row > 0 && pos == map.map[index - map.width]) {
      var attrs = table.nodeAt(pos).attrs;
      tr.setNodeMarkup(tr.mapping.slice(mapFrom).map(pos + tableStart), null, Object.assign({}, attrs,
        {rowspan: attrs.rowspan - 1}));
      col += attrs.colspan - 1;
    } else if (row < map.height && pos == map.map[index + map.width]) {
      var cell = table.nodeAt(pos);
      var attrs$1 = cell.attrs;
      var copy = cell.type.create(
        Object.assign({}, attrs$1, {rowspan: cell.attrs.rowspan - 1}),
        cell.content
      );
      var newPos = map.positionAt(row + 1, col, table);
      tr.insert(tr.mapping.slice(mapFrom).map(tableStart + newPos), copy);
      col += attrs$1.colspan - 1;
    }
  }
}
function deleteRow(state, dispatch) {
  if (!isInTable(state))
    { return false; }
  if (dispatch) {
    var rect = selectedRect(state), tr = state.tr;
    if (rect.top == 0 && rect.bottom == rect.map.height)
      { return false; }
    for (var i = rect.bottom - 1; ; i--) {
      removeRow(tr, rect, i);
      if (i == rect.top)
        { break; }
      var table = rect.tableStart ? tr.doc.nodeAt(rect.tableStart - 1) : tr.doc;
      if (!table) {
        throw RangeError("No table found");
      }
      rect.table = table;
      rect.map = TableMap.get(rect.table);
    }
    dispatch(tr);
  }
  return true;
}
function deprecated_toggleHeader(type) {
  return function(state, dispatch) {
    if (!isInTable(state))
      { return false; }
    if (dispatch) {
      var types = tableNodeTypes(state.schema);
      var rect = selectedRect(state), tr = state.tr;
      var cells = rect.map.cellsInRect(
        type == "column" ? {
          left: rect.left,
          top: 0,
          right: rect.right,
          bottom: rect.map.height
        } : type == "row" ? {
          left: 0,
          top: rect.top,
          right: rect.map.width,
          bottom: rect.bottom
        } : rect
      );
      var nodes = cells.map(function (pos) { return rect.table.nodeAt(pos); });
      for (var i = 0; i < cells.length; i++)
        { if (nodes[i].type == types.header_cell)
          { tr.setNodeMarkup(
            rect.tableStart + cells[i],
            types.cell,
            nodes[i].attrs
          ); } }
      if (tr.steps.length == 0)
        { for (var i$1 = 0; i$1 < cells.length; i$1++)
          { tr.setNodeMarkup(
            rect.tableStart + cells[i$1],
            types.header_cell,
            nodes[i$1].attrs
          ); } }
      dispatch(tr);
    }
    return true;
  };
}
function isHeaderEnabledByType(type, rect, types) {
  var cellPositions = rect.map.cellsInRect({
    left: 0,
    top: 0,
    right: type == "row" ? rect.map.width : 1,
    bottom: type == "column" ? rect.map.height : 1
  });
  for (var i = 0; i < cellPositions.length; i++) {
    var cell = rect.table.nodeAt(cellPositions[i]);
    if (cell && cell.type !== types.header_cell) {
      return false;
    }
  }
  return true;
}
function toggleHeader(type, options) {
  options = options || { useDeprecatedLogic: false };
  if (options.useDeprecatedLogic)
    { return deprecated_toggleHeader(type); }
  return function(state, dispatch) {
    if (!isInTable(state))
      { return false; }
    if (dispatch) {
      var types = tableNodeTypes(state.schema);
      var rect = selectedRect(state), tr = state.tr;
      var isHeaderRowEnabled = isHeaderEnabledByType("row", rect, types);
      var isHeaderColumnEnabled = isHeaderEnabledByType(
        "column",
        rect,
        types
      );
      var isHeaderEnabled = type === "column" ? isHeaderRowEnabled : type === "row" ? isHeaderColumnEnabled : false;
      var selectionStartsAt = isHeaderEnabled ? 1 : 0;
      var cellsRect = type == "column" ? {
        left: 0,
        top: selectionStartsAt,
        right: 1,
        bottom: rect.map.height
      } : type == "row" ? {
        left: selectionStartsAt,
        top: 0,
        right: rect.map.width,
        bottom: 1
      } : rect;
      var newType = type == "column" ? isHeaderColumnEnabled ? types.cell : types.header_cell : type == "row" ? isHeaderRowEnabled ? types.cell : types.header_cell : types.cell;
      rect.map.cellsInRect(cellsRect).forEach(function (relativeCellPos) {
        var cellPos = relativeCellPos + rect.tableStart;
        var cell = tr.doc.nodeAt(cellPos);
        if (cell) {
          tr.setNodeMarkup(cellPos, newType, cell.attrs);
        }
      });
      dispatch(tr);
    }
    return true;
  };
}
var toggleHeaderRow = toggleHeader("row", {
  useDeprecatedLogic: true
});
toggleHeader("column", {
  useDeprecatedLogic: true
});
toggleHeader("cell", {
  useDeprecatedLogic: true
});
function findNextCell($cell, dir) {
  if (dir < 0) {
    var before = $cell.nodeBefore;
    if (before)
      { return $cell.pos - before.nodeSize; }
    for (var row = $cell.index(-1) - 1, rowEnd = $cell.before(); row >= 0; row--) {
      var rowNode = $cell.node(-1).child(row);
      var lastChild = rowNode.lastChild;
      if (lastChild) {
        return rowEnd - 1 - lastChild.nodeSize;
      }
      rowEnd -= rowNode.nodeSize;
    }
  } else {
    if ($cell.index() < $cell.parent.childCount - 1) {
      return $cell.pos + $cell.nodeAfter.nodeSize;
    }
    var table = $cell.node(-1);
    for (var row$1 = $cell.indexAfter(-1), rowStart = $cell.after(); row$1 < table.childCount; row$1++) {
      var rowNode$1 = table.child(row$1);
      if (rowNode$1.childCount)
        { return rowStart + 1; }
      rowStart += rowNode$1.nodeSize;
    }
  }
  return null;
}
function goToNextCell(direction) {
  return function(state, dispatch) {
    if (!isInTable(state))
      { return false; }
    var cell = findNextCell(selectionCell(state), direction);
    if (cell == null)
      { return false; }
    if (dispatch) {
      var $cell = state.doc.resolve(cell);
      dispatch(
        state.tr.setSelection(TextSelection.between($cell, moveCellForward($cell))).scrollIntoView()
      );
    }
    return true;
  };
}
function deleteTable(state, dispatch) {
  var $pos = state.selection.$anchor;
  for (var d = $pos.depth; d > 0; d--) {
    var node = $pos.node(d);
    if (node.type.spec.tableRole == "table") {
      if (dispatch)
        { dispatch(
          state.tr.delete($pos.before(d), $pos.after(d)).scrollIntoView()
        ); }
      return true;
    }
  }
  return false;
}
function deleteCellSelection(state, dispatch) {
  var sel = state.selection;
  if (!(sel instanceof CellSelection))
    { return false; }
  if (dispatch) {
    var tr = state.tr;
    var baseContent = tableNodeTypes(state.schema).cell.createAndFill().content;
    sel.forEachCell(function (cell, pos) {
      if (!cell.content.eq(baseContent))
        { tr.replace(
          tr.mapping.map(pos + 1),
          tr.mapping.map(pos + cell.nodeSize - 1),
          new Slice(baseContent, 0, 0)
        ); }
    });
    if (tr.docChanged)
      { dispatch(tr); }
  }
  return true;
}
function pastedCells(slice) {
  if (!slice.size)
    { return null; }
  var content = slice.content;
  var openStart = slice.openStart;
  var openEnd = slice.openEnd;
  while (content.childCount == 1 && (openStart > 0 && openEnd > 0 || content.child(0).type.spec.tableRole == "table")) {
    openStart--;
    openEnd--;
    content = content.child(0).content;
  }
  var first = content.child(0);
  var role = first.type.spec.tableRole;
  var schema = first.type.schema, rows = [];
  if (role == "row") {
    for (var i = 0; i < content.childCount; i++) {
      var cells = content.child(i).content;
      var left = i ? 0 : Math.max(0, openStart - 1);
      var right = i < content.childCount - 1 ? 0 : Math.max(0, openEnd - 1);
      if (left || right)
        { cells = fitSlice(
          tableNodeTypes(schema).row,
          new Slice(cells, left, right)
        ).content; }
      rows.push(cells);
    }
  } else if (role == "cell" || role == "header_cell") {
    rows.push(
      openStart || openEnd ? fitSlice(
        tableNodeTypes(schema).row,
        new Slice(content, openStart, openEnd)
      ).content : content
    );
  } else {
    return null;
  }
  return ensureRectangular(schema, rows);
}
function ensureRectangular(schema, rows) {
  var widths = [];
  for (var i = 0; i < rows.length; i++) {
    var row = rows[i];
    for (var j = row.childCount - 1; j >= 0; j--) {
      var ref = row.child(j).attrs;
      var rowspan = ref.rowspan;
      var colspan = ref.colspan;
      for (var r = i; r < i + rowspan; r++)
        { widths[r] = (widths[r] || 0) + colspan; }
    }
  }
  var width = 0;
  for (var r$1 = 0; r$1 < widths.length; r$1++)
    { width = Math.max(width, widths[r$1]); }
  for (var r$2 = 0; r$2 < widths.length; r$2++) {
    if (r$2 >= rows.length)
      { rows.push(Fragment.empty); }
    if (widths[r$2] < width) {
      var empty = tableNodeTypes(schema).cell.createAndFill();
      var cells = [];
      for (var i$1 = widths[r$2]; i$1 < width; i$1++) {
        cells.push(empty);
      }
      rows[r$2] = rows[r$2].append(Fragment.from(cells));
    }
  }
  return { height: rows.length, width: width, rows: rows };
}
function fitSlice(nodeType, slice) {
  var node = nodeType.createAndFill();
  var tr = new Transform(node).replace(0, node.content.size, slice);
  return tr.doc;
}
function clipCells(ref, newWidth, newHeight) {
  var width = ref.width;
  var height = ref.height;
  var rows = ref.rows;

  if (width != newWidth) {
    var added = [];
    var newRows = [];
    for (var row = 0; row < rows.length; row++) {
      var frag = rows[row], cells = [];
      for (var col = added[row] || 0, i = 0; col < newWidth; i++) {
        var cell = frag.child(i % frag.childCount);
        if (col + cell.attrs.colspan > newWidth)
          { cell = cell.type.createChecked(
            removeColSpan(
              cell.attrs,
              cell.attrs.colspan,
              col + cell.attrs.colspan - newWidth
            ),
            cell.content
          ); }
        cells.push(cell);
        col += cell.attrs.colspan;
        for (var j = 1; j < cell.attrs.rowspan; j++)
          { added[row + j] = (added[row + j] || 0) + cell.attrs.colspan; }
      }
      newRows.push(Fragment.from(cells));
    }
    rows = newRows;
    width = newWidth;
  }
  if (height != newHeight) {
    var newRows$1 = [];
    for (var row$1 = 0, i$1 = 0; row$1 < newHeight; row$1++, i$1++) {
      var cells$1 = [], source = rows[i$1 % height];
      for (var j$1 = 0; j$1 < source.childCount; j$1++) {
        var cell$1 = source.child(j$1);
        if (row$1 + cell$1.attrs.rowspan > newHeight)
          { cell$1 = cell$1.type.create(
            Object.assign({}, cell$1.attrs,
              {rowspan: Math.max(1, newHeight - cell$1.attrs.rowspan)}),
            cell$1.content
          ); }
        cells$1.push(cell$1);
      }
      newRows$1.push(Fragment.from(cells$1));
    }
    rows = newRows$1;
    height = newHeight;
  }
  return { width: width, height: height, rows: rows };
}
function growTable(tr, map, table, start, width, height, mapFrom) {
  var schema = tr.doc.type.schema;
  var types = tableNodeTypes(schema);
  var empty;
  var emptyHead;
  if (width > map.width) {
    for (var row = 0, rowEnd = 0; row < map.height; row++) {
      var rowNode = table.child(row);
      rowEnd += rowNode.nodeSize;
      var cells = [];
      var add = (void 0);
      if (rowNode.lastChild == null || rowNode.lastChild.type == types.cell)
        { add = empty || (empty = types.cell.createAndFill()); }
      else
        { add = emptyHead || (emptyHead = types.header_cell.createAndFill()); }
      for (var i = map.width; i < width; i++)
        { cells.push(add); }
      tr.insert(tr.mapping.slice(mapFrom).map(rowEnd - 1 + start), cells);
    }
  }
  if (height > map.height) {
    var cells$1 = [];
    for (var i$1 = 0, start2 = (map.height - 1) * map.width; i$1 < Math.max(map.width, width); i$1++) {
      var header = i$1 >= map.width ? false : table.nodeAt(map.map[start2 + i$1]).type == types.header_cell;
      cells$1.push(
        header ? emptyHead || (emptyHead = types.header_cell.createAndFill()) : empty || (empty = types.cell.createAndFill())
      );
    }
    var emptyRow = types.row.create(null, Fragment.from(cells$1)), rows = [];
    for (var i$2 = map.height; i$2 < height; i$2++)
      { rows.push(emptyRow); }
    tr.insert(tr.mapping.slice(mapFrom).map(start + table.nodeSize - 2), rows);
  }
  return !!(empty || emptyHead);
}
function isolateHorizontal(tr, map, table, start, left, right, top, mapFrom) {
  if (top == 0 || top == map.height)
    { return false; }
  var found = false;
  for (var col = left; col < right; col++) {
    var index = top * map.width + col, pos = map.map[index];
    if (map.map[index - map.width] == pos) {
      found = true;
      var cell = table.nodeAt(pos);
      var ref = map.findCell(pos);
      var cellTop = ref.top;
      var cellLeft = ref.left;
      tr.setNodeMarkup(tr.mapping.slice(mapFrom).map(pos + start), null, Object.assign({}, cell.attrs,
        {rowspan: top - cellTop}));
      tr.insert(
        tr.mapping.slice(mapFrom).map(map.positionAt(top, cellLeft, table)),
        cell.type.createAndFill(Object.assign({}, cell.attrs,
          {rowspan: cellTop + cell.attrs.rowspan - top}))
      );
      col += cell.attrs.colspan - 1;
    }
  }
  return found;
}
function isolateVertical(tr, map, table, start, top, bottom, left, mapFrom) {
  if (left == 0 || left == map.width)
    { return false; }
  var found = false;
  for (var row = top; row < bottom; row++) {
    var index = row * map.width + left, pos = map.map[index];
    if (map.map[index - 1] == pos) {
      found = true;
      var cell = table.nodeAt(pos);
      var cellLeft = map.colCount(pos);
      var updatePos = tr.mapping.slice(mapFrom).map(pos + start);
      tr.setNodeMarkup(
        updatePos,
        null,
        removeColSpan(
          cell.attrs,
          left - cellLeft,
          cell.attrs.colspan - (left - cellLeft)
        )
      );
      tr.insert(
        updatePos + cell.nodeSize,
        cell.type.createAndFill(
          removeColSpan(cell.attrs, 0, left - cellLeft)
        )
      );
      row += cell.attrs.rowspan - 1;
    }
  }
  return found;
}
function insertCells(state, dispatch, tableStart, rect, cells) {
  var table = tableStart ? state.doc.nodeAt(tableStart - 1) : state.doc;
  if (!table) {
    throw new Error("No table found");
  }
  var map = TableMap.get(table);
  var top = rect.top;
  var left = rect.left;
  var right = left + cells.width, bottom = top + cells.height;
  var tr = state.tr;
  var mapFrom = 0;
  function recomp() {
    table = tableStart ? tr.doc.nodeAt(tableStart - 1) : tr.doc;
    if (!table) {
      throw new Error("No table found");
    }
    map = TableMap.get(table);
    mapFrom = tr.mapping.maps.length;
  }
  if (growTable(tr, map, table, tableStart, right, bottom, mapFrom))
    { recomp(); }
  if (isolateHorizontal(tr, map, table, tableStart, left, right, top, mapFrom))
    { recomp(); }
  if (isolateHorizontal(tr, map, table, tableStart, left, right, bottom, mapFrom))
    { recomp(); }
  if (isolateVertical(tr, map, table, tableStart, top, bottom, left, mapFrom))
    { recomp(); }
  if (isolateVertical(tr, map, table, tableStart, top, bottom, right, mapFrom))
    { recomp(); }
  for (var row = top; row < bottom; row++) {
    var from = map.positionAt(row, left, table), to = map.positionAt(row, right, table);
    tr.replace(
      tr.mapping.slice(mapFrom).map(from + tableStart),
      tr.mapping.slice(mapFrom).map(to + tableStart),
      new Slice(cells.rows[row - top], 0, 0)
    );
  }
  recomp();
  tr.setSelection(
    new CellSelection(
      tr.doc.resolve(tableStart + map.positionAt(top, left, table)),
      tr.doc.resolve(tableStart + map.positionAt(bottom - 1, right - 1, table))
    )
  );
  dispatch(tr);
}

// src/input.ts
var handleKeyDown$1 = keydownHandler({
  ArrowLeft: arrow$1("horiz", -1),
  ArrowRight: arrow$1("horiz", 1),
  ArrowUp: arrow$1("vert", -1),
  ArrowDown: arrow$1("vert", 1),
  "Shift-ArrowLeft": shiftArrow("horiz", -1),
  "Shift-ArrowRight": shiftArrow("horiz", 1),
  "Shift-ArrowUp": shiftArrow("vert", -1),
  "Shift-ArrowDown": shiftArrow("vert", 1),
  Backspace: deleteCellSelection,
  "Mod-Backspace": deleteCellSelection,
  Delete: deleteCellSelection,
  "Mod-Delete": deleteCellSelection
});
function maybeSetSelection(state, dispatch, selection) {
  if (selection.eq(state.selection))
    { return false; }
  if (dispatch)
    { dispatch(state.tr.setSelection(selection).scrollIntoView()); }
  return true;
}
function arrow$1(axis, dir) {
  return function (state, dispatch, view) {
    if (!view)
      { return false; }
    var sel = state.selection;
    if (sel instanceof CellSelection) {
      return maybeSetSelection(
        state,
        dispatch,
        Selection.near(sel.$headCell, dir)
      );
    }
    if (axis != "horiz" && !sel.empty)
      { return false; }
    var end = atEndOfCell(view, axis, dir);
    if (end == null)
      { return false; }
    if (axis == "horiz") {
      return maybeSetSelection(
        state,
        dispatch,
        Selection.near(state.doc.resolve(sel.head + dir), dir)
      );
    } else {
      var $cell = state.doc.resolve(end);
      var $next = nextCell($cell, axis, dir);
      var newSel;
      if ($next)
        { newSel = Selection.near($next, 1); }
      else if (dir < 0)
        { newSel = Selection.near(state.doc.resolve($cell.before(-1)), -1); }
      else
        { newSel = Selection.near(state.doc.resolve($cell.after(-1)), 1); }
      return maybeSetSelection(state, dispatch, newSel);
    }
  };
}
function shiftArrow(axis, dir) {
  return function (state, dispatch, view) {
    if (!view)
      { return false; }
    var sel = state.selection;
    var cellSel;
    if (sel instanceof CellSelection) {
      cellSel = sel;
    } else {
      var end = atEndOfCell(view, axis, dir);
      if (end == null)
        { return false; }
      cellSel = new CellSelection(state.doc.resolve(end));
    }
    var $head = nextCell(cellSel.$headCell, axis, dir);
    if (!$head)
      { return false; }
    return maybeSetSelection(
      state,
      dispatch,
      new CellSelection(cellSel.$anchorCell, $head)
    );
  };
}
function handleTripleClick(view, pos) {
  var doc = view.state.doc, $cell = cellAround(doc.resolve(pos));
  if (!$cell)
    { return false; }
  view.dispatch(view.state.tr.setSelection(new CellSelection($cell)));
  return true;
}
function handlePaste(view, _, slice) {
  if (!isInTable(view.state))
    { return false; }
  var cells = pastedCells(slice);
  var sel = view.state.selection;
  if (sel instanceof CellSelection) {
    if (!cells)
      { cells = {
        width: 1,
        height: 1,
        rows: [
          Fragment.from(
            fitSlice(tableNodeTypes(view.state.schema).cell, slice)
          )
        ]
      }; }
    var table = sel.$anchorCell.node(-1);
    var start = sel.$anchorCell.start(-1);
    var rect = TableMap.get(table).rectBetween(
      sel.$anchorCell.pos - start,
      sel.$headCell.pos - start
    );
    cells = clipCells(cells, rect.right - rect.left, rect.bottom - rect.top);
    insertCells(view.state, view.dispatch, start, rect, cells);
    return true;
  } else if (cells) {
    var $cell = selectionCell(view.state);
    var start$1 = $cell.start(-1);
    insertCells(
      view.state,
      view.dispatch,
      start$1,
      TableMap.get($cell.node(-1)).findCell($cell.pos - start$1),
      cells
    );
    return true;
  } else {
    return false;
  }
}
function handleMouseDown(view, startEvent) {
  var _a;
  if (startEvent.ctrlKey || startEvent.metaKey)
    { return; }
  var startDOMCell = domInCell(view, startEvent.target);
  var $anchor;
  if (startEvent.shiftKey && view.state.selection instanceof CellSelection) {
    setCellSelection(view.state.selection.$anchorCell, startEvent);
    startEvent.preventDefault();
  } else if (startEvent.shiftKey && startDOMCell && ($anchor = cellAround(view.state.selection.$anchor)) != null && ((_a = cellUnderMouse(view, startEvent)) == null ? void 0 : _a.pos) != $anchor.pos) {
    setCellSelection($anchor, startEvent);
    startEvent.preventDefault();
  } else if (!startDOMCell) {
    return;
  }
  function setCellSelection($anchor2, event) {
    var $head = cellUnderMouse(view, event);
    var starting = tableEditingKey.getState(view.state) == null;
    if (!$head || !inSameTable($anchor2, $head)) {
      if (starting)
        { $head = $anchor2; }
      else
        { return; }
    }
    var selection = new CellSelection($anchor2, $head);
    if (starting || !view.state.selection.eq(selection)) {
      var tr = view.state.tr.setSelection(selection);
      if (starting)
        { tr.setMeta(tableEditingKey, $anchor2.pos); }
      view.dispatch(tr);
    }
  }
  function stop() {
    view.root.removeEventListener("mouseup", stop);
    view.root.removeEventListener("dragstart", stop);
    view.root.removeEventListener("mousemove", move);
    if (tableEditingKey.getState(view.state) != null)
      { view.dispatch(view.state.tr.setMeta(tableEditingKey, -1)); }
  }
  function move(_event) {
    var event = _event;
    var anchor = tableEditingKey.getState(view.state);
    var $anchor2;
    if (anchor != null) {
      $anchor2 = view.state.doc.resolve(anchor);
    } else if (domInCell(view, event.target) != startDOMCell) {
      $anchor2 = cellUnderMouse(view, startEvent);
      if (!$anchor2)
        { return stop(); }
    }
    if ($anchor2)
      { setCellSelection($anchor2, event); }
  }
  view.root.addEventListener("mouseup", stop);
  view.root.addEventListener("dragstart", stop);
  view.root.addEventListener("mousemove", move);
}
function atEndOfCell(view, axis, dir) {
  if (!(view.state.selection instanceof TextSelection))
    { return null; }
  var ref = view.state.selection;
  var $head = ref.$head;
  for (var d = $head.depth - 1; d >= 0; d--) {
    var parent = $head.node(d), index = dir < 0 ? $head.index(d) : $head.indexAfter(d);
    if (index != (dir < 0 ? 0 : parent.childCount))
      { return null; }
    if (parent.type.spec.tableRole == "cell" || parent.type.spec.tableRole == "header_cell") {
      var cellPos = $head.before(d);
      var dirStr = axis == "vert" ? dir > 0 ? "down" : "up" : dir > 0 ? "right" : "left";
      return view.endOfTextblock(dirStr) ? cellPos : null;
    }
  }
  return null;
}
function domInCell(view, dom) {
  for (; dom && dom != view.dom; dom = dom.parentNode) {
    if (dom.nodeName == "TD" || dom.nodeName == "TH") {
      return dom;
    }
  }
  return null;
}
function cellUnderMouse(view, event) {
  var mousePos = view.posAtCoords({
    left: event.clientX,
    top: event.clientY
  });
  if (!mousePos)
    { return null; }
  return mousePos ? cellAround(view.state.doc.resolve(mousePos.pos)) : null;
}

// src/columnresizing.ts
new PluginKey(
  "tableColumnResizing"
);

// src/index.ts
function tableEditing(ref) {
  if ( ref === void 0 ) ref = {};
  var allowTableNodeSelection = ref.allowTableNodeSelection; if ( allowTableNodeSelection === void 0 ) allowTableNodeSelection = false;

  return new Plugin({
    key: tableEditingKey,
    // This piece of state is used to remember when a mouse-drag
    // cell-selection is happening, so that it can continue even as
    // transactions (which might move its anchor cell) come in.
    state: {
      init: function init() {
        return null;
      },
      apply: function apply(tr, cur) {
        var set = tr.getMeta(tableEditingKey);
        if (set != null)
          { return set == -1 ? null : set; }
        if (cur == null || !tr.docChanged)
          { return cur; }
        var ref = tr.mapping.mapResult(cur);
        var deleted = ref.deleted;
        var pos = ref.pos;
        return deleted ? null : pos;
      }
    },
    props: {
      decorations: drawCellSelection,
      handleDOMEvents: {
        mousedown: handleMouseDown
      },
      createSelectionBetween: function createSelectionBetween(view) {
        return tableEditingKey.getState(view.state) != null ? view.state.selection : null;
      },
      handleTripleClick: handleTripleClick,
      handleKeyDown: handleKeyDown$1,
      handlePaste: handlePaste
    },
    appendTransaction: function appendTransaction(_, oldState, state) {
      return normalizeSelection(
        state,
        fixTables(state, oldState),
        allowTableNodeSelection
      );
    }
  });
}/**
Returns a command function that wraps the selection in a list with
the given type an attributes. If `dispatch` is null, only return a
value to indicate whether this is possible, but don't actually
perform the change.
*/
function wrapInList(listType, attrs) {
    if ( attrs === void 0 ) attrs = null;

    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        var range = $from.blockRange($to), doJoin = false, outerRange = range;
        if (!range)
            { return false; }
        // This is at the top of an existing list item
        if (range.depth >= 2 && $from.node(range.depth - 1).type.compatibleContent(listType) && range.startIndex == 0) {
            // Don't do anything if this is the top of the list
            if ($from.index(range.depth - 1) == 0)
                { return false; }
            var $insert = state.doc.resolve(range.start - 2);
            outerRange = new NodeRange($insert, $insert, range.depth);
            if (range.endIndex < range.parent.childCount)
                { range = new NodeRange($from, state.doc.resolve($to.end(range.depth)), range.depth); }
            doJoin = true;
        }
        var wrap = findWrapping(outerRange, listType, attrs, range);
        if (!wrap)
            { return false; }
        if (dispatch)
            { dispatch(doWrapInList(state.tr, range, wrap, doJoin, listType).scrollIntoView()); }
        return true;
    };
}
function doWrapInList(tr, range, wrappers, joinBefore, listType) {
    var content = Fragment.empty;
    for (var i = wrappers.length - 1; i >= 0; i--)
        { content = Fragment.from(wrappers[i].type.create(wrappers[i].attrs, content)); }
    tr.step(new ReplaceAroundStep(range.start - (joinBefore ? 2 : 0), range.end, range.start, range.end, new Slice(content, 0, 0), wrappers.length, true));
    var found = 0;
    for (var i$1 = 0; i$1 < wrappers.length; i$1++)
        { if (wrappers[i$1].type == listType)
            { found = i$1 + 1; } }
    var splitDepth = wrappers.length - found;
    var splitPos = range.start + wrappers.length - (joinBefore ? 2 : 0), parent = range.parent;
    for (var i$2 = range.startIndex, e = range.endIndex, first = true; i$2 < e; i$2++, first = false) {
        if (!first && canSplit(tr.doc, splitPos, splitDepth)) {
            tr.split(splitPos, splitDepth);
            splitPos += 2 * splitDepth;
        }
        splitPos += parent.child(i$2).nodeSize;
    }
    return tr;
}
/**
Build a command that splits a non-empty textblock at the top level
of a list item by also splitting that list item.
*/
function splitListItem(itemType, itemAttrs) {
    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        var node = ref.node;
        if ((node && node.isBlock) || $from.depth < 2 || !$from.sameParent($to))
            { return false; }
        var grandParent = $from.node(-1);
        if (grandParent.type != itemType)
            { return false; }
        if ($from.parent.content.size == 0 && $from.node(-1).childCount == $from.indexAfter(-1)) {
            // In an empty block. If this is a nested list, the wrapping
            // list item should be split. Otherwise, bail out and let next
            // command handle lifting.
            if ($from.depth == 3 || $from.node(-3).type != itemType ||
                $from.index(-2) != $from.node(-2).childCount - 1)
                { return false; }
            if (dispatch) {
                var wrap = Fragment.empty;
                var depthBefore = $from.index(-1) ? 1 : $from.index(-2) ? 2 : 3;
                // Build a fragment containing empty versions of the structure
                // from the outer list item to the parent node of the cursor
                for (var d = $from.depth - depthBefore; d >= $from.depth - 3; d--)
                    { wrap = Fragment.from($from.node(d).copy(wrap)); }
                var depthAfter = $from.indexAfter(-1) < $from.node(-2).childCount ? 1
                    : $from.indexAfter(-2) < $from.node(-3).childCount ? 2 : 3;
                // Add a second list item with an empty default start node
                wrap = wrap.append(Fragment.from(itemType.createAndFill()));
                var start = $from.before($from.depth - (depthBefore - 1));
                var tr$1 = state.tr.replace(start, $from.after(-depthAfter), new Slice(wrap, 4 - depthBefore, 0));
                var sel = -1;
                tr$1.doc.nodesBetween(start, tr$1.doc.content.size, function (node, pos) {
                    if (sel > -1)
                        { return false; }
                    if (node.isTextblock && node.content.size == 0)
                        { sel = pos + 1; }
                });
                if (sel > -1)
                    { tr$1.setSelection(Selection.near(tr$1.doc.resolve(sel))); }
                dispatch(tr$1.scrollIntoView());
            }
            return true;
        }
        var nextType = $to.pos == $from.end() ? grandParent.contentMatchAt(0).defaultType : null;
        var tr = state.tr.delete($from.pos, $to.pos);
        var types = nextType ? [itemAttrs ? { type: itemType, attrs: itemAttrs } : null, { type: nextType }] : undefined;
        if (!canSplit(tr.doc, $from.pos, 2, types))
            { return false; }
        if (dispatch)
            { dispatch(tr.split($from.pos, 2, types).scrollIntoView()); }
        return true;
    };
}
/**
Create a command to lift the list item around the selection up into
a wrapping list.
*/
function liftListItem(itemType) {
    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        var range = $from.blockRange($to, function (node) { return node.childCount > 0 && node.firstChild.type == itemType; });
        if (!range)
            { return false; }
        if (!dispatch)
            { return true; }
        if ($from.node(range.depth - 1).type == itemType) // Inside a parent list
            { return liftToOuterList(state, dispatch, itemType, range); }
        else // Outer list node
            { return liftOutOfList(state, dispatch, range); }
    };
}
function liftToOuterList(state, dispatch, itemType, range) {
    var tr = state.tr, end = range.end, endOfList = range.$to.end(range.depth);
    if (end < endOfList) {
        // There are siblings after the lifted items, which must become
        // children of the last item
        tr.step(new ReplaceAroundStep(end - 1, endOfList, end, endOfList, new Slice(Fragment.from(itemType.create(null, range.parent.copy())), 1, 0), 1, true));
        range = new NodeRange(tr.doc.resolve(range.$from.pos), tr.doc.resolve(endOfList), range.depth);
    }
    var target = liftTarget(range);
    if (target == null)
        { return false; }
    tr.lift(range, target);
    var after = tr.mapping.map(end, -1) - 1;
    if (canJoin(tr.doc, after))
        { tr.join(after); }
    dispatch(tr.scrollIntoView());
    return true;
}
function liftOutOfList(state, dispatch, range) {
    var tr = state.tr, list = range.parent;
    // Merge the list items into a single big item
    for (var pos = range.end, i = range.endIndex - 1, e = range.startIndex; i > e; i--) {
        pos -= list.child(i).nodeSize;
        tr.delete(pos - 1, pos + 1);
    }
    var $start = tr.doc.resolve(range.start), item = $start.nodeAfter;
    if (tr.mapping.map(range.end) != range.start + $start.nodeAfter.nodeSize)
        { return false; }
    var atStart = range.startIndex == 0, atEnd = range.endIndex == list.childCount;
    var parent = $start.node(-1), indexBefore = $start.index(-1);
    if (!parent.canReplace(indexBefore + (atStart ? 0 : 1), indexBefore + 1, item.content.append(atEnd ? Fragment.empty : Fragment.from(list))))
        { return false; }
    var start = $start.pos, end = start + item.nodeSize;
    // Strip off the surrounding list. At the sides where we're not at
    // the end of the list, the existing list is closed. At sides where
    // this is the end, it is overwritten to its end.
    tr.step(new ReplaceAroundStep(start - (atStart ? 1 : 0), end + (atEnd ? 1 : 0), start + 1, end - 1, new Slice((atStart ? Fragment.empty : Fragment.from(list.copy(Fragment.empty)))
        .append(atEnd ? Fragment.empty : Fragment.from(list.copy(Fragment.empty))), atStart ? 0 : 1, atEnd ? 0 : 1), atStart ? 0 : 1));
    dispatch(tr.scrollIntoView());
    return true;
}
/**
Create a command to sink the list item around the selection down
into an inner list.
*/
function sinkListItem(itemType) {
    return function (state, dispatch) {
        var ref = state.selection;
        var $from = ref.$from;
        var $to = ref.$to;
        var range = $from.blockRange($to, function (node) { return node.childCount > 0 && node.firstChild.type == itemType; });
        if (!range)
            { return false; }
        var startIndex = range.startIndex;
        if (startIndex == 0)
            { return false; }
        var parent = range.parent, nodeBefore = parent.child(startIndex - 1);
        if (nodeBefore.type != itemType)
            { return false; }
        if (dispatch) {
            var nestedBefore = nodeBefore.lastChild && nodeBefore.lastChild.type == parent.type;
            var inner = Fragment.from(nestedBefore ? itemType.create() : null);
            var slice = new Slice(Fragment.from(itemType.create(null, Fragment.from(parent.type.create(null, inner)))), nestedBefore ? 3 : 1, 0);
            var before = range.start, after = range.end;
            dispatch(state.tr.step(new ReplaceAroundStep(before - (nestedBefore ? 3 : 1), after, before, after, slice, 1, true))
                .scrollIntoView());
        }
        return true;
    };
}var SVG = "http://www.w3.org/2000/svg";
var XLINK = "http://www.w3.org/1999/xlink";

var prefix$2 = "ProseMirror-icon";

var getIconSize = function (name) {
    var veryBigIcons = ['table', 'enlarge', 'shrink'];
    var biggerIcons = ['emoji'];
    var bigIcons = ['headline', 'join', 'image', 'undo', 'redo', 'markdown', 'angleDoubleRight', 'angleDoubleLeft'];
    var smallIcons = ['bulletList', 'orderedList', 'indent', 'outdent'];

    return smallIcons.includes(name)
        ? '18' : bigIcons.includes(name)
            ? '16' : biggerIcons.includes(name)
                ? '15' : veryBigIcons.includes(name) ? '14' : '17';
};

function getIcon(icon, htmlNode) {
    htmlNode = htmlNode || 'div';
    var node = document.createElement(htmlNode);
    node.className = prefix$2;

    if (htmlNode === "button") {
        node.setAttribute("type", "button");
    }

    if (icon.path) {
        var name = "pm-icon-" + icon.name;
        if (!document.getElementById(name)) { buildSVG(name, icon); }
        var svg = node.appendChild(document.createElementNS(SVG, "svg"));
        var iconSize = getIconSize(icon.name);
        svg.setAttribute('height', iconSize /*icon.height*/);
        svg.setAttribute('width', iconSize /*icon.width*/);
        var use = svg.appendChild(document.createElementNS(SVG, "use"));
        use.setAttributeNS(XLINK, "href", /([^#]*)/.exec(document.location)[1] + "#" + name);
    } else if (icon.dom) {
        node.appendChild(icon.dom.cloneNode(true));
    } else {
        node.appendChild(document.createElement("span")).textContent = icon.text || '';
        if (icon.css) { node.firstChild.style.cssText = icon.css; }
    }
    return node;
}

function buildSVG(name, data) {
    var collection = document.getElementById(prefix$2 + "-collection");
    if (!collection) {
        collection = document.createElementNS(SVG, "svg");
        collection.id = prefix$2 + "-collection";
        collection.style.display = "none";
        document.body.insertBefore(collection, document.body.firstChild);
    }
    var sym = document.createElementNS(SVG, "symbol");
    sym.id = name;
    sym.setAttribute("viewBox", "0 0 " + data.width + " " + data.height);

    var pathData = Array.isArray(data.path) ? data.path : [data.path];

    pathData.forEach(function (path) {
        var pathDom = sym.appendChild(document.createElementNS(SVG, "path"));
        pathDom.setAttribute("d", path);
        collection.appendChild(sym);
    });
}

// :: Object
// A set of basic editor-related icons. Contains the properties
// `join`, `lift`, `selectParentNode`, `undo`, `redo`, `strong`, `em`,
// `code`, `link`, `bulletList`, `orderedList`, and `blockquote`, each
// holding an object that can be used as the `icon` option to
// `MenuItem`.
var icons = {
    headline: {
        name: 'headline',
        width: 27, height: 27,
        path: "M26.281 26c-1.375 0-2.766-0.109-4.156-0.109-1.375 0-2.75 0.109-4.125 0.109-0.531 0-0.781-0.578-0.781-1.031 0-1.391 1.563-0.797 2.375-1.328 0.516-0.328 0.516-1.641 0.516-2.188l-0.016-6.109c0-0.172 0-0.328-0.016-0.484-0.25-0.078-0.531-0.063-0.781-0.063h-10.547c-0.266 0-0.547-0.016-0.797 0.063-0.016 0.156-0.016 0.313-0.016 0.484l-0.016 5.797c0 0.594 0 2.219 0.578 2.562 0.812 0.5 2.656-0.203 2.656 1.203 0 0.469-0.219 1.094-0.766 1.094-1.453 0-2.906-0.109-4.344-0.109-1.328 0-2.656 0.109-3.984 0.109-0.516 0-0.75-0.594-0.75-1.031 0-1.359 1.437-0.797 2.203-1.328 0.5-0.344 0.516-1.687 0.516-2.234l-0.016-0.891v-12.703c0-0.75 0.109-3.156-0.594-3.578-0.781-0.484-2.453 0.266-2.453-1.141 0-0.453 0.203-1.094 0.75-1.094 1.437 0 2.891 0.109 4.328 0.109 1.313 0 2.641-0.109 3.953-0.109 0.562 0 0.781 0.625 0.781 1.094 0 1.344-1.547 0.688-2.312 1.172-0.547 0.328-0.547 1.937-0.547 2.5l0.016 5c0 0.172 0 0.328 0.016 0.5 0.203 0.047 0.406 0.047 0.609 0.047h10.922c0.187 0 0.391 0 0.594-0.047 0.016-0.172 0.016-0.328 0.016-0.5l0.016-5c0-0.578 0-2.172-0.547-2.5-0.781-0.469-2.344 0.156-2.344-1.172 0-0.469 0.219-1.094 0.781-1.094 1.375 0 2.75 0.109 4.125 0.109 1.344 0 2.688-0.109 4.031-0.109 0.562 0 0.781 0.625 0.781 1.094 0 1.359-1.609 0.672-2.391 1.156-0.531 0.344-0.547 1.953-0.547 2.516l0.016 14.734c0 0.516 0.031 1.875 0.531 2.188 0.797 0.5 2.484-0.141 2.484 1.219 0 0.453-0.203 1.094-0.75 1.094z"
    },
    plus: {
        name: 'plus',
        width: 16, height: 16,
        path: "M8 2a.5.5 0 0 1 .5.5v5h5a.5.5 0 0 1 0 1h-5v5a.5.5 0 0 1-1 0v-5h-5a.5.5 0 0 1 0-1h5v-5A.5.5 0 0 1 8 2Z"
    },
    add: {
        name: 'add',
        width: 16, height: 16,
        path: [
            "M14 1a1 1 0 0 1 1 1v12a1 1 0 0 1-1 1H2a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h12zM2 0a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V2a2 2 0 0 0-2-2H2z",
            "M8 4a.5.5 0 0 1 .5.5v3h3a.5.5 0 0 1 0 1h-3v3a.5.5 0 0 1-1 0v-3h-3a.5.5 0 0 1 0-1h3v-3A.5.5 0 0 1 8 4z"
        ]
    },
    join: {
        name: 'join',
        width: 800, height: 900,
        path: "M0 75h800v125h-800z M0 825h800v-125h-800z M250 400h100v-100h100v100h100v100h-100v100h-100v-100h-100z"
    },
    text: {
        name: 'text',
        width: 16, height: 16,
        path: 'm2.244 13.081.943-2.803H6.66l.944 2.803H8.86L5.54 3.75H4.322L1 13.081h1.244zm2.7-7.923L6.34 9.314H3.51l1.4-4.156h.034zm9.146 7.027h.035v.896h1.128V8.125c0-1.51-1.114-2.345-2.646-2.345-1.736 0-2.59.916-2.666 2.174h1.108c.068-.718.595-1.19 1.517-1.19.971 0 1.518.52 1.518 1.464v.731H12.19c-1.647.007-2.522.8-2.522 2.058 0 1.319.957 2.18 2.345 2.18 1.06 0 1.716-.43 2.078-1.011zm-1.763.035c-.752 0-1.456-.397-1.456-1.244 0-.65.424-1.115 1.408-1.115h1.805v.834c0 .896-.752 1.525-1.757 1.525z'
    },
    strong: {
        name: 'strong',
        width: 16, height: 16,
        path: "M8.21 13c2.106 0 3.412-1.087 3.412-2.823 0-1.306-.984-2.283-2.324-2.386v-.055a2.176 2.176 0 0 0 1.852-2.14c0-1.51-1.162-2.46-3.014-2.46H3.843V13H8.21zM5.908 4.674h1.696c.963 0 1.517.451 1.517 1.244 0 .834-.629 1.32-1.73 1.32H5.908V4.673zm0 6.788V8.598h1.73c1.217 0 1.88.492 1.88 1.415 0 .943-.643 1.449-1.832 1.449H5.907z"
    },
    em: {
        name: 'em',
        width: 16, height: 16,
        path: "M7.991 11.674 9.53 4.455c.123-.595.246-.71 1.347-.807l.11-.52H7.211l-.11.52c1.06.096 1.128.212 1.005.807L6.57 11.674c-.123.595-.246.71-1.346.806l-.11.52h3.774l.11-.52c-1.06-.095-1.129-.211-1.006-.806z"
    },
    strikethrough: {
        name: 'strikethrough',
        width: 16, height: 16,
        path: "M6.333 5.686c0 .31.083.581.27.814H5.166a2.776 2.776 0 0 1-.099-.76c0-1.627 1.436-2.768 3.48-2.768 1.969 0 3.39 1.175 3.445 2.85h-1.23c-.11-1.08-.964-1.743-2.25-1.743-1.23 0-2.18.602-2.18 1.607zm2.194 7.478c-2.153 0-3.589-1.107-3.705-2.81h1.23c.144 1.06 1.129 1.703 2.544 1.703 1.34 0 2.31-.705 2.31-1.675 0-.827-.547-1.374-1.914-1.675L8.046 8.5H1v-1h14v1h-3.504c.468.437.675.994.675 1.697 0 1.826-1.436 2.967-3.644 2.967z"
    },
    code: {
        name: 'code',
        width: 16, height: 16,
        path: "M5.854 4.854a.5.5 0 1 0-.708-.708l-3.5 3.5a.5.5 0 0 0 0 .708l3.5 3.5a.5.5 0 0 0 .708-.708L2.707 8l3.147-3.146zm4.292 0a.5.5 0 0 1 .708-.708l3.5 3.5a.5.5 0 0 1 0 .708l-3.5 3.5a.5.5 0 0 1-.708-.708L13.293 8l-3.147-3.146z"
    },
    link: {
        name: 'link',
        width: 16, height: 16,
        path: [
            "M4.715 6.542 3.343 7.914a3 3 0 1 0 4.243 4.243l1.828-1.829A3 3 0 0 0 8.586 5.5L8 6.086a1.002 1.002 0 0 0-.154.199 2 2 0 0 1 .861 3.337L6.88 11.45a2 2 0 1 1-2.83-2.83l.793-.792a4.018 4.018 0 0 1-.128-1.287z",
            "M6.586 4.672A3 3 0 0 0 7.414 9.5l.775-.776a2 2 0 0 1-.896-3.346L9.12 3.55a2 2 0 1 1 2.83 2.83l-.793.792c.112.42.155.855.128 1.287l1.372-1.372a3 3 0 1 0-4.243-4.243L6.586 4.672z"
        ]
    },
    bulletList: {
        name: 'bulletList',
        width: 16, height: 16,
        path: "M5 11.5a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5zm-3 1a1 1 0 1 0 0-2 1 1 0 0 0 0 2zm0 4a1 1 0 1 0 0-2 1 1 0 0 0 0 2zm0 4a1 1 0 1 0 0-2 1 1 0 0 0 0 2z"
    },
    orderedList: {
        name: 'orderedList',
        width: 16, height: 16,
        path: [
            "M5 11.5a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5zm0-4a.5.5 0 0 1 .5-.5h9a.5.5 0 0 1 0 1h-9a.5.5 0 0 1-.5-.5z",
            "M1.713 11.865v-.474H2c.217 0 .363-.137.363-.317 0-.185-.158-.31-.361-.31-.223 0-.367.152-.373.31h-.59c.016-.467.373-.787.986-.787.588-.002.954.291.957.703a.595.595 0 0 1-.492.594v.033a.615.615 0 0 1 .569.631c.003.533-.502.8-1.051.8-.656 0-1-.37-1.008-.794h.582c.008.178.186.306.422.309.254 0 .424-.145.422-.35-.002-.195-.155-.348-.414-.348h-.3zm-.004-4.699h-.604v-.035c0-.408.295-.844.958-.844.583 0 .96.326.96.756 0 .389-.257.617-.476.848l-.537.572v.03h1.054V9H1.143v-.395l.957-.99c.138-.142.293-.304.293-.508 0-.18-.147-.32-.342-.32a.33.33 0 0 0-.342.338v.041zM2.564 5h-.635V2.924h-.031l-.598.42v-.567l.629-.443h.635V5z"
        ]
    },
    indent: {
        name: 'indent',
        width: 16, height: 16,
        path: "M2 3.5a.5.5 0 0 1 .5-.5h11a.5.5 0 0 1 0 1h-11a.5.5 0 0 1-.5-.5zm.646 2.146a.5.5 0 0 1 .708 0l2 2a.5.5 0 0 1 0 .708l-2 2a.5.5 0 0 1-.708-.708L4.293 8 2.646 6.354a.5.5 0 0 1 0-.708zM7 6.5a.5.5 0 0 1 .5-.5h6a.5.5 0 0 1 0 1h-6a.5.5 0 0 1-.5-.5zm0 3a.5.5 0 0 1 .5-.5h6a.5.5 0 0 1 0 1h-6a.5.5 0 0 1-.5-.5zm-5 3a.5.5 0 0 1 .5-.5h11a.5.5 0 0 1 0 1h-11a.5.5 0 0 1-.5-.5z"
    },
    outdent: {
        name: 'outdent',
        width: 16, height: 16,
        path: "M2 3.5a.5.5 0 0 1 .5-.5h11a.5.5 0 0 1 0 1h-11a.5.5 0 0 1-.5-.5zm10.646 2.146a.5.5 0 0 1 .708.708L11.707 8l1.647 1.646a.5.5 0 0 1-.708.708l-2-2a.5.5 0 0 1 0-.708l2-2zM2 6.5a.5.5 0 0 1 .5-.5h6a.5.5 0 0 1 0 1h-6a.5.5 0 0 1-.5-.5zm0 3a.5.5 0 0 1 .5-.5h6a.5.5 0 0 1 0 1h-6a.5.5 0 0 1-.5-.5zm0 3a.5.5 0 0 1 .5-.5h11a.5.5 0 0 1 0 1h-11a.5.5 0 0 1-.5-.5z"
    },
    blockquote: {
        name: 'blockquote',
        width: 16, height: 16,
        path: "M12 12a1 1 0 0 0 1-1V8.558a1 1 0 0 0-1-1h-1.388c0-.351.021-.703.062-1.054.062-.372.166-.703.31-.992.145-.29.331-.517.559-.683.227-.186.516-.279.868-.279V3c-.579 0-1.085.124-1.52.372a3.322 3.322 0 0 0-1.085.992 4.92 4.92 0 0 0-.62 1.458A7.712 7.712 0 0 0 9 7.558V11a1 1 0 0 0 1 1h2Zm-6 0a1 1 0 0 0 1-1V8.558a1 1 0 0 0-1-1H4.612c0-.351.021-.703.062-1.054.062-.372.166-.703.31-.992.145-.29.331-.517.559-.683.227-.186.516-.279.868-.279V3c-.579 0-1.085.124-1.52.372a3.322 3.322 0 0 0-1.085.992 4.92 4.92 0 0 0-.62 1.458A7.712 7.712 0 0 0 3 7.558V11a1 1 0 0 0 1 1h2Z"
    },
    table: {
        name: 'table',
        width: 18, height: 18,
        path: "M0 2a2 2 0 0 1 2-2h12a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V2zm15 2h-4v3h4V4zm0 4h-4v3h4V8zm0 4h-4v3h3a1 1 0 0 0 1-1v-2zm-5 3v-3H6v3h4zm-5 0v-3H1v2a1 1 0 0 0 1 1h3zm-4-4h4V8H1v3zm0-4h4V4H1v3zm5-3v3h4V4H6zm4 4H6v3h4V8z"
    },
    emoji: {
        name: 'emoji',
        width: 18, height: 18,
        path: [
            "M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z",
            "M4.285 9.567a.5.5 0 0 1 .683.183A3.498 3.498 0 0 0 8 11.5a3.498 3.498 0 0 0 3.032-1.75.5.5 0 1 1 .866.5A4.498 4.498 0 0 1 8 12.5a4.498 4.498 0 0 1-3.898-2.25.5.5 0 0 1 .183-.683zM7 6.5C7 7.328 6.552 8 6 8s-1-.672-1-1.5S5.448 5 6 5s1 .672 1 1.5zm4 0c0 .828-.448 1.5-1 1.5s-1-.672-1-1.5S9.448 5 10 5s1 .672 1 1.5z"
        ]
    },
    upload: {
        name: 'upload',
        width: 16, height: 16,
        path: [
            "M9.344 8.656h2l-3.344-3.313-3.344 3.313h2v2.688h2.688v-2.688zM12.906 6.688q1.281 0.094 2.188 1.047t0.906 2.266q0 1.375-0.984 2.359t-2.359 0.984h-8.656q-1.656 0-2.828-1.172t-1.172-2.828q0-1.469 1.047-2.641t2.516-1.328q0.656-1.219 1.844-1.969t2.594-0.75q1.688 0 3.141 1.188t1.766 2.844z"
        ]
    },
    undo: {
        name: 'undo',
        width: 16, height: 16,
        path: [
            "M8 3a5 5 0 1 1-4.546 2.914.5.5 0 0 0-.908-.417A6 6 0 1 0 8 2v1z",
            "M8 4.466V.534a.25.25 0 0 0-.41-.192L5.23 2.308a.25.25 0 0 0 0 .384l2.36 1.966A.25.25 0 0 0 8 4.466z"
        ]
    },
    redo: {
        name: 'redo',
        width: 16, height: 16,
        path: [
            "M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2v1z",
            "M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466z"
        ]
    },
    enlarge: {
        name: 'enlarge',
        width: 16, height: 16,
        path: "M5.828 10.172a.5.5 0 0 0-.707 0l-4.096 4.096V11.5a.5.5 0 0 0-1 0v3.975a.5.5 0 0 0 .5.5H4.5a.5.5 0 0 0 0-1H1.732l4.096-4.096a.5.5 0 0 0 0-.707zm4.344-4.344a.5.5 0 0 0 .707 0l4.096-4.096V4.5a.5.5 0 1 0 1 0V.525a.5.5 0 0 0-.5-.5H11.5a.5.5 0 0 0 0 1h2.768l-4.096 4.096a.5.5 0 0 0 0 .707z"
    },
    shrink: {
        name: 'shrink',
        width: 16, height: 16,
        path: "M.172 15.828a.5.5 0 0 0 .707 0l4.096-4.096V14.5a.5.5 0 1 0 1 0v-3.975a.5.5 0 0 0-.5-.5H1.5a.5.5 0 0 0 0 1h2.768L.172 15.121a.5.5 0 0 0 0 .707zM15.828.172a.5.5 0 0 0-.707 0l-4.096 4.096V1.5a.5.5 0 1 0-1 0v3.975a.5.5 0 0 0 .5.5H14.5a.5.5 0 0 0 0-1h-2.768L15.828.879a.5.5 0 0 0 0-.707z"
    },
    angleDoubleRight: {
        name: 'angleDoubleRight',
        width: 16, height: 16,
        path: [
            "M3.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L9.293 8 3.646 2.354a.5.5 0 0 1 0-.708z",
            "M7.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L13.293 8 7.646 2.354a.5.5 0 0 1 0-.708z"
        ]
    },
    angleDoubleLeft: {
        name: 'angleDoubleLeft',
        width: 16, height: 16,
        path: [
            "M8.354 1.646a.5.5 0 0 1 0 .708L2.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0z",
            "M12.354 1.646a.5.5 0 0 1 0 .708L6.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0z"
        ]
    },
    markdown: {
        name: 'markdown',
        width: 16, height: 16,
        path: [
            "M14 3a1 1 0 0 1 1 1v8a1 1 0 0 1-1 1H2a1 1 0 0 1-1-1V4a1 1 0 0 1 1-1h12zM2 2a2 2 0 0 0-2 2v8a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V4a2 2 0 0 0-2-2H2z",
            "M9.146 8.146a.5.5 0 0 1 .708 0L11.5 9.793l1.646-1.647a.5.5 0 0 1 .708.708l-2 2a.5.5 0 0 1-.708 0l-2-2a.5.5 0 0 1 0-.708z",
            "M11.5 5a.5.5 0 0 1 .5.5v4a.5.5 0 0 1-1 0v-4a.5.5 0 0 1 .5-.5z",
            "M3.56 11V7.01h.056l1.428 3.239h.774l1.42-3.24h.056V11h1.073V5.001h-1.2l-1.71 3.894h-.039l-1.71-3.894H2.5V11h1.06z"
        ]
    },
    horizontalRule: {
        name: 'horizontalRule',
        width: 16, height: 16,
        path: [
            "M0 6.5v3c0 0.276 0.224 0.5 0.5 0.5h15c0.276 0 0.5-0.224 0.5-0.5v-3c0-0.276-0.224-0.5-0.5-0.5h-15c-0.276 0-0.5 0.224-0.5 0.5z"
        ]
    },
    selectParentNode: {name: 'selectParentNode',text: "\u2b1a", css: "font-weight: bold"}
};var PREFIX = "ProseMirror-menu";

// Work around classList.toggle being broken in IE11
function setClass(dom, cls, on) {
    if (on) { dom.classList.add(cls); }
    else { dom.classList.remove(cls); }
}

function setAttribute(dom, attr, value, on) {
    if (on) { dom.setAttribute(attr, value); }
    else { dom.removeAttribute(attr); }
}

function translate$1(view, text) {
    return view._props.translate ? view._props.translate(text) : text;
}

function addClassId(dom, options) {
    if (options.id) {
        addMenuClass(dom, options);
    }
}

function initMenuItemTrigger(view, options) {
    var trigger = null;
    if (options.icon) {
        trigger = getIcon(options.icon, options.htmlNode);
    } else if (options.label) {
        trigger = document.createElement(options.htmlNode);
        trigger.innerHTML = translate$1(view, options.label).replace('</i>', '</i> ');
    } else {
        return null;
    }

    trigger.classList.add(buildMenuClass('trigger'));

    if (trigger) {
        setAttributesFromOptions(trigger, options);
    }

    return trigger;
}

function setTabindex(dom, options) {
    if (typeof options.tabindex !== 'undefined') {
        dom.setAttribute('tabindex', options.tabindex);
    }
}

function setTitle(dom, options, view) {
    if (options.title !== 'undefined') {
        var title = (typeof options.title === "function" ? options.title(view.state) : options.title);
        dom.setAttribute("title", translate$1(view, title));
    }
}

function setAttributesFromOptions(dom, options) {
    if (options.class) { dom.classList.add(options.class); }
    if (options.css) { dom.style.cssText += options.css; }
    if (options.seperator) {
        dom.classList.add('seperator');
    }
    addClassId(dom, options);
}

function buildMenuClass(suffix) {
    if (typeof suffix === 'object') {
        suffix = suffix.id;
    }

    return PREFIX + '-' + suffix;
}

function addMenuClass(dom, suffix) {
    dom.classList.add(buildMenuClass(suffix));
}

var lastMenuEvent = {time: 0, node: null};

function markMenuEvent(e) {
    lastMenuEvent.time = Date.now();
    lastMenuEvent.node = e.target;
}

function isMenuEvent(wrapper) {
    return Date.now() - 100 < lastMenuEvent.time &&
        lastMenuEvent.node && wrapper.contains(lastMenuEvent.node);
}// ::- An icon or label that, when clicked, executes a command.

// MenuItemSpec:: interface
// The configuration object passed to the `MenuItem` constructor.
//
//   run:: (EditorState, (Transaction), EditorView, dom.Event)
//   The function to execute when the menu item is activated.
//
//   select:: ?(EditorState) → bool
//   Optional function that is used to determine whether the item is
//   appropriate at the moment. Deselected items will be hidden.
//
//   enable:: ?(EditorState) → bool
//   Function that is used to determine if the item is enabled. If
//   given and returning false, the item will be given a disabled
//   styling.
//
//   active:: ?(EditorState) → bool
//   A predicate function to determine whether the item is 'active' (for
//   example, the item for toggling the strong mark might be active then
//   the cursor is in strong text).
//
//   render:: ?(EditorView) → dom.Node
//   A function that renders the item. You must provide either this,
//   [`icon`](#menu.MenuItemSpec.icon), or [`label`](#MenuItemSpec.label).
//
//   icon:: ?Object
//   Describes an icon to show for this item. The object may specify
//   an SVG icon, in which case its `path` property should be an [SVG
//   path
//   spec](https://developer.mozilla.org/en-US/docs/Web/SVG/Attribute/d),
//   and `width` and `height` should provide the viewbox in which that
//   path exists. Alternatively, it may have a `text` property
//   specifying a string of text that makes up the icon, with an
//   optional `css` property giving additional CSS styling for the
//   text. _Or_ it may contain `dom` property containing a DOM node.
//
//   label:: ?string
//   Makes the item show up as a text label. Mostly useful for items
//   wrapped in a [drop-down](#menu.Dropdown) or similar menu. The object
//   should have a `label` property providing the text to display.
//
//   title:: ?union<string, (EditorState) → string>
//   Defines DOM title (mouseover) text for the item.
//
//   class:: string
//   Optionally adds a CSS class to the item's DOM representation.
//
//   css:: string
//   Optionally adds a string of inline CSS to the item's DOM
//   representation.
//
//   execEvent:: string
//   Defines which event on the command's DOM representation should
//   trigger the execution of the command. Defaults to mousedown.
var MenuItem = function MenuItem(options) {
    // :: MenuItemSpec
    // The options used to create the menu item.
    this.options = options || {};
    this.sortOrder = this.options.sortOrder;
    this.options.htmlNode = 'button';
};

// :: (EditorView) → {dom: dom.Node, update: (EditorState) → bool}
// Renders the icon according to its [display
// options](#menu.MenuItemSpec.display), and adds an event handler which
// executes the command when the representation is clicked.
MenuItem.prototype.render = function render (view, renderOptions) {
        if ( renderOptions === void 0 ) renderOptions = {};

    this.options = $.extend(this.options, renderOptions);

    if (typeof this.options.render === 'function') {
        return this.options.render.apply(this, [this.options]);
    }

    this.dom = initMenuItemTrigger(view, this.options);

    if (!this.dom) {
        throw new RangeError("MenuItem without icon or label property");
    }

    this.$ = $(this.dom);

    setAttributesFromOptions(this.dom, this.options);
    setTabindex(this.dom, renderOptions);
    setTitle(this.dom, this.options, view);
    this.initEvents(view);

    return this.dom;
};

MenuItem.prototype.initEvents = function initEvents (view) {
        var this$1$1 = this;

    var runHandler = function (e) {
        e.preventDefault();
        if (!this$1$1.$.hasClass(buildMenuClass('disabled'))) {
            this$1$1.options.run.call(this$1$1, view.state, view.dispatch, view, e);
        }
    };

    $(this.dom).on("mousedown", runHandler);
    $(this.dom).on("keydown", function (e) {
        var keyCode = e.keyCode || e.which;

        switch (keyCode) {
            case 13: // Enter
                e.preventDefault();
                runHandler(e);
                break;
        }
    });
};

MenuItem.prototype.getMenuBar = function getMenuBar () {
    if (!this.menuBar) {
        this.menuBar = $(this.dom).closest('.ProseMirror-menubar').data('menuBarInstance');
    }

    return this.menuBar;
};

MenuItem.prototype.switchIcon = function switchIcon (icon, title) {
    if (title) {
        $(this.dom).attr('title', title);
    }
    $(this.dom).find('svg').replaceWith($(getIcon(icon, this.options.htmlNode)).find('svg'));
};

MenuItem.prototype.update = function update (state) {
    this.adoptItemState(state);
    return this.selected;
};

MenuItem.prototype.adoptItemState = function adoptItemState (state, forceEnable, forceActive) {
    this.setEnabledItemState(state, forceEnable);
    this.setActiveItemState(state, forceActive);
    this.setSelectedItemState(state, forceEnable);
};

MenuItem.prototype.setActiveItemState = function setActiveItemState (state, forceActive) {
    this.active = false;
    if (this.options.active) {
        this.active = (this.options.active(state) || forceActive) || false;
        setClass(this.dom, buildMenuClass('active'), this.active);
        setAttribute(this.dom, 'aria-pressed', 'true', this.active);
    }
};

MenuItem.prototype.setEnabledItemState = function setEnabledItemState (state, forceEnable) {
    this.enabled = true;
    if (this.options.enable) {
        this.enabled = this.options.enable(state) || forceEnable || false;
        setClass(this.dom, buildMenuClass('disabled'), !this.enabled);
        setAttribute(this.dom, 'aria-disabled', 'true', !this.enabled);
    }
};

MenuItem.prototype.setSelectedItemState = function setSelectedItemState (state, forceEnable) {
    this.selected = true;
    if (this.options.select) {
        this.selected = this.options.select(state);
        this.dom.style.display = this.selected || forceEnable ? "" : "none";
        this.setHidden(!this.selected);
    }
};

MenuItem.prototype.setHidden = function setHidden (isHidden) {
    setAttribute(this.dom, 'aria-hidden', 'true', isHidden);
    setClass(this.dom, 'd-none', isHidden);

    if (this.isFocusable() && isHidden) {
        setAttribute(this.dom, 'tabindex', '-1', isHidden);
    }
};

MenuItem.prototype.isFocusable = function isFocusable () {
    return ['a', 'button', 'select', 'input'].includes(this.dom.tagName.toLocaleLowerCase());
};var MENU_SUFFIX_GROUP = 'group';

var MenuItemGroup = /*@__PURE__*/(function (MenuItem) {
    function MenuItemGroup(content, options) {
        MenuItem.call(this, options);
        this.options.htmlNode = 'div';
        this.initContent(content);
    }

    if ( MenuItem ) MenuItemGroup.__proto__ = MenuItem;
    MenuItemGroup.prototype = Object.create( MenuItem && MenuItem.prototype );
    MenuItemGroup.prototype.constructor = MenuItemGroup;

    MenuItemGroup.prototype.initContent = function initContent (content) {
        var this$1$1 = this;

        this.content = {
            items: sortItems(Array.isArray(content) ? content : [content]),
            update: function (state) {
                var result = false;

                sortItems(this$1$1.content.items).forEach(function (item, i) {
                    var updateResult = item.update(state);
                    var $item = $(item.dom);

                    if (!updateResult) {
                        $item.hide();
                    } else {
                        $item.show();
                    }

                    // Mark the last item in the group
                    if ((i === this$1$1.content.items.length - 1)) {
                        $item.addClass('last');
                    }

                    // If one item is visible the whole group is visible
                    result = result || updateResult;
                });

                return result;
            }
        };
    };

    MenuItemGroup.prototype.forEachItem = function forEachItem (callable) {
        this.content.items.forEach(callable);
    };

    MenuItemGroup.prototype.render = function render (view, renderOptions) {
        var this$1$1 = this;

        this.$ = $('<' + this.options.htmlNode + '>').addClass(buildMenuClass(MENU_SUFFIX_GROUP));
        this.dom = this.$[0];

        setAttributesFromOptions(this.dom, this.options);

        this.renderItems(view).forEach(function (itemDom) {
            this$1$1.$.append(itemDom);
        });

        return this.dom;
    };

    MenuItemGroup.prototype.update = function update (state) {
        var result = this.content.update(state);
        return result && MenuItem.prototype.update.call(this, state);
    };

    MenuItemGroup.prototype.setHidden = function setHidden (isHidden) {
        MenuItem.prototype.setHidden.call(this, isHidden);

        if (isHidden) {
            this.forEachItem(function (item) {
                item.setHidden(isHidden);
            });
        }
    };

    MenuItemGroup.prototype.renderItems = function renderItems (view) {
        var rendered = [];

        this.forEachItem(function (item) {
            var dom = item.render(view);

            // Note the PREFIX + item is here for compatibility reasons
            var itemDom = crelt("div", {class: PREFIX + 'item'}, dom);
            addMenuClass(itemDom, 'item');
            rendered.push(itemDom);
        });

        return rendered;
    };

    return MenuItemGroup;
}(MenuItem));

function sortItems(items) {
    var result = [];
    items.forEach(function (item) {
        if (item && item.type && item.type === 'dropdown') {
            result.push(new Dropdown(sortItems(item.items), item));
        } else if (item && item.type && item.type === 'group') {
            result.push(new MenuItemGroup(sortItems(item.items), item));
        } else if (item) {
            result.push(item);
        }
    });

    return result.sort(function (a, b) {
        if (typeof a.sortOrder === 'undefined') {
            return 1;
        }
        if (typeof b.sortOrder === 'undefined') {
            return -1;
        }
        return a.sortOrder - b.sortOrder;
    });
}// Holds currently opened dropdown item with close function
var opened = null;

// ::- A drop-down menu, displayed as a label with a downwards-pointing
// triangle to the right of it.
var Dropdown = /*@__PURE__*/(function (MenuItemGroup) {
    function Dropdown(content, options) {
        MenuItemGroup.call(this, content, options);
        this.options.htmlNode = 'button';
    }

    if ( MenuItemGroup ) Dropdown.__proto__ = MenuItemGroup;
    Dropdown.prototype = Object.create( MenuItemGroup && MenuItemGroup.prototype );
    Dropdown.prototype.constructor = Dropdown;

    // :: (EditorView) → {dom: dom.Node, update: (EditorState)}
    // Render the dropdown menu and sub-items.
    Dropdown.prototype.render = function render (view, renderOptions) {
        this.view = view;
        this.getContentDom(view);
        this.initTrigger(view);
        this.initWrapper(view);
        this.initEvents(view);
        return this.dom;
    };

    Dropdown.prototype.initTrigger = function initTrigger (view) {
        this.trigger = initMenuItemTrigger(view, this.options);

        if (!this.trigger) {
            throw new RangeError("Dropdown without icon or label property")
        }

        addMenuClass(this.trigger, 'dropdown');

        this.trigger.setAttribute('aria-haspopup', 'true');
        this.trigger.setAttribute('aria-expanded', 'false');

        setTitle(this.trigger, this.options, view);
    };

    Dropdown.prototype.initWrapper = function initWrapper () {
        this.dom = crelt("div", {}, this.trigger);
        addMenuClass(this.dom, 'dropdown-wrapper');
        this.$ = $(this.dom);
    };

    Dropdown.prototype.initContent = function initContent (content) {
        var this$1$1 = this;

        MenuItemGroup.prototype.initContent.call(this, content);
        this.content.update = function (state) {
            var result = false;
            this$1$1.forEachItem(function (item) {
                var updateResult = item.update(state);
                item.dom.style.display = updateResult ? "" : "none";
                result = result || updateResult;
            });
            return result;
        };
    };

    Dropdown.prototype.initEvents = function initEvents (view) {
        var this$1$1 = this;

        this.$.on('keydown', function (e) {
            var keyCode = e.keyCode || e.which;

            switch (keyCode) {
                case 9: // Enter
                    this$1$1.onTab(e);
                    break;
                case 13: // Enter
                    this$1$1.onEnter(e);
                    break;
                case 27: // Escape
                    this$1$1.onEscape(e);
                    break;
                case 40: // ArrowDown
                    this$1$1.onArrowDown(e);
                    break;
                case 38: // ArrowUp
                    this$1$1.onArrowUp(e);
                    break;
                case 37: // ArrowLeft
                    this$1$1.onArrowLeft(e);
                    break;
                case 39: // ArrowLeft
                    this$1$1.onArrowRight(e);
                    break;
            }
        });

        $(this.trigger).on('mousedown', function (e) {
            this$1$1.onClickTrigger(e);
        });
    };

    Dropdown.prototype.getContentDom = function getContentDom (view) {
        if (!this.contentDom) {
            this.contentDom = this.renderItems(view);
        }

        return this.contentDom;
    };

    Dropdown.prototype.open = function open () {
        var this$1$1 = this;

        opened = this.expand(this.dom);

        $(window).on('mousedown.richtextMenu', function () {
            if (!isMenuEvent(this$1$1.dom)) { this$1$1.close(); }
        });

        closeSubMenues();

        this.trigger.setAttribute('aria-expanded', 'true');
    };

    Dropdown.prototype.close = function close () {
        if (opened && opened.close()) {
            opened = null;
            $(window).off("mousedown.richtextMenu");
        }

        closeSubMenues();
        this.trigger.setAttribute('aria-expanded', 'false');
    };

    Dropdown.prototype.isOpen = function isOpen () {
        return this.$.find('.' + buildMenuClass('dropdown-menu')).is(':visible');
    };

    Dropdown.prototype.onTab = function onTab (e) {
        if (this.isOpen()) {
            this.close();
        }
    };

    Dropdown.prototype.onEnter = function onEnter (e) {
        e.preventDefault();
        if (!this.isOpen()) {
            if (opened) {
                this.close();
            }

            this.open();
            return;
        }

        if (this.getSubMenu().find('a:focus').length) {
            this.getMenuBar().context.editor.focus();
        }

        this.close();
    };

    Dropdown.prototype.getSubMenu = function getSubMenu () {
        return $(this.menu);
    };

    Dropdown.prototype.onEscape = function onEscape (e) {
        e.preventDefault();
        e.stopPropagation();

        var wasOpen = this.isOpen();

        this.close();

        if (wasOpen) {
            $(this.trigger).focus();
        }
    };

    Dropdown.prototype.onArrowDown = function onArrowDown (e) {
        e.preventDefault();
        e.stopPropagation();

        if (this.isOpen()) {
            this.focusNext();
        } else {
            if (opened) {
                this.close();
            }

            $(this.trigger).trigger('mousedown');
        }
    };

    Dropdown.prototype.focusNext = function focusNext () {
        var $focused = this.getFocused();
        var $parent = $focused.parent('.' + buildMenuClass('dropdown-item'));
        var $next = $parent.next();
        $next = $next.length ? $next.find('a:visible:first') : this.getFirstLink();
        $next.focus();
    };

    Dropdown.prototype.getFocused = function getFocused () {
        return this.getSubMenu().find('a:focus');
    };

    Dropdown.prototype.getFirstLink = function getFirstLink () {
        return $(this.menu).find('.' + buildMenuClass('dropdown-item') + ':first').find('a:first');
    };

    Dropdown.prototype.onArrowUp = function onArrowUp (e) {
        e.preventDefault();
        e.stopPropagation();

        if (this.isOpen()) {
            this.focusPrev();
        } else {
            if (opened) {
                this.close();
            }

            this.open();
        }
    };

    Dropdown.prototype.onArrowLeft = function onArrowLeft (e) {
        // Prevent main nav switch
        if (this.isOpen()) {
            e.preventDefault();
            e.stopPropagation();
        }
    };

    Dropdown.prototype.onArrowRight = function onArrowRight (e) {
        // Prevent main nav switch
        if (this.isOpen()) {
            e.preventDefault();
            e.stopPropagation();
        }
    };

    Dropdown.prototype.focusPrev = function focusPrev () {
        var $focused = this.getFocused();
        var $parent = $focused.closest('.' + buildMenuClass('dropdown-item'));
        var $prev = $parent.prev();
        $prev = $prev.length ? $prev.find('a:visible:first') : this.getLastLink();
        $prev.focus();
    };

    Dropdown.prototype.getLastLink = function getLastLink () {
        return $(this.menu).children('.' + buildMenuClass('dropdown-item:last')).find('a:first');
    };

    Dropdown.prototype.onClickTrigger = function onClickTrigger (e) {
        e.preventDefault();
        if (!this.selected || !this.enabled) { return; }
        markMenuEvent(e);

        var wasOpen = this.isOpen();

        if (opened) {
            this.close();
        }

        if (!wasOpen) {
            this.open();
        }
    };

    Dropdown.prototype.renderItems = function renderItems (view) {
        var rendered = [];
        this.content.items.forEach(function (item) {
            var dom = item.render(view, {htmlNode: 'a', tabindex: 0});
            var itemDom = crelt("div", {}, dom);
            addMenuClass(itemDom, 'dropdown-item');
            rendered.push(itemDom);
        });

        return rendered;
    };

    Dropdown.prototype.update = function update (state) {
        var contentUpdateResult = this.content.update(state);

        var forceEnable = false;
        var forceActive = false;

        this.content.items.forEach(function (item) {
            forceEnable = forceEnable || item.enabled;
            forceActive = forceActive || item.active;
        });

        this.adoptItemState(state, forceEnable, (forceActive && this.options.bubbleActive));
        return contentUpdateResult;
    };

    Dropdown.prototype.expand = function expand (dom) {

        var menuDOM = crelt("div", {}, this.getContentDom());
        addMenuClass(menuDOM, 'dropdown-menu');

        var done = false;

        function close() {
            closeSubMenues();
            if (done) { return; }
            done = true;
            dom.removeChild(menuDOM);
            return true;
        }

        dom.appendChild(menuDOM);

        var $menuDom = $(menuDOM);
        var right = $menuDom.offset().left + $menuDom.width();
        var rightAlignClass = buildMenuClass('dropdown-right');

        setClass($menuDom[0], rightAlignClass, (right > $(window).width() / 2));

        this.menu = menuDOM;
        return {close: close, node: menuDOM};
    };

    return Dropdown;
}(MenuItemGroup));

function closeSubMenues() {
    $('.' + buildMenuClass('submenu')).css('display', '');
}// ::- Represents a submenu wrapping a group of elements that start
// hidden and expand to the right when hovered over or tapped.
var DropdownSubmenu = /*@__PURE__*/(function (Dropdown) {
    function DropdownSubmenu(content, options) {
        Dropdown.call(this, content, options);
        this.options.bubbleActive = true;
    }

    if ( Dropdown ) DropdownSubmenu.__proto__ = Dropdown;
    DropdownSubmenu.prototype = Object.create( Dropdown && Dropdown.prototype );
    DropdownSubmenu.prototype.constructor = DropdownSubmenu;

    DropdownSubmenu.prototype.initTrigger = function initTrigger (view) {
        this.trigger = $('<a tabindex="0" aria-haspopup="listbox" aria-expanded="false"></a>')
            .text(translate$1(view, this.options.label))[0];
        setAttributesFromOptions(this.trigger, this.options);
    };

    /**
     * <div class="ProseMirror-menu-submenu-wrap">
     *     <div class="ProseMirror-menu-submenu-label">
     *         <a tabindex="0">Label</a>
     *     </div>
     *     <div class="ProseMirror-menu-submenu">
     *         <div class="ProseMirror-menu-dropdown-item"></div>
     *         <div class="ProseMirror-menu-dropdown-item"></div>
     *     </div>>
     * </div>
     */
    DropdownSubmenu.prototype.initWrapper = function initWrapper (view) {
        var label = $('<div>').addClass(buildMenuClass('submenu-label')).html(this.trigger)[0];

        this.menu = crelt("div", {class: buildMenuClass("submenu")},  this.getContentDom(view));

        this.dom = crelt("div", {class: buildMenuClass("submenu-wrap")}, label, this.menu);

        this.$ = $(this.dom);
    };

    DropdownSubmenu.prototype.onEnter = function onEnter (e) {
        if(this.getFocused().length) {
            return;
        }

        e.preventDefault();
        e.stopPropagation();

        this.getSubMenu().show();

        this.getFirstLink().focus();
    };

    DropdownSubmenu.prototype.onArrowLeft = function onArrowLeft (e) {
        if(this.isOpen()) {
            e.preventDefault();
            e.stopPropagation();
            this.close();
            $(this.trigger).focus();
        }
    };

    DropdownSubmenu.prototype.onArrowRight = function onArrowRight (e) {
        e.preventDefault();
        e.stopPropagation();
        this.getSubMenu().show();
        this.getFirstLink().focus();
    };

    DropdownSubmenu.prototype.onArrowUp = function onArrowUp (e) {
        if(!this.isOpen()) {
            // Let parent dropdown handle
            return;
        }

        Dropdown.prototype.onArrowUp.call(this, e);
    };

    DropdownSubmenu.prototype.onArrowDown = function onArrowDown (e) {
        if(!this.isOpen()) {
            // Let parent dropdown handle
            return;
        }

        Dropdown.prototype.onArrowDown.call(this, e);
    };

    DropdownSubmenu.prototype.open = function open () {
        this.getSubMenu().show();
        this.trigger.setAttribute('aria-expanded', 'true');
    };

    DropdownSubmenu.prototype.close = function close () {
        this.getSubMenu().css('display', '');
        this.trigger.setAttribute('aria-expanded', 'false');
    };

    DropdownSubmenu.prototype.getSubMenu = function getSubMenu () {
        return this.$.find('.ProseMirror-menu-submenu');
    };

    DropdownSubmenu.prototype.focusNext = function focusNext () {
        var $focused = this.getSubMenu().find('a:focus');
        var $parent = $focused.parent('.'+buildMenuClass('dropdown-item'));
        var $next = $parent.next();
        $next = $next.length ? $next.find('a:visible:first') : this.getFirstLink();
        $next.focus();
    };

    DropdownSubmenu.prototype.isOpen = function isOpen () {
        return this.getSubMenu().is(':visible');
    };

    return DropdownSubmenu;
}(Dropdown));// Helpers to create specific types of items
function cmdItem(cmd, options) {
    var passedOptions = {
        label: options.title,
        run: cmd
    };
    for (var prop in options) { passedOptions[prop] = options[prop]; }
    if ((!options.enable || options.enable === true) && !options.select)
        { passedOptions[options.enable ? "enable" : "select"] = function (state) { return cmd(state); }; }

    return new MenuItem(passedOptions)
}

function markItem(markType, options, context) {
    var passedOptions = {
        active: function active(state) {
            return markActive(state, markType)
        },
        enable: true
    };
    for (var prop in options) { passedOptions[prop] = options[prop]; }
    var menuItem = cmdItem(toggleMark(markType), passedOptions);

    if (options.runSource) {
        menuItem.runSource = options.runSource;
    } else if (context && markType.spec.toMarkdown
        && typeof markType.spec.toMarkdown.open === 'string'
        && typeof markType.spec.toMarkdown.close === 'string') {
        menuItem.runSource = wrapSourceTextMark(context, markType);
    }

    return menuItem;
}

function wrapSourceTextMark(context, markType, open, close) {
    if (!open && markType.spec.toMarkdown && markType.spec.toMarkdown.open) {
        open = markType.spec.toMarkdown.open;
    }

    if (!close && markType.spec.toMarkdown && markType.spec.toMarkdown.close) {
        close = markType.spec.toMarkdown.close;
    }

    if (!open) {
        return;
    }

    if (!close) {
        close = open;
    }

    return function () {
        var $source = context.$source;
        var length = $source.val().length;
        var start = $source[0].selectionStart;
        var selectionDirection = $source[0].selectionDirection;
        var end = $source[0].selectionEnd;
        var selectedText = $source.val().substring(start, end);

        var preSelection = $source.val().substring((start - open.length), start);
        var postSelection = $source.val().substring(end, (end + close.length));

        if (preSelection === open && postSelection === close) {
            // Revert mark
            $source.val($source.val().substring(0, start - open.length) + selectedText + $source.val().substring(end + close.length, length));
            $source[0].setSelectionRange((start - open.length), end - open.length, selectionDirection);
        } else {
            var leadingSpaceCount = Math.max(selectedText.search(/\S/), 0);
            var leadingSpaces = leadingSpaceCount > 0
                ? ' '.repeat(leadingSpaceCount)
                : '';

            var trailingSpaceCount = selectedText.search(/ +$/) > -1
                ? selectedText.length - selectedText.search(/ +$/)
                : 0;
            var trailingSpaces = trailingSpaceCount > 0
                ? ' '.repeat(trailingSpaceCount)
                : '';

            selectedText = selectedText.trim();

            var replacement = open + selectedText + close;
            $source.val($source.val().substring(0, start) + leadingSpaces + replacement + trailingSpaces + $source.val().substring(end, length));
            $source[0].setSelectionRange((start + leadingSpaceCount + open.length), (end + open.length - trailingSpaceCount), selectionDirection);
        }
    }
}

function markActive(state, type) {
    var ref = state.selection;
    var from = ref.from;
    var $from = ref.$from;
    var to = ref.to;
    var empty = ref.empty;
    if (empty) { return type.isInSet(state.storedMarks || $from.marks()) }
    else { return state.doc.rangeHasMark(from, to, type) }
}

function wrapListItem(nodeType, options) {
    return cmdItem(wrapInList(nodeType, options.attrs), options)
}

// :: MenuItem
// Menu item for the `joinUp` command.
var joinUpItem = function () {
    return new MenuItem({
        title: "Join with above block",
        run: joinUp,
        select: function (state) { return joinUp(state); },
        icon: icons.join
    });
};

// :: MenuItem
// Menu item for the `lift` command.
var liftItem = function () {
    return new MenuItem({
        title: "Lift out of enclosing block",
        run: lift,
        select: function (state) { return lift(state); },
        icon: icons.outdent
    });
};

function lift(state, dispatch) {
    var ref = state.selection;
    var $from = ref.$from;
    var $to = ref.$to;

    var inList = $from.blockRange($to, function (node) {
        return node.childCount && node.firstChild.type.name === 'list_item';
    });

    if (inList) {
        return false;
    }

    var range = $from.blockRange($to), target = range && liftTarget(range);
    if (target == null) {
        return false;
    }
    if (dispatch) {
        dispatch(state.tr.lift(range, target).scrollIntoView());
    }
    return true;
}

// :: MenuItem
// Menu item for the `selectParentNode` command.
var selectParentNodeItem = function () {
    return new MenuItem({
        title: "Select parent node",
        run: selectParentNode,
        select: function (state) { return selectParentNode(state); },
        icon: icons.selectParentNode
    });
};

// :: (Object) → MenuItem
// Menu item for the `undo` command.
var undoItem = function () {
    return new MenuItem({
        title: "Undo last change",
        run: undo,
        enable: function (state) { return undo(state); },
        icon: icons.undo
    });
};

// :: (Object) → MenuItem
// Menu item for the `redo` command.
var redoItem = function () {
    return new MenuItem({
        title: "Redo last undone change",
        run: redo,
        enable: function (state) { return redo(state); },
        icon: icons.redo
    });
};

// :: (NodeType, Object) → MenuItem
// Build a menu item for wrapping the selection in a given node type.
// Adds `run` and `select` properties to the ones present in
// `options`. `options.attrs` may be an object or a function, as in
// `toggleMarkItem`.
function wrapItem(nodeType, options) {
    var passedOptions = {
        run: function run(state, dispatch) {
            // FIXME if (options.attrs instanceof Function) options.attrs(state, attrs => wrapIn(nodeType, attrs)(state))
            return wrapIn(nodeType, options.attrs)(state, dispatch)
        },
        select: function select(state) {
            return wrapIn(nodeType, options.attrs instanceof Function ? null : options.attrs)(state)
        }
    };
    for (var prop in options) { passedOptions[prop] = options[prop]; }
    return new MenuItem(passedOptions)
}

// :: (NodeType, Object) → MenuItem
// Build a menu item for changing the type of the textblock around the
// selection to the given type. Provides `run`, `active`, and `select`
// properties. Others must be given in `options`. `options.attrs` may
// be an object to provide the attributes for the textblock node.
function blockTypeItem(nodeType, options) {
    var command = setBlockType(nodeType, options.attrs);
    var passedOptions = {
        run: command,
        enable: function enable(state) {
            return command(state)
        },
        active: function active(state) {
            var ref = state.selection;
            var $from = ref.$from;
            var to = ref.to;
            var node = ref.node;
            if (node) { return node.hasMarkup(nodeType, options.attrs) }
            return to <= $from.end() && $from.parent.hasMarkup(nodeType, options.attrs)
        }
    };
    for (var prop in options) { passedOptions[prop] = options[prop]; }
    return new MenuItem(passedOptions)
}

function canInsert(state, nodeType) {
    var $from = state.selection.$from;
    for (var d = $from.depth; d >= 0; d--) {
        var index = $from.index(d);
        if ($from.node(d).canReplaceWith(index, index, nodeType)) { return true }
    }
    return false
}

function canInsertLink(state) {
    var allowLink = true;
    state.doc.nodesBetween(state.selection.$from.pos, state.selection.$to.pos, function (node) {
        if (node.type.spec.code) {
            allowLink = false;
        } else {
            node.marks.forEach(function (mark) {
                var spec = mark.type.spec;
                if (spec.preventMarks && $.inArray('link', spec.preventMarks) >= 0) {
                    allowLink = false;
                }
            });
        }
    });

    return allowLink;
}var prefix$1 = "ProseMirror-menubar";

function buildMenuItems(context) {
    var groups = {
        types: {
            type: 'dropdown',
            id: 'type',
            toggleSelect: false,
            sortOrder: 100,
            title: context.translate("Type"),
            seperator: true,
            icon: icons.text,
            items: []
        },
        marks: {type: 'group', id: 'marks-group', sortOrder: 200, items: []},
        format: {type: 'group', id: 'format-group', sortOrder: 300, items: [liftItem()]},
        insert: {
            type: 'dropdown',
            id: 'insert-dropdown',
            sortOrder: 400,
            title: context.translate("Insert"),
            seperator: true,
            icon: icons.upload,
            items: []
        },
        helper: {
            type: 'group',
            id: 'helper-group',
            hideOnCollapse: true,
            sortOrder: 500,
            items: []
        },
        resize: {type: 'group', id: 'resize-group', sortOrder: 600, items: []},
    };

    var definitions = [groups.types, groups.marks, groups.format, groups.insert, groups.helper, groups.mode, groups.resize];

    var menuGroupPlugins = [];
    var menuWrapperPlugins = [];

    context.plugins.forEach(function (plugin) {
        if (plugin.menu) {
            plugin.menu(context).forEach(function (menuDefinition) {
                if (checkMenuDefinition(context, menuDefinition)) {
                    if (menuDefinition.type && menuDefinition.type === 'group') {
                        definitions.push(menuDefinition);
                        return;
                    }

                    if (menuDefinition.item && menuDefinition.id) {
                        // transfer the id of the definition to the item itself
                        menuDefinition.item.options.id = menuDefinition.id;
                    }

                    if (menuDefinition.group && groups[menuDefinition.group]) {
                        groups[menuDefinition.group].items.push(menuDefinition.item);
                    } else if (menuDefinition.item && !menuDefinition.group) {
                        definitions.push(menuDefinition.item);
                    }
                }
            });
        }

        if (plugin.menuGroups) {
            menuGroupPlugins.push(plugin);
        }

        if (plugin.menuWrapper) {
            menuWrapperPlugins.push(plugin);
        }
    });

    // Execute after all menu items are assembled
    menuGroupPlugins.forEach(function (plugin) {
        definitions = plugin.menuGroups(definitions, context);
    });

    context.menuWrapperPlugins = menuWrapperPlugins;

    return filterOutExcludes(context, definitions);
}

function filterOutExcludes(context, definition) {
    if (!definition) {
        return false;
    }

    if (Array.isArray(definition)) {
        return definition.filter(function (item) {
            return filterOutExcludes(context, item);
        });
    }

    if (definition instanceof MenuItemGroup) {
        definition.content.items = definition.content.items.filter(function (item) {
            return filterOutExcludes(context, item);
        });
    }

    if (typeof definition === 'object' && definition.items) {
        definition.items = definition.items.filter(function (item) {
            return filterOutExcludes(context, item);
        });
    }

    var id = definition.id;
    if (!id && definition.options) {
        id = definition.options.id;
    }

    return definition && !isExcludedMenuItem(context, id);
}

function wrapMenuItem(plugin, context, menuItem) {
    if (!menuItem) {
        return;
    }

    if (!plugin.menuWrapper) {
        return;
    }

    if ($.isArray(menuItem)) {
        menuItem.forEach(function (item) {
            wrapMenuItem(plugin, context, item);
        });
    }

    var wrapper = plugin.menuWrapper(context);

    if (menuItem instanceof MenuItem) {
        if (wrapper.run) {
            var origCallback = menuItem.options.run;
            menuItem.options.run = function (state, dispatch, view, evt) {
                var result = wrapper.run(menuItem, state, dispatch, view, evt);
                if (!result) {
                    origCallback.call(menuItem, state, dispatch, view, evt);
                }
            };
        }

        if (wrapper.active) {
            var origCallback$1 = menuItem.options.active;
            menuItem.options.active = function (state) {
                var origValue = origCallback$1 ? origCallback$1.call(menuItem, state) : false;
                return wrapper.active(menuItem, state, origValue);
            };
        }

        if (wrapper.enable) {
            var origCallback$2 = menuItem.options.enable;
            menuItem.options.enable = function (state) {
                var origValue = origCallback$2 ? origCallback$2.call(menuItem, state) : true;
                return wrapper.enable(menuItem, state, origValue);
            };
        }

        if (wrapper.select) {
            var origCallback$3 = menuItem.options.select;
            menuItem.options.select = function (state) {
                var origValue = origCallback$3 ? origCallback$3.call(menuItem, state) : true;
                return wrapper.select(menuItem, state, origValue);
            };
        }
    }

    if (menuItem.items) {
        wrapMenuItem(plugin, context, menuItem.items);
    }

    if (menuItem instanceof MenuItemGroup) {
        wrapMenuItem(plugin, context, menuItem.content.items);
    }
}

function checkMenuDefinition(context, menuDefinition) {
    if (!menuDefinition || menuDefinition.node && !context.schema.nodes[menuDefinition.node]) {
        return false;
    }

    return !(menuDefinition.mark && !context.schema.marks[menuDefinition.mark]);
}

function isExcludedMenuItem(context, id) {
    var presetOption = context.getPresetOption('menu', 'exclude', []);
    if (Array.isArray(presetOption) && presetOption.includes(id)) {
        return true;
    }

    var globalOption = context.getGlobalOption('menu', 'exclude', []);
    if (Array.isArray(globalOption) && globalOption.includes(id)) {
        return true;
    }

    var contextOption = context.getPluginOption('menu', 'exclude', []);
    return Array.isArray(contextOption) && contextOption.includes(id);
}

function buildMenuBar(context) {
    context.menu = menuBar({
        content: buildMenuItems(context),
        floating: false,
        context: context
    });

    return context.menu;
}


function isIOS() {
    if (typeof navigator == "undefined") { return false; }
    var agent = navigator.userAgent;
    return !/Edge\/\d/.test(agent) && /AppleWebKit/.test(agent) && /Mobile\/\w+/.test(agent);
}

// :: (Object) → Plugin
// A plugin that will place a menu bar above the editor. Note that
// this involves wrapping the editor in an additional `<div>`.
//
//   options::-
//   Supports the following options:
//
//     content:: [[MenuElement]]
//     Provides the content of the menu, as a nested array to be
//     passed to `renderGrouped`.
//
//     floating:: ?bool
//     Determines whether the menu floats, i.e. whether it sticks to
//     the top of the viewport when the editor is partially scrolled
//     out of view.
function menuBar(options) {
    return new Plugin({
        view: function view(editorView) {
            options.context.menu = new MenuBarView(editorView, options);
            options.context.event.trigger('afterMenuBarInit', options.context.menu);
            return options.context.menu;
        }
    })
}

function translate(view, text) {
    return view._props.translate ? view._props.translate(text) : text;
}

var lastFocusedElement = null;

var MenuBarView = function MenuBarView(editorView, options) {
    var this$1$1 = this;

    this.editorView = editorView;
    this.options = options;
    this.context = this.options.context;
    this.focusIconIndex = 0;
    var $editor = this.context.editor.$;
    this.wrapper = crelt("div", {class: prefix$1 + "-wrapper"});

    this.menu = this.wrapper.appendChild(crelt("div", {
        class: prefix$1,
        'aria-label': translate(editorView, 'Text Formatting'),
        'aria-controls': $editor.attr('id'),
        role: 'toolbar'
    }));

    this.menu.className = prefix$1;
    this.spacer = null;

    editorView.dom.parentNode.replaceChild(this.wrapper, editorView.dom);
    this.wrapper.appendChild(editorView.dom);

    this.maxHeight = 0;
    this.widthForMaxHeight = 0;
    this.floating = false;
    this.eventType = 'click';

    this.groupItem = new MenuItemGroup(this.options.content, {id: 'main-menu-group'});

    this.context.menuWrapperPlugins.forEach(function (plugin) {
        wrapMenuItem(plugin, this$1$1.context, this$1$1.groupItem);
    });

    // TODO: In case of focus menu render only on first focus
    this.menu.appendChild(this.groupItem.render(this.editorView));

    this.$ = $(this.menu);

    // Focus and blur editor handler
    if ($editor.is('.focusMenu')) {
        this.$.addClass('d-none');

        $editor.off('focus', '.ProseMirror, textarea').off('blur', '.ProseMirror, textarea')
            .on('focus', '.ProseMirror, textarea', function (event) {
                if (this$1$1.$.data('isHiding')) {
                    this$1$1.$.data('isHiding', false)
                        .addClass('d-none')
                        .stop(true, false);
                }
                if (this$1$1.$.hasClass('d-none')) {
                    this$1$1.$.removeClass('d-none');
                    var that = this$1$1;

                    $editor.on('keyup', function (e) {
                        var code = e.keyCode ? e.keyCode : e.which;
                        if (code === 9 && lastFocusedElement !== $(lastFocusedElement).closest(event.target).prevObject[0]) {
                            setTimeout(function () {
                                $(that.groupItem.dom).find('.' + buildMenuClass('trigger:first')).attr('tabindex', 0).focus();
                                $editor.off('keyup');
                            },0);
                        }
                    });
                }
            })
            .on('blur', '.ProseMirror, textarea', function (e) {
                var targetHasMenuBtn = e.relatedTarget ? e.relatedTarget.classList.contains('ProseMirror-menu-trigger') : false;
                lastFocusedElement = e.target;

                if (!$editor.is('.fullscreen') && !targetHasMenuBtn && !$(e.target).hasClass('cm-editor')) {
                    lastFocusedElement = null;
                    var that = this$1$1.$;
                    that.data('isHiding', true)
                        // Don't apply the d-none immediately to allow clicking on links which might move after hiding the toolbar
                        .hide(100, function(){that.addClass('d-none');});
                }
            });
    }

    this.$.on('mousedown', function (e) {
        // Prevent focusout if we click outside a menu item, but still inside menu container
        e.preventDefault();
    }).on("keydown", function (e) {
        var keyCode = e.keyCode || e.which;

        switch (keyCode) {
            case 39: // ArrowRight
                e.preventDefault();
                this$1$1.focusNext();
                break;
            case 37: // ArrowLeft
                e.preventDefault();
                this$1$1.focusPrev();
                break;
        }
    });

    this.$.data('menuBarInstance', this);

    this.update();

    if (options.floating && !isIOS()) {
        this.updateFloat();
        this.scrollFunc = function () {
            var root = this$1$1.editorView.root;
            if (!(root.body || root).contains(this$1$1.wrapper))
                { window.removeEventListener("scroll", this$1$1.scrollFunc); }
            else
                { this$1$1.updateFloat(); }
        };
        window.addEventListener("scroll", this.scrollFunc);
    }
};

MenuBarView.prototype.update = function update () {
    this.groupItem.update(this.editorView.state, this.context);

    if (this.floating) {
        this.updateScrollCursor();
    } else {
        if (this.menu.offsetWidth !== this.widthForMaxHeight) {
            this.widthForMaxHeight = this.menu.offsetWidth;
            this.maxHeight = 0;
        }
        if (this.menu.offsetHeight > this.maxHeight) {
            this.maxHeight = this.menu.offsetHeight;
        }
    }

    this.context.event.trigger('afterMenuBarUpdate', this);
};

MenuBarView.prototype.focusPrev = function focusPrev () {
    var $prev = null;
    var $focus = null;
    var newFocusIconIndex = 0;
    var $current = this.$.find('.' + buildMenuClass('trigger:focus'));
    var $items = this.$.find('.' + buildMenuClass('trigger'));

    $items.each(function (index) {
        var $this = $(this);

        if ($this.is($current)) {
            $focus = $prev;
            newFocusIconIndex = index - 1;
        }

        $this.attr('tabindex', -1);

        if ($this.is(':visible')) {
            $prev = $this;
        }
    });

    if (!$focus) {
        $focus = $items.last();
        newFocusIconIndex = $items.length - 1;
    }

    this.focusIconIndex = newFocusIconIndex;

    $focus.attr('tabindex', 0).focus();
};

MenuBarView.prototype.focusNext = function focusNext () {
    var $next = null;
    var newFocusIconIndex = 0;
    var focusNextItem = false;
    var $current = this.$.find('.' + buildMenuClass('trigger:focus'));

    this.$.find('.' + buildMenuClass('trigger')).each(function (index) {
        var $this = $(this);
        if (!$this.is(':visible')) {
            return;
        }

        if (focusNextItem) {
            $next = $this;
            focusNextItem = false;
            newFocusIconIndex = index;
        } else {
            $this.attr('tabindex', -1);
            focusNextItem = $this.is($current);
        }
    });

    if (!$next) {
        $next = this.$.find('.' + buildMenuClass('trigger:first'));
    }

    this.focusIconIndex = newFocusIconIndex;

    $next.attr('tabindex', 0).focus();
};

MenuBarView.prototype.updateScrollCursor = function updateScrollCursor () {
    var selection = this.editorView.root.getSelection();
    if (!selection.focusNode) { return; }
    var rects = selection.getRangeAt(0).getClientRects();
    var selRect = rects[selectionIsInverted(selection) ? 0 : rects.length - 1];
    if (!selRect) { return; }
    var menuRect = this.menu.getBoundingClientRect();
    if (selRect.top < menuRect.bottom && selRect.bottom > menuRect.top) {
        var scrollable = findWrappingScrollable(this.wrapper);
        if (scrollable) { scrollable.scrollTop -= (menuRect.bottom - selRect.top); }
    }
};

MenuBarView.prototype.updateFloat = function updateFloat () {
    var parent = this.wrapper, editorRect = parent.getBoundingClientRect();
    if (this.floating) {
        if (editorRect.top >= 0 || editorRect.bottom < this.menu.offsetHeight + 10) {
            this.floating = false;
            this.menu.style.position = this.menu.style.left = this.menu.style.width = "";
            this.menu.style.display = "";
            this.spacer.parentNode.removeChild(this.spacer);
            this.spacer = null;
        } else {
            var border = (parent.offsetWidth - parent.clientWidth) / 2;
            this.menu.style.left = (editorRect.left + border) + "px";
            this.menu.style.display = (editorRect.top > window.innerHeight ? "none" : "");
        }
    } else {
        if (editorRect.top < 0 && editorRect.bottom >= this.menu.offsetHeight + 10) {
            this.floating = true;
            var menuRect = this.menu.getBoundingClientRect();
            this.menu.style.left = menuRect.left + "px";
            this.menu.style.width = menuRect.width + "px";
            this.menu.style.position = "fixed";
            this.spacer = crel("div", {class: prefix$1 + "-spacer", style: ("height: " + (menuRect.height) + "px")});
            parent.insertBefore(this.spacer, this.menu);
        }
    }
};

MenuBarView.prototype.destroy = function destroy () {
    if (this.wrapper.parentNode)
        { this.wrapper.parentNode.replaceChild(this.editorView.dom, this.wrapper); }
};

// Not precise, but close enough
function selectionIsInverted(selection) {
    if (selection.anchorNode === selection.focusNode)
        { return selection.anchorOffset > selection.focusOffset; }
    return selection.anchorNode.compareDocumentPosition(selection.focusNode) === Node.DOCUMENT_POSITION_FOLLOWING;
}

function findWrappingScrollable(node) {
    for (var cur = node.parentNode; cur; cur = cur.parentNode)
        { if (cur.scrollHeight > cur.clientHeight) { return cur; } }
}var menu$k=/*#__PURE__*/Object.freeze({__proto__:null,Dropdown:Dropdown,DropdownSubmenu:DropdownSubmenu,MenuItem:MenuItem,MenuItemGroup:MenuItemGroup,blockTypeItem:blockTypeItem,canInsert:canInsert,canInsertLink:canInsertLink,cmdItem:cmdItem,icons:icons,joinUpItem:joinUpItem,liftItem:liftItem,markActive:markActive,markItem:markItem,menuBar:menuBar,redoItem:redoItem,selectParentNodeItem:selectParentNodeItem,undoItem:undoItem,wrapItem:wrapItem,wrapListItem:wrapListItem,wrapSourceTextMark:wrapSourceTextMark});var prefix = "ProseMirror-prompt";

var Promt = function Promt(options) {
    this.options = options;
    this.render();
};

Promt.prototype.render = function render () {
    $('.ProseMirror-prompt').remove();
    this.$wrapper = $('<div>').addClass(prefix).appendTo($('body'));
    this.buildForm();
    this.initEvents();
    this.initWrapper();
};

Promt.prototype.initWrapper = function initWrapper () {
    var box = this.$wrapper[0].getBoundingClientRect();
    this.$wrapper.css({
        top: ((window.innerHeight - box.height) / 2) + "px",
        left: ((window.innerWidth - box.width) / 2) + "px"
    });

    this.$wrapper.find('select:visible, input[type="text"]:visible, textarea:visible, [contenteditable="true"]:visible').first().focus();
};

Promt.prototype.buildForm = function buildForm () {
        var this$1$1 = this;

    this.$form = $('<form>').appendTo(this.$wrapper);

    if (this.options.title) {
        this.$form.append('<h5>' + this.options.title + '</h5>');
    }

    this.buildFormFields();

    this.domFields.forEach(function (field) {
        this$1$1.$form.append(field);
    });

    this.$form.on('submit', function (e) {
        e.preventDefault();
        this$1$1.submit();
    });

    this.buildButtons();
};

Promt.prototype.buildFormFields = function buildFormFields () {
    this.domFields = [];
    for (var name in this.options.fields) {
        var field = this.options.fields[name];
        var $field = $('<div>').append('<label>' + (field.options.label || name) + ':</label><br>').append(this.options.fields[name].render());
        this.domFields.push($field[0]);
    }
};

Promt.prototype.buildButtons = function buildButtons () {
        var this$1$1 = this;

    this.$buttons = $('<div>').addClass(prefix + "-buttons");
    // TODO: translate text!
    $('<button type="submit" class="btn btn-primary">').addClass(prefix + "-submit").text('OK').appendTo(this.$buttons);

    this.$buttons.append(document.createTextNode(' '));

    $('<button type="button" class="btn btn-secondary">').addClass(prefix + "-cancel")
        .text('Cancel').appendTo(this.$buttons)
        .on('click', function () { return this$1$1.close(); });

    this.$form.append(this.$buttons);
};

Promt.prototype.close = function close () {
    this.$wrapper.remove();
};

Promt.prototype.submit = function submit () {
    var params = this.getValues();
    if (params) {
        this.close();
        this.options.callback(params);
    }
};

Promt.prototype.getValues = function getValues () {
    var result = Object.create(null);
    var i = 0;

    for (var name in this.options.fields) {
        var field = this.options.fields[name];
        var dom = this.domFields[i++];

        var value = field.read(dom);
        var bad = field.validate(value);

        if (bad) {
            this.reportInvalid(dom, bad);
            return null;
        }
        result[name] = field.clean(value);
    }

    return result;
};

Promt.prototype.reportInvalid = function reportInvalid (dom, message) {
    // FIXME this is awful and needs a lot more work
    var parent = dom.parentNode;
    var msg = parent.appendChild(document.createElement("div"));
    msg.style.left = (dom.offsetLeft + dom.offsetWidth + 2) + "px";
    msg.style.top = (dom.offsetTop - 5) + "px";
    msg.className = "ProseMirror-invalid";
    msg.textContent = message;
    setTimeout(function () { return parent.removeChild(msg); }, 1500);
};

Promt.prototype.initEvents = function initEvents () {
        var this$1$1 = this;

    this.$form.on("keydown", function (e) {
        if (e.keyCode === 27) {
            e.preventDefault();
            this$1$1.close();
        } else if (e.keyCode === 13 && !(e.ctrlKey || e.metaKey || e.shiftKey)) {
            e.preventDefault();
            this$1$1.submit();
        } else if (e.keyCode === 9) {
            window.setTimeout(function () {
                if (!$.contains(this$1$1.$wrapper[0], document.activeElement)) {
                    this$1$1.close();
                }
            }, 500);
        }
    }).on('mousedown', function (e) {
        if (!$.contains(this$1$1.$wrapper[0], e.target)) {
            this$1$1.close();
        }
    });
};

function openPrompt(options) {
    return new Promt(options);
}

// ::- The type of field that `FieldPrompt` expects to be passed to it.
var Field = function Field(options) {
    this.options = options;
};

// render:: (state: EditorState, props: Object) → dom.Node
// Render the field to the DOM. Should be implemented by all subclasses.

// :: (dom.Node) → any
// Read the field's value from its DOM node.
Field.prototype.read = function read (dom) {
    if (dom.value) {
        return dom.value;
    } else {
        return $(dom).find('input, select')[0].value;
    }
};

// :: (any) → ?string
// A field-type-specific validation function.
Field.prototype.validateType = function validateType (_value) {
};

Field.prototype.validate = function validate (value) {
    if (!value && this.options.required)
        { return "Required field"; }
    return this.validateType(value) || (this.options.validate && this.options.validate(value));
};

Field.prototype.clean = function clean (value) {
    return this.options.clean ? this.options.clean(value) : value;
};

// ::- A field class for single-line text fields.
var TextField = /*@__PURE__*/(function (Field) {
    function TextField () {
        Field.apply(this, arguments);
    }

    if ( Field ) TextField.__proto__ = Field;
    TextField.prototype = Object.create( Field && Field.prototype );
    TextField.prototype.constructor = TextField;

    TextField.prototype.render = function render () {
        var input = document.createElement("input");
        input.type = "text";
        input.className = 'form-control';
        input.value = this.options.value || "";
        input.autocomplete = "off";
        return input;
    };

    return TextField;
}(Field));


// ::- A field class for dropdown fields based on a plain `<select>`
// tag. Expects an option `options`, which should be an array of
// `{value: string, label: string}` objects, or a function taking a
// `ProseMirror` instance and returning such an array.
var SelectField = /*@__PURE__*/(function (Field) {
    function SelectField () {
        Field.apply(this, arguments);
    }

    if ( Field ) SelectField.__proto__ = Field;
    SelectField.prototype = Object.create( Field && Field.prototype );
    SelectField.prototype.constructor = SelectField;

    SelectField.prototype.render = function render () {
        var this$1$1 = this;

        var select = document.createElement("select");
        select.className = 'form-control';

        this.options.options.forEach(function (o) {
            var opt = select.appendChild(document.createElement("option"));
            opt.value = o.value;
            opt.selected = o.value === this$1$1.options.value;
            opt.label = o.label;
        });
        return select;
    };

    return SelectField;
}(Field));var prompt=/*#__PURE__*/Object.freeze({__proto__:null,Field:Field,SelectField:SelectField,TextField:TextField,openPrompt:openPrompt});// TODO: enable global default config e.g. for emoji, locale, etc

var isHumhub = function () {
    return humhub && humhub.modules;
};

var showSuccessNotify = function (message) {
    if (isHumhub) {
        humhub.modules.ui.status.success(message);
    }
};

function getEmojiConfig() {
    if (!window.humhub || !window.humhub.config) {
        return {};
    }

    return humhub.config.get('ui.richtext.prosemirror', 'emoji', {twemoji: {}});
}

function isSmallView() {
    if (!window.humhub) {
        return getClientWidth() <= 767;
    }

    return isHumhub() ? humhub.modules.ui.view.isSmall() : null;
}

var getClientWidth = function () {
    return window.innerWidth || document.documentElement.clientWidth || document.body.clientWidth;
};

function getUserLocale() {
    if (!window.humhub) {
        return (navigator.languages && navigator.languages.length) ? navigator.languages[0] : navigator.language;
    }

    return isHumhub() ? humhub.modules.user.config.locale.split("-")[0] : null;
}

function filterFileUrl(url, mode) {
    if (!window.humhub) {
        return {url: url, guid: null};
    }

    return isHumhub() ? humhub.modules.file.filterFileUrl(url, mode) : url;
}

function getLoaderWidget() {
    if (!window.humhub) {
        return $('<div class="loader">loading...</div>');
    }

    return humhub.require('ui.loader').set($('<span class="ProseMirror-placeholder">'), {
        span: true,
        size: '8px',
        css: {
            padding: '0px',
            width: '60px'
        }
    });
}

function encode$4(str) {
    if (!window.humhub) {
        return $('<div/>').text(str).html();
    }

    return isHumhub() ? humhub.modules.util.string.encode(str) : str;
}

// TODO: Implement oembed provider interface
function loadOembeds(urls) {
    if (!window.humhub) {
        return Promise.resolve([]);
    }

    return humhub.require('oembed').load(urls);
}

function getOembed(url) {
    if (!window.humhub) {
        return null;
    }

    return humhub.require('oembed').get(url);
}/*
 * @link https://www.humhub.org/
 * @copyright Copyright (c) 2017 HumHub GmbH & Co. KG
 * @license https://www.humhub.com/licences
 *
 */



var loaderPlugin = function (context) {
    return new Plugin({
        state: {
            init: function init() {
                return DecorationSet.empty
            },
            apply: function apply(tr, set) {
                // Adjust decoration positions to changes made by the transaction
                set = set.map(tr.mapping, tr.doc);
                // See if the transaction adds or removes any placeholders
                var action = tr.getMeta(this);
                if (action && action.add) {
                    var deco = Decoration$1.widget(action.add.pos, getLoaderWidget()[0], {id: action.add.id, content: true});
                    set = set.add(tr.doc, [deco]);
                    context.addContentDecoration('loader');
                } else if (action && action.remove) {
                    set = set.remove(set.find(null, null, function (spec) { return spec.id === action.remove.id; }));
                    context.removeContentDecoration('loader');
                }
                return set
            }
        },
        props: {
            decorations: function decorations(state) {
                return this.getState(state)
            }
        }
    });
};

function findLoader(context, id) {
    var decos = context.getProsemirrorPlugins('loader')[0].getState(context.editor.view.state);
    var found = decos.find(null, null, function (spec) { return spec.id === id; });
    return found.length ? found[0].from : null
}

function loaderStart(context, id, dispatch) {
    var view = context.editor.view;
    var tr = view.state.tr;

    if (!tr.selection.empty) {
        tr.deleteSelection();
    }

    tr.setMeta(context.getProsemirrorPlugins('loader')[0], {add: {id: id, pos: tr.selection.from}});

    if(dispatch) {
        view.dispatch(tr);
    }

    return tr;
}

function replaceLoader(context, id, content, dispatch) {
    var view = context.editor.view;
    var pos = findLoader(context, id);

    // If the content around the placeholder has been deleted, drop the image
    if (pos === null) {
        return;
    }

    var tr = view.state.tr.replaceWith(pos, pos, content).setMeta(context.getProsemirrorPlugins('loader')[0], {remove: {id: id}});

    if(dispatch) {
        view.dispatch(tr);
    }

    return tr;
}

function removeLoader(context, id, dispatch) {
    var view = context.editor.view;
    var pos = findLoader(context, id);

    // Focus the editor in order to synchronized changes into hidden textarea
    // for case when before file uploading the editor was not focused
    view.focus();

    // If the content around the placeholder has been deleted, drop the image
    if (pos === null) {
        return;
    }

    var tr = view.state.tr.setMeta(context.getProsemirrorPlugins('loader')[0], {remove: {id: id}});

    if(dispatch) {
        view.dispatch(tr);
    }

    return tr;
}var loader$1=/*#__PURE__*/Object.freeze({__proto__:null,loaderPlugin:loaderPlugin,loaderStart:loaderStart,removeLoader:removeLoader,replaceLoader:replaceLoader});/* eslint-disable no-bitwise */

var decodeCache$1 = {};

function getDecodeCache$1 (exclude) {
  var cache = decodeCache$1[exclude];
  if (cache) { return cache }

  cache = decodeCache$1[exclude] = [];

  for (var i = 0; i < 128; i++) {
    var ch = String.fromCharCode(i);
    cache.push(ch);
  }

  for (var i$1 = 0; i$1 < exclude.length; i$1++) {
    var ch$1 = exclude.charCodeAt(i$1);
    cache[ch$1] = '%' + ('0' + ch$1.toString(16).toUpperCase()).slice(-2);
  }

  return cache
}

// Decode percent-encoded string.
//
function decode$3 (string, exclude) {
  if (typeof exclude !== 'string') {
    exclude = decode$3.defaultChars;
  }

  var cache = getDecodeCache$1(exclude);

  return string.replace(/(%[a-f0-9]{2})+/gi, function (seq) {
    var result = '';

    for (var i = 0, l = seq.length; i < l; i += 3) {
      var b1 = parseInt(seq.slice(i + 1, i + 3), 16);

      if (b1 < 0x80) {
        result += cache[b1];
        continue
      }

      if ((b1 & 0xE0) === 0xC0 && (i + 3 < l)) {
        // 110xxxxx 10xxxxxx
        var b2 = parseInt(seq.slice(i + 4, i + 6), 16);

        if ((b2 & 0xC0) === 0x80) {
          var chr = ((b1 << 6) & 0x7C0) | (b2 & 0x3F);

          if (chr < 0x80) {
            result += '\ufffd\ufffd';
          } else {
            result += String.fromCharCode(chr);
          }

          i += 3;
          continue
        }
      }

      if ((b1 & 0xF0) === 0xE0 && (i + 6 < l)) {
        // 1110xxxx 10xxxxxx 10xxxxxx
        var b2$1 = parseInt(seq.slice(i + 4, i + 6), 16);
        var b3 = parseInt(seq.slice(i + 7, i + 9), 16);

        if ((b2$1 & 0xC0) === 0x80 && (b3 & 0xC0) === 0x80) {
          var chr$1 = ((b1 << 12) & 0xF000) | ((b2$1 << 6) & 0xFC0) | (b3 & 0x3F);

          if (chr$1 < 0x800 || (chr$1 >= 0xD800 && chr$1 <= 0xDFFF)) {
            result += '\ufffd\ufffd\ufffd';
          } else {
            result += String.fromCharCode(chr$1);
          }

          i += 6;
          continue
        }
      }

      if ((b1 & 0xF8) === 0xF0 && (i + 9 < l)) {
        // 111110xx 10xxxxxx 10xxxxxx 10xxxxxx
        var b2$2 = parseInt(seq.slice(i + 4, i + 6), 16);
        var b3$1 = parseInt(seq.slice(i + 7, i + 9), 16);
        var b4 = parseInt(seq.slice(i + 10, i + 12), 16);

        if ((b2$2 & 0xC0) === 0x80 && (b3$1 & 0xC0) === 0x80 && (b4 & 0xC0) === 0x80) {
          var chr$2 = ((b1 << 18) & 0x1C0000) | ((b2$2 << 12) & 0x3F000) | ((b3$1 << 6) & 0xFC0) | (b4 & 0x3F);

          if (chr$2 < 0x10000 || chr$2 > 0x10FFFF) {
            result += '\ufffd\ufffd\ufffd\ufffd';
          } else {
            chr$2 -= 0x10000;
            result += String.fromCharCode(0xD800 + (chr$2 >> 10), 0xDC00 + (chr$2 & 0x3FF));
          }

          i += 9;
          continue
        }
      }

      result += '\ufffd';
    }

    return result
  })
}

decode$3.defaultChars = ';/?:@&=+$,#';
decode$3.componentChars = '';var encodeCache$1 = {};

// Create a lookup array where anything but characters in `chars` string
// and alphanumeric chars is percent-encoded.
//
function getEncodeCache$1 (exclude) {
  var cache = encodeCache$1[exclude];
  if (cache) { return cache }

  cache = encodeCache$1[exclude] = [];

  for (var i = 0; i < 128; i++) {
    var ch = String.fromCharCode(i);

    if (/^[0-9a-z]$/i.test(ch)) {
      // always allow unencoded alphanumeric characters
      cache.push(ch);
    } else {
      cache.push('%' + ('0' + i.toString(16).toUpperCase()).slice(-2));
    }
  }

  for (var i$1 = 0; i$1 < exclude.length; i$1++) {
    cache[exclude.charCodeAt(i$1)] = exclude[i$1];
  }

  return cache
}

// Encode unsafe characters with percent-encoding, skipping already
// encoded sequences.
//
//  - string       - string to encode
//  - exclude      - list of characters to ignore (in addition to a-zA-Z0-9)
//  - keepEscaped  - don't encode '%' in a correct escape sequence (default: true)
//
function encode$3 (string, exclude, keepEscaped) {
  if (typeof exclude !== 'string') {
    // encode(string, keepEscaped)
    keepEscaped = exclude;
    exclude = encode$3.defaultChars;
  }

  if (typeof keepEscaped === 'undefined') {
    keepEscaped = true;
  }

  var cache = getEncodeCache$1(exclude);
  var result = '';

  for (var i = 0, l = string.length; i < l; i++) {
    var code = string.charCodeAt(i);

    if (keepEscaped && code === 0x25 /* % */ && i + 2 < l) {
      if (/^[0-9a-f]{2}$/i.test(string.slice(i + 1, i + 3))) {
        result += string.slice(i, i + 3);
        i += 2;
        continue
      }
    }

    if (code < 128) {
      result += cache[code];
      continue
    }

    if (code >= 0xD800 && code <= 0xDFFF) {
      if (code >= 0xD800 && code <= 0xDBFF && i + 1 < l) {
        var nextCode = string.charCodeAt(i + 1);
        if (nextCode >= 0xDC00 && nextCode <= 0xDFFF) {
          result += encodeURIComponent(string[i] + string[i + 1]);
          i++;
          continue
        }
      }
      result += '%EF%BF%BD';
      continue
    }

    result += encodeURIComponent(string[i]);
  }

  return result
}

encode$3.defaultChars = ";/?:@&=+$,-_.!~*'()#";
encode$3.componentChars = "-_.!~*'()";function format$1 (url) {
  var result = '';

  result += url.protocol || '';
  result += url.slashes ? '//' : '';
  result += url.auth ? url.auth + '@' : '';

  if (url.hostname && url.hostname.indexOf(':') !== -1) {
    // ipv6 address
    result += '[' + url.hostname + ']';
  } else {
    result += url.hostname || '';
  }

  result += url.port ? ':' + url.port : '';
  result += url.pathname || '';
  result += url.search || '';
  result += url.hash || '';

  return result
}// Copyright Joyent, Inc. and other Node contributors.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to permit
// persons to whom the Software is furnished to do so, subject to the
// following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
// USE OR OTHER DEALINGS IN THE SOFTWARE.

//
// Changes from joyent/node:
//
// 1. No leading slash in paths,
//    e.g. in `url.parse('http://foo?bar')` pathname is ``, not `/`
//
// 2. Backslashes are not replaced with slashes,
//    so `http:\\example.org\` is treated like a relative path
//
// 3. Trailing colon is treated like a part of the path,
//    i.e. in `http://example.org:foo` pathname is `:foo`
//
// 4. Nothing is URL-encoded in the resulting object,
//    (in joyent/node some chars in auth and paths are encoded)
//
// 5. `url.parse()` does not have `parseQueryString` argument
//
// 6. Removed extraneous result properties: `host`, `path`, `query`, etc.,
//    which can be constructed using other parts of the url.
//

function Url$1 () {
  this.protocol = null;
  this.slashes = null;
  this.auth = null;
  this.port = null;
  this.hostname = null;
  this.hash = null;
  this.search = null;
  this.pathname = null;
}

// Reference: RFC 3986, RFC 1808, RFC 2396

// define these here so at least they only have to be
// compiled once on the first module load.
var protocolPattern$1 = /^([a-z0-9.+-]+:)/i;
var portPattern$1 = /:[0-9]*$/;

// Special case for a simple path URL
/* eslint-disable-next-line no-useless-escape */
var simplePathPattern$1 = /^(\/\/?(?!\/)[^\?\s]*)(\?[^\s]*)?$/;

// RFC 2396: characters reserved for delimiting URLs.
// We actually just auto-escape these.
var delims$1 = ['<', '>', '"', '`', ' ', '\r', '\n', '\t'];

// RFC 2396: characters not allowed for various reasons.
var unwise$1 = ['{', '}', '|', '\\', '^', '`'].concat(delims$1);

// Allowed by RFCs, but cause of XSS attacks.  Always escape these.
var autoEscape$1 = ['\''].concat(unwise$1);
// Characters that are never ever allowed in a hostname.
// Note that any invalid chars are also handled, but these
// are the ones that are *expected* to be seen, so we fast-path
// them.
var nonHostChars$1 = ['%', '/', '?', ';', '#'].concat(autoEscape$1);
var hostEndingChars$1 = ['/', '?', '#'];
var hostnameMaxLen$1 = 255;
var hostnamePartPattern$1 = /^[+a-z0-9A-Z_-]{0,63}$/;
var hostnamePartStart$1 = /^([+a-z0-9A-Z_-]{0,63})(.*)$/;
// protocols that can allow "unsafe" and "unwise" chars.
// protocols that never have a hostname.
var hostlessProtocol$1 = {
  javascript: true,
  'javascript:': true
};
// protocols that always contain a // bit.
var slashedProtocol$1 = {
  http: true,
  https: true,
  ftp: true,
  gopher: true,
  file: true,
  'http:': true,
  'https:': true,
  'ftp:': true,
  'gopher:': true,
  'file:': true
};

function urlParse$1 (url, slashesDenoteHost) {
  if (url && url instanceof Url$1) { return url }

  var u = new Url$1();
  u.parse(url, slashesDenoteHost);
  return u
}

Url$1.prototype.parse = function (url, slashesDenoteHost) {
  var lowerProto, hec, slashes;
  var rest = url;

  // trim before proceeding.
  // This is to support parse stuff like "  http://foo.com  \n"
  rest = rest.trim();

  if (!slashesDenoteHost && url.split('#').length === 1) {
    // Try fast path regexp
    var simplePath = simplePathPattern$1.exec(rest);
    if (simplePath) {
      this.pathname = simplePath[1];
      if (simplePath[2]) {
        this.search = simplePath[2];
      }
      return this
    }
  }

  var proto = protocolPattern$1.exec(rest);
  if (proto) {
    proto = proto[0];
    lowerProto = proto.toLowerCase();
    this.protocol = proto;
    rest = rest.substr(proto.length);
  }

  // figure out if it's got a host
  // user@server is *always* interpreted as a hostname, and url
  // resolution will treat //foo/bar as host=foo,path=bar because that's
  // how the browser resolves relative URLs.
  /* eslint-disable-next-line no-useless-escape */
  if (slashesDenoteHost || proto || rest.match(/^\/\/[^@\/]+@[^@\/]+/)) {
    slashes = rest.substr(0, 2) === '//';
    if (slashes && !(proto && hostlessProtocol$1[proto])) {
      rest = rest.substr(2);
      this.slashes = true;
    }
  }

  if (!hostlessProtocol$1[proto] &&
      (slashes || (proto && !slashedProtocol$1[proto]))) {
    // there's a hostname.
    // the first instance of /, ?, ;, or # ends the host.
    //
    // If there is an @ in the hostname, then non-host chars *are* allowed
    // to the left of the last @ sign, unless some host-ending character
    // comes *before* the @-sign.
    // URLs are obnoxious.
    //
    // ex:
    // http://a@b@c/ => user:a@b host:c
    // http://a@b?@c => user:a host:c path:/?@c

    // v0.12 TODO(isaacs): This is not quite how Chrome does things.
    // Review our test case against browsers more comprehensively.

    // find the first instance of any hostEndingChars
    var hostEnd = -1;
    for (var i = 0; i < hostEndingChars$1.length; i++) {
      hec = rest.indexOf(hostEndingChars$1[i]);
      if (hec !== -1 && (hostEnd === -1 || hec < hostEnd)) {
        hostEnd = hec;
      }
    }

    // at this point, either we have an explicit point where the
    // auth portion cannot go past, or the last @ char is the decider.
    var auth, atSign;
    if (hostEnd === -1) {
      // atSign can be anywhere.
      atSign = rest.lastIndexOf('@');
    } else {
      // atSign must be in auth portion.
      // http://a@b/c@d => host:b auth:a path:/c@d
      atSign = rest.lastIndexOf('@', hostEnd);
    }

    // Now we have a portion which is definitely the auth.
    // Pull that off.
    if (atSign !== -1) {
      auth = rest.slice(0, atSign);
      rest = rest.slice(atSign + 1);
      this.auth = auth;
    }

    // the host is the remaining to the left of the first non-host char
    hostEnd = -1;
    for (var i$1 = 0; i$1 < nonHostChars$1.length; i$1++) {
      hec = rest.indexOf(nonHostChars$1[i$1]);
      if (hec !== -1 && (hostEnd === -1 || hec < hostEnd)) {
        hostEnd = hec;
      }
    }
    // if we still have not hit it, then the entire thing is a host.
    if (hostEnd === -1) {
      hostEnd = rest.length;
    }

    if (rest[hostEnd - 1] === ':') { hostEnd--; }
    var host = rest.slice(0, hostEnd);
    rest = rest.slice(hostEnd);

    // pull out port.
    this.parseHost(host);

    // we've indicated that there is a hostname,
    // so even if it's empty, it has to be present.
    this.hostname = this.hostname || '';

    // if hostname begins with [ and ends with ]
    // assume that it's an IPv6 address.
    var ipv6Hostname = this.hostname[0] === '[' &&
        this.hostname[this.hostname.length - 1] === ']';

    // validate a little.
    if (!ipv6Hostname) {
      var hostparts = this.hostname.split(/\./);
      for (var i$2 = 0, l = hostparts.length; i$2 < l; i$2++) {
        var part = hostparts[i$2];
        if (!part) { continue }
        if (!part.match(hostnamePartPattern$1)) {
          var newpart = '';
          for (var j = 0, k = part.length; j < k; j++) {
            if (part.charCodeAt(j) > 127) {
              // we replace non-ASCII char with a temporary placeholder
              // we need this to make sure size of hostname is not
              // broken by replacing non-ASCII by nothing
              newpart += 'x';
            } else {
              newpart += part[j];
            }
          }
          // we test again with ASCII char only
          if (!newpart.match(hostnamePartPattern$1)) {
            var validParts = hostparts.slice(0, i$2);
            var notHost = hostparts.slice(i$2 + 1);
            var bit = part.match(hostnamePartStart$1);
            if (bit) {
              validParts.push(bit[1]);
              notHost.unshift(bit[2]);
            }
            if (notHost.length) {
              rest = notHost.join('.') + rest;
            }
            this.hostname = validParts.join('.');
            break
          }
        }
      }
    }

    if (this.hostname.length > hostnameMaxLen$1) {
      this.hostname = '';
    }

    // strip [ and ] from the hostname
    // the host field still retains them, though
    if (ipv6Hostname) {
      this.hostname = this.hostname.substr(1, this.hostname.length - 2);
    }
  }

  // chop off from the tail first.
  var hash = rest.indexOf('#');
  if (hash !== -1) {
    // got a fragment string.
    this.hash = rest.substr(hash);
    rest = rest.slice(0, hash);
  }
  var qm = rest.indexOf('?');
  if (qm !== -1) {
    this.search = rest.substr(qm);
    rest = rest.slice(0, qm);
  }
  if (rest) { this.pathname = rest; }
  if (slashedProtocol$1[lowerProto] &&
      this.hostname && !this.pathname) {
    this.pathname = '';
  }

  return this
};

Url$1.prototype.parseHost = function (host) {
  var port = portPattern$1.exec(host);
  if (port) {
    port = port[0];
    if (port !== ':') {
      this.port = port.substr(1);
    }
    host = host.substr(0, host.length - port.length);
  }
  if (host) { this.hostname = host; }
};var mdurl$2=/*#__PURE__*/Object.freeze({__proto__:null,decode:decode$3,encode:encode$3,format:format$1,parse:urlParse$1});var Any = /[\0-\uD7FF\uE000-\uFFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;var Cc = /[\0-\x1F\x7F-\x9F]/;var regex$6 = /[\xAD\u0600-\u0605\u061C\u06DD\u070F\u0890\u0891\u08E2\u180E\u200B-\u200F\u202A-\u202E\u2060-\u2064\u2066-\u206F\uFEFF\uFFF9-\uFFFB]|\uD804[\uDCBD\uDCCD]|\uD80D[\uDC30-\uDC3F]|\uD82F[\uDCA0-\uDCA3]|\uD834[\uDD73-\uDD7A]|\uDB40[\uDC01\uDC20-\uDC7F]/;var P = /[!-#%-\*,-\/:;\?@\[-\]_\{\}\xA1\xA7\xAB\xB6\xB7\xBB\xBF\u037E\u0387\u055A-\u055F\u0589\u058A\u05BE\u05C0\u05C3\u05C6\u05F3\u05F4\u0609\u060A\u060C\u060D\u061B\u061D-\u061F\u066A-\u066D\u06D4\u0700-\u070D\u07F7-\u07F9\u0830-\u083E\u085E\u0964\u0965\u0970\u09FD\u0A76\u0AF0\u0C77\u0C84\u0DF4\u0E4F\u0E5A\u0E5B\u0F04-\u0F12\u0F14\u0F3A-\u0F3D\u0F85\u0FD0-\u0FD4\u0FD9\u0FDA\u104A-\u104F\u10FB\u1360-\u1368\u1400\u166E\u169B\u169C\u16EB-\u16ED\u1735\u1736\u17D4-\u17D6\u17D8-\u17DA\u1800-\u180A\u1944\u1945\u1A1E\u1A1F\u1AA0-\u1AA6\u1AA8-\u1AAD\u1B5A-\u1B60\u1B7D\u1B7E\u1BFC-\u1BFF\u1C3B-\u1C3F\u1C7E\u1C7F\u1CC0-\u1CC7\u1CD3\u2010-\u2027\u2030-\u2043\u2045-\u2051\u2053-\u205E\u207D\u207E\u208D\u208E\u2308-\u230B\u2329\u232A\u2768-\u2775\u27C5\u27C6\u27E6-\u27EF\u2983-\u2998\u29D8-\u29DB\u29FC\u29FD\u2CF9-\u2CFC\u2CFE\u2CFF\u2D70\u2E00-\u2E2E\u2E30-\u2E4F\u2E52-\u2E5D\u3001-\u3003\u3008-\u3011\u3014-\u301F\u3030\u303D\u30A0\u30FB\uA4FE\uA4FF\uA60D-\uA60F\uA673\uA67E\uA6F2-\uA6F7\uA874-\uA877\uA8CE\uA8CF\uA8F8-\uA8FA\uA8FC\uA92E\uA92F\uA95F\uA9C1-\uA9CD\uA9DE\uA9DF\uAA5C-\uAA5F\uAADE\uAADF\uAAF0\uAAF1\uABEB\uFD3E\uFD3F\uFE10-\uFE19\uFE30-\uFE52\uFE54-\uFE61\uFE63\uFE68\uFE6A\uFE6B\uFF01-\uFF03\uFF05-\uFF0A\uFF0C-\uFF0F\uFF1A\uFF1B\uFF1F\uFF20\uFF3B-\uFF3D\uFF3F\uFF5B\uFF5D\uFF5F-\uFF65]|\uD800[\uDD00-\uDD02\uDF9F\uDFD0]|\uD801\uDD6F|\uD802[\uDC57\uDD1F\uDD3F\uDE50-\uDE58\uDE7F\uDEF0-\uDEF6\uDF39-\uDF3F\uDF99-\uDF9C]|\uD803[\uDEAD\uDF55-\uDF59\uDF86-\uDF89]|\uD804[\uDC47-\uDC4D\uDCBB\uDCBC\uDCBE-\uDCC1\uDD40-\uDD43\uDD74\uDD75\uDDC5-\uDDC8\uDDCD\uDDDB\uDDDD-\uDDDF\uDE38-\uDE3D\uDEA9]|\uD805[\uDC4B-\uDC4F\uDC5A\uDC5B\uDC5D\uDCC6\uDDC1-\uDDD7\uDE41-\uDE43\uDE60-\uDE6C\uDEB9\uDF3C-\uDF3E]|\uD806[\uDC3B\uDD44-\uDD46\uDDE2\uDE3F-\uDE46\uDE9A-\uDE9C\uDE9E-\uDEA2\uDF00-\uDF09]|\uD807[\uDC41-\uDC45\uDC70\uDC71\uDEF7\uDEF8\uDF43-\uDF4F\uDFFF]|\uD809[\uDC70-\uDC74]|\uD80B[\uDFF1\uDFF2]|\uD81A[\uDE6E\uDE6F\uDEF5\uDF37-\uDF3B\uDF44]|\uD81B[\uDE97-\uDE9A\uDFE2]|\uD82F\uDC9F|\uD836[\uDE87-\uDE8B]|\uD83A[\uDD5E\uDD5F]/;var regex$5 = /[\$\+<->\^`\|~\xA2-\xA6\xA8\xA9\xAC\xAE-\xB1\xB4\xB8\xD7\xF7\u02C2-\u02C5\u02D2-\u02DF\u02E5-\u02EB\u02ED\u02EF-\u02FF\u0375\u0384\u0385\u03F6\u0482\u058D-\u058F\u0606-\u0608\u060B\u060E\u060F\u06DE\u06E9\u06FD\u06FE\u07F6\u07FE\u07FF\u0888\u09F2\u09F3\u09FA\u09FB\u0AF1\u0B70\u0BF3-\u0BFA\u0C7F\u0D4F\u0D79\u0E3F\u0F01-\u0F03\u0F13\u0F15-\u0F17\u0F1A-\u0F1F\u0F34\u0F36\u0F38\u0FBE-\u0FC5\u0FC7-\u0FCC\u0FCE\u0FCF\u0FD5-\u0FD8\u109E\u109F\u1390-\u1399\u166D\u17DB\u1940\u19DE-\u19FF\u1B61-\u1B6A\u1B74-\u1B7C\u1FBD\u1FBF-\u1FC1\u1FCD-\u1FCF\u1FDD-\u1FDF\u1FED-\u1FEF\u1FFD\u1FFE\u2044\u2052\u207A-\u207C\u208A-\u208C\u20A0-\u20C0\u2100\u2101\u2103-\u2106\u2108\u2109\u2114\u2116-\u2118\u211E-\u2123\u2125\u2127\u2129\u212E\u213A\u213B\u2140-\u2144\u214A-\u214D\u214F\u218A\u218B\u2190-\u2307\u230C-\u2328\u232B-\u2426\u2440-\u244A\u249C-\u24E9\u2500-\u2767\u2794-\u27C4\u27C7-\u27E5\u27F0-\u2982\u2999-\u29D7\u29DC-\u29FB\u29FE-\u2B73\u2B76-\u2B95\u2B97-\u2BFF\u2CE5-\u2CEA\u2E50\u2E51\u2E80-\u2E99\u2E9B-\u2EF3\u2F00-\u2FD5\u2FF0-\u2FFF\u3004\u3012\u3013\u3020\u3036\u3037\u303E\u303F\u309B\u309C\u3190\u3191\u3196-\u319F\u31C0-\u31E3\u31EF\u3200-\u321E\u322A-\u3247\u3250\u3260-\u327F\u328A-\u32B0\u32C0-\u33FF\u4DC0-\u4DFF\uA490-\uA4C6\uA700-\uA716\uA720\uA721\uA789\uA78A\uA828-\uA82B\uA836-\uA839\uAA77-\uAA79\uAB5B\uAB6A\uAB6B\uFB29\uFBB2-\uFBC2\uFD40-\uFD4F\uFDCF\uFDFC-\uFDFF\uFE62\uFE64-\uFE66\uFE69\uFF04\uFF0B\uFF1C-\uFF1E\uFF3E\uFF40\uFF5C\uFF5E\uFFE0-\uFFE6\uFFE8-\uFFEE\uFFFC\uFFFD]|\uD800[\uDD37-\uDD3F\uDD79-\uDD89\uDD8C-\uDD8E\uDD90-\uDD9C\uDDA0\uDDD0-\uDDFC]|\uD802[\uDC77\uDC78\uDEC8]|\uD805\uDF3F|\uD807[\uDFD5-\uDFF1]|\uD81A[\uDF3C-\uDF3F\uDF45]|\uD82F\uDC9C|\uD833[\uDF50-\uDFC3]|\uD834[\uDC00-\uDCF5\uDD00-\uDD26\uDD29-\uDD64\uDD6A-\uDD6C\uDD83\uDD84\uDD8C-\uDDA9\uDDAE-\uDDEA\uDE00-\uDE41\uDE45\uDF00-\uDF56]|\uD835[\uDEC1\uDEDB\uDEFB\uDF15\uDF35\uDF4F\uDF6F\uDF89\uDFA9\uDFC3]|\uD836[\uDC00-\uDDFF\uDE37-\uDE3A\uDE6D-\uDE74\uDE76-\uDE83\uDE85\uDE86]|\uD838[\uDD4F\uDEFF]|\uD83B[\uDCAC\uDCB0\uDD2E\uDEF0\uDEF1]|\uD83C[\uDC00-\uDC2B\uDC30-\uDC93\uDCA0-\uDCAE\uDCB1-\uDCBF\uDCC1-\uDCCF\uDCD1-\uDCF5\uDD0D-\uDDAD\uDDE6-\uDE02\uDE10-\uDE3B\uDE40-\uDE48\uDE50\uDE51\uDE60-\uDE65\uDF00-\uDFFF]|\uD83D[\uDC00-\uDED7\uDEDC-\uDEEC\uDEF0-\uDEFC\uDF00-\uDF76\uDF7B-\uDFD9\uDFE0-\uDFEB\uDFF0]|\uD83E[\uDC00-\uDC0B\uDC10-\uDC47\uDC50-\uDC59\uDC60-\uDC87\uDC90-\uDCAD\uDCB0\uDCB1\uDD00-\uDE53\uDE60-\uDE6D\uDE70-\uDE7C\uDE80-\uDE88\uDE90-\uDEBD\uDEBF-\uDEC5\uDECE-\uDEDB\uDEE0-\uDEE8\uDEF0-\uDEF8\uDF00-\uDF92\uDF94-\uDFCA]/;var Z = /[ \xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000]/;var ucmicro=/*#__PURE__*/Object.freeze({__proto__:null,Any:Any,Cc:Cc,Cf:regex$6,P:P,S:regex$5,Z:Z});// Generated using scripts/write-decode-map.ts
var htmlDecodeTree = new Uint16Array(
// prettier-ignore
"\u1d41<\xd5\u0131\u028a\u049d\u057b\u05d0\u0675\u06de\u07a2\u07d6\u080f\u0a4a\u0a91\u0da1\u0e6d\u0f09\u0f26\u10ca\u1228\u12e1\u1415\u149d\u14c3\u14df\u1525\0\0\0\0\0\0\u156b\u16cd\u198d\u1c12\u1ddd\u1f7e\u2060\u21b0\u228d\u23c0\u23fb\u2442\u2824\u2912\u2d08\u2e48\u2fce\u3016\u32ba\u3639\u37ac\u38fe\u3a28\u3a71\u3ae0\u3b2e\u0800EMabcfglmnoprstu\\bfms\x7f\x84\x8b\x90\x95\x98\xa6\xb3\xb9\xc8\xcflig\u803b\xc6\u40c6P\u803b&\u4026cute\u803b\xc1\u40c1reve;\u4102\u0100iyx}rc\u803b\xc2\u40c2;\u4410r;\uc000\ud835\udd04rave\u803b\xc0\u40c0pha;\u4391acr;\u4100d;\u6a53\u0100gp\x9d\xa1on;\u4104f;\uc000\ud835\udd38plyFunction;\u6061ing\u803b\xc5\u40c5\u0100cs\xbe\xc3r;\uc000\ud835\udc9cign;\u6254ilde\u803b\xc3\u40c3ml\u803b\xc4\u40c4\u0400aceforsu\xe5\xfb\xfe\u0117\u011c\u0122\u0127\u012a\u0100cr\xea\xf2kslash;\u6216\u0176\xf6\xf8;\u6ae7ed;\u6306y;\u4411\u0180crt\u0105\u010b\u0114ause;\u6235noullis;\u612ca;\u4392r;\uc000\ud835\udd05pf;\uc000\ud835\udd39eve;\u42d8c\xf2\u0113mpeq;\u624e\u0700HOacdefhilorsu\u014d\u0151\u0156\u0180\u019e\u01a2\u01b5\u01b7\u01ba\u01dc\u0215\u0273\u0278\u027ecy;\u4427PY\u803b\xa9\u40a9\u0180cpy\u015d\u0162\u017aute;\u4106\u0100;i\u0167\u0168\u62d2talDifferentialD;\u6145leys;\u612d\u0200aeio\u0189\u018e\u0194\u0198ron;\u410cdil\u803b\xc7\u40c7rc;\u4108nint;\u6230ot;\u410a\u0100dn\u01a7\u01adilla;\u40b8terDot;\u40b7\xf2\u017fi;\u43a7rcle\u0200DMPT\u01c7\u01cb\u01d1\u01d6ot;\u6299inus;\u6296lus;\u6295imes;\u6297o\u0100cs\u01e2\u01f8kwiseContourIntegral;\u6232eCurly\u0100DQ\u0203\u020foubleQuote;\u601duote;\u6019\u0200lnpu\u021e\u0228\u0247\u0255on\u0100;e\u0225\u0226\u6237;\u6a74\u0180git\u022f\u0236\u023aruent;\u6261nt;\u622fourIntegral;\u622e\u0100fr\u024c\u024e;\u6102oduct;\u6210nterClockwiseContourIntegral;\u6233oss;\u6a2fcr;\uc000\ud835\udc9ep\u0100;C\u0284\u0285\u62d3ap;\u624d\u0580DJSZacefios\u02a0\u02ac\u02b0\u02b4\u02b8\u02cb\u02d7\u02e1\u02e6\u0333\u048d\u0100;o\u0179\u02a5trahd;\u6911cy;\u4402cy;\u4405cy;\u440f\u0180grs\u02bf\u02c4\u02c7ger;\u6021r;\u61a1hv;\u6ae4\u0100ay\u02d0\u02d5ron;\u410e;\u4414l\u0100;t\u02dd\u02de\u6207a;\u4394r;\uc000\ud835\udd07\u0100af\u02eb\u0327\u0100cm\u02f0\u0322ritical\u0200ADGT\u0300\u0306\u0316\u031ccute;\u40b4o\u0174\u030b\u030d;\u42d9bleAcute;\u42ddrave;\u4060ilde;\u42dcond;\u62c4ferentialD;\u6146\u0470\u033d\0\0\0\u0342\u0354\0\u0405f;\uc000\ud835\udd3b\u0180;DE\u0348\u0349\u034d\u40a8ot;\u60dcqual;\u6250ble\u0300CDLRUV\u0363\u0372\u0382\u03cf\u03e2\u03f8ontourIntegra\xec\u0239o\u0274\u0379\0\0\u037b\xbb\u0349nArrow;\u61d3\u0100eo\u0387\u03a4ft\u0180ART\u0390\u0396\u03a1rrow;\u61d0ightArrow;\u61d4e\xe5\u02cang\u0100LR\u03ab\u03c4eft\u0100AR\u03b3\u03b9rrow;\u67f8ightArrow;\u67faightArrow;\u67f9ight\u0100AT\u03d8\u03derrow;\u61d2ee;\u62a8p\u0241\u03e9\0\0\u03efrrow;\u61d1ownArrow;\u61d5erticalBar;\u6225n\u0300ABLRTa\u0412\u042a\u0430\u045e\u047f\u037crrow\u0180;BU\u041d\u041e\u0422\u6193ar;\u6913pArrow;\u61f5reve;\u4311eft\u02d2\u043a\0\u0446\0\u0450ightVector;\u6950eeVector;\u695eector\u0100;B\u0459\u045a\u61bdar;\u6956ight\u01d4\u0467\0\u0471eeVector;\u695fector\u0100;B\u047a\u047b\u61c1ar;\u6957ee\u0100;A\u0486\u0487\u62a4rrow;\u61a7\u0100ct\u0492\u0497r;\uc000\ud835\udc9frok;\u4110\u0800NTacdfglmopqstux\u04bd\u04c0\u04c4\u04cb\u04de\u04e2\u04e7\u04ee\u04f5\u0521\u052f\u0536\u0552\u055d\u0560\u0565G;\u414aH\u803b\xd0\u40d0cute\u803b\xc9\u40c9\u0180aiy\u04d2\u04d7\u04dcron;\u411arc\u803b\xca\u40ca;\u442dot;\u4116r;\uc000\ud835\udd08rave\u803b\xc8\u40c8ement;\u6208\u0100ap\u04fa\u04fecr;\u4112ty\u0253\u0506\0\0\u0512mallSquare;\u65fberySmallSquare;\u65ab\u0100gp\u0526\u052aon;\u4118f;\uc000\ud835\udd3csilon;\u4395u\u0100ai\u053c\u0549l\u0100;T\u0542\u0543\u6a75ilde;\u6242librium;\u61cc\u0100ci\u0557\u055ar;\u6130m;\u6a73a;\u4397ml\u803b\xcb\u40cb\u0100ip\u056a\u056fsts;\u6203onentialE;\u6147\u0280cfios\u0585\u0588\u058d\u05b2\u05ccy;\u4424r;\uc000\ud835\udd09lled\u0253\u0597\0\0\u05a3mallSquare;\u65fcerySmallSquare;\u65aa\u0370\u05ba\0\u05bf\0\0\u05c4f;\uc000\ud835\udd3dAll;\u6200riertrf;\u6131c\xf2\u05cb\u0600JTabcdfgorst\u05e8\u05ec\u05ef\u05fa\u0600\u0612\u0616\u061b\u061d\u0623\u066c\u0672cy;\u4403\u803b>\u403emma\u0100;d\u05f7\u05f8\u4393;\u43dcreve;\u411e\u0180eiy\u0607\u060c\u0610dil;\u4122rc;\u411c;\u4413ot;\u4120r;\uc000\ud835\udd0a;\u62d9pf;\uc000\ud835\udd3eeater\u0300EFGLST\u0635\u0644\u064e\u0656\u065b\u0666qual\u0100;L\u063e\u063f\u6265ess;\u62dbullEqual;\u6267reater;\u6aa2ess;\u6277lantEqual;\u6a7eilde;\u6273cr;\uc000\ud835\udca2;\u626b\u0400Aacfiosu\u0685\u068b\u0696\u069b\u069e\u06aa\u06be\u06caRDcy;\u442a\u0100ct\u0690\u0694ek;\u42c7;\u405eirc;\u4124r;\u610clbertSpace;\u610b\u01f0\u06af\0\u06b2f;\u610dizontalLine;\u6500\u0100ct\u06c3\u06c5\xf2\u06a9rok;\u4126mp\u0144\u06d0\u06d8ownHum\xf0\u012fqual;\u624f\u0700EJOacdfgmnostu\u06fa\u06fe\u0703\u0707\u070e\u071a\u071e\u0721\u0728\u0744\u0778\u078b\u078f\u0795cy;\u4415lig;\u4132cy;\u4401cute\u803b\xcd\u40cd\u0100iy\u0713\u0718rc\u803b\xce\u40ce;\u4418ot;\u4130r;\u6111rave\u803b\xcc\u40cc\u0180;ap\u0720\u072f\u073f\u0100cg\u0734\u0737r;\u412ainaryI;\u6148lie\xf3\u03dd\u01f4\u0749\0\u0762\u0100;e\u074d\u074e\u622c\u0100gr\u0753\u0758ral;\u622bsection;\u62c2isible\u0100CT\u076c\u0772omma;\u6063imes;\u6062\u0180gpt\u077f\u0783\u0788on;\u412ef;\uc000\ud835\udd40a;\u4399cr;\u6110ilde;\u4128\u01eb\u079a\0\u079ecy;\u4406l\u803b\xcf\u40cf\u0280cfosu\u07ac\u07b7\u07bc\u07c2\u07d0\u0100iy\u07b1\u07b5rc;\u4134;\u4419r;\uc000\ud835\udd0dpf;\uc000\ud835\udd41\u01e3\u07c7\0\u07ccr;\uc000\ud835\udca5rcy;\u4408kcy;\u4404\u0380HJacfos\u07e4\u07e8\u07ec\u07f1\u07fd\u0802\u0808cy;\u4425cy;\u440cppa;\u439a\u0100ey\u07f6\u07fbdil;\u4136;\u441ar;\uc000\ud835\udd0epf;\uc000\ud835\udd42cr;\uc000\ud835\udca6\u0580JTaceflmost\u0825\u0829\u082c\u0850\u0863\u09b3\u09b8\u09c7\u09cd\u0a37\u0a47cy;\u4409\u803b<\u403c\u0280cmnpr\u0837\u083c\u0841\u0844\u084dute;\u4139bda;\u439bg;\u67ealacetrf;\u6112r;\u619e\u0180aey\u0857\u085c\u0861ron;\u413ddil;\u413b;\u441b\u0100fs\u0868\u0970t\u0500ACDFRTUVar\u087e\u08a9\u08b1\u08e0\u08e6\u08fc\u092f\u095b\u0390\u096a\u0100nr\u0883\u088fgleBracket;\u67e8row\u0180;BR\u0899\u089a\u089e\u6190ar;\u61e4ightArrow;\u61c6eiling;\u6308o\u01f5\u08b7\0\u08c3bleBracket;\u67e6n\u01d4\u08c8\0\u08d2eeVector;\u6961ector\u0100;B\u08db\u08dc\u61c3ar;\u6959loor;\u630aight\u0100AV\u08ef\u08f5rrow;\u6194ector;\u694e\u0100er\u0901\u0917e\u0180;AV\u0909\u090a\u0910\u62a3rrow;\u61a4ector;\u695aiangle\u0180;BE\u0924\u0925\u0929\u62b2ar;\u69cfqual;\u62b4p\u0180DTV\u0937\u0942\u094cownVector;\u6951eeVector;\u6960ector\u0100;B\u0956\u0957\u61bfar;\u6958ector\u0100;B\u0965\u0966\u61bcar;\u6952ight\xe1\u039cs\u0300EFGLST\u097e\u098b\u0995\u099d\u09a2\u09adqualGreater;\u62daullEqual;\u6266reater;\u6276ess;\u6aa1lantEqual;\u6a7dilde;\u6272r;\uc000\ud835\udd0f\u0100;e\u09bd\u09be\u62d8ftarrow;\u61daidot;\u413f\u0180npw\u09d4\u0a16\u0a1bg\u0200LRlr\u09de\u09f7\u0a02\u0a10eft\u0100AR\u09e6\u09ecrrow;\u67f5ightArrow;\u67f7ightArrow;\u67f6eft\u0100ar\u03b3\u0a0aight\xe1\u03bfight\xe1\u03caf;\uc000\ud835\udd43er\u0100LR\u0a22\u0a2ceftArrow;\u6199ightArrow;\u6198\u0180cht\u0a3e\u0a40\u0a42\xf2\u084c;\u61b0rok;\u4141;\u626a\u0400acefiosu\u0a5a\u0a5d\u0a60\u0a77\u0a7c\u0a85\u0a8b\u0a8ep;\u6905y;\u441c\u0100dl\u0a65\u0a6fiumSpace;\u605flintrf;\u6133r;\uc000\ud835\udd10nusPlus;\u6213pf;\uc000\ud835\udd44c\xf2\u0a76;\u439c\u0480Jacefostu\u0aa3\u0aa7\u0aad\u0ac0\u0b14\u0b19\u0d91\u0d97\u0d9ecy;\u440acute;\u4143\u0180aey\u0ab4\u0ab9\u0aberon;\u4147dil;\u4145;\u441d\u0180gsw\u0ac7\u0af0\u0b0eative\u0180MTV\u0ad3\u0adf\u0ae8ediumSpace;\u600bhi\u0100cn\u0ae6\u0ad8\xeb\u0ad9eryThi\xee\u0ad9ted\u0100GL\u0af8\u0b06reaterGreate\xf2\u0673essLes\xf3\u0a48Line;\u400ar;\uc000\ud835\udd11\u0200Bnpt\u0b22\u0b28\u0b37\u0b3areak;\u6060BreakingSpace;\u40a0f;\u6115\u0680;CDEGHLNPRSTV\u0b55\u0b56\u0b6a\u0b7c\u0ba1\u0beb\u0c04\u0c5e\u0c84\u0ca6\u0cd8\u0d61\u0d85\u6aec\u0100ou\u0b5b\u0b64ngruent;\u6262pCap;\u626doubleVerticalBar;\u6226\u0180lqx\u0b83\u0b8a\u0b9bement;\u6209ual\u0100;T\u0b92\u0b93\u6260ilde;\uc000\u2242\u0338ists;\u6204reater\u0380;EFGLST\u0bb6\u0bb7\u0bbd\u0bc9\u0bd3\u0bd8\u0be5\u626fqual;\u6271ullEqual;\uc000\u2267\u0338reater;\uc000\u226b\u0338ess;\u6279lantEqual;\uc000\u2a7e\u0338ilde;\u6275ump\u0144\u0bf2\u0bfdownHump;\uc000\u224e\u0338qual;\uc000\u224f\u0338e\u0100fs\u0c0a\u0c27tTriangle\u0180;BE\u0c1a\u0c1b\u0c21\u62eaar;\uc000\u29cf\u0338qual;\u62ecs\u0300;EGLST\u0c35\u0c36\u0c3c\u0c44\u0c4b\u0c58\u626equal;\u6270reater;\u6278ess;\uc000\u226a\u0338lantEqual;\uc000\u2a7d\u0338ilde;\u6274ested\u0100GL\u0c68\u0c79reaterGreater;\uc000\u2aa2\u0338essLess;\uc000\u2aa1\u0338recedes\u0180;ES\u0c92\u0c93\u0c9b\u6280qual;\uc000\u2aaf\u0338lantEqual;\u62e0\u0100ei\u0cab\u0cb9verseElement;\u620cghtTriangle\u0180;BE\u0ccb\u0ccc\u0cd2\u62ebar;\uc000\u29d0\u0338qual;\u62ed\u0100qu\u0cdd\u0d0cuareSu\u0100bp\u0ce8\u0cf9set\u0100;E\u0cf0\u0cf3\uc000\u228f\u0338qual;\u62e2erset\u0100;E\u0d03\u0d06\uc000\u2290\u0338qual;\u62e3\u0180bcp\u0d13\u0d24\u0d4eset\u0100;E\u0d1b\u0d1e\uc000\u2282\u20d2qual;\u6288ceeds\u0200;EST\u0d32\u0d33\u0d3b\u0d46\u6281qual;\uc000\u2ab0\u0338lantEqual;\u62e1ilde;\uc000\u227f\u0338erset\u0100;E\u0d58\u0d5b\uc000\u2283\u20d2qual;\u6289ilde\u0200;EFT\u0d6e\u0d6f\u0d75\u0d7f\u6241qual;\u6244ullEqual;\u6247ilde;\u6249erticalBar;\u6224cr;\uc000\ud835\udca9ilde\u803b\xd1\u40d1;\u439d\u0700Eacdfgmoprstuv\u0dbd\u0dc2\u0dc9\u0dd5\u0ddb\u0de0\u0de7\u0dfc\u0e02\u0e20\u0e22\u0e32\u0e3f\u0e44lig;\u4152cute\u803b\xd3\u40d3\u0100iy\u0dce\u0dd3rc\u803b\xd4\u40d4;\u441eblac;\u4150r;\uc000\ud835\udd12rave\u803b\xd2\u40d2\u0180aei\u0dee\u0df2\u0df6cr;\u414cga;\u43a9cron;\u439fpf;\uc000\ud835\udd46enCurly\u0100DQ\u0e0e\u0e1aoubleQuote;\u601cuote;\u6018;\u6a54\u0100cl\u0e27\u0e2cr;\uc000\ud835\udcaaash\u803b\xd8\u40d8i\u016c\u0e37\u0e3cde\u803b\xd5\u40d5es;\u6a37ml\u803b\xd6\u40d6er\u0100BP\u0e4b\u0e60\u0100ar\u0e50\u0e53r;\u603eac\u0100ek\u0e5a\u0e5c;\u63deet;\u63b4arenthesis;\u63dc\u0480acfhilors\u0e7f\u0e87\u0e8a\u0e8f\u0e92\u0e94\u0e9d\u0eb0\u0efcrtialD;\u6202y;\u441fr;\uc000\ud835\udd13i;\u43a6;\u43a0usMinus;\u40b1\u0100ip\u0ea2\u0eadncareplan\xe5\u069df;\u6119\u0200;eio\u0eb9\u0eba\u0ee0\u0ee4\u6abbcedes\u0200;EST\u0ec8\u0ec9\u0ecf\u0eda\u627aqual;\u6aaflantEqual;\u627cilde;\u627eme;\u6033\u0100dp\u0ee9\u0eeeuct;\u620fortion\u0100;a\u0225\u0ef9l;\u621d\u0100ci\u0f01\u0f06r;\uc000\ud835\udcab;\u43a8\u0200Ufos\u0f11\u0f16\u0f1b\u0f1fOT\u803b\"\u4022r;\uc000\ud835\udd14pf;\u611acr;\uc000\ud835\udcac\u0600BEacefhiorsu\u0f3e\u0f43\u0f47\u0f60\u0f73\u0fa7\u0faa\u0fad\u1096\u10a9\u10b4\u10bearr;\u6910G\u803b\xae\u40ae\u0180cnr\u0f4e\u0f53\u0f56ute;\u4154g;\u67ebr\u0100;t\u0f5c\u0f5d\u61a0l;\u6916\u0180aey\u0f67\u0f6c\u0f71ron;\u4158dil;\u4156;\u4420\u0100;v\u0f78\u0f79\u611cerse\u0100EU\u0f82\u0f99\u0100lq\u0f87\u0f8eement;\u620builibrium;\u61cbpEquilibrium;\u696fr\xbb\u0f79o;\u43a1ght\u0400ACDFTUVa\u0fc1\u0feb\u0ff3\u1022\u1028\u105b\u1087\u03d8\u0100nr\u0fc6\u0fd2gleBracket;\u67e9row\u0180;BL\u0fdc\u0fdd\u0fe1\u6192ar;\u61e5eftArrow;\u61c4eiling;\u6309o\u01f5\u0ff9\0\u1005bleBracket;\u67e7n\u01d4\u100a\0\u1014eeVector;\u695dector\u0100;B\u101d\u101e\u61c2ar;\u6955loor;\u630b\u0100er\u102d\u1043e\u0180;AV\u1035\u1036\u103c\u62a2rrow;\u61a6ector;\u695biangle\u0180;BE\u1050\u1051\u1055\u62b3ar;\u69d0qual;\u62b5p\u0180DTV\u1063\u106e\u1078ownVector;\u694feeVector;\u695cector\u0100;B\u1082\u1083\u61bear;\u6954ector\u0100;B\u1091\u1092\u61c0ar;\u6953\u0100pu\u109b\u109ef;\u611dndImplies;\u6970ightarrow;\u61db\u0100ch\u10b9\u10bcr;\u611b;\u61b1leDelayed;\u69f4\u0680HOacfhimoqstu\u10e4\u10f1\u10f7\u10fd\u1119\u111e\u1151\u1156\u1161\u1167\u11b5\u11bb\u11bf\u0100Cc\u10e9\u10eeHcy;\u4429y;\u4428FTcy;\u442ccute;\u415a\u0280;aeiy\u1108\u1109\u110e\u1113\u1117\u6abcron;\u4160dil;\u415erc;\u415c;\u4421r;\uc000\ud835\udd16ort\u0200DLRU\u112a\u1134\u113e\u1149ownArrow\xbb\u041eeftArrow\xbb\u089aightArrow\xbb\u0fddpArrow;\u6191gma;\u43a3allCircle;\u6218pf;\uc000\ud835\udd4a\u0272\u116d\0\0\u1170t;\u621aare\u0200;ISU\u117b\u117c\u1189\u11af\u65a1ntersection;\u6293u\u0100bp\u118f\u119eset\u0100;E\u1197\u1198\u628fqual;\u6291erset\u0100;E\u11a8\u11a9\u6290qual;\u6292nion;\u6294cr;\uc000\ud835\udcaear;\u62c6\u0200bcmp\u11c8\u11db\u1209\u120b\u0100;s\u11cd\u11ce\u62d0et\u0100;E\u11cd\u11d5qual;\u6286\u0100ch\u11e0\u1205eeds\u0200;EST\u11ed\u11ee\u11f4\u11ff\u627bqual;\u6ab0lantEqual;\u627dilde;\u627fTh\xe1\u0f8c;\u6211\u0180;es\u1212\u1213\u1223\u62d1rset\u0100;E\u121c\u121d\u6283qual;\u6287et\xbb\u1213\u0580HRSacfhiors\u123e\u1244\u1249\u1255\u125e\u1271\u1276\u129f\u12c2\u12c8\u12d1ORN\u803b\xde\u40deADE;\u6122\u0100Hc\u124e\u1252cy;\u440by;\u4426\u0100bu\u125a\u125c;\u4009;\u43a4\u0180aey\u1265\u126a\u126fron;\u4164dil;\u4162;\u4422r;\uc000\ud835\udd17\u0100ei\u127b\u1289\u01f2\u1280\0\u1287efore;\u6234a;\u4398\u0100cn\u128e\u1298kSpace;\uc000\u205f\u200aSpace;\u6009lde\u0200;EFT\u12ab\u12ac\u12b2\u12bc\u623cqual;\u6243ullEqual;\u6245ilde;\u6248pf;\uc000\ud835\udd4bipleDot;\u60db\u0100ct\u12d6\u12dbr;\uc000\ud835\udcafrok;\u4166\u0ae1\u12f7\u130e\u131a\u1326\0\u132c\u1331\0\0\0\0\0\u1338\u133d\u1377\u1385\0\u13ff\u1404\u140a\u1410\u0100cr\u12fb\u1301ute\u803b\xda\u40dar\u0100;o\u1307\u1308\u619fcir;\u6949r\u01e3\u1313\0\u1316y;\u440eve;\u416c\u0100iy\u131e\u1323rc\u803b\xdb\u40db;\u4423blac;\u4170r;\uc000\ud835\udd18rave\u803b\xd9\u40d9acr;\u416a\u0100di\u1341\u1369er\u0100BP\u1348\u135d\u0100ar\u134d\u1350r;\u405fac\u0100ek\u1357\u1359;\u63dfet;\u63b5arenthesis;\u63ddon\u0100;P\u1370\u1371\u62c3lus;\u628e\u0100gp\u137b\u137fon;\u4172f;\uc000\ud835\udd4c\u0400ADETadps\u1395\u13ae\u13b8\u13c4\u03e8\u13d2\u13d7\u13f3rrow\u0180;BD\u1150\u13a0\u13a4ar;\u6912ownArrow;\u61c5ownArrow;\u6195quilibrium;\u696eee\u0100;A\u13cb\u13cc\u62a5rrow;\u61a5own\xe1\u03f3er\u0100LR\u13de\u13e8eftArrow;\u6196ightArrow;\u6197i\u0100;l\u13f9\u13fa\u43d2on;\u43a5ing;\u416ecr;\uc000\ud835\udcb0ilde;\u4168ml\u803b\xdc\u40dc\u0480Dbcdefosv\u1427\u142c\u1430\u1433\u143e\u1485\u148a\u1490\u1496ash;\u62abar;\u6aeby;\u4412ash\u0100;l\u143b\u143c\u62a9;\u6ae6\u0100er\u1443\u1445;\u62c1\u0180bty\u144c\u1450\u147aar;\u6016\u0100;i\u144f\u1455cal\u0200BLST\u1461\u1465\u146a\u1474ar;\u6223ine;\u407ceparator;\u6758ilde;\u6240ThinSpace;\u600ar;\uc000\ud835\udd19pf;\uc000\ud835\udd4dcr;\uc000\ud835\udcb1dash;\u62aa\u0280cefos\u14a7\u14ac\u14b1\u14b6\u14bcirc;\u4174dge;\u62c0r;\uc000\ud835\udd1apf;\uc000\ud835\udd4ecr;\uc000\ud835\udcb2\u0200fios\u14cb\u14d0\u14d2\u14d8r;\uc000\ud835\udd1b;\u439epf;\uc000\ud835\udd4fcr;\uc000\ud835\udcb3\u0480AIUacfosu\u14f1\u14f5\u14f9\u14fd\u1504\u150f\u1514\u151a\u1520cy;\u442fcy;\u4407cy;\u442ecute\u803b\xdd\u40dd\u0100iy\u1509\u150drc;\u4176;\u442br;\uc000\ud835\udd1cpf;\uc000\ud835\udd50cr;\uc000\ud835\udcb4ml;\u4178\u0400Hacdefos\u1535\u1539\u153f\u154b\u154f\u155d\u1560\u1564cy;\u4416cute;\u4179\u0100ay\u1544\u1549ron;\u417d;\u4417ot;\u417b\u01f2\u1554\0\u155boWidt\xe8\u0ad9a;\u4396r;\u6128pf;\u6124cr;\uc000\ud835\udcb5\u0be1\u1583\u158a\u1590\0\u15b0\u15b6\u15bf\0\0\0\0\u15c6\u15db\u15eb\u165f\u166d\0\u1695\u169b\u16b2\u16b9\0\u16becute\u803b\xe1\u40e1reve;\u4103\u0300;Ediuy\u159c\u159d\u15a1\u15a3\u15a8\u15ad\u623e;\uc000\u223e\u0333;\u623frc\u803b\xe2\u40e2te\u80bb\xb4\u0306;\u4430lig\u803b\xe6\u40e6\u0100;r\xb2\u15ba;\uc000\ud835\udd1erave\u803b\xe0\u40e0\u0100ep\u15ca\u15d6\u0100fp\u15cf\u15d4sym;\u6135\xe8\u15d3ha;\u43b1\u0100ap\u15dfc\u0100cl\u15e4\u15e7r;\u4101g;\u6a3f\u0264\u15f0\0\0\u160a\u0280;adsv\u15fa\u15fb\u15ff\u1601\u1607\u6227nd;\u6a55;\u6a5clope;\u6a58;\u6a5a\u0380;elmrsz\u1618\u1619\u161b\u161e\u163f\u164f\u1659\u6220;\u69a4e\xbb\u1619sd\u0100;a\u1625\u1626\u6221\u0461\u1630\u1632\u1634\u1636\u1638\u163a\u163c\u163e;\u69a8;\u69a9;\u69aa;\u69ab;\u69ac;\u69ad;\u69ae;\u69aft\u0100;v\u1645\u1646\u621fb\u0100;d\u164c\u164d\u62be;\u699d\u0100pt\u1654\u1657h;\u6222\xbb\xb9arr;\u637c\u0100gp\u1663\u1667on;\u4105f;\uc000\ud835\udd52\u0380;Eaeiop\u12c1\u167b\u167d\u1682\u1684\u1687\u168a;\u6a70cir;\u6a6f;\u624ad;\u624bs;\u4027rox\u0100;e\u12c1\u1692\xf1\u1683ing\u803b\xe5\u40e5\u0180cty\u16a1\u16a6\u16a8r;\uc000\ud835\udcb6;\u402amp\u0100;e\u12c1\u16af\xf1\u0288ilde\u803b\xe3\u40e3ml\u803b\xe4\u40e4\u0100ci\u16c2\u16c8onin\xf4\u0272nt;\u6a11\u0800Nabcdefiklnoprsu\u16ed\u16f1\u1730\u173c\u1743\u1748\u1778\u177d\u17e0\u17e6\u1839\u1850\u170d\u193d\u1948\u1970ot;\u6aed\u0100cr\u16f6\u171ek\u0200ceps\u1700\u1705\u170d\u1713ong;\u624cpsilon;\u43f6rime;\u6035im\u0100;e\u171a\u171b\u623dq;\u62cd\u0176\u1722\u1726ee;\u62bded\u0100;g\u172c\u172d\u6305e\xbb\u172drk\u0100;t\u135c\u1737brk;\u63b6\u0100oy\u1701\u1741;\u4431quo;\u601e\u0280cmprt\u1753\u175b\u1761\u1764\u1768aus\u0100;e\u010a\u0109ptyv;\u69b0s\xe9\u170cno\xf5\u0113\u0180ahw\u176f\u1771\u1773;\u43b2;\u6136een;\u626cr;\uc000\ud835\udd1fg\u0380costuvw\u178d\u179d\u17b3\u17c1\u17d5\u17db\u17de\u0180aiu\u1794\u1796\u179a\xf0\u0760rc;\u65efp\xbb\u1371\u0180dpt\u17a4\u17a8\u17adot;\u6a00lus;\u6a01imes;\u6a02\u0271\u17b9\0\0\u17becup;\u6a06ar;\u6605riangle\u0100du\u17cd\u17d2own;\u65bdp;\u65b3plus;\u6a04e\xe5\u1444\xe5\u14adarow;\u690d\u0180ako\u17ed\u1826\u1835\u0100cn\u17f2\u1823k\u0180lst\u17fa\u05ab\u1802ozenge;\u69ebriangle\u0200;dlr\u1812\u1813\u1818\u181d\u65b4own;\u65beeft;\u65c2ight;\u65b8k;\u6423\u01b1\u182b\0\u1833\u01b2\u182f\0\u1831;\u6592;\u65914;\u6593ck;\u6588\u0100eo\u183e\u184d\u0100;q\u1843\u1846\uc000=\u20e5uiv;\uc000\u2261\u20e5t;\u6310\u0200ptwx\u1859\u185e\u1867\u186cf;\uc000\ud835\udd53\u0100;t\u13cb\u1863om\xbb\u13cctie;\u62c8\u0600DHUVbdhmptuv\u1885\u1896\u18aa\u18bb\u18d7\u18db\u18ec\u18ff\u1905\u190a\u1910\u1921\u0200LRlr\u188e\u1890\u1892\u1894;\u6557;\u6554;\u6556;\u6553\u0280;DUdu\u18a1\u18a2\u18a4\u18a6\u18a8\u6550;\u6566;\u6569;\u6564;\u6567\u0200LRlr\u18b3\u18b5\u18b7\u18b9;\u655d;\u655a;\u655c;\u6559\u0380;HLRhlr\u18ca\u18cb\u18cd\u18cf\u18d1\u18d3\u18d5\u6551;\u656c;\u6563;\u6560;\u656b;\u6562;\u655fox;\u69c9\u0200LRlr\u18e4\u18e6\u18e8\u18ea;\u6555;\u6552;\u6510;\u650c\u0280;DUdu\u06bd\u18f7\u18f9\u18fb\u18fd;\u6565;\u6568;\u652c;\u6534inus;\u629flus;\u629eimes;\u62a0\u0200LRlr\u1919\u191b\u191d\u191f;\u655b;\u6558;\u6518;\u6514\u0380;HLRhlr\u1930\u1931\u1933\u1935\u1937\u1939\u193b\u6502;\u656a;\u6561;\u655e;\u653c;\u6524;\u651c\u0100ev\u0123\u1942bar\u803b\xa6\u40a6\u0200ceio\u1951\u1956\u195a\u1960r;\uc000\ud835\udcb7mi;\u604fm\u0100;e\u171a\u171cl\u0180;bh\u1968\u1969\u196b\u405c;\u69c5sub;\u67c8\u016c\u1974\u197el\u0100;e\u1979\u197a\u6022t\xbb\u197ap\u0180;Ee\u012f\u1985\u1987;\u6aae\u0100;q\u06dc\u06db\u0ce1\u19a7\0\u19e8\u1a11\u1a15\u1a32\0\u1a37\u1a50\0\0\u1ab4\0\0\u1ac1\0\0\u1b21\u1b2e\u1b4d\u1b52\0\u1bfd\0\u1c0c\u0180cpr\u19ad\u19b2\u19ddute;\u4107\u0300;abcds\u19bf\u19c0\u19c4\u19ca\u19d5\u19d9\u6229nd;\u6a44rcup;\u6a49\u0100au\u19cf\u19d2p;\u6a4bp;\u6a47ot;\u6a40;\uc000\u2229\ufe00\u0100eo\u19e2\u19e5t;\u6041\xee\u0693\u0200aeiu\u19f0\u19fb\u1a01\u1a05\u01f0\u19f5\0\u19f8s;\u6a4don;\u410ddil\u803b\xe7\u40e7rc;\u4109ps\u0100;s\u1a0c\u1a0d\u6a4cm;\u6a50ot;\u410b\u0180dmn\u1a1b\u1a20\u1a26il\u80bb\xb8\u01adptyv;\u69b2t\u8100\xa2;e\u1a2d\u1a2e\u40a2r\xe4\u01b2r;\uc000\ud835\udd20\u0180cei\u1a3d\u1a40\u1a4dy;\u4447ck\u0100;m\u1a47\u1a48\u6713ark\xbb\u1a48;\u43c7r\u0380;Ecefms\u1a5f\u1a60\u1a62\u1a6b\u1aa4\u1aaa\u1aae\u65cb;\u69c3\u0180;el\u1a69\u1a6a\u1a6d\u42c6q;\u6257e\u0261\u1a74\0\0\u1a88rrow\u0100lr\u1a7c\u1a81eft;\u61baight;\u61bb\u0280RSacd\u1a92\u1a94\u1a96\u1a9a\u1a9f\xbb\u0f47;\u64c8st;\u629birc;\u629aash;\u629dnint;\u6a10id;\u6aefcir;\u69c2ubs\u0100;u\u1abb\u1abc\u6663it\xbb\u1abc\u02ec\u1ac7\u1ad4\u1afa\0\u1b0aon\u0100;e\u1acd\u1ace\u403a\u0100;q\xc7\xc6\u026d\u1ad9\0\0\u1ae2a\u0100;t\u1ade\u1adf\u402c;\u4040\u0180;fl\u1ae8\u1ae9\u1aeb\u6201\xee\u1160e\u0100mx\u1af1\u1af6ent\xbb\u1ae9e\xf3\u024d\u01e7\u1afe\0\u1b07\u0100;d\u12bb\u1b02ot;\u6a6dn\xf4\u0246\u0180fry\u1b10\u1b14\u1b17;\uc000\ud835\udd54o\xe4\u0254\u8100\xa9;s\u0155\u1b1dr;\u6117\u0100ao\u1b25\u1b29rr;\u61b5ss;\u6717\u0100cu\u1b32\u1b37r;\uc000\ud835\udcb8\u0100bp\u1b3c\u1b44\u0100;e\u1b41\u1b42\u6acf;\u6ad1\u0100;e\u1b49\u1b4a\u6ad0;\u6ad2dot;\u62ef\u0380delprvw\u1b60\u1b6c\u1b77\u1b82\u1bac\u1bd4\u1bf9arr\u0100lr\u1b68\u1b6a;\u6938;\u6935\u0270\u1b72\0\0\u1b75r;\u62dec;\u62dfarr\u0100;p\u1b7f\u1b80\u61b6;\u693d\u0300;bcdos\u1b8f\u1b90\u1b96\u1ba1\u1ba5\u1ba8\u622arcap;\u6a48\u0100au\u1b9b\u1b9ep;\u6a46p;\u6a4aot;\u628dr;\u6a45;\uc000\u222a\ufe00\u0200alrv\u1bb5\u1bbf\u1bde\u1be3rr\u0100;m\u1bbc\u1bbd\u61b7;\u693cy\u0180evw\u1bc7\u1bd4\u1bd8q\u0270\u1bce\0\0\u1bd2re\xe3\u1b73u\xe3\u1b75ee;\u62ceedge;\u62cfen\u803b\xa4\u40a4earrow\u0100lr\u1bee\u1bf3eft\xbb\u1b80ight\xbb\u1bbde\xe4\u1bdd\u0100ci\u1c01\u1c07onin\xf4\u01f7nt;\u6231lcty;\u632d\u0980AHabcdefhijlorstuwz\u1c38\u1c3b\u1c3f\u1c5d\u1c69\u1c75\u1c8a\u1c9e\u1cac\u1cb7\u1cfb\u1cff\u1d0d\u1d7b\u1d91\u1dab\u1dbb\u1dc6\u1dcdr\xf2\u0381ar;\u6965\u0200glrs\u1c48\u1c4d\u1c52\u1c54ger;\u6020eth;\u6138\xf2\u1133h\u0100;v\u1c5a\u1c5b\u6010\xbb\u090a\u016b\u1c61\u1c67arow;\u690fa\xe3\u0315\u0100ay\u1c6e\u1c73ron;\u410f;\u4434\u0180;ao\u0332\u1c7c\u1c84\u0100gr\u02bf\u1c81r;\u61catseq;\u6a77\u0180glm\u1c91\u1c94\u1c98\u803b\xb0\u40b0ta;\u43b4ptyv;\u69b1\u0100ir\u1ca3\u1ca8sht;\u697f;\uc000\ud835\udd21ar\u0100lr\u1cb3\u1cb5\xbb\u08dc\xbb\u101e\u0280aegsv\u1cc2\u0378\u1cd6\u1cdc\u1ce0m\u0180;os\u0326\u1cca\u1cd4nd\u0100;s\u0326\u1cd1uit;\u6666amma;\u43ddin;\u62f2\u0180;io\u1ce7\u1ce8\u1cf8\u40f7de\u8100\xf7;o\u1ce7\u1cf0ntimes;\u62c7n\xf8\u1cf7cy;\u4452c\u026f\u1d06\0\0\u1d0arn;\u631eop;\u630d\u0280lptuw\u1d18\u1d1d\u1d22\u1d49\u1d55lar;\u4024f;\uc000\ud835\udd55\u0280;emps\u030b\u1d2d\u1d37\u1d3d\u1d42q\u0100;d\u0352\u1d33ot;\u6251inus;\u6238lus;\u6214quare;\u62a1blebarwedg\xe5\xfan\u0180adh\u112e\u1d5d\u1d67ownarrow\xf3\u1c83arpoon\u0100lr\u1d72\u1d76ef\xf4\u1cb4igh\xf4\u1cb6\u0162\u1d7f\u1d85karo\xf7\u0f42\u026f\u1d8a\0\0\u1d8ern;\u631fop;\u630c\u0180cot\u1d98\u1da3\u1da6\u0100ry\u1d9d\u1da1;\uc000\ud835\udcb9;\u4455l;\u69f6rok;\u4111\u0100dr\u1db0\u1db4ot;\u62f1i\u0100;f\u1dba\u1816\u65bf\u0100ah\u1dc0\u1dc3r\xf2\u0429a\xf2\u0fa6angle;\u69a6\u0100ci\u1dd2\u1dd5y;\u445fgrarr;\u67ff\u0900Dacdefglmnopqrstux\u1e01\u1e09\u1e19\u1e38\u0578\u1e3c\u1e49\u1e61\u1e7e\u1ea5\u1eaf\u1ebd\u1ee1\u1f2a\u1f37\u1f44\u1f4e\u1f5a\u0100Do\u1e06\u1d34o\xf4\u1c89\u0100cs\u1e0e\u1e14ute\u803b\xe9\u40e9ter;\u6a6e\u0200aioy\u1e22\u1e27\u1e31\u1e36ron;\u411br\u0100;c\u1e2d\u1e2e\u6256\u803b\xea\u40ealon;\u6255;\u444dot;\u4117\u0100Dr\u1e41\u1e45ot;\u6252;\uc000\ud835\udd22\u0180;rs\u1e50\u1e51\u1e57\u6a9aave\u803b\xe8\u40e8\u0100;d\u1e5c\u1e5d\u6a96ot;\u6a98\u0200;ils\u1e6a\u1e6b\u1e72\u1e74\u6a99nters;\u63e7;\u6113\u0100;d\u1e79\u1e7a\u6a95ot;\u6a97\u0180aps\u1e85\u1e89\u1e97cr;\u4113ty\u0180;sv\u1e92\u1e93\u1e95\u6205et\xbb\u1e93p\u01001;\u1e9d\u1ea4\u0133\u1ea1\u1ea3;\u6004;\u6005\u6003\u0100gs\u1eaa\u1eac;\u414bp;\u6002\u0100gp\u1eb4\u1eb8on;\u4119f;\uc000\ud835\udd56\u0180als\u1ec4\u1ece\u1ed2r\u0100;s\u1eca\u1ecb\u62d5l;\u69e3us;\u6a71i\u0180;lv\u1eda\u1edb\u1edf\u43b5on\xbb\u1edb;\u43f5\u0200csuv\u1eea\u1ef3\u1f0b\u1f23\u0100io\u1eef\u1e31rc\xbb\u1e2e\u0269\u1ef9\0\0\u1efb\xed\u0548ant\u0100gl\u1f02\u1f06tr\xbb\u1e5dess\xbb\u1e7a\u0180aei\u1f12\u1f16\u1f1als;\u403dst;\u625fv\u0100;D\u0235\u1f20D;\u6a78parsl;\u69e5\u0100Da\u1f2f\u1f33ot;\u6253rr;\u6971\u0180cdi\u1f3e\u1f41\u1ef8r;\u612fo\xf4\u0352\u0100ah\u1f49\u1f4b;\u43b7\u803b\xf0\u40f0\u0100mr\u1f53\u1f57l\u803b\xeb\u40ebo;\u60ac\u0180cip\u1f61\u1f64\u1f67l;\u4021s\xf4\u056e\u0100eo\u1f6c\u1f74ctatio\xee\u0559nential\xe5\u0579\u09e1\u1f92\0\u1f9e\0\u1fa1\u1fa7\0\0\u1fc6\u1fcc\0\u1fd3\0\u1fe6\u1fea\u2000\0\u2008\u205allingdotse\xf1\u1e44y;\u4444male;\u6640\u0180ilr\u1fad\u1fb3\u1fc1lig;\u8000\ufb03\u0269\u1fb9\0\0\u1fbdg;\u8000\ufb00ig;\u8000\ufb04;\uc000\ud835\udd23lig;\u8000\ufb01lig;\uc000fj\u0180alt\u1fd9\u1fdc\u1fe1t;\u666dig;\u8000\ufb02ns;\u65b1of;\u4192\u01f0\u1fee\0\u1ff3f;\uc000\ud835\udd57\u0100ak\u05bf\u1ff7\u0100;v\u1ffc\u1ffd\u62d4;\u6ad9artint;\u6a0d\u0100ao\u200c\u2055\u0100cs\u2011\u2052\u03b1\u201a\u2030\u2038\u2045\u2048\0\u2050\u03b2\u2022\u2025\u2027\u202a\u202c\0\u202e\u803b\xbd\u40bd;\u6153\u803b\xbc\u40bc;\u6155;\u6159;\u615b\u01b3\u2034\0\u2036;\u6154;\u6156\u02b4\u203e\u2041\0\0\u2043\u803b\xbe\u40be;\u6157;\u615c5;\u6158\u01b6\u204c\0\u204e;\u615a;\u615d8;\u615el;\u6044wn;\u6322cr;\uc000\ud835\udcbb\u0880Eabcdefgijlnorstv\u2082\u2089\u209f\u20a5\u20b0\u20b4\u20f0\u20f5\u20fa\u20ff\u2103\u2112\u2138\u0317\u213e\u2152\u219e\u0100;l\u064d\u2087;\u6a8c\u0180cmp\u2090\u2095\u209dute;\u41f5ma\u0100;d\u209c\u1cda\u43b3;\u6a86reve;\u411f\u0100iy\u20aa\u20aerc;\u411d;\u4433ot;\u4121\u0200;lqs\u063e\u0642\u20bd\u20c9\u0180;qs\u063e\u064c\u20c4lan\xf4\u0665\u0200;cdl\u0665\u20d2\u20d5\u20e5c;\u6aa9ot\u0100;o\u20dc\u20dd\u6a80\u0100;l\u20e2\u20e3\u6a82;\u6a84\u0100;e\u20ea\u20ed\uc000\u22db\ufe00s;\u6a94r;\uc000\ud835\udd24\u0100;g\u0673\u061bmel;\u6137cy;\u4453\u0200;Eaj\u065a\u210c\u210e\u2110;\u6a92;\u6aa5;\u6aa4\u0200Eaes\u211b\u211d\u2129\u2134;\u6269p\u0100;p\u2123\u2124\u6a8arox\xbb\u2124\u0100;q\u212e\u212f\u6a88\u0100;q\u212e\u211bim;\u62e7pf;\uc000\ud835\udd58\u0100ci\u2143\u2146r;\u610am\u0180;el\u066b\u214e\u2150;\u6a8e;\u6a90\u8300>;cdlqr\u05ee\u2160\u216a\u216e\u2173\u2179\u0100ci\u2165\u2167;\u6aa7r;\u6a7aot;\u62d7Par;\u6995uest;\u6a7c\u0280adels\u2184\u216a\u2190\u0656\u219b\u01f0\u2189\0\u218epro\xf8\u209er;\u6978q\u0100lq\u063f\u2196les\xf3\u2088i\xed\u066b\u0100en\u21a3\u21adrtneqq;\uc000\u2269\ufe00\xc5\u21aa\u0500Aabcefkosy\u21c4\u21c7\u21f1\u21f5\u21fa\u2218\u221d\u222f\u2268\u227dr\xf2\u03a0\u0200ilmr\u21d0\u21d4\u21d7\u21dbrs\xf0\u1484f\xbb\u2024il\xf4\u06a9\u0100dr\u21e0\u21e4cy;\u444a\u0180;cw\u08f4\u21eb\u21efir;\u6948;\u61adar;\u610firc;\u4125\u0180alr\u2201\u220e\u2213rts\u0100;u\u2209\u220a\u6665it\xbb\u220alip;\u6026con;\u62b9r;\uc000\ud835\udd25s\u0100ew\u2223\u2229arow;\u6925arow;\u6926\u0280amopr\u223a\u223e\u2243\u225e\u2263rr;\u61fftht;\u623bk\u0100lr\u2249\u2253eftarrow;\u61a9ightarrow;\u61aaf;\uc000\ud835\udd59bar;\u6015\u0180clt\u226f\u2274\u2278r;\uc000\ud835\udcbdas\xe8\u21f4rok;\u4127\u0100bp\u2282\u2287ull;\u6043hen\xbb\u1c5b\u0ae1\u22a3\0\u22aa\0\u22b8\u22c5\u22ce\0\u22d5\u22f3\0\0\u22f8\u2322\u2367\u2362\u237f\0\u2386\u23aa\u23b4cute\u803b\xed\u40ed\u0180;iy\u0771\u22b0\u22b5rc\u803b\xee\u40ee;\u4438\u0100cx\u22bc\u22bfy;\u4435cl\u803b\xa1\u40a1\u0100fr\u039f\u22c9;\uc000\ud835\udd26rave\u803b\xec\u40ec\u0200;ino\u073e\u22dd\u22e9\u22ee\u0100in\u22e2\u22e6nt;\u6a0ct;\u622dfin;\u69dcta;\u6129lig;\u4133\u0180aop\u22fe\u231a\u231d\u0180cgt\u2305\u2308\u2317r;\u412b\u0180elp\u071f\u230f\u2313in\xe5\u078ear\xf4\u0720h;\u4131f;\u62b7ed;\u41b5\u0280;cfot\u04f4\u232c\u2331\u233d\u2341are;\u6105in\u0100;t\u2338\u2339\u621eie;\u69dddo\xf4\u2319\u0280;celp\u0757\u234c\u2350\u235b\u2361al;\u62ba\u0100gr\u2355\u2359er\xf3\u1563\xe3\u234darhk;\u6a17rod;\u6a3c\u0200cgpt\u236f\u2372\u2376\u237by;\u4451on;\u412ff;\uc000\ud835\udd5aa;\u43b9uest\u803b\xbf\u40bf\u0100ci\u238a\u238fr;\uc000\ud835\udcben\u0280;Edsv\u04f4\u239b\u239d\u23a1\u04f3;\u62f9ot;\u62f5\u0100;v\u23a6\u23a7\u62f4;\u62f3\u0100;i\u0777\u23aelde;\u4129\u01eb\u23b8\0\u23bccy;\u4456l\u803b\xef\u40ef\u0300cfmosu\u23cc\u23d7\u23dc\u23e1\u23e7\u23f5\u0100iy\u23d1\u23d5rc;\u4135;\u4439r;\uc000\ud835\udd27ath;\u4237pf;\uc000\ud835\udd5b\u01e3\u23ec\0\u23f1r;\uc000\ud835\udcbfrcy;\u4458kcy;\u4454\u0400acfghjos\u240b\u2416\u2422\u2427\u242d\u2431\u2435\u243bppa\u0100;v\u2413\u2414\u43ba;\u43f0\u0100ey\u241b\u2420dil;\u4137;\u443ar;\uc000\ud835\udd28reen;\u4138cy;\u4445cy;\u445cpf;\uc000\ud835\udd5ccr;\uc000\ud835\udcc0\u0b80ABEHabcdefghjlmnoprstuv\u2470\u2481\u2486\u248d\u2491\u250e\u253d\u255a\u2580\u264e\u265e\u2665\u2679\u267d\u269a\u26b2\u26d8\u275d\u2768\u278b\u27c0\u2801\u2812\u0180art\u2477\u247a\u247cr\xf2\u09c6\xf2\u0395ail;\u691barr;\u690e\u0100;g\u0994\u248b;\u6a8bar;\u6962\u0963\u24a5\0\u24aa\0\u24b1\0\0\0\0\0\u24b5\u24ba\0\u24c6\u24c8\u24cd\0\u24f9ute;\u413amptyv;\u69b4ra\xee\u084cbda;\u43bbg\u0180;dl\u088e\u24c1\u24c3;\u6991\xe5\u088e;\u6a85uo\u803b\xab\u40abr\u0400;bfhlpst\u0899\u24de\u24e6\u24e9\u24eb\u24ee\u24f1\u24f5\u0100;f\u089d\u24e3s;\u691fs;\u691d\xeb\u2252p;\u61abl;\u6939im;\u6973l;\u61a2\u0180;ae\u24ff\u2500\u2504\u6aabil;\u6919\u0100;s\u2509\u250a\u6aad;\uc000\u2aad\ufe00\u0180abr\u2515\u2519\u251drr;\u690crk;\u6772\u0100ak\u2522\u252cc\u0100ek\u2528\u252a;\u407b;\u405b\u0100es\u2531\u2533;\u698bl\u0100du\u2539\u253b;\u698f;\u698d\u0200aeuy\u2546\u254b\u2556\u2558ron;\u413e\u0100di\u2550\u2554il;\u413c\xec\u08b0\xe2\u2529;\u443b\u0200cqrs\u2563\u2566\u256d\u257da;\u6936uo\u0100;r\u0e19\u1746\u0100du\u2572\u2577har;\u6967shar;\u694bh;\u61b2\u0280;fgqs\u258b\u258c\u0989\u25f3\u25ff\u6264t\u0280ahlrt\u2598\u25a4\u25b7\u25c2\u25e8rrow\u0100;t\u0899\u25a1a\xe9\u24f6arpoon\u0100du\u25af\u25b4own\xbb\u045ap\xbb\u0966eftarrows;\u61c7ight\u0180ahs\u25cd\u25d6\u25derrow\u0100;s\u08f4\u08a7arpoon\xf3\u0f98quigarro\xf7\u21f0hreetimes;\u62cb\u0180;qs\u258b\u0993\u25falan\xf4\u09ac\u0280;cdgs\u09ac\u260a\u260d\u261d\u2628c;\u6aa8ot\u0100;o\u2614\u2615\u6a7f\u0100;r\u261a\u261b\u6a81;\u6a83\u0100;e\u2622\u2625\uc000\u22da\ufe00s;\u6a93\u0280adegs\u2633\u2639\u263d\u2649\u264bppro\xf8\u24c6ot;\u62d6q\u0100gq\u2643\u2645\xf4\u0989gt\xf2\u248c\xf4\u099bi\xed\u09b2\u0180ilr\u2655\u08e1\u265asht;\u697c;\uc000\ud835\udd29\u0100;E\u099c\u2663;\u6a91\u0161\u2669\u2676r\u0100du\u25b2\u266e\u0100;l\u0965\u2673;\u696alk;\u6584cy;\u4459\u0280;acht\u0a48\u2688\u268b\u2691\u2696r\xf2\u25c1orne\xf2\u1d08ard;\u696bri;\u65fa\u0100io\u269f\u26a4dot;\u4140ust\u0100;a\u26ac\u26ad\u63b0che\xbb\u26ad\u0200Eaes\u26bb\u26bd\u26c9\u26d4;\u6268p\u0100;p\u26c3\u26c4\u6a89rox\xbb\u26c4\u0100;q\u26ce\u26cf\u6a87\u0100;q\u26ce\u26bbim;\u62e6\u0400abnoptwz\u26e9\u26f4\u26f7\u271a\u272f\u2741\u2747\u2750\u0100nr\u26ee\u26f1g;\u67ecr;\u61fdr\xeb\u08c1g\u0180lmr\u26ff\u270d\u2714eft\u0100ar\u09e6\u2707ight\xe1\u09f2apsto;\u67fcight\xe1\u09fdparrow\u0100lr\u2725\u2729ef\xf4\u24edight;\u61ac\u0180afl\u2736\u2739\u273dr;\u6985;\uc000\ud835\udd5dus;\u6a2dimes;\u6a34\u0161\u274b\u274fst;\u6217\xe1\u134e\u0180;ef\u2757\u2758\u1800\u65cange\xbb\u2758ar\u0100;l\u2764\u2765\u4028t;\u6993\u0280achmt\u2773\u2776\u277c\u2785\u2787r\xf2\u08a8orne\xf2\u1d8car\u0100;d\u0f98\u2783;\u696d;\u600eri;\u62bf\u0300achiqt\u2798\u279d\u0a40\u27a2\u27ae\u27bbquo;\u6039r;\uc000\ud835\udcc1m\u0180;eg\u09b2\u27aa\u27ac;\u6a8d;\u6a8f\u0100bu\u252a\u27b3o\u0100;r\u0e1f\u27b9;\u601arok;\u4142\u8400<;cdhilqr\u082b\u27d2\u2639\u27dc\u27e0\u27e5\u27ea\u27f0\u0100ci\u27d7\u27d9;\u6aa6r;\u6a79re\xe5\u25f2mes;\u62c9arr;\u6976uest;\u6a7b\u0100Pi\u27f5\u27f9ar;\u6996\u0180;ef\u2800\u092d\u181b\u65c3r\u0100du\u2807\u280dshar;\u694ahar;\u6966\u0100en\u2817\u2821rtneqq;\uc000\u2268\ufe00\xc5\u281e\u0700Dacdefhilnopsu\u2840\u2845\u2882\u288e\u2893\u28a0\u28a5\u28a8\u28da\u28e2\u28e4\u0a83\u28f3\u2902Dot;\u623a\u0200clpr\u284e\u2852\u2863\u287dr\u803b\xaf\u40af\u0100et\u2857\u2859;\u6642\u0100;e\u285e\u285f\u6720se\xbb\u285f\u0100;s\u103b\u2868to\u0200;dlu\u103b\u2873\u2877\u287bow\xee\u048cef\xf4\u090f\xf0\u13d1ker;\u65ae\u0100oy\u2887\u288cmma;\u6a29;\u443cash;\u6014asuredangle\xbb\u1626r;\uc000\ud835\udd2ao;\u6127\u0180cdn\u28af\u28b4\u28c9ro\u803b\xb5\u40b5\u0200;acd\u1464\u28bd\u28c0\u28c4s\xf4\u16a7ir;\u6af0ot\u80bb\xb7\u01b5us\u0180;bd\u28d2\u1903\u28d3\u6212\u0100;u\u1d3c\u28d8;\u6a2a\u0163\u28de\u28e1p;\u6adb\xf2\u2212\xf0\u0a81\u0100dp\u28e9\u28eeels;\u62a7f;\uc000\ud835\udd5e\u0100ct\u28f8\u28fdr;\uc000\ud835\udcc2pos\xbb\u159d\u0180;lm\u2909\u290a\u290d\u43bctimap;\u62b8\u0c00GLRVabcdefghijlmoprstuvw\u2942\u2953\u297e\u2989\u2998\u29da\u29e9\u2a15\u2a1a\u2a58\u2a5d\u2a83\u2a95\u2aa4\u2aa8\u2b04\u2b07\u2b44\u2b7f\u2bae\u2c34\u2c67\u2c7c\u2ce9\u0100gt\u2947\u294b;\uc000\u22d9\u0338\u0100;v\u2950\u0bcf\uc000\u226b\u20d2\u0180elt\u295a\u2972\u2976ft\u0100ar\u2961\u2967rrow;\u61cdightarrow;\u61ce;\uc000\u22d8\u0338\u0100;v\u297b\u0c47\uc000\u226a\u20d2ightarrow;\u61cf\u0100Dd\u298e\u2993ash;\u62afash;\u62ae\u0280bcnpt\u29a3\u29a7\u29ac\u29b1\u29ccla\xbb\u02deute;\u4144g;\uc000\u2220\u20d2\u0280;Eiop\u0d84\u29bc\u29c0\u29c5\u29c8;\uc000\u2a70\u0338d;\uc000\u224b\u0338s;\u4149ro\xf8\u0d84ur\u0100;a\u29d3\u29d4\u666el\u0100;s\u29d3\u0b38\u01f3\u29df\0\u29e3p\u80bb\xa0\u0b37mp\u0100;e\u0bf9\u0c00\u0280aeouy\u29f4\u29fe\u2a03\u2a10\u2a13\u01f0\u29f9\0\u29fb;\u6a43on;\u4148dil;\u4146ng\u0100;d\u0d7e\u2a0aot;\uc000\u2a6d\u0338p;\u6a42;\u443dash;\u6013\u0380;Aadqsx\u0b92\u2a29\u2a2d\u2a3b\u2a41\u2a45\u2a50rr;\u61d7r\u0100hr\u2a33\u2a36k;\u6924\u0100;o\u13f2\u13f0ot;\uc000\u2250\u0338ui\xf6\u0b63\u0100ei\u2a4a\u2a4ear;\u6928\xed\u0b98ist\u0100;s\u0ba0\u0b9fr;\uc000\ud835\udd2b\u0200Eest\u0bc5\u2a66\u2a79\u2a7c\u0180;qs\u0bbc\u2a6d\u0be1\u0180;qs\u0bbc\u0bc5\u2a74lan\xf4\u0be2i\xed\u0bea\u0100;r\u0bb6\u2a81\xbb\u0bb7\u0180Aap\u2a8a\u2a8d\u2a91r\xf2\u2971rr;\u61aear;\u6af2\u0180;sv\u0f8d\u2a9c\u0f8c\u0100;d\u2aa1\u2aa2\u62fc;\u62facy;\u445a\u0380AEadest\u2ab7\u2aba\u2abe\u2ac2\u2ac5\u2af6\u2af9r\xf2\u2966;\uc000\u2266\u0338rr;\u619ar;\u6025\u0200;fqs\u0c3b\u2ace\u2ae3\u2aeft\u0100ar\u2ad4\u2ad9rro\xf7\u2ac1ightarro\xf7\u2a90\u0180;qs\u0c3b\u2aba\u2aealan\xf4\u0c55\u0100;s\u0c55\u2af4\xbb\u0c36i\xed\u0c5d\u0100;r\u0c35\u2afei\u0100;e\u0c1a\u0c25i\xe4\u0d90\u0100pt\u2b0c\u2b11f;\uc000\ud835\udd5f\u8180\xac;in\u2b19\u2b1a\u2b36\u40acn\u0200;Edv\u0b89\u2b24\u2b28\u2b2e;\uc000\u22f9\u0338ot;\uc000\u22f5\u0338\u01e1\u0b89\u2b33\u2b35;\u62f7;\u62f6i\u0100;v\u0cb8\u2b3c\u01e1\u0cb8\u2b41\u2b43;\u62fe;\u62fd\u0180aor\u2b4b\u2b63\u2b69r\u0200;ast\u0b7b\u2b55\u2b5a\u2b5flle\xec\u0b7bl;\uc000\u2afd\u20e5;\uc000\u2202\u0338lint;\u6a14\u0180;ce\u0c92\u2b70\u2b73u\xe5\u0ca5\u0100;c\u0c98\u2b78\u0100;e\u0c92\u2b7d\xf1\u0c98\u0200Aait\u2b88\u2b8b\u2b9d\u2ba7r\xf2\u2988rr\u0180;cw\u2b94\u2b95\u2b99\u619b;\uc000\u2933\u0338;\uc000\u219d\u0338ghtarrow\xbb\u2b95ri\u0100;e\u0ccb\u0cd6\u0380chimpqu\u2bbd\u2bcd\u2bd9\u2b04\u0b78\u2be4\u2bef\u0200;cer\u0d32\u2bc6\u0d37\u2bc9u\xe5\u0d45;\uc000\ud835\udcc3ort\u026d\u2b05\0\0\u2bd6ar\xe1\u2b56m\u0100;e\u0d6e\u2bdf\u0100;q\u0d74\u0d73su\u0100bp\u2beb\u2bed\xe5\u0cf8\xe5\u0d0b\u0180bcp\u2bf6\u2c11\u2c19\u0200;Ees\u2bff\u2c00\u0d22\u2c04\u6284;\uc000\u2ac5\u0338et\u0100;e\u0d1b\u2c0bq\u0100;q\u0d23\u2c00c\u0100;e\u0d32\u2c17\xf1\u0d38\u0200;Ees\u2c22\u2c23\u0d5f\u2c27\u6285;\uc000\u2ac6\u0338et\u0100;e\u0d58\u2c2eq\u0100;q\u0d60\u2c23\u0200gilr\u2c3d\u2c3f\u2c45\u2c47\xec\u0bd7lde\u803b\xf1\u40f1\xe7\u0c43iangle\u0100lr\u2c52\u2c5ceft\u0100;e\u0c1a\u2c5a\xf1\u0c26ight\u0100;e\u0ccb\u2c65\xf1\u0cd7\u0100;m\u2c6c\u2c6d\u43bd\u0180;es\u2c74\u2c75\u2c79\u4023ro;\u6116p;\u6007\u0480DHadgilrs\u2c8f\u2c94\u2c99\u2c9e\u2ca3\u2cb0\u2cb6\u2cd3\u2ce3ash;\u62adarr;\u6904p;\uc000\u224d\u20d2ash;\u62ac\u0100et\u2ca8\u2cac;\uc000\u2265\u20d2;\uc000>\u20d2nfin;\u69de\u0180Aet\u2cbd\u2cc1\u2cc5rr;\u6902;\uc000\u2264\u20d2\u0100;r\u2cca\u2ccd\uc000<\u20d2ie;\uc000\u22b4\u20d2\u0100At\u2cd8\u2cdcrr;\u6903rie;\uc000\u22b5\u20d2im;\uc000\u223c\u20d2\u0180Aan\u2cf0\u2cf4\u2d02rr;\u61d6r\u0100hr\u2cfa\u2cfdk;\u6923\u0100;o\u13e7\u13e5ear;\u6927\u1253\u1a95\0\0\0\0\0\0\0\0\0\0\0\0\0\u2d2d\0\u2d38\u2d48\u2d60\u2d65\u2d72\u2d84\u1b07\0\0\u2d8d\u2dab\0\u2dc8\u2dce\0\u2ddc\u2e19\u2e2b\u2e3e\u2e43\u0100cs\u2d31\u1a97ute\u803b\xf3\u40f3\u0100iy\u2d3c\u2d45r\u0100;c\u1a9e\u2d42\u803b\xf4\u40f4;\u443e\u0280abios\u1aa0\u2d52\u2d57\u01c8\u2d5alac;\u4151v;\u6a38old;\u69bclig;\u4153\u0100cr\u2d69\u2d6dir;\u69bf;\uc000\ud835\udd2c\u036f\u2d79\0\0\u2d7c\0\u2d82n;\u42dbave\u803b\xf2\u40f2;\u69c1\u0100bm\u2d88\u0df4ar;\u69b5\u0200acit\u2d95\u2d98\u2da5\u2da8r\xf2\u1a80\u0100ir\u2d9d\u2da0r;\u69beoss;\u69bbn\xe5\u0e52;\u69c0\u0180aei\u2db1\u2db5\u2db9cr;\u414dga;\u43c9\u0180cdn\u2dc0\u2dc5\u01cdron;\u43bf;\u69b6pf;\uc000\ud835\udd60\u0180ael\u2dd4\u2dd7\u01d2r;\u69b7rp;\u69b9\u0380;adiosv\u2dea\u2deb\u2dee\u2e08\u2e0d\u2e10\u2e16\u6228r\xf2\u1a86\u0200;efm\u2df7\u2df8\u2e02\u2e05\u6a5dr\u0100;o\u2dfe\u2dff\u6134f\xbb\u2dff\u803b\xaa\u40aa\u803b\xba\u40bagof;\u62b6r;\u6a56lope;\u6a57;\u6a5b\u0180clo\u2e1f\u2e21\u2e27\xf2\u2e01ash\u803b\xf8\u40f8l;\u6298i\u016c\u2e2f\u2e34de\u803b\xf5\u40f5es\u0100;a\u01db\u2e3as;\u6a36ml\u803b\xf6\u40f6bar;\u633d\u0ae1\u2e5e\0\u2e7d\0\u2e80\u2e9d\0\u2ea2\u2eb9\0\0\u2ecb\u0e9c\0\u2f13\0\0\u2f2b\u2fbc\0\u2fc8r\u0200;ast\u0403\u2e67\u2e72\u0e85\u8100\xb6;l\u2e6d\u2e6e\u40b6le\xec\u0403\u0269\u2e78\0\0\u2e7bm;\u6af3;\u6afdy;\u443fr\u0280cimpt\u2e8b\u2e8f\u2e93\u1865\u2e97nt;\u4025od;\u402eil;\u6030enk;\u6031r;\uc000\ud835\udd2d\u0180imo\u2ea8\u2eb0\u2eb4\u0100;v\u2ead\u2eae\u43c6;\u43d5ma\xf4\u0a76ne;\u660e\u0180;tv\u2ebf\u2ec0\u2ec8\u43c0chfork\xbb\u1ffd;\u43d6\u0100au\u2ecf\u2edfn\u0100ck\u2ed5\u2eddk\u0100;h\u21f4\u2edb;\u610e\xf6\u21f4s\u0480;abcdemst\u2ef3\u2ef4\u1908\u2ef9\u2efd\u2f04\u2f06\u2f0a\u2f0e\u402bcir;\u6a23ir;\u6a22\u0100ou\u1d40\u2f02;\u6a25;\u6a72n\u80bb\xb1\u0e9dim;\u6a26wo;\u6a27\u0180ipu\u2f19\u2f20\u2f25ntint;\u6a15f;\uc000\ud835\udd61nd\u803b\xa3\u40a3\u0500;Eaceinosu\u0ec8\u2f3f\u2f41\u2f44\u2f47\u2f81\u2f89\u2f92\u2f7e\u2fb6;\u6ab3p;\u6ab7u\xe5\u0ed9\u0100;c\u0ece\u2f4c\u0300;acens\u0ec8\u2f59\u2f5f\u2f66\u2f68\u2f7eppro\xf8\u2f43urlye\xf1\u0ed9\xf1\u0ece\u0180aes\u2f6f\u2f76\u2f7approx;\u6ab9qq;\u6ab5im;\u62e8i\xed\u0edfme\u0100;s\u2f88\u0eae\u6032\u0180Eas\u2f78\u2f90\u2f7a\xf0\u2f75\u0180dfp\u0eec\u2f99\u2faf\u0180als\u2fa0\u2fa5\u2faalar;\u632eine;\u6312urf;\u6313\u0100;t\u0efb\u2fb4\xef\u0efbrel;\u62b0\u0100ci\u2fc0\u2fc5r;\uc000\ud835\udcc5;\u43c8ncsp;\u6008\u0300fiopsu\u2fda\u22e2\u2fdf\u2fe5\u2feb\u2ff1r;\uc000\ud835\udd2epf;\uc000\ud835\udd62rime;\u6057cr;\uc000\ud835\udcc6\u0180aeo\u2ff8\u3009\u3013t\u0100ei\u2ffe\u3005rnion\xf3\u06b0nt;\u6a16st\u0100;e\u3010\u3011\u403f\xf1\u1f19\xf4\u0f14\u0a80ABHabcdefhilmnoprstux\u3040\u3051\u3055\u3059\u30e0\u310e\u312b\u3147\u3162\u3172\u318e\u3206\u3215\u3224\u3229\u3258\u326e\u3272\u3290\u32b0\u32b7\u0180art\u3047\u304a\u304cr\xf2\u10b3\xf2\u03ddail;\u691car\xf2\u1c65ar;\u6964\u0380cdenqrt\u3068\u3075\u3078\u307f\u308f\u3094\u30cc\u0100eu\u306d\u3071;\uc000\u223d\u0331te;\u4155i\xe3\u116emptyv;\u69b3g\u0200;del\u0fd1\u3089\u308b\u308d;\u6992;\u69a5\xe5\u0fd1uo\u803b\xbb\u40bbr\u0580;abcfhlpstw\u0fdc\u30ac\u30af\u30b7\u30b9\u30bc\u30be\u30c0\u30c3\u30c7\u30cap;\u6975\u0100;f\u0fe0\u30b4s;\u6920;\u6933s;\u691e\xeb\u225d\xf0\u272el;\u6945im;\u6974l;\u61a3;\u619d\u0100ai\u30d1\u30d5il;\u691ao\u0100;n\u30db\u30dc\u6236al\xf3\u0f1e\u0180abr\u30e7\u30ea\u30eer\xf2\u17e5rk;\u6773\u0100ak\u30f3\u30fdc\u0100ek\u30f9\u30fb;\u407d;\u405d\u0100es\u3102\u3104;\u698cl\u0100du\u310a\u310c;\u698e;\u6990\u0200aeuy\u3117\u311c\u3127\u3129ron;\u4159\u0100di\u3121\u3125il;\u4157\xec\u0ff2\xe2\u30fa;\u4440\u0200clqs\u3134\u3137\u313d\u3144a;\u6937dhar;\u6969uo\u0100;r\u020e\u020dh;\u61b3\u0180acg\u314e\u315f\u0f44l\u0200;ips\u0f78\u3158\u315b\u109cn\xe5\u10bbar\xf4\u0fa9t;\u65ad\u0180ilr\u3169\u1023\u316esht;\u697d;\uc000\ud835\udd2f\u0100ao\u3177\u3186r\u0100du\u317d\u317f\xbb\u047b\u0100;l\u1091\u3184;\u696c\u0100;v\u318b\u318c\u43c1;\u43f1\u0180gns\u3195\u31f9\u31fcht\u0300ahlrst\u31a4\u31b0\u31c2\u31d8\u31e4\u31eerrow\u0100;t\u0fdc\u31ada\xe9\u30c8arpoon\u0100du\u31bb\u31bfow\xee\u317ep\xbb\u1092eft\u0100ah\u31ca\u31d0rrow\xf3\u0feaarpoon\xf3\u0551ightarrows;\u61c9quigarro\xf7\u30cbhreetimes;\u62ccg;\u42daingdotse\xf1\u1f32\u0180ahm\u320d\u3210\u3213r\xf2\u0feaa\xf2\u0551;\u600foust\u0100;a\u321e\u321f\u63b1che\xbb\u321fmid;\u6aee\u0200abpt\u3232\u323d\u3240\u3252\u0100nr\u3237\u323ag;\u67edr;\u61fer\xeb\u1003\u0180afl\u3247\u324a\u324er;\u6986;\uc000\ud835\udd63us;\u6a2eimes;\u6a35\u0100ap\u325d\u3267r\u0100;g\u3263\u3264\u4029t;\u6994olint;\u6a12ar\xf2\u31e3\u0200achq\u327b\u3280\u10bc\u3285quo;\u603ar;\uc000\ud835\udcc7\u0100bu\u30fb\u328ao\u0100;r\u0214\u0213\u0180hir\u3297\u329b\u32a0re\xe5\u31f8mes;\u62cai\u0200;efl\u32aa\u1059\u1821\u32ab\u65b9tri;\u69celuhar;\u6968;\u611e\u0d61\u32d5\u32db\u32df\u332c\u3338\u3371\0\u337a\u33a4\0\0\u33ec\u33f0\0\u3428\u3448\u345a\u34ad\u34b1\u34ca\u34f1\0\u3616\0\0\u3633cute;\u415bqu\xef\u27ba\u0500;Eaceinpsy\u11ed\u32f3\u32f5\u32ff\u3302\u330b\u330f\u331f\u3326\u3329;\u6ab4\u01f0\u32fa\0\u32fc;\u6ab8on;\u4161u\xe5\u11fe\u0100;d\u11f3\u3307il;\u415frc;\u415d\u0180Eas\u3316\u3318\u331b;\u6ab6p;\u6abaim;\u62e9olint;\u6a13i\xed\u1204;\u4441ot\u0180;be\u3334\u1d47\u3335\u62c5;\u6a66\u0380Aacmstx\u3346\u334a\u3357\u335b\u335e\u3363\u336drr;\u61d8r\u0100hr\u3350\u3352\xeb\u2228\u0100;o\u0a36\u0a34t\u803b\xa7\u40a7i;\u403bwar;\u6929m\u0100in\u3369\xf0nu\xf3\xf1t;\u6736r\u0100;o\u3376\u2055\uc000\ud835\udd30\u0200acoy\u3382\u3386\u3391\u33a0rp;\u666f\u0100hy\u338b\u338fcy;\u4449;\u4448rt\u026d\u3399\0\0\u339ci\xe4\u1464ara\xec\u2e6f\u803b\xad\u40ad\u0100gm\u33a8\u33b4ma\u0180;fv\u33b1\u33b2\u33b2\u43c3;\u43c2\u0400;deglnpr\u12ab\u33c5\u33c9\u33ce\u33d6\u33de\u33e1\u33e6ot;\u6a6a\u0100;q\u12b1\u12b0\u0100;E\u33d3\u33d4\u6a9e;\u6aa0\u0100;E\u33db\u33dc\u6a9d;\u6a9fe;\u6246lus;\u6a24arr;\u6972ar\xf2\u113d\u0200aeit\u33f8\u3408\u340f\u3417\u0100ls\u33fd\u3404lsetm\xe9\u336ahp;\u6a33parsl;\u69e4\u0100dl\u1463\u3414e;\u6323\u0100;e\u341c\u341d\u6aaa\u0100;s\u3422\u3423\u6aac;\uc000\u2aac\ufe00\u0180flp\u342e\u3433\u3442tcy;\u444c\u0100;b\u3438\u3439\u402f\u0100;a\u343e\u343f\u69c4r;\u633ff;\uc000\ud835\udd64a\u0100dr\u344d\u0402es\u0100;u\u3454\u3455\u6660it\xbb\u3455\u0180csu\u3460\u3479\u349f\u0100au\u3465\u346fp\u0100;s\u1188\u346b;\uc000\u2293\ufe00p\u0100;s\u11b4\u3475;\uc000\u2294\ufe00u\u0100bp\u347f\u348f\u0180;es\u1197\u119c\u3486et\u0100;e\u1197\u348d\xf1\u119d\u0180;es\u11a8\u11ad\u3496et\u0100;e\u11a8\u349d\xf1\u11ae\u0180;af\u117b\u34a6\u05b0r\u0165\u34ab\u05b1\xbb\u117car\xf2\u1148\u0200cemt\u34b9\u34be\u34c2\u34c5r;\uc000\ud835\udcc8tm\xee\xf1i\xec\u3415ar\xe6\u11be\u0100ar\u34ce\u34d5r\u0100;f\u34d4\u17bf\u6606\u0100an\u34da\u34edight\u0100ep\u34e3\u34eapsilo\xee\u1ee0h\xe9\u2eafs\xbb\u2852\u0280bcmnp\u34fb\u355e\u1209\u358b\u358e\u0480;Edemnprs\u350e\u350f\u3511\u3515\u351e\u3523\u352c\u3531\u3536\u6282;\u6ac5ot;\u6abd\u0100;d\u11da\u351aot;\u6ac3ult;\u6ac1\u0100Ee\u3528\u352a;\u6acb;\u628alus;\u6abfarr;\u6979\u0180eiu\u353d\u3552\u3555t\u0180;en\u350e\u3545\u354bq\u0100;q\u11da\u350feq\u0100;q\u352b\u3528m;\u6ac7\u0100bp\u355a\u355c;\u6ad5;\u6ad3c\u0300;acens\u11ed\u356c\u3572\u3579\u357b\u3326ppro\xf8\u32faurlye\xf1\u11fe\xf1\u11f3\u0180aes\u3582\u3588\u331bppro\xf8\u331aq\xf1\u3317g;\u666a\u0680123;Edehlmnps\u35a9\u35ac\u35af\u121c\u35b2\u35b4\u35c0\u35c9\u35d5\u35da\u35df\u35e8\u35ed\u803b\xb9\u40b9\u803b\xb2\u40b2\u803b\xb3\u40b3;\u6ac6\u0100os\u35b9\u35bct;\u6abeub;\u6ad8\u0100;d\u1222\u35c5ot;\u6ac4s\u0100ou\u35cf\u35d2l;\u67c9b;\u6ad7arr;\u697bult;\u6ac2\u0100Ee\u35e4\u35e6;\u6acc;\u628blus;\u6ac0\u0180eiu\u35f4\u3609\u360ct\u0180;en\u121c\u35fc\u3602q\u0100;q\u1222\u35b2eq\u0100;q\u35e7\u35e4m;\u6ac8\u0100bp\u3611\u3613;\u6ad4;\u6ad6\u0180Aan\u361c\u3620\u362drr;\u61d9r\u0100hr\u3626\u3628\xeb\u222e\u0100;o\u0a2b\u0a29war;\u692alig\u803b\xdf\u40df\u0be1\u3651\u365d\u3660\u12ce\u3673\u3679\0\u367e\u36c2\0\0\0\0\0\u36db\u3703\0\u3709\u376c\0\0\0\u3787\u0272\u3656\0\0\u365bget;\u6316;\u43c4r\xeb\u0e5f\u0180aey\u3666\u366b\u3670ron;\u4165dil;\u4163;\u4442lrec;\u6315r;\uc000\ud835\udd31\u0200eiko\u3686\u369d\u36b5\u36bc\u01f2\u368b\0\u3691e\u01004f\u1284\u1281a\u0180;sv\u3698\u3699\u369b\u43b8ym;\u43d1\u0100cn\u36a2\u36b2k\u0100as\u36a8\u36aeppro\xf8\u12c1im\xbb\u12acs\xf0\u129e\u0100as\u36ba\u36ae\xf0\u12c1rn\u803b\xfe\u40fe\u01ec\u031f\u36c6\u22e7es\u8180\xd7;bd\u36cf\u36d0\u36d8\u40d7\u0100;a\u190f\u36d5r;\u6a31;\u6a30\u0180eps\u36e1\u36e3\u3700\xe1\u2a4d\u0200;bcf\u0486\u36ec\u36f0\u36f4ot;\u6336ir;\u6af1\u0100;o\u36f9\u36fc\uc000\ud835\udd65rk;\u6ada\xe1\u3362rime;\u6034\u0180aip\u370f\u3712\u3764d\xe5\u1248\u0380adempst\u3721\u374d\u3740\u3751\u3757\u375c\u375fngle\u0280;dlqr\u3730\u3731\u3736\u3740\u3742\u65b5own\xbb\u1dbbeft\u0100;e\u2800\u373e\xf1\u092e;\u625cight\u0100;e\u32aa\u374b\xf1\u105aot;\u65ecinus;\u6a3alus;\u6a39b;\u69cdime;\u6a3bezium;\u63e2\u0180cht\u3772\u377d\u3781\u0100ry\u3777\u377b;\uc000\ud835\udcc9;\u4446cy;\u445brok;\u4167\u0100io\u378b\u378ex\xf4\u1777head\u0100lr\u3797\u37a0eftarro\xf7\u084fightarrow\xbb\u0f5d\u0900AHabcdfghlmoprstuw\u37d0\u37d3\u37d7\u37e4\u37f0\u37fc\u380e\u381c\u3823\u3834\u3851\u385d\u386b\u38a9\u38cc\u38d2\u38ea\u38f6r\xf2\u03edar;\u6963\u0100cr\u37dc\u37e2ute\u803b\xfa\u40fa\xf2\u1150r\u01e3\u37ea\0\u37edy;\u445eve;\u416d\u0100iy\u37f5\u37farc\u803b\xfb\u40fb;\u4443\u0180abh\u3803\u3806\u380br\xf2\u13adlac;\u4171a\xf2\u13c3\u0100ir\u3813\u3818sht;\u697e;\uc000\ud835\udd32rave\u803b\xf9\u40f9\u0161\u3827\u3831r\u0100lr\u382c\u382e\xbb\u0957\xbb\u1083lk;\u6580\u0100ct\u3839\u384d\u026f\u383f\0\0\u384arn\u0100;e\u3845\u3846\u631cr\xbb\u3846op;\u630fri;\u65f8\u0100al\u3856\u385acr;\u416b\u80bb\xa8\u0349\u0100gp\u3862\u3866on;\u4173f;\uc000\ud835\udd66\u0300adhlsu\u114b\u3878\u387d\u1372\u3891\u38a0own\xe1\u13b3arpoon\u0100lr\u3888\u388cef\xf4\u382digh\xf4\u382fi\u0180;hl\u3899\u389a\u389c\u43c5\xbb\u13faon\xbb\u389aparrows;\u61c8\u0180cit\u38b0\u38c4\u38c8\u026f\u38b6\0\0\u38c1rn\u0100;e\u38bc\u38bd\u631dr\xbb\u38bdop;\u630eng;\u416fri;\u65f9cr;\uc000\ud835\udcca\u0180dir\u38d9\u38dd\u38e2ot;\u62f0lde;\u4169i\u0100;f\u3730\u38e8\xbb\u1813\u0100am\u38ef\u38f2r\xf2\u38a8l\u803b\xfc\u40fcangle;\u69a7\u0780ABDacdeflnoprsz\u391c\u391f\u3929\u392d\u39b5\u39b8\u39bd\u39df\u39e4\u39e8\u39f3\u39f9\u39fd\u3a01\u3a20r\xf2\u03f7ar\u0100;v\u3926\u3927\u6ae8;\u6ae9as\xe8\u03e1\u0100nr\u3932\u3937grt;\u699c\u0380eknprst\u34e3\u3946\u394b\u3952\u395d\u3964\u3996app\xe1\u2415othin\xe7\u1e96\u0180hir\u34eb\u2ec8\u3959op\xf4\u2fb5\u0100;h\u13b7\u3962\xef\u318d\u0100iu\u3969\u396dgm\xe1\u33b3\u0100bp\u3972\u3984setneq\u0100;q\u397d\u3980\uc000\u228a\ufe00;\uc000\u2acb\ufe00setneq\u0100;q\u398f\u3992\uc000\u228b\ufe00;\uc000\u2acc\ufe00\u0100hr\u399b\u399fet\xe1\u369ciangle\u0100lr\u39aa\u39afeft\xbb\u0925ight\xbb\u1051y;\u4432ash\xbb\u1036\u0180elr\u39c4\u39d2\u39d7\u0180;be\u2dea\u39cb\u39cfar;\u62bbq;\u625alip;\u62ee\u0100bt\u39dc\u1468a\xf2\u1469r;\uc000\ud835\udd33tr\xe9\u39aesu\u0100bp\u39ef\u39f1\xbb\u0d1c\xbb\u0d59pf;\uc000\ud835\udd67ro\xf0\u0efbtr\xe9\u39b4\u0100cu\u3a06\u3a0br;\uc000\ud835\udccb\u0100bp\u3a10\u3a18n\u0100Ee\u3980\u3a16\xbb\u397en\u0100Ee\u3992\u3a1e\xbb\u3990igzag;\u699a\u0380cefoprs\u3a36\u3a3b\u3a56\u3a5b\u3a54\u3a61\u3a6airc;\u4175\u0100di\u3a40\u3a51\u0100bg\u3a45\u3a49ar;\u6a5fe\u0100;q\u15fa\u3a4f;\u6259erp;\u6118r;\uc000\ud835\udd34pf;\uc000\ud835\udd68\u0100;e\u1479\u3a66at\xe8\u1479cr;\uc000\ud835\udccc\u0ae3\u178e\u3a87\0\u3a8b\0\u3a90\u3a9b\0\0\u3a9d\u3aa8\u3aab\u3aaf\0\0\u3ac3\u3ace\0\u3ad8\u17dc\u17dftr\xe9\u17d1r;\uc000\ud835\udd35\u0100Aa\u3a94\u3a97r\xf2\u03c3r\xf2\u09f6;\u43be\u0100Aa\u3aa1\u3aa4r\xf2\u03b8r\xf2\u09eba\xf0\u2713is;\u62fb\u0180dpt\u17a4\u3ab5\u3abe\u0100fl\u3aba\u17a9;\uc000\ud835\udd69im\xe5\u17b2\u0100Aa\u3ac7\u3acar\xf2\u03cer\xf2\u0a01\u0100cq\u3ad2\u17b8r;\uc000\ud835\udccd\u0100pt\u17d6\u3adcr\xe9\u17d4\u0400acefiosu\u3af0\u3afd\u3b08\u3b0c\u3b11\u3b15\u3b1b\u3b21c\u0100uy\u3af6\u3afbte\u803b\xfd\u40fd;\u444f\u0100iy\u3b02\u3b06rc;\u4177;\u444bn\u803b\xa5\u40a5r;\uc000\ud835\udd36cy;\u4457pf;\uc000\ud835\udd6acr;\uc000\ud835\udcce\u0100cm\u3b26\u3b29y;\u444el\u803b\xff\u40ff\u0500acdefhiosw\u3b42\u3b48\u3b54\u3b58\u3b64\u3b69\u3b6d\u3b74\u3b7a\u3b80cute;\u417a\u0100ay\u3b4d\u3b52ron;\u417e;\u4437ot;\u417c\u0100et\u3b5d\u3b61tr\xe6\u155fa;\u43b6r;\uc000\ud835\udd37cy;\u4436grarr;\u61ddpf;\uc000\ud835\udd6bcr;\uc000\ud835\udccf\u0100jn\u3b85\u3b87;\u600dj;\u600c"
    .split("")
    .map(function (c) { return c.charCodeAt(0); }));// Generated using scripts/write-decode-map.ts
var xmlDecodeTree = new Uint16Array(
// prettier-ignore
"\u0200aglq\t\x15\x18\x1b\u026d\x0f\0\0\x12p;\u4026os;\u4027t;\u403et;\u403cuot;\u4022"
    .split("")
    .map(function (c) { return c.charCodeAt(0); }));// Adapted from https://github.com/mathiasbynens/he/blob/36afe179392226cf1b6ccdb16ebbb7a5a844d93a/src/he.js#L106-L134
var _a$1;
var decodeMap = new Map([
    [0, 65533],
    // C1 Unicode control character reference replacements
    [128, 8364],
    [130, 8218],
    [131, 402],
    [132, 8222],
    [133, 8230],
    [134, 8224],
    [135, 8225],
    [136, 710],
    [137, 8240],
    [138, 352],
    [139, 8249],
    [140, 338],
    [142, 381],
    [145, 8216],
    [146, 8217],
    [147, 8220],
    [148, 8221],
    [149, 8226],
    [150, 8211],
    [151, 8212],
    [152, 732],
    [153, 8482],
    [154, 353],
    [155, 8250],
    [156, 339],
    [158, 382],
    [159, 376] ]);
/**
 * Polyfill for `String.fromCodePoint`. It is used to create a string from a Unicode code point.
 */
var fromCodePoint$3 = 
// eslint-disable-next-line @typescript-eslint/no-unnecessary-condition, node/no-unsupported-features/es-builtins
(_a$1 = String.fromCodePoint) !== null && _a$1 !== void 0 ? _a$1 : function (codePoint) {
    var output = "";
    if (codePoint > 0xffff) {
        codePoint -= 0x10000;
        output += String.fromCharCode(((codePoint >>> 10) & 0x3ff) | 0xd800);
        codePoint = 0xdc00 | (codePoint & 0x3ff);
    }
    output += String.fromCharCode(codePoint);
    return output;
};
/**
 * Replace the given code point with a replacement character if it is a
 * surrogate or is outside the valid range. Otherwise return the code
 * point unchanged.
 */
function replaceCodePoint(codePoint) {
    var _a;
    if ((codePoint >= 0xd800 && codePoint <= 0xdfff) || codePoint > 0x10ffff) {
        return 0xfffd;
    }
    return (_a = decodeMap.get(codePoint)) !== null && _a !== void 0 ? _a : codePoint;
}var CharCodes;
(function (CharCodes) {
    CharCodes[CharCodes["NUM"] = 35] = "NUM";
    CharCodes[CharCodes["SEMI"] = 59] = "SEMI";
    CharCodes[CharCodes["EQUALS"] = 61] = "EQUALS";
    CharCodes[CharCodes["ZERO"] = 48] = "ZERO";
    CharCodes[CharCodes["NINE"] = 57] = "NINE";
    CharCodes[CharCodes["LOWER_A"] = 97] = "LOWER_A";
    CharCodes[CharCodes["LOWER_F"] = 102] = "LOWER_F";
    CharCodes[CharCodes["LOWER_X"] = 120] = "LOWER_X";
    CharCodes[CharCodes["LOWER_Z"] = 122] = "LOWER_Z";
    CharCodes[CharCodes["UPPER_A"] = 65] = "UPPER_A";
    CharCodes[CharCodes["UPPER_F"] = 70] = "UPPER_F";
    CharCodes[CharCodes["UPPER_Z"] = 90] = "UPPER_Z";
})(CharCodes || (CharCodes = {}));
/** Bit that needs to be set to convert an upper case ASCII character to lower case */
var TO_LOWER_BIT = 32;
var BinTrieFlags;
(function (BinTrieFlags) {
    BinTrieFlags[BinTrieFlags["VALUE_LENGTH"] = 49152] = "VALUE_LENGTH";
    BinTrieFlags[BinTrieFlags["BRANCH_LENGTH"] = 16256] = "BRANCH_LENGTH";
    BinTrieFlags[BinTrieFlags["JUMP_TABLE"] = 127] = "JUMP_TABLE";
})(BinTrieFlags || (BinTrieFlags = {}));
function isNumber(code) {
    return code >= CharCodes.ZERO && code <= CharCodes.NINE;
}
function isHexadecimalCharacter(code) {
    return ((code >= CharCodes.UPPER_A && code <= CharCodes.UPPER_F) ||
        (code >= CharCodes.LOWER_A && code <= CharCodes.LOWER_F));
}
function isAsciiAlphaNumeric(code) {
    return ((code >= CharCodes.UPPER_A && code <= CharCodes.UPPER_Z) ||
        (code >= CharCodes.LOWER_A && code <= CharCodes.LOWER_Z) ||
        isNumber(code));
}
/**
 * Checks if the given character is a valid end character for an entity in an attribute.
 *
 * Attribute values that aren't terminated properly aren't parsed, and shouldn't lead to a parser error.
 * See the example in https://html.spec.whatwg.org/multipage/parsing.html#named-character-reference-state
 */
function isEntityInAttributeInvalidEnd(code) {
    return code === CharCodes.EQUALS || isAsciiAlphaNumeric(code);
}
var EntityDecoderState;
(function (EntityDecoderState) {
    EntityDecoderState[EntityDecoderState["EntityStart"] = 0] = "EntityStart";
    EntityDecoderState[EntityDecoderState["NumericStart"] = 1] = "NumericStart";
    EntityDecoderState[EntityDecoderState["NumericDecimal"] = 2] = "NumericDecimal";
    EntityDecoderState[EntityDecoderState["NumericHex"] = 3] = "NumericHex";
    EntityDecoderState[EntityDecoderState["NamedEntity"] = 4] = "NamedEntity";
})(EntityDecoderState || (EntityDecoderState = {}));
var DecodingMode;
(function (DecodingMode) {
    /** Entities in text nodes that can end with any character. */
    DecodingMode[DecodingMode["Legacy"] = 0] = "Legacy";
    /** Only allow entities terminated with a semicolon. */
    DecodingMode[DecodingMode["Strict"] = 1] = "Strict";
    /** Entities in attributes have limitations on ending characters. */
    DecodingMode[DecodingMode["Attribute"] = 2] = "Attribute";
})(DecodingMode || (DecodingMode = {}));
/**
 * Token decoder with support of writing partial entities.
 */
var EntityDecoder = function EntityDecoder(
/** The tree used to decode entities. */
decodeTree, 
/**
 * The function that is called when a codepoint is decoded.
 *
 * For multi-byte named entities, this will be called multiple times,
 * with the second codepoint, and the same `consumed` value.
 *
 * @param codepoint The decoded codepoint.
 * @param consumed The number of bytes consumed by the decoder.
 */
emitCodePoint, 
/** An object that is used to produce errors. */
errors) {
    this.decodeTree = decodeTree;
    this.emitCodePoint = emitCodePoint;
    this.errors = errors;
    /** The current state of the decoder. */
    this.state = EntityDecoderState.EntityStart;
    /** Characters that were consumed while parsing an entity. */
    this.consumed = 1;
    /**
     * The result of the entity.
     *
     * Either the result index of a numeric entity, or the codepoint of a
     * numeric entity.
     */
    this.result = 0;
    /** The current index in the decode tree. */
    this.treeIndex = 0;
    /** The number of characters that were consumed in excess. */
    this.excess = 1;
    /** The mode in which the decoder is operating. */
    this.decodeMode = DecodingMode.Strict;
};
/** Resets the instance to make it reusable. */
EntityDecoder.prototype.startEntity = function startEntity (decodeMode) {
    this.decodeMode = decodeMode;
    this.state = EntityDecoderState.EntityStart;
    this.result = 0;
    this.treeIndex = 0;
    this.excess = 1;
    this.consumed = 1;
};
/**
 * Write an entity to the decoder. This can be called multiple times with partial entities.
 * If the entity is incomplete, the decoder will return -1.
 *
 * Mirrors the implementation of `getDecoder`, but with the ability to stop decoding if the
 * entity is incomplete, and resume when the next string is written.
 *
 * @param string The string containing the entity (or a continuation of the entity).
 * @param offset The offset at which the entity begins. Should be 0 if this is not the first call.
 * @returns The number of characters that were consumed, or -1 if the entity is incomplete.
 */
EntityDecoder.prototype.write = function write (str, offset) {
    switch (this.state) {
        case EntityDecoderState.EntityStart: {
            if (str.charCodeAt(offset) === CharCodes.NUM) {
                this.state = EntityDecoderState.NumericStart;
                this.consumed += 1;
                return this.stateNumericStart(str, offset + 1);
            }
            this.state = EntityDecoderState.NamedEntity;
            return this.stateNamedEntity(str, offset);
        }
        case EntityDecoderState.NumericStart: {
            return this.stateNumericStart(str, offset);
        }
        case EntityDecoderState.NumericDecimal: {
            return this.stateNumericDecimal(str, offset);
        }
        case EntityDecoderState.NumericHex: {
            return this.stateNumericHex(str, offset);
        }
        case EntityDecoderState.NamedEntity: {
            return this.stateNamedEntity(str, offset);
        }
    }
};
/**
 * Switches between the numeric decimal and hexadecimal states.
 *
 * Equivalent to the `Numeric character reference state` in the HTML spec.
 *
 * @param str The string containing the entity (or a continuation of the entity).
 * @param offset The current offset.
 * @returns The number of characters that were consumed, or -1 if the entity is incomplete.
 */
EntityDecoder.prototype.stateNumericStart = function stateNumericStart (str, offset) {
    if (offset >= str.length) {
        return -1;
    }
    if ((str.charCodeAt(offset) | TO_LOWER_BIT) === CharCodes.LOWER_X) {
        this.state = EntityDecoderState.NumericHex;
        this.consumed += 1;
        return this.stateNumericHex(str, offset + 1);
    }
    this.state = EntityDecoderState.NumericDecimal;
    return this.stateNumericDecimal(str, offset);
};
EntityDecoder.prototype.addToNumericResult = function addToNumericResult (str, start, end, base) {
    if (start !== end) {
        var digitCount = end - start;
        this.result =
            this.result * Math.pow(base, digitCount) +
                parseInt(str.substr(start, digitCount), base);
        this.consumed += digitCount;
    }
};
/**
 * Parses a hexadecimal numeric entity.
 *
 * Equivalent to the `Hexademical character reference state` in the HTML spec.
 *
 * @param str The string containing the entity (or a continuation of the entity).
 * @param offset The current offset.
 * @returns The number of characters that were consumed, or -1 if the entity is incomplete.
 */
EntityDecoder.prototype.stateNumericHex = function stateNumericHex (str, offset) {
    var startIdx = offset;
    while (offset < str.length) {
        var char = str.charCodeAt(offset);
        if (isNumber(char) || isHexadecimalCharacter(char)) {
            offset += 1;
        }
        else {
            this.addToNumericResult(str, startIdx, offset, 16);
            return this.emitNumericEntity(char, 3);
        }
    }
    this.addToNumericResult(str, startIdx, offset, 16);
    return -1;
};
/**
 * Parses a decimal numeric entity.
 *
 * Equivalent to the `Decimal character reference state` in the HTML spec.
 *
 * @param str The string containing the entity (or a continuation of the entity).
 * @param offset The current offset.
 * @returns The number of characters that were consumed, or -1 if the entity is incomplete.
 */
EntityDecoder.prototype.stateNumericDecimal = function stateNumericDecimal (str, offset) {
    var startIdx = offset;
    while (offset < str.length) {
        var char = str.charCodeAt(offset);
        if (isNumber(char)) {
            offset += 1;
        }
        else {
            this.addToNumericResult(str, startIdx, offset, 10);
            return this.emitNumericEntity(char, 2);
        }
    }
    this.addToNumericResult(str, startIdx, offset, 10);
    return -1;
};
/**
 * Validate and emit a numeric entity.
 *
 * Implements the logic from the `Hexademical character reference start
 * state` and `Numeric character reference end state` in the HTML spec.
 *
 * @param lastCp The last code point of the entity. Used to see if the
 *           entity was terminated with a semicolon.
 * @param expectedLength The minimum number of characters that should be
 *                   consumed. Used to validate that at least one digit
 *                   was consumed.
 * @returns The number of characters that were consumed.
 */
EntityDecoder.prototype.emitNumericEntity = function emitNumericEntity (lastCp, expectedLength) {
    var _a;
    // Ensure we consumed at least one digit.
    if (this.consumed <= expectedLength) {
        (_a = this.errors) === null || _a === void 0 ? void 0 : _a.absenceOfDigitsInNumericCharacterReference(this.consumed);
        return 0;
    }
    // Figure out if this is a legit end of the entity
    if (lastCp === CharCodes.SEMI) {
        this.consumed += 1;
    }
    else if (this.decodeMode === DecodingMode.Strict) {
        return 0;
    }
    this.emitCodePoint(replaceCodePoint(this.result), this.consumed);
    if (this.errors) {
        if (lastCp !== CharCodes.SEMI) {
            this.errors.missingSemicolonAfterCharacterReference();
        }
        this.errors.validateNumericCharacterReference(this.result);
    }
    return this.consumed;
};
/**
 * Parses a named entity.
 *
 * Equivalent to the `Named character reference state` in the HTML spec.
 *
 * @param str The string containing the entity (or a continuation of the entity).
 * @param offset The current offset.
 * @returns The number of characters that were consumed, or -1 if the entity is incomplete.
 */
EntityDecoder.prototype.stateNamedEntity = function stateNamedEntity (str, offset) {
    var ref = this;
        var decodeTree = ref.decodeTree;
    var current = decodeTree[this.treeIndex];
    // The mask is the number of bytes of the value, including the current byte.
    var valueLength = (current & BinTrieFlags.VALUE_LENGTH) >> 14;
    for (; offset < str.length; offset++, this.excess++) {
        var char = str.charCodeAt(offset);
        this.treeIndex = determineBranch(decodeTree, current, this.treeIndex + Math.max(1, valueLength), char);
        if (this.treeIndex < 0) {
            return this.result === 0 ||
                // If we are parsing an attribute
                (this.decodeMode === DecodingMode.Attribute &&
                    // We shouldn't have consumed any characters after the entity,
                    (valueLength === 0 ||
                        // And there should be no invalid characters.
                        isEntityInAttributeInvalidEnd(char)))
                ? 0
                : this.emitNotTerminatedNamedEntity();
        }
        current = decodeTree[this.treeIndex];
        valueLength = (current & BinTrieFlags.VALUE_LENGTH) >> 14;
        // If the branch is a value, store it and continue
        if (valueLength !== 0) {
            // If the entity is terminated by a semicolon, we are done.
            if (char === CharCodes.SEMI) {
                return this.emitNamedEntityData(this.treeIndex, valueLength, this.consumed + this.excess);
            }
            // If we encounter a non-terminated (legacy) entity while parsing strictly, then ignore it.
            if (this.decodeMode !== DecodingMode.Strict) {
                this.result = this.treeIndex;
                this.consumed += this.excess;
                this.excess = 0;
            }
        }
    }
    return -1;
};
/**
 * Emit a named entity that was not terminated with a semicolon.
 *
 * @returns The number of characters consumed.
 */
EntityDecoder.prototype.emitNotTerminatedNamedEntity = function emitNotTerminatedNamedEntity () {
    var _a;
    var ref = this;
        var result = ref.result;
        var decodeTree = ref.decodeTree;
    var valueLength = (decodeTree[result] & BinTrieFlags.VALUE_LENGTH) >> 14;
    this.emitNamedEntityData(result, valueLength, this.consumed);
    (_a = this.errors) === null || _a === void 0 ? void 0 : _a.missingSemicolonAfterCharacterReference();
    return this.consumed;
};
/**
 * Emit a named entity.
 *
 * @param result The index of the entity in the decode tree.
 * @param valueLength The number of bytes in the entity.
 * @param consumed The number of characters consumed.
 *
 * @returns The number of characters consumed.
 */
EntityDecoder.prototype.emitNamedEntityData = function emitNamedEntityData (result, valueLength, consumed) {
    var ref = this;
        var decodeTree = ref.decodeTree;
    this.emitCodePoint(valueLength === 1
        ? decodeTree[result] & ~BinTrieFlags.VALUE_LENGTH
        : decodeTree[result + 1], consumed);
    if (valueLength === 3) {
        // For multi-byte values, we need to emit the second byte.
        this.emitCodePoint(decodeTree[result + 2], consumed);
    }
    return consumed;
};
/**
 * Signal to the parser that the end of the input was reached.
 *
 * Remaining data will be emitted and relevant errors will be produced.
 *
 * @returns The number of characters consumed.
 */
EntityDecoder.prototype.end = function end () {
    var _a;
    switch (this.state) {
        case EntityDecoderState.NamedEntity: {
            // Emit a named entity if we have one.
            return this.result !== 0 &&
                (this.decodeMode !== DecodingMode.Attribute ||
                    this.result === this.treeIndex)
                ? this.emitNotTerminatedNamedEntity()
                : 0;
        }
        // Otherwise, emit a numeric entity if we have one.
        case EntityDecoderState.NumericDecimal: {
            return this.emitNumericEntity(0, 2);
        }
        case EntityDecoderState.NumericHex: {
            return this.emitNumericEntity(0, 3);
        }
        case EntityDecoderState.NumericStart: {
            (_a = this.errors) === null || _a === void 0 ? void 0 : _a.absenceOfDigitsInNumericCharacterReference(this.consumed);
            return 0;
        }
        case EntityDecoderState.EntityStart: {
            // Return 0 if we have no entity.
            return 0;
        }
    }
};
/**
 * Creates a function that decodes entities in a string.
 *
 * @param decodeTree The decode tree.
 * @returns A function that decodes entities in a string.
 */
function getDecoder(decodeTree) {
    var ret = "";
    var decoder = new EntityDecoder(decodeTree, function (str) { return (ret += fromCodePoint$3(str)); });
    return function decodeWithTrie(str, decodeMode) {
        var lastIndex = 0;
        var offset = 0;
        while ((offset = str.indexOf("&", offset)) >= 0) {
            ret += str.slice(lastIndex, offset);
            decoder.startEntity(decodeMode);
            var len = decoder.write(str, 
            // Skip the "&"
            offset + 1);
            if (len < 0) {
                lastIndex = offset + decoder.end();
                break;
            }
            lastIndex = offset + len;
            // If `len` is 0, skip the current `&` and continue.
            offset = len === 0 ? lastIndex + 1 : lastIndex;
        }
        var result = ret + str.slice(lastIndex);
        // Make sure we don't keep a reference to the final string.
        ret = "";
        return result;
    };
}
/**
 * Determines the branch of the current node that is taken given the current
 * character. This function is used to traverse the trie.
 *
 * @param decodeTree The trie.
 * @param current The current node.
 * @param nodeIdx The index right after the current node and its value.
 * @param char The current character.
 * @returns The index of the next node, or -1 if no branch is taken.
 */
function determineBranch(decodeTree, current, nodeIdx, char) {
    var branchCount = (current & BinTrieFlags.BRANCH_LENGTH) >> 7;
    var jumpOffset = current & BinTrieFlags.JUMP_TABLE;
    // Case 1: Single branch encoded in jump offset
    if (branchCount === 0) {
        return jumpOffset !== 0 && char === jumpOffset ? nodeIdx : -1;
    }
    // Case 2: Multiple branches encoded in jump table
    if (jumpOffset) {
        var value = char - jumpOffset;
        return value < 0 || value >= branchCount
            ? -1
            : decodeTree[nodeIdx + value] - 1;
    }
    // Case 3: Multiple branches encoded in dictionary
    // Binary search for the character.
    var lo = nodeIdx;
    var hi = lo + branchCount - 1;
    while (lo <= hi) {
        var mid = (lo + hi) >>> 1;
        var midVal = decodeTree[mid];
        if (midVal < char) {
            lo = mid + 1;
        }
        else if (midVal > char) {
            hi = mid - 1;
        }
        else {
            return decodeTree[mid + branchCount];
        }
    }
    return -1;
}
var htmlDecoder = getDecoder(htmlDecodeTree);
getDecoder(xmlDecodeTree);
/**
 * Decodes an HTML string.
 *
 * @param str The string to decode.
 * @param mode The decoding mode.
 * @returns The decoded string.
 */
function decodeHTML(str, mode) {
    if ( mode === void 0 ) mode = DecodingMode.Legacy;

    return htmlDecoder(str, mode);
}// Utilities
//


function _class$2 (obj) { return Object.prototype.toString.call(obj) }

function isString$2 (obj) { return _class$2(obj) === '[object String]' }

var _hasOwnProperty = Object.prototype.hasOwnProperty;

function has$1 (object, key) {
  return _hasOwnProperty.call(object, key)
}

// Merge objects
//
function assign$3 (obj /* from1, from2, from3, ... */) {
  var sources = Array.prototype.slice.call(arguments, 1);

  sources.forEach(function (source) {
    if (!source) { return }

    if (typeof source !== 'object') {
      throw new TypeError(source + 'must be object')
    }

    Object.keys(source).forEach(function (key) {
      obj[key] = source[key];
    });
  });

  return obj
}

// Remove element from array and put another array at those position.
// Useful for some operations with tokens
function arrayReplaceAt$1 (src, pos, newElements) {
  return [].concat(src.slice(0, pos), newElements, src.slice(pos + 1))
}

function isValidEntityCode$1 (c) {
  /* eslint no-bitwise:0 */
  // broken sequence
  if (c >= 0xD800 && c <= 0xDFFF) { return false }
  // never used
  if (c >= 0xFDD0 && c <= 0xFDEF) { return false }
  if ((c & 0xFFFF) === 0xFFFF || (c & 0xFFFF) === 0xFFFE) { return false }
  // control codes
  if (c >= 0x00 && c <= 0x08) { return false }
  if (c === 0x0B) { return false }
  if (c >= 0x0E && c <= 0x1F) { return false }
  if (c >= 0x7F && c <= 0x9F) { return false }
  // out of range
  if (c > 0x10FFFF) { return false }
  return true
}

function fromCodePoint$2 (c) {
  /* eslint no-bitwise:0 */
  if (c > 0xffff) {
    c -= 0x10000;
    var surrogate1 = 0xd800 + (c >> 10);
    var surrogate2 = 0xdc00 + (c & 0x3ff);

    return String.fromCharCode(surrogate1, surrogate2)
  }
  return String.fromCharCode(c)
}

var UNESCAPE_MD_RE  = /\\([!"#$%&'()*+,\-./:;<=>?@[\\\]^_`{|}~])/g;
var ENTITY_RE       = /&([a-z#][a-z0-9]{1,31});/gi;
var UNESCAPE_ALL_RE = new RegExp(UNESCAPE_MD_RE.source + '|' + ENTITY_RE.source, 'gi');

var DIGITAL_ENTITY_TEST_RE = /^#((?:x[a-f0-9]{1,8}|[0-9]{1,8}))$/i;

function replaceEntityPattern (match, name) {
  if (name.charCodeAt(0) === 0x23/* # */ && DIGITAL_ENTITY_TEST_RE.test(name)) {
    var code = name[1].toLowerCase() === 'x'
      ? parseInt(name.slice(2), 16)
      : parseInt(name.slice(1), 10);

    if (isValidEntityCode$1(code)) {
      return fromCodePoint$2(code)
    }

    return match
  }

  var decoded = decodeHTML(match);
  if (decoded !== match) {
    return decoded
  }

  return match
}

/* function replaceEntities(str) {
  if (str.indexOf('&') < 0) { return str; }

  return str.replace(ENTITY_RE, replaceEntityPattern);
} */

function unescapeMd (str) {
  if (str.indexOf('\\') < 0) { return str }
  return str.replace(UNESCAPE_MD_RE, '$1')
}

function unescapeAll$3 (str) {
  if (str.indexOf('\\') < 0 && str.indexOf('&') < 0) { return str }

  return str.replace(UNESCAPE_ALL_RE, function (match, escaped, entity) {
    if (escaped) { return escaped }
    return replaceEntityPattern(match, entity)
  })
}

var HTML_ESCAPE_TEST_RE = /[&<>"]/;
var HTML_ESCAPE_REPLACE_RE = /[&<>"]/g;
var HTML_REPLACEMENTS = {
  '&': '&amp;',
  '<': '&lt;',
  '>': '&gt;',
  '"': '&quot;'
};

function replaceUnsafeChar (ch) {
  return HTML_REPLACEMENTS[ch]
}

function escapeHtml$1 (str) {
  if (HTML_ESCAPE_TEST_RE.test(str)) {
    return str.replace(HTML_ESCAPE_REPLACE_RE, replaceUnsafeChar)
  }
  return str
}

var REGEXP_ESCAPE_RE = /[.?*+^$[\]\\(){}|-]/g;

function escapeRE$2 (str) {
  return str.replace(REGEXP_ESCAPE_RE, '\\$&')
}

function isSpace$e (code) {
  switch (code) {
    case 0x09:
    case 0x20:
      return true
  }
  return false
}

// Zs (unicode class) || [\t\f\v\r\n]
function isWhiteSpace$2 (code) {
  if (code >= 0x2000 && code <= 0x200A) { return true }
  switch (code) {
    case 0x09: // \t
    case 0x0A: // \n
    case 0x0B: // \v
    case 0x0C: // \f
    case 0x0D: // \r
    case 0x20:
    case 0xA0:
    case 0x1680:
    case 0x202F:
    case 0x205F:
    case 0x3000:
      return true
  }
  return false
}

/* eslint-disable max-len */

// Currently without astral characters support.
function isPunctChar$2 (ch) {
  return P.test(ch) || regex$5.test(ch)
}

// Markdown ASCII punctuation characters.
//
// !, ", #, $, %, &, ', (, ), *, +, ,, -, ., /, :, ;, <, =, >, ?, @, [, \, ], ^, _, `, {, |, }, or ~
// http://spec.commonmark.org/0.15/#ascii-punctuation-character
//
// Don't confuse with unicode punctuation !!! It lacks some chars in ascii range.
//
function isMdAsciiPunct$2 (ch) {
  switch (ch) {
    case 0x21/* ! */:
    case 0x22/* " */:
    case 0x23/* # */:
    case 0x24/* $ */:
    case 0x25/* % */:
    case 0x26/* & */:
    case 0x27/* ' */:
    case 0x28/* ( */:
    case 0x29/* ) */:
    case 0x2A/* * */:
    case 0x2B/* + */:
    case 0x2C/* , */:
    case 0x2D/* - */:
    case 0x2E/* . */:
    case 0x2F/* / */:
    case 0x3A/* : */:
    case 0x3B/* ; */:
    case 0x3C/* < */:
    case 0x3D/* = */:
    case 0x3E/* > */:
    case 0x3F/* ? */:
    case 0x40/* @ */:
    case 0x5B/* [ */:
    case 0x5C/* \ */:
    case 0x5D/* ] */:
    case 0x5E/* ^ */:
    case 0x5F/* _ */:
    case 0x60/* ` */:
    case 0x7B/* { */:
    case 0x7C/* | */:
    case 0x7D/* } */:
    case 0x7E/* ~ */:
      return true
    default:
      return false
  }
}

// Hepler to unify [reference labels].
//
function normalizeReference$3 (str) {
  // Trim and collapse whitespace
  //
  str = str.trim().replace(/\s+/g, ' ');

  // In node v10 'ẞ'.toLowerCase() === 'Ṿ', which is presumed to be a bug
  // fixed in v12 (couldn't find any details).
  //
  // So treat this one as a special case
  // (remove this when node v10 is no longer supported).
  //
  if ('ẞ'.toLowerCase() === 'Ṿ') {
    str = str.replace(/ẞ/g, 'ß');
  }

  // .toLowerCase().toUpperCase() should get rid of all differences
  // between letter variants.
  //
  // Simple .toLowerCase() doesn't normalize 125 code points correctly,
  // and .toUpperCase doesn't normalize 6 of them (list of exceptions:
  // İ, ϴ, ẞ, Ω, K, Å - those are already uppercased, but have differently
  // uppercased versions).
  //
  // Here's an example showing how it happens. Lets take greek letter omega:
  // uppercase U+0398 (Θ), U+03f4 (ϴ) and lowercase U+03b8 (θ), U+03d1 (ϑ)
  //
  // Unicode entries:
  // 0398;GREEK CAPITAL LETTER THETA;Lu;0;L;;;;;N;;;;03B8;
  // 03B8;GREEK SMALL LETTER THETA;Ll;0;L;;;;;N;;;0398;;0398
  // 03D1;GREEK THETA SYMBOL;Ll;0;L;<compat> 03B8;;;;N;GREEK SMALL LETTER SCRIPT THETA;;0398;;0398
  // 03F4;GREEK CAPITAL THETA SYMBOL;Lu;0;L;<compat> 0398;;;;N;;;;03B8;
  //
  // Case-insensitive comparison should treat all of them as equivalent.
  //
  // But .toLowerCase() doesn't change ϑ (it's already lowercase),
  // and .toUpperCase() doesn't change ϴ (already uppercase).
  //
  // Applying first lower then upper case normalizes any character:
  // '\u0398\u03f4\u03b8\u03d1'.toLowerCase().toUpperCase() === '\u0398\u0398\u0398\u0398'
  //
  // Note: this is equivalent to unicode case folding; unicode normalization
  // is a different step that is not required here.
  //
  // Final result should be uppercased, because it's later stored in an object
  // (this avoid a conflict with Object.prototype members,
  // most notably, `__proto__`)
  //
  return str.toLowerCase().toUpperCase()
}

// Re-export libraries commonly used in both markdown-it and its plugins,
// so plugins won't have to depend on them explicitly, which reduces their
// bundled size (e.g. a browser build).
//
var lib$4 = { mdurl: mdurl$2, ucmicro: ucmicro };var utils$2=/*#__PURE__*/Object.freeze({__proto__:null,arrayReplaceAt:arrayReplaceAt$1,assign:assign$3,escapeHtml:escapeHtml$1,escapeRE:escapeRE$2,fromCodePoint:fromCodePoint$2,has:has$1,isMdAsciiPunct:isMdAsciiPunct$2,isPunctChar:isPunctChar$2,isSpace:isSpace$e,isString:isString$2,isValidEntityCode:isValidEntityCode$1,isWhiteSpace:isWhiteSpace$2,lib:lib$4,normalizeReference:normalizeReference$3,unescapeAll:unescapeAll$3,unescapeMd:unescapeMd});// Parse link label
//
// this function assumes that first character ("[") already matches;
// returns the end of the label
//

function parseLinkLabel (state, start, disableNested) {
  var level, found, marker, prevPos;

  var max = state.posMax;
  var oldPos = state.pos;

  state.pos = start + 1;
  level = 1;

  while (state.pos < max) {
    marker = state.src.charCodeAt(state.pos);
    if (marker === 0x5D /* ] */) {
      level--;
      if (level === 0) {
        found = true;
        break
      }
    }

    prevPos = state.pos;
    state.md.inline.skipToken(state);
    if (marker === 0x5B /* [ */) {
      if (prevPos === state.pos - 1) {
        // increase level if we find text `[`, which is not a part of any token
        level++;
      } else if (disableNested) {
        state.pos = oldPos;
        return -1
      }
    }
  }

  var labelEnd = -1;

  if (found) {
    labelEnd = state.pos;
  }

  // restore old state
  state.pos = oldPos;

  return labelEnd
}// Parse link destination
//


function parseLinkDestination (str, start, max) {
  var code;
  var pos = start;

  var result = {
    ok: false,
    pos: 0,
    str: ''
  };

  if (str.charCodeAt(pos) === 0x3C /* < */) {
    pos++;
    while (pos < max) {
      code = str.charCodeAt(pos);
      if (code === 0x0A /* \n */) { return result }
      if (code === 0x3C /* < */) { return result }
      if (code === 0x3E /* > */) {
        result.pos = pos + 1;
        result.str = unescapeAll$3(str.slice(start + 1, pos));
        result.ok = true;
        return result
      }
      if (code === 0x5C /* \ */ && pos + 1 < max) {
        pos += 2;
        continue
      }

      pos++;
    }

    // no closing '>'
    return result
  }

  // this should be ... } else { ... branch

  var level = 0;
  while (pos < max) {
    code = str.charCodeAt(pos);

    if (code === 0x20) { break }

    // ascii control characters
    if (code < 0x20 || code === 0x7F) { break }

    if (code === 0x5C /* \ */ && pos + 1 < max) {
      if (str.charCodeAt(pos + 1) === 0x20) { break }
      pos += 2;
      continue
    }

    if (code === 0x28 /* ( */) {
      level++;
      if (level > 32) { return result }
    }

    if (code === 0x29 /* ) */) {
      if (level === 0) { break }
      level--;
    }

    pos++;
  }

  if (start === pos) { return result }
  if (level !== 0) { return result }

  result.str = unescapeAll$3(str.slice(start, pos));
  result.pos = pos;
  result.ok = true;
  return result
}// Parse link title
//


// Parse link title within `str` in [start, max] range,
// or continue previous parsing if `prev_state` is defined (equal to result of last execution).
//
function parseLinkTitle (str, start, max, prev_state) {
  var code;
  var pos = start;

  var state = {
    // if `true`, this is a valid link title
    ok: false,
    // if `true`, this link can be continued on the next line
    can_continue: false,
    // if `ok`, it's the position of the first character after the closing marker
    pos: 0,
    // if `ok`, it's the unescaped title
    str: '',
    // expected closing marker character code
    marker: 0
  };

  if (prev_state) {
    // this is a continuation of a previous parseLinkTitle call on the next line,
    // used in reference links only
    state.str = prev_state.str;
    state.marker = prev_state.marker;
  } else {
    if (pos >= max) { return state }

    var marker = str.charCodeAt(pos);
    if (marker !== 0x22 /* " */ && marker !== 0x27 /* ' */ && marker !== 0x28 /* ( */) { return state }

    start++;
    pos++;

    // if opening marker is "(", switch it to closing marker ")"
    if (marker === 0x28) { marker = 0x29; }

    state.marker = marker;
  }

  while (pos < max) {
    code = str.charCodeAt(pos);
    if (code === state.marker) {
      state.pos = pos + 1;
      state.str += unescapeAll$3(str.slice(start, pos));
      state.ok = true;
      return state
    } else if (code === 0x28 /* ( */ && state.marker === 0x29 /* ) */) {
      return state
    } else if (code === 0x5C /* \ */ && pos + 1 < max) {
      pos++;
    }

    pos++;
  }

  // no closing marker found, but this link title may continue on the next line (for references)
  state.can_continue = true;
  state.str += unescapeAll$3(str.slice(start, pos));
  return state
}// Just a shortcut for bulk export
var helpers$2=/*#__PURE__*/Object.freeze({__proto__:null,parseLinkDestination:parseLinkDestination,parseLinkLabel:parseLinkLabel,parseLinkTitle:parseLinkTitle});/**
 * class Renderer
 *
 * Generates HTML from parsed token stream. Each instance has independent
 * copy of rules. Those can be rewritten with ease. Also, you can add new
 * rules if you create plugin and adds new token types.
 **/


var default_rules$1 = {};

default_rules$1.code_inline = function (tokens, idx, options, env, slf) {
  var token = tokens[idx];

  return  '<code' + slf.renderAttrs(token) + '>' +
          escapeHtml$1(token.content) +
          '</code>'
};

default_rules$1.code_block = function (tokens, idx, options, env, slf) {
  var token = tokens[idx];

  return  '<pre' + slf.renderAttrs(token) + '><code>' +
          escapeHtml$1(tokens[idx].content) +
          '</code></pre>\n'
};

default_rules$1.fence = function (tokens, idx, options, env, slf) {
  var token = tokens[idx];
  var info = token.info ? unescapeAll$3(token.info).trim() : '';
  var langName = '';
  var langAttrs = '';

  if (info) {
    var arr = info.split(/(\s+)/g);
    langName = arr[0];
    langAttrs = arr.slice(2).join('');
  }

  var highlighted;
  if (options.highlight) {
    highlighted = options.highlight(token.content, langName, langAttrs) || escapeHtml$1(token.content);
  } else {
    highlighted = escapeHtml$1(token.content);
  }

  if (highlighted.indexOf('<pre') === 0) {
    return highlighted + '\n'
  }

  // If language exists, inject class gently, without modifying original token.
  // May be, one day we will add .deepClone() for token and simplify this part, but
  // now we prefer to keep things local.
  if (info) {
    var i = token.attrIndex('class');
    var tmpAttrs = token.attrs ? token.attrs.slice() : [];

    if (i < 0) {
      tmpAttrs.push(['class', options.langPrefix + langName]);
    } else {
      tmpAttrs[i] = tmpAttrs[i].slice();
      tmpAttrs[i][1] += ' ' + options.langPrefix + langName;
    }

    // Fake token just to render attributes
    var tmpToken = {
      attrs: tmpAttrs
    };

    return ("<pre><code" + (slf.renderAttrs(tmpToken)) + ">" + highlighted + "</code></pre>\n")
  }

  return ("<pre><code" + (slf.renderAttrs(token)) + ">" + highlighted + "</code></pre>\n")
};

default_rules$1.image = function (tokens, idx, options, env, slf) {
  var token = tokens[idx];

  // "alt" attr MUST be set, even if empty. Because it's mandatory and
  // should be placed on proper position for tests.
  //
  // Replace content with actual value

  token.attrs[token.attrIndex('alt')][1] =
    slf.renderInlineAsText(token.children, options, env);

  return slf.renderToken(tokens, idx, options)
};

default_rules$1.hardbreak = function (tokens, idx, options /*, env */) {
  return options.xhtmlOut ? '<br />\n' : '<br>\n'
};
default_rules$1.softbreak = function (tokens, idx, options /*, env */) {
  return options.breaks ? (options.xhtmlOut ? '<br />\n' : '<br>\n') : '\n'
};

default_rules$1.text = function (tokens, idx /*, options, env */) {
  return escapeHtml$1(tokens[idx].content)
};

default_rules$1.html_block = function (tokens, idx /*, options, env */) {
  return tokens[idx].content
};
default_rules$1.html_inline = function (tokens, idx /*, options, env */) {
  return tokens[idx].content
};

/**
 * new Renderer()
 *
 * Creates new [[Renderer]] instance and fill [[Renderer#rules]] with defaults.
 **/
function Renderer$2 () {
  /**
   * Renderer#rules -> Object
   *
   * Contains render rules for tokens. Can be updated and extended.
   *
   * ##### Example
   *
   * ```javascript
   * var md = require('markdown-it')();
   *
   * md.renderer.rules.strong_open  = function () { return '<b>'; };
   * md.renderer.rules.strong_close = function () { return '</b>'; };
   *
   * var result = md.renderInline(...);
   * ```
   *
   * Each rule is called as independent static function with fixed signature:
   *
   * ```javascript
   * function my_token_render(tokens, idx, options, env, renderer) {
   *   // ...
   *   return renderedHTML;
   * }
   * ```
   *
   * See [source code](https://github.com/markdown-it/markdown-it/blob/master/lib/renderer.mjs)
   * for more details and examples.
   **/
  this.rules = assign$3({}, default_rules$1);
}

/**
 * Renderer.renderAttrs(token) -> String
 *
 * Render token attributes to string.
 **/
Renderer$2.prototype.renderAttrs = function renderAttrs (token) {
  var i, l, result;

  if (!token.attrs) { return '' }

  result = '';

  for (i = 0, l = token.attrs.length; i < l; i++) {
    result += ' ' + escapeHtml$1(token.attrs[i][0]) + '="' + escapeHtml$1(token.attrs[i][1]) + '"';
  }

  return result
};

/**
 * Renderer.renderToken(tokens, idx, options) -> String
 * - tokens (Array): list of tokens
 * - idx (Numbed): token index to render
 * - options (Object): params of parser instance
 *
 * Default token renderer. Can be overriden by custom function
 * in [[Renderer#rules]].
 **/
Renderer$2.prototype.renderToken = function renderToken (tokens, idx, options) {
  var token = tokens[idx];
  var result = '';

  // Tight list paragraphs
  if (token.hidden) {
    return ''
  }

  // Insert a newline between hidden paragraph and subsequent opening
  // block-level tag.
  //
  // For example, here we should insert a newline before blockquote:
  //  - a
  //    >
  //
  if (token.block && token.nesting !== -1 && idx && tokens[idx - 1].hidden) {
    result += '\n';
  }

  // Add token name, e.g. `<img`
  result += (token.nesting === -1 ? '</' : '<') + token.tag;

  // Encode attributes, e.g. `<img src="foo"`
  result += this.renderAttrs(token);

  // Add a slash for self-closing tags, e.g. `<img src="foo" /`
  if (token.nesting === 0 && options.xhtmlOut) {
    result += ' /';
  }

  // Check if we need to add a newline after this tag
  var needLf = false;
  if (token.block) {
    needLf = true;

    if (token.nesting === 1) {
      if (idx + 1 < tokens.length) {
        var nextToken = tokens[idx + 1];

        if (nextToken.type === 'inline' || nextToken.hidden) {
          // Block-level tag containing an inline tag.
          //
          needLf = false;
        } else if (nextToken.nesting === -1 && nextToken.tag === token.tag) {
          // Opening tag + closing tag of the same type. E.g. `<li></li>`.
          //
          needLf = false;
        }
      }
    }
  }

  result += needLf ? '>\n' : '>';

  return result
};

/**
 * Renderer.renderInline(tokens, options, env) -> String
 * - tokens (Array): list on block tokens to render
 * - options (Object): params of parser instance
 * - env (Object): additional data from parsed input (references, for example)
 *
 * The same as [[Renderer.render]], but for single token of `inline` type.
 **/
Renderer$2.prototype.renderInline = function (tokens, options, env) {
  var result = '';
  var rules = this.rules;

  for (var i = 0, len = tokens.length; i < len; i++) {
    var type = tokens[i].type;

    if (typeof rules[type] !== 'undefined') {
      result += rules[type](tokens, i, options, env, this);
    } else {
      result += this.renderToken(tokens, i, options);
    }
  }

  return result
};

/** internal
 * Renderer.renderInlineAsText(tokens, options, env) -> String
 * - tokens (Array): list on block tokens to render
 * - options (Object): params of parser instance
 * - env (Object): additional data from parsed input (references, for example)
 *
 * Special kludge for image `alt` attributes to conform CommonMark spec.
 * Don't try to use it! Spec requires to show `alt` content with stripped markup,
 * instead of simple escaping.
 **/
Renderer$2.prototype.renderInlineAsText = function (tokens, options, env) {
  var result = '';

  for (var i = 0, len = tokens.length; i < len; i++) {
    switch (tokens[i].type) {
      case 'text':
        result += tokens[i].content;
        break
      case 'image':
        result += this.renderInlineAsText(tokens[i].children, options, env);
        break
      case 'html_inline':
      case 'html_block':
        result += tokens[i].content;
        break
      case 'softbreak':
      case 'hardbreak':
        result += '\n';
        break
        // all other tokens are skipped
    }
  }

  return result
};

/**
 * Renderer.render(tokens, options, env) -> String
 * - tokens (Array): list on block tokens to render
 * - options (Object): params of parser instance
 * - env (Object): additional data from parsed input (references, for example)
 *
 * Takes token stream and generates HTML. Probably, you will never need to call
 * this method directly.
 **/
Renderer$2.prototype.render = function (tokens, options, env) {
  var result = '';
  var rules = this.rules;

  for (var i = 0, len = tokens.length; i < len; i++) {
    var type = tokens[i].type;

    if (type === 'inline') {
      result += this.renderInline(tokens[i].children, options, env);
    } else if (typeof rules[type] !== 'undefined') {
      result += rules[type](tokens, i, options, env, this);
    } else {
      result += this.renderToken(tokens, i, options, env);
    }
  }

  return result
};/**
 * class Ruler
 *
 * Helper class, used by [[MarkdownIt#core]], [[MarkdownIt#block]] and
 * [[MarkdownIt#inline]] to manage sequences of functions (rules):
 *
 * - keep rules in defined order
 * - assign the name to each rule
 * - enable/disable rules
 * - add/replace rules
 * - allow assign rules to additional named chains (in the same)
 * - cacheing lists of active rules
 *
 * You will not need use this class directly until write plugins. For simple
 * rules control use [[MarkdownIt.disable]], [[MarkdownIt.enable]] and
 * [[MarkdownIt.use]].
 **/

/**
 * new Ruler()
 **/
function Ruler$4 () {
  // List of added rules. Each element is:
  //
  // {
  //   name: XXX,
  //   enabled: Boolean,
  //   fn: Function(),
  //   alt: [ name2, name3 ]
  // }
  //
  this.__rules__ = [];

  // Cached rule chains.
  //
  // First level - chain name, '' for default.
  // Second level - diginal anchor for fast filtering by charcodes.
  //
  this.__cache__ = null;
}

// Helper methods, should not be used directly

// Find rule index by name
//
Ruler$4.prototype.__find__ = function (name) {
  for (var i = 0; i < this.__rules__.length; i++) {
    if (this.__rules__[i].name === name) {
      return i
    }
  }
  return -1
};

// Build rules lookup cache
//
Ruler$4.prototype.__compile__ = function () {
  var self = this;
  var chains = [''];

  // collect unique names
  self.__rules__.forEach(function (rule) {
    if (!rule.enabled) { return }

    rule.alt.forEach(function (altName) {
      if (chains.indexOf(altName) < 0) {
        chains.push(altName);
      }
    });
  });

  self.__cache__ = {};

  chains.forEach(function (chain) {
    self.__cache__[chain] = [];
    self.__rules__.forEach(function (rule) {
      if (!rule.enabled) { return }

      if (chain && rule.alt.indexOf(chain) < 0) { return }

      self.__cache__[chain].push(rule.fn);
    });
  });
};

/**
 * Ruler.at(name, fn [, options])
 * - name (String): rule name to replace.
 * - fn (Function): new rule function.
 * - options (Object): new rule options (not mandatory).
 *
 * Replace rule by name with new function & options. Throws error if name not
 * found.
 *
 * ##### Options:
 *
 * - __alt__ - array with names of "alternate" chains.
 *
 * ##### Example
 *
 * Replace existing typographer replacement rule with new one:
 *
 * ```javascript
 * var md = require('markdown-it')();
 *
 * md.core.ruler.at('replacements', function replace(state) {
 *   //...
 * });
 * ```
 **/
Ruler$4.prototype.at = function (name, fn, options) {
  var index = this.__find__(name);
  var opt = options || {};

  if (index === -1) { throw new Error('Parser rule not found: ' + name) }

  this.__rules__[index].fn = fn;
  this.__rules__[index].alt = opt.alt || [];
  this.__cache__ = null;
};

/**
 * Ruler.before(beforeName, ruleName, fn [, options])
 * - beforeName (String): new rule will be added before this one.
 * - ruleName (String): name of added rule.
 * - fn (Function): rule function.
 * - options (Object): rule options (not mandatory).
 *
 * Add new rule to chain before one with given name. See also
 * [[Ruler.after]], [[Ruler.push]].
 *
 * ##### Options:
 *
 * - __alt__ - array with names of "alternate" chains.
 *
 * ##### Example
 *
 * ```javascript
 * var md = require('markdown-it')();
 *
 * md.block.ruler.before('paragraph', 'my_rule', function replace(state) {
 *   //...
 * });
 * ```
 **/
Ruler$4.prototype.before = function (beforeName, ruleName, fn, options) {
  var index = this.__find__(beforeName);
  var opt = options || {};

  if (index === -1) { throw new Error('Parser rule not found: ' + beforeName) }

  this.__rules__.splice(index, 0, {
    name: ruleName,
    enabled: true,
    fn: fn,
    alt: opt.alt || []
  });

  this.__cache__ = null;
};

/**
 * Ruler.after(afterName, ruleName, fn [, options])
 * - afterName (String): new rule will be added after this one.
 * - ruleName (String): name of added rule.
 * - fn (Function): rule function.
 * - options (Object): rule options (not mandatory).
 *
 * Add new rule to chain after one with given name. See also
 * [[Ruler.before]], [[Ruler.push]].
 *
 * ##### Options:
 *
 * - __alt__ - array with names of "alternate" chains.
 *
 * ##### Example
 *
 * ```javascript
 * var md = require('markdown-it')();
 *
 * md.inline.ruler.after('text', 'my_rule', function replace(state) {
 *   //...
 * });
 * ```
 **/
Ruler$4.prototype.after = function (afterName, ruleName, fn, options) {
  var index = this.__find__(afterName);
  var opt = options || {};

  if (index === -1) { throw new Error('Parser rule not found: ' + afterName) }

  this.__rules__.splice(index + 1, 0, {
    name: ruleName,
    enabled: true,
    fn: fn,
    alt: opt.alt || []
  });

  this.__cache__ = null;
};

/**
 * Ruler.push(ruleName, fn [, options])
 * - ruleName (String): name of added rule.
 * - fn (Function): rule function.
 * - options (Object): rule options (not mandatory).
 *
 * Push new rule to the end of chain. See also
 * [[Ruler.before]], [[Ruler.after]].
 *
 * ##### Options:
 *
 * - __alt__ - array with names of "alternate" chains.
 *
 * ##### Example
 *
 * ```javascript
 * var md = require('markdown-it')();
 *
 * md.core.ruler.push('my_rule', function replace(state) {
 *   //...
 * });
 * ```
 **/
Ruler$4.prototype.push = function (ruleName, fn, options) {
  var opt = options || {};

  this.__rules__.push({
    name: ruleName,
    enabled: true,
    fn: fn,
    alt: opt.alt || []
  });

  this.__cache__ = null;
};

/**
 * Ruler.enable(list [, ignoreInvalid]) -> Array
 * - list (String|Array): list of rule names to enable.
 * - ignoreInvalid (Boolean): set `true` to ignore errors when rule not found.
 *
 * Enable rules with given names. If any rule name not found - throw Error.
 * Errors can be disabled by second param.
 *
 * Returns list of found rule names (if no exception happened).
 *
 * See also [[Ruler.disable]], [[Ruler.enableOnly]].
 **/
Ruler$4.prototype.enable = function (list, ignoreInvalid) {
  if (!Array.isArray(list)) { list = [list]; }

  var result = [];

  // Search by name and enable
  list.forEach(function (name) {
    var idx = this.__find__(name);

    if (idx < 0) {
      if (ignoreInvalid) { return }
      throw new Error('Rules manager: invalid rule name ' + name)
    }
    this.__rules__[idx].enabled = true;
    result.push(name);
  }, this);

  this.__cache__ = null;
  return result
};

/**
 * Ruler.enableOnly(list [, ignoreInvalid])
 * - list (String|Array): list of rule names to enable (whitelist).
 * - ignoreInvalid (Boolean): set `true` to ignore errors when rule not found.
 *
 * Enable rules with given names, and disable everything else. If any rule name
 * not found - throw Error. Errors can be disabled by second param.
 *
 * See also [[Ruler.disable]], [[Ruler.enable]].
 **/
Ruler$4.prototype.enableOnly = function (list, ignoreInvalid) {
  if (!Array.isArray(list)) { list = [list]; }

  this.__rules__.forEach(function (rule) { rule.enabled = false; });

  this.enable(list, ignoreInvalid);
};

/**
 * Ruler.disable(list [, ignoreInvalid]) -> Array
 * - list (String|Array): list of rule names to disable.
 * - ignoreInvalid (Boolean): set `true` to ignore errors when rule not found.
 *
 * Disable rules with given names. If any rule name not found - throw Error.
 * Errors can be disabled by second param.
 *
 * Returns list of found rule names (if no exception happened).
 *
 * See also [[Ruler.enable]], [[Ruler.enableOnly]].
 **/
Ruler$4.prototype.disable = function (list, ignoreInvalid) {
  if (!Array.isArray(list)) { list = [list]; }

  var result = [];

  // Search by name and disable
  list.forEach(function (name) {
    var idx = this.__find__(name);

    if (idx < 0) {
      if (ignoreInvalid) { return }
      throw new Error('Rules manager: invalid rule name ' + name)
    }
    this.__rules__[idx].enabled = false;
    result.push(name);
  }, this);

  this.__cache__ = null;
  return result
};

/**
 * Ruler.getRules(chainName) -> Array
 *
 * Return array of active functions (rules) for given chain name. It analyzes
 * rules configuration, compiles caches if not exists and returns result.
 *
 * Default chain name is `''` (empty string). It can't be skipped. That's
 * done intentionally, to keep signature monomorphic for high speed.
 **/
Ruler$4.prototype.getRules = function (chainName) {
  if (this.__cache__ === null) {
    this.__compile__();
  }

  // Chain can be empty, if rules disabled. But we still have to return Array.
  return this.__cache__[chainName] || []
};// Token class

/**
 * class Token
 **/

/**
 * new Token(type, tag, nesting)
 *
 * Create new token and fill passed properties.
 **/
function Token$4 (type, tag, nesting) {
  /**
   * Token#type -> String
   *
   * Type of the token (string, e.g. "paragraph_open")
   **/
  this.type     = type;

  /**
   * Token#tag -> String
   *
   * html tag name, e.g. "p"
   **/
  this.tag      = tag;

  /**
   * Token#attrs -> Array
   *
   * Html attributes. Format: `[ [ name1, value1 ], [ name2, value2 ] ]`
   **/
  this.attrs    = null;

  /**
   * Token#map -> Array
   *
   * Source map info. Format: `[ line_begin, line_end ]`
   **/
  this.map      = null;

  /**
   * Token#nesting -> Number
   *
   * Level change (number in {-1, 0, 1} set), where:
   *
   * -  `1` means the tag is opening
   * -  `0` means the tag is self-closing
   * - `-1` means the tag is closing
   **/
  this.nesting  = nesting;

  /**
   * Token#level -> Number
   *
   * nesting level, the same as `state.level`
   **/
  this.level    = 0;

  /**
   * Token#children -> Array
   *
   * An array of child nodes (inline and img tokens)
   **/
  this.children = null;

  /**
   * Token#content -> String
   *
   * In a case of self-closing tag (code, html, fence, etc.),
   * it has contents of this tag.
   **/
  this.content  = '';

  /**
   * Token#markup -> String
   *
   * '*' or '_' for emphasis, fence string for fence, etc.
   **/
  this.markup   = '';

  /**
   * Token#info -> String
   *
   * Additional information:
   *
   * - Info string for "fence" tokens
   * - The value "auto" for autolink "link_open" and "link_close" tokens
   * - The string value of the item marker for ordered-list "list_item_open" tokens
   **/
  this.info     = '';

  /**
   * Token#meta -> Object
   *
   * A place for plugins to store an arbitrary data
   **/
  this.meta     = null;

  /**
   * Token#block -> Boolean
   *
   * True for block-level tokens, false for inline tokens.
   * Used in renderer to calculate line breaks
   **/
  this.block    = false;

  /**
   * Token#hidden -> Boolean
   *
   * If it's true, ignore this element when rendering. Used for tight lists
   * to hide paragraphs.
   **/
  this.hidden   = false;
}

/**
 * Token.attrIndex(name) -> Number
 *
 * Search attribute index by name.
 **/
Token$4.prototype.attrIndex = function attrIndex (name) {
  if (!this.attrs) { return -1 }

  var attrs = this.attrs;

  for (var i = 0, len = attrs.length; i < len; i++) {
    if (attrs[i][0] === name) { return i }
  }
  return -1
};

/**
 * Token.attrPush(attrData)
 *
 * Add `[ name, value ]` attribute to list. Init attrs if necessary
 **/
Token$4.prototype.attrPush = function attrPush (attrData) {
  if (this.attrs) {
    this.attrs.push(attrData);
  } else {
    this.attrs = [attrData];
  }
};

/**
 * Token.attrSet(name, value)
 *
 * Set `name` attribute to `value`. Override old value if exists.
 **/
Token$4.prototype.attrSet = function attrSet (name, value) {
  var idx = this.attrIndex(name);
  var attrData = [name, value];

  if (idx < 0) {
    this.attrPush(attrData);
  } else {
    this.attrs[idx] = attrData;
  }
};

/**
 * Token.attrGet(name)
 *
 * Get the value of attribute `name`, or null if it does not exist.
 **/
Token$4.prototype.attrGet = function attrGet (name) {
  var idx = this.attrIndex(name);
  var value = null;
  if (idx >= 0) {
    value = this.attrs[idx][1];
  }
  return value
};

/**
 * Token.attrJoin(name, value)
 *
 * Join value to existing attribute via space. Or create new attribute if not
 * exists. Useful to operate with token classes.
 **/
Token$4.prototype.attrJoin = function attrJoin (name, value) {
  var idx = this.attrIndex(name);

  if (idx < 0) {
    this.attrPush([name, value]);
  } else {
    this.attrs[idx][1] = this.attrs[idx][1] + ' ' + value;
  }
};// Core state object
//


function StateCore$1 (src, md, env) {
  this.src = src;
  this.env = env;
  this.tokens = [];
  this.inlineMode = false;
  this.md = md; // link to parser instance
}

// re-export Token class to use in core rules
StateCore$1.prototype.Token = Token$4;// Normalize input string

// https://spec.commonmark.org/0.29/#line-ending
var NEWLINES_RE$1  = /\r\n?|\n/g;
var NULL_RE$1      = /\0/g;

function normalize$1 (state) {
  var str;

  // Normalize newlines
  str = state.src.replace(NEWLINES_RE$1, '\n');

  // Replace NULL characters
  str = str.replace(NULL_RE$1, '\uFFFD');

  state.src = str;
}function block$2 (state) {
  var token;

  if (state.inlineMode) {
    token          = new state.Token('inline', '', 0);
    token.content  = state.src;
    token.map      = [0, 1];
    token.children = [];
    state.tokens.push(token);
  } else {
    state.md.block.parse(state.src, state.md, state.env, state.tokens);
  }
}function inline$1 (state) {
  var tokens = state.tokens;

  // Parse inlines
  for (var i = 0, l = tokens.length; i < l; i++) {
    var tok = tokens[i];
    if (tok.type === 'inline') {
      state.md.inline.parse(tok.content, state.md, state.env, tok.children);
    }
  }
}// Replace link-like texts with link nodes.
//
// Currently restricted by `md.validateLink()` to http/https/ftp
//


function isLinkOpen$3 (str) {
  return /^<a[>\s]/i.test(str)
}
function isLinkClose$3 (str) {
  return /^<\/a\s*>/i.test(str)
}

function linkify$4 (state) {
  var blockTokens = state.tokens;

  if (!state.md.options.linkify) { return }

  for (var j = 0, l = blockTokens.length; j < l; j++) {
    if (blockTokens[j].type !== 'inline' ||
        !state.md.linkify.pretest(blockTokens[j].content)) {
      continue
    }

    var tokens = blockTokens[j].children;

    var htmlLinkLevel = 0;

    // We scan from the end, to keep position when new tags added.
    // Use reversed logic in links start/end match
    for (var i = tokens.length - 1; i >= 0; i--) {
      var currentToken = tokens[i];

      // Skip content of markdown links
      if (currentToken.type === 'link_close') {
        i--;
        while (tokens[i].level !== currentToken.level && tokens[i].type !== 'link_open') {
          i--;
        }
        continue
      }

      // Skip content of html tag links
      if (currentToken.type === 'html_inline') {
        if (isLinkOpen$3(currentToken.content) && htmlLinkLevel > 0) {
          htmlLinkLevel--;
        }
        if (isLinkClose$3(currentToken.content)) {
          htmlLinkLevel++;
        }
      }
      if (htmlLinkLevel > 0) { continue }

      if (currentToken.type === 'text' && state.md.linkify.test(currentToken.content)) {
        var text = currentToken.content;
        var links = state.md.linkify.match(text);

        // Now split string to nodes
        var nodes = [];
        var level = currentToken.level;
        var lastPos = 0;

        // forbid escape sequence at the start of the string,
        // this avoids http\://example.com/ from being linkified as
        // http:<a href="//example.com/">//example.com/</a>
        if (links.length > 0 &&
            links[0].index === 0 &&
            i > 0 &&
            tokens[i - 1].type === 'text_special') {
          links = links.slice(1);
        }

        for (var ln = 0; ln < links.length; ln++) {
          var url = links[ln].url;
          var fullUrl = state.md.normalizeLink(url);
          if (!state.md.validateLink(fullUrl)) { continue }

          var urlText = links[ln].text;

          // Linkifier might send raw hostnames like "example.com", where url
          // starts with domain name. So we prepend http:// in those cases,
          // and remove it afterwards.
          //
          if (!links[ln].schema) {
            urlText = state.md.normalizeLinkText('http://' + urlText).replace(/^http:\/\//, '');
          } else if (links[ln].schema === 'mailto:' && !/^mailto:/i.test(urlText)) {
            urlText = state.md.normalizeLinkText('mailto:' + urlText).replace(/^mailto:/, '');
          } else {
            urlText = state.md.normalizeLinkText(urlText);
          }

          var pos = links[ln].index;

          if (pos > lastPos) {
            var token   = new state.Token('text', '', 0);
            token.content = text.slice(lastPos, pos);
            token.level   = level;
            nodes.push(token);
          }

          var token_o   = new state.Token('link_open', 'a', 1);
          token_o.attrs   = [['href', fullUrl]];
          token_o.level   = level++;
          token_o.markup  = 'linkify';
          token_o.info    = 'auto';
          nodes.push(token_o);

          var token_t   = new state.Token('text', '', 0);
          token_t.content = urlText;
          token_t.level   = level;
          nodes.push(token_t);

          var token_c   = new state.Token('link_close', 'a', -1);
          token_c.level   = --level;
          token_c.markup  = 'linkify';
          token_c.info    = 'auto';
          nodes.push(token_c);

          lastPos = links[ln].lastIndex;
        }
        if (lastPos < text.length) {
          var token$1   = new state.Token('text', '', 0);
          token$1.content = text.slice(lastPos);
          token$1.level   = level;
          nodes.push(token$1);
        }

        // replace current node
        blockTokens[j].children = tokens = arrayReplaceAt$1(tokens, i, nodes);
      }
    }
  }
}// Simple typographic replacements
//
// (c) (C) → ©
// (tm) (TM) → ™
// (r) (R) → ®
// +- → ±
// ... → … (also ?.... → ?.., !.... → !..)
// ???????? → ???, !!!!! → !!!, `,,` → `,`
// -- → &ndash;, --- → &mdash;
//

// TODO:
// - fractionals 1/2, 1/4, 3/4 -> ½, ¼, ¾
// - multiplications 2 x 4 -> 2 × 4

var RARE_RE$1 = /\+-|\.\.|\?\?\?\?|!!!!|,,|--/;

// Workaround for phantomjs - need regex without /g flag,
// or root check will fail every second time
var SCOPED_ABBR_TEST_RE$1 = /\((c|tm|r)\)/i;

var SCOPED_ABBR_RE$1 = /\((c|tm|r)\)/ig;
var SCOPED_ABBR$1 = {
  c: '©',
  r: '®',
  tm: '™'
};

function replaceFn$1 (match, name) {
  return SCOPED_ABBR$1[name.toLowerCase()]
}

function replace_scoped$1 (inlineTokens) {
  var inside_autolink = 0;

  for (var i = inlineTokens.length - 1; i >= 0; i--) {
    var token = inlineTokens[i];

    if (token.type === 'text' && !inside_autolink) {
      token.content = token.content.replace(SCOPED_ABBR_RE$1, replaceFn$1);
    }

    if (token.type === 'link_open' && token.info === 'auto') {
      inside_autolink--;
    }

    if (token.type === 'link_close' && token.info === 'auto') {
      inside_autolink++;
    }
  }
}

function replace_rare$1 (inlineTokens) {
  var inside_autolink = 0;

  for (var i = inlineTokens.length - 1; i >= 0; i--) {
    var token = inlineTokens[i];

    if (token.type === 'text' && !inside_autolink) {
      if (RARE_RE$1.test(token.content)) {
        token.content = token.content
          .replace(/\+-/g, '±')
          // .., ..., ....... -> …
          // but ?..... & !..... -> ?.. & !..
          .replace(/\.{2,}/g, '…').replace(/([?!])…/g, '$1..')
          .replace(/([?!]){4,}/g, '$1$1$1').replace(/,{2,}/g, ',')
          // em-dash
          .replace(/(^|[^-])---(?=[^-]|$)/mg, '$1\u2014')
          // en-dash
          .replace(/(^|\s)--(?=\s|$)/mg, '$1\u2013')
          .replace(/(^|[^-\s])--(?=[^-\s]|$)/mg, '$1\u2013');
      }
    }

    if (token.type === 'link_open' && token.info === 'auto') {
      inside_autolink--;
    }

    if (token.type === 'link_close' && token.info === 'auto') {
      inside_autolink++;
    }
  }
}

function replace$2 (state) {
  var blkIdx;

  if (!state.md.options.typographer) { return }

  for (blkIdx = state.tokens.length - 1; blkIdx >= 0; blkIdx--) {
    if (state.tokens[blkIdx].type !== 'inline') { continue }

    if (SCOPED_ABBR_TEST_RE$1.test(state.tokens[blkIdx].content)) {
      replace_scoped$1(state.tokens[blkIdx].children);
    }

    if (RARE_RE$1.test(state.tokens[blkIdx].content)) {
      replace_rare$1(state.tokens[blkIdx].children);
    }
  }
}// Convert straight quotation marks to typographic ones
//


var QUOTE_TEST_RE$1 = /['"]/;
var QUOTE_RE$1 = /['"]/g;
var APOSTROPHE$1 = '\u2019'; /* ’ */

function replaceAt$1 (str, index, ch) {
  return str.slice(0, index) + ch + str.slice(index + 1)
}

function process_inlines$1 (tokens, state) {
  var j;

  var stack = [];

  for (var i = 0; i < tokens.length; i++) {
    var token = tokens[i];

    var thisLevel = tokens[i].level;

    for (j = stack.length - 1; j >= 0; j--) {
      if (stack[j].level <= thisLevel) { break }
    }
    stack.length = j + 1;

    if (token.type !== 'text') { continue }

    var text = token.content;
    var pos = 0;
    var max = text.length;

    /* eslint no-labels:0,block-scoped-var:0 */
    OUTER:
    while (pos < max) {
      QUOTE_RE$1.lastIndex = pos;
      var t = QUOTE_RE$1.exec(text);
      if (!t) { break }

      var canOpen = true;
      var canClose = true;
      pos = t.index + 1;
      var isSingle = (t[0] === "'");

      // Find previous character,
      // default to space if it's the beginning of the line
      //
      var lastChar = 0x20;

      if (t.index - 1 >= 0) {
        lastChar = text.charCodeAt(t.index - 1);
      } else {
        for (j = i - 1; j >= 0; j--) {
          if (tokens[j].type === 'softbreak' || tokens[j].type === 'hardbreak') { break } // lastChar defaults to 0x20
          if (!tokens[j].content) { continue } // should skip all tokens except 'text', 'html_inline' or 'code_inline'

          lastChar = tokens[j].content.charCodeAt(tokens[j].content.length - 1);
          break
        }
      }

      // Find next character,
      // default to space if it's the end of the line
      //
      var nextChar = 0x20;

      if (pos < max) {
        nextChar = text.charCodeAt(pos);
      } else {
        for (j = i + 1; j < tokens.length; j++) {
          if (tokens[j].type === 'softbreak' || tokens[j].type === 'hardbreak') { break } // nextChar defaults to 0x20
          if (!tokens[j].content) { continue } // should skip all tokens except 'text', 'html_inline' or 'code_inline'

          nextChar = tokens[j].content.charCodeAt(0);
          break
        }
      }

      var isLastPunctChar = isMdAsciiPunct$2(lastChar) || isPunctChar$2(String.fromCharCode(lastChar));
      var isNextPunctChar = isMdAsciiPunct$2(nextChar) || isPunctChar$2(String.fromCharCode(nextChar));

      var isLastWhiteSpace = isWhiteSpace$2(lastChar);
      var isNextWhiteSpace = isWhiteSpace$2(nextChar);

      if (isNextWhiteSpace) {
        canOpen = false;
      } else if (isNextPunctChar) {
        if (!(isLastWhiteSpace || isLastPunctChar)) {
          canOpen = false;
        }
      }

      if (isLastWhiteSpace) {
        canClose = false;
      } else if (isLastPunctChar) {
        if (!(isNextWhiteSpace || isNextPunctChar)) {
          canClose = false;
        }
      }

      if (nextChar === 0x22 /* " */ && t[0] === '"') {
        if (lastChar >= 0x30 /* 0 */ && lastChar <= 0x39 /* 9 */) {
          // special case: 1"" - count first quote as an inch
          canClose = canOpen = false;
        }
      }

      if (canOpen && canClose) {
        // Replace quotes in the middle of punctuation sequence, but not
        // in the middle of the words, i.e.:
        //
        // 1. foo " bar " baz - not replaced
        // 2. foo-"-bar-"-baz - replaced
        // 3. foo"bar"baz     - not replaced
        //
        canOpen = isLastPunctChar;
        canClose = isNextPunctChar;
      }

      if (!canOpen && !canClose) {
        // middle of word
        if (isSingle) {
          token.content = replaceAt$1(token.content, t.index, APOSTROPHE$1);
        }
        continue
      }

      if (canClose) {
        // this could be a closing quote, rewind the stack to get a match
        for (j = stack.length - 1; j >= 0; j--) {
          var item = stack[j];
          if (stack[j].level < thisLevel) { break }
          if (item.single === isSingle && stack[j].level === thisLevel) {
            item = stack[j];

            var openQuote = (void 0);
            var closeQuote = (void 0);
            if (isSingle) {
              openQuote = state.md.options.quotes[2];
              closeQuote = state.md.options.quotes[3];
            } else {
              openQuote = state.md.options.quotes[0];
              closeQuote = state.md.options.quotes[1];
            }

            // replace token.content *before* tokens[item.token].content,
            // because, if they are pointing at the same token, replaceAt
            // could mess up indices when quote length != 1
            token.content = replaceAt$1(token.content, t.index, closeQuote);
            tokens[item.token].content = replaceAt$1(
              tokens[item.token].content, item.pos, openQuote);

            pos += closeQuote.length - 1;
            if (item.token === i) { pos += openQuote.length - 1; }

            text = token.content;
            max = text.length;

            stack.length = j;
            continue OUTER
          }
        }
      }

      if (canOpen) {
        stack.push({
          token: i,
          pos: t.index,
          single: isSingle,
          level: thisLevel
        });
      } else if (canClose && isSingle) {
        token.content = replaceAt$1(token.content, t.index, APOSTROPHE$1);
      }
    }
  }
}

function smartquotes$1 (state) {
  /* eslint max-depth:0 */
  if (!state.md.options.typographer) { return }

  for (var blkIdx = state.tokens.length - 1; blkIdx >= 0; blkIdx--) {
    if (state.tokens[blkIdx].type !== 'inline' ||
        !QUOTE_TEST_RE$1.test(state.tokens[blkIdx].content)) {
      continue
    }

    process_inlines$1(state.tokens[blkIdx].children, state);
  }
}// Join raw text tokens with the rest of the text
//
// This is set as a separate rule to provide an opportunity for plugins
// to run text replacements after text join, but before escape join.
//
// For example, `\:)` shouldn't be replaced with an emoji.
//

function text_join$1 (state) {
  var curr, last;
  var blockTokens = state.tokens;
  var l = blockTokens.length;

  for (var j = 0; j < l; j++) {
    if (blockTokens[j].type !== 'inline') { continue }

    var tokens = blockTokens[j].children;
    var max = tokens.length;

    for (curr = 0; curr < max; curr++) {
      if (tokens[curr].type === 'text_special') {
        tokens[curr].type = 'text';
      }
    }

    for (curr = last = 0; curr < max; curr++) {
      if (tokens[curr].type === 'text' &&
          curr + 1 < max &&
          tokens[curr + 1].type === 'text') {
        // collapse two adjacent text nodes
        tokens[curr + 1].content = tokens[curr].content + tokens[curr + 1].content;
      } else {
        if (curr !== last) { tokens[last] = tokens[curr]; }

        last++;
      }
    }

    if (curr !== last) {
      tokens.length = last;
    }
  }
}/** internal
 * class Core
 *
 * Top-level rules executor. Glues block/inline parsers and does intermediate
 * transformations.
 **/


var _rules$5 = [
  ['normalize',      normalize$1],
  ['block',          block$2],
  ['inline',         inline$1],
  ['linkify',        linkify$4],
  ['replacements',   replace$2],
  ['smartquotes',    smartquotes$1],
  // `text_join` finds `text_special` tokens (for escape sequences)
  // and joins them with the rest of the text
  ['text_join',      text_join$1]
];

/**
 * new Core()
 **/
function Core$1 () {
  /**
   * Core#ruler -> Ruler
   *
   * [[Ruler]] instance. Keep configuration of core rules.
   **/
  this.ruler = new Ruler$4();

  for (var i = 0; i < _rules$5.length; i++) {
    this.ruler.push(_rules$5[i][0], _rules$5[i][1]);
  }
}

/**
 * Core.process(state)
 *
 * Executes core chain rules.
 **/
Core$1.prototype.process = function (state) {
  var rules = this.ruler.getRules('');

  for (var i = 0, l = rules.length; i < l; i++) {
    rules[i](state);
  }
};

Core$1.prototype.State = StateCore$1;// Parser state class


function StateBlock$1 (src, md, env, tokens) {
  this.src = src;

  // link to parser instance
  this.md     = md;

  this.env = env;

  //
  // Internal state vartiables
  //

  this.tokens = tokens;

  this.bMarks = [];  // line begin offsets for fast jumps
  this.eMarks = [];  // line end offsets for fast jumps
  this.tShift = [];  // offsets of the first non-space characters (tabs not expanded)
  this.sCount = [];  // indents for each line (tabs expanded)

  // An amount of virtual spaces (tabs expanded) between beginning
  // of each line (bMarks) and real beginning of that line.
  //
  // It exists only as a hack because blockquotes override bMarks
  // losing information in the process.
  //
  // It's used only when expanding tabs, you can think about it as
  // an initial tab length, e.g. bsCount=21 applied to string `\t123`
  // means first tab should be expanded to 4-21%4 === 3 spaces.
  //
  this.bsCount = [];

  // block parser variables

  // required block content indent (for example, if we are
  // inside a list, it would be positioned after list marker)
  this.blkIndent  = 0;
  this.line       = 0; // line index in src
  this.lineMax    = 0; // lines count
  this.tight      = false;  // loose/tight mode for lists
  this.ddIndent   = -1; // indent of the current dd block (-1 if there isn't any)
  this.listIndent = -1; // indent of the current list block (-1 if there isn't any)

  // can be 'blockquote', 'list', 'root', 'paragraph' or 'reference'
  // used in lists to determine if they interrupt a paragraph
  this.parentType = 'root';

  this.level = 0;

  // Create caches
  // Generate markers.
  var s = this.src;

  for (var start = 0, pos = 0, indent = 0, offset = 0, len = s.length, indent_found = false; pos < len; pos++) {
    var ch = s.charCodeAt(pos);

    if (!indent_found) {
      if (isSpace$e(ch)) {
        indent++;

        if (ch === 0x09) {
          offset += 4 - offset % 4;
        } else {
          offset++;
        }
        continue
      } else {
        indent_found = true;
      }
    }

    if (ch === 0x0A || pos === len - 1) {
      if (ch !== 0x0A) { pos++; }
      this.bMarks.push(start);
      this.eMarks.push(pos);
      this.tShift.push(indent);
      this.sCount.push(offset);
      this.bsCount.push(0);

      indent_found = false;
      indent = 0;
      offset = 0;
      start = pos + 1;
    }
  }

  // Push fake entry to simplify cache bounds checks
  this.bMarks.push(s.length);
  this.eMarks.push(s.length);
  this.tShift.push(0);
  this.sCount.push(0);
  this.bsCount.push(0);

  this.lineMax = this.bMarks.length - 1; // don't count last fake line
}

// Push new token to "stream".
//
StateBlock$1.prototype.push = function (type, tag, nesting) {
  var token = new Token$4(type, tag, nesting);
  token.block = true;

  if (nesting < 0) { this.level--; } // closing tag
  token.level = this.level;
  if (nesting > 0) { this.level++; } // opening tag

  this.tokens.push(token);
  return token
};

StateBlock$1.prototype.isEmpty = function isEmpty (line) {
  return this.bMarks[line] + this.tShift[line] >= this.eMarks[line]
};

StateBlock$1.prototype.skipEmptyLines = function skipEmptyLines (from) {
  for (var max = this.lineMax; from < max; from++) {
    if (this.bMarks[from] + this.tShift[from] < this.eMarks[from]) {
      break
    }
  }
  return from
};

// Skip spaces from given position.
StateBlock$1.prototype.skipSpaces = function skipSpaces (pos) {
  for (var max = this.src.length; pos < max; pos++) {
    var ch = this.src.charCodeAt(pos);
    if (!isSpace$e(ch)) { break }
  }
  return pos
};

// Skip spaces from given position in reverse.
StateBlock$1.prototype.skipSpacesBack = function skipSpacesBack (pos, min) {
  if (pos <= min) { return pos }

  while (pos > min) {
    if (!isSpace$e(this.src.charCodeAt(--pos))) { return pos + 1 }
  }
  return pos
};

// Skip char codes from given position
StateBlock$1.prototype.skipChars = function skipChars (pos, code) {
  for (var max = this.src.length; pos < max; pos++) {
    if (this.src.charCodeAt(pos) !== code) { break }
  }
  return pos
};

// Skip char codes reverse from given position - 1
StateBlock$1.prototype.skipCharsBack = function skipCharsBack (pos, code, min) {
  if (pos <= min) { return pos }

  while (pos > min) {
    if (code !== this.src.charCodeAt(--pos)) { return pos + 1 }
  }
  return pos
};

// cut lines range from source.
StateBlock$1.prototype.getLines = function getLines (begin, end, indent, keepLastLF) {
  if (begin >= end) {
    return ''
  }

  var queue = new Array(end - begin);

  for (var i = 0, line = begin; line < end; line++, i++) {
    var lineIndent = 0;
    var lineStart = this.bMarks[line];
    var first = lineStart;
    var last = (void 0);

    if (line + 1 < end || keepLastLF) {
      // No need for bounds check because we have fake entry on tail.
      last = this.eMarks[line] + 1;
    } else {
      last = this.eMarks[line];
    }

    while (first < last && lineIndent < indent) {
      var ch = this.src.charCodeAt(first);

      if (isSpace$e(ch)) {
        if (ch === 0x09) {
          lineIndent += 4 - (lineIndent + this.bsCount[line]) % 4;
        } else {
          lineIndent++;
        }
      } else if (first - lineStart < this.tShift[line]) {
        // patched tShift masked characters to look like spaces (blockquotes, list markers)
        lineIndent++;
      } else {
        break
      }

      first++;
    }

    if (lineIndent > indent) {
      // partially expanding tabs in code blocks, e.g '\t\tfoobar'
      // with indent=2 becomes '  \tfoobar'
      queue[i] = new Array(lineIndent - indent + 1).join(' ') + this.src.slice(first, last);
    } else {
      queue[i] = this.src.slice(first, last);
    }
  }

  return queue.join('')
};

// re-export Token class to use in block rules
StateBlock$1.prototype.Token = Token$4;// GFM table, https://github.github.com/gfm/#tables-extension-


// Limit the amount of empty autocompleted cells in a table,
// see https://github.com/markdown-it/markdown-it/issues/1000,
//
// Both pulldown-cmark and commonmark-hs limit the number of cells this way to ~200k.
// We set it to 65k, which can expand user input by a factor of x370
// (256x256 square is 1.8kB expanded into 650kB).
var MAX_AUTOCOMPLETED_CELLS = 0x10000;

function getLine$2 (state, line) {
  var pos = state.bMarks[line] + state.tShift[line];
  var max = state.eMarks[line];

  return state.src.slice(pos, max)
}

function escapedSplit$2 (str) {
  var result = [];
  var max = str.length;

  var pos = 0;
  var ch = str.charCodeAt(pos);
  var isEscaped = false;
  var lastPos = 0;
  var current = '';

  while (pos < max) {
    if (ch === 0x7c/* | */) {
      if (!isEscaped) {
        // pipe separating cells, '|'
        result.push(current + str.substring(lastPos, pos));
        current = '';
        lastPos = pos + 1;
      } else {
        // escaped pipe, '\|'
        current += str.substring(lastPos, pos - 1);
        lastPos = pos;
      }
    }

    isEscaped = (ch === 0x5c/* \ */);
    pos++;

    ch = str.charCodeAt(pos);
  }

  result.push(current + str.substring(lastPos));

  return result
}

function table$2 (state, startLine, endLine, silent) {
  // should have at least two lines
  if (startLine + 2 > endLine) { return false }

  var nextLine = startLine + 1;

  if (state.sCount[nextLine] < state.blkIndent) { return false }

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[nextLine] - state.blkIndent >= 4) { return false }

  // first character of the second line should be '|', '-', ':',
  // and no other characters are allowed but spaces;
  // basically, this is the equivalent of /^[-:|][-:|\s]*$/ regexp

  var pos = state.bMarks[nextLine] + state.tShift[nextLine];
  if (pos >= state.eMarks[nextLine]) { return false }

  var firstCh = state.src.charCodeAt(pos++);
  if (firstCh !== 0x7C/* | */ && firstCh !== 0x2D/* - */ && firstCh !== 0x3A/* : */) { return false }

  if (pos >= state.eMarks[nextLine]) { return false }

  var secondCh = state.src.charCodeAt(pos++);
  if (secondCh !== 0x7C/* | */ && secondCh !== 0x2D/* - */ && secondCh !== 0x3A/* : */ && !isSpace$e(secondCh)) {
    return false
  }

  // if first character is '-', then second character must not be a space
  // (due to parsing ambiguity with list)
  if (firstCh === 0x2D/* - */ && isSpace$e(secondCh)) { return false }

  while (pos < state.eMarks[nextLine]) {
    var ch = state.src.charCodeAt(pos);

    if (ch !== 0x7C/* | */ && ch !== 0x2D/* - */ && ch !== 0x3A/* : */ && !isSpace$e(ch)) { return false }

    pos++;
  }

  var lineText = getLine$2(state, startLine + 1);
  var columns = lineText.split('|');
  var aligns = [];
  for (var i = 0; i < columns.length; i++) {
    var t = columns[i].trim();
    if (!t) {
      // allow empty columns before and after table, but not in between columns;
      // e.g. allow ` |---| `, disallow ` ---||--- `
      if (i === 0 || i === columns.length - 1) {
        continue
      } else {
        return false
      }
    }

    if (!/^:?-+:?$/.test(t)) { return false }
    if (t.charCodeAt(t.length - 1) === 0x3A/* : */) {
      aligns.push(t.charCodeAt(0) === 0x3A/* : */ ? 'center' : 'right');
    } else if (t.charCodeAt(0) === 0x3A/* : */) {
      aligns.push('left');
    } else {
      aligns.push('');
    }
  }

  lineText = getLine$2(state, startLine).trim();
  if (lineText.indexOf('|') === -1) { return false }
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }
  columns = escapedSplit$2(lineText);
  if (columns.length && columns[0] === '') { columns.shift(); }
  if (columns.length && columns[columns.length - 1] === '') { columns.pop(); }

  // header row will define an amount of columns in the entire table,
  // and align row should be exactly the same (the rest of the rows can differ)
  var columnCount = columns.length;
  if (columnCount === 0 || columnCount !== aligns.length) { return false }

  if (silent) { return true }

  var oldParentType = state.parentType;
  state.parentType = 'table';

  // use 'blockquote' lists for termination because it's
  // the most similar to tables
  var terminatorRules = state.md.block.ruler.getRules('blockquote');

  var token_to = state.push('table_open', 'table', 1);
  var tableLines = [startLine, 0];
  token_to.map = tableLines;

  var token_tho = state.push('thead_open', 'thead', 1);
  token_tho.map = [startLine, startLine + 1];

  var token_htro = state.push('tr_open', 'tr', 1);
  token_htro.map = [startLine, startLine + 1];

  for (var i$1 = 0; i$1 < columns.length; i$1++) {
    var token_ho = state.push('th_open', 'th', 1);
    if (aligns[i$1]) {
      token_ho.attrs  = [['style', 'text-align:' + aligns[i$1]]];
    }

    var token_il = state.push('inline', '', 0);
    token_il.content  = columns[i$1].trim();
    token_il.children = [];

    state.push('th_close', 'th', -1);
  }

  state.push('tr_close', 'tr', -1);
  state.push('thead_close', 'thead', -1);

  var tbodyLines;
  var autocompletedCells = 0;

  for (nextLine = startLine + 2; nextLine < endLine; nextLine++) {
    if (state.sCount[nextLine] < state.blkIndent) { break }

    var terminate = false;
    for (var i$2 = 0, l = terminatorRules.length; i$2 < l; i$2++) {
      if (terminatorRules[i$2](state, nextLine, endLine, true)) {
        terminate = true;
        break
      }
    }

    if (terminate) { break }
    lineText = getLine$2(state, nextLine).trim();
    if (!lineText) { break }
    if (state.sCount[nextLine] - state.blkIndent >= 4) { break }
    columns = escapedSplit$2(lineText);
    if (columns.length && columns[0] === '') { columns.shift(); }
    if (columns.length && columns[columns.length - 1] === '') { columns.pop(); }

    // note: autocomplete count can be negative if user specifies more columns than header,
    // but that does not affect intended use (which is limiting expansion)
    autocompletedCells += columnCount - columns.length;
    if (autocompletedCells > MAX_AUTOCOMPLETED_CELLS) { break }

    if (nextLine === startLine + 2) {
      var token_tbo = state.push('tbody_open', 'tbody', 1);
      token_tbo.map = tbodyLines = [startLine + 2, 0];
    }

    var token_tro = state.push('tr_open', 'tr', 1);
    token_tro.map = [nextLine, nextLine + 1];

    for (var i$3 = 0; i$3 < columnCount; i$3++) {
      var token_tdo = state.push('td_open', 'td', 1);
      if (aligns[i$3]) {
        token_tdo.attrs  = [['style', 'text-align:' + aligns[i$3]]];
      }

      var token_il$1 = state.push('inline', '', 0);
      token_il$1.content  = columns[i$3] ? columns[i$3].trim() : '';
      token_il$1.children = [];

      state.push('td_close', 'td', -1);
    }
    state.push('tr_close', 'tr', -1);
  }

  if (tbodyLines) {
    state.push('tbody_close', 'tbody', -1);
    tbodyLines[1] = nextLine;
  }

  state.push('table_close', 'table', -1);
  tableLines[1] = nextLine;

  state.parentType = oldParentType;
  state.line = nextLine;
  return true
}// Code block (4 spaces padded)

function code$2 (state, startLine, endLine/*, silent */) {
  if (state.sCount[startLine] - state.blkIndent < 4) { return false }

  var nextLine = startLine + 1;
  var last = nextLine;

  while (nextLine < endLine) {
    if (state.isEmpty(nextLine)) {
      nextLine++;
      continue
    }

    if (state.sCount[nextLine] - state.blkIndent >= 4) {
      nextLine++;
      last = nextLine;
      continue
    }
    break
  }

  state.line = last;

  var token   = state.push('code_block', 'code', 0);
  token.content = state.getLines(startLine, last, 4 + state.blkIndent, false) + '\n';
  token.map     = [startLine, state.line];

  return true
}// fences (``` lang, ~~~ lang)

function fence$2 (state, startLine, endLine, silent) {
  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  if (pos + 3 > max) { return false }

  var marker = state.src.charCodeAt(pos);

  if (marker !== 0x7E/* ~ */ && marker !== 0x60 /* ` */) {
    return false
  }

  // scan marker length
  var mem = pos;
  pos = state.skipChars(pos, marker);

  var len = pos - mem;

  if (len < 3) { return false }

  var markup = state.src.slice(mem, pos);
  var params = state.src.slice(pos, max);

  if (marker === 0x60 /* ` */) {
    if (params.indexOf(String.fromCharCode(marker)) >= 0) {
      return false
    }
  }

  // Since start is found, we can report success here in validation mode
  if (silent) { return true }

  // search end of block
  var nextLine = startLine;
  var haveEndMarker = false;

  for (;;) {
    nextLine++;
    if (nextLine >= endLine) {
      // unclosed block should be autoclosed by end of document.
      // also block seems to be autoclosed by end of parent
      break
    }

    pos = mem = state.bMarks[nextLine] + state.tShift[nextLine];
    max = state.eMarks[nextLine];

    if (pos < max && state.sCount[nextLine] < state.blkIndent) {
      // non-empty line with negative indent should stop the list:
      // - ```
      //  test
      break
    }

    if (state.src.charCodeAt(pos) !== marker) { continue }

    if (state.sCount[nextLine] - state.blkIndent >= 4) {
      // closing fence should be indented less than 4 spaces
      continue
    }

    pos = state.skipChars(pos, marker);

    // closing code fence must be at least as long as the opening one
    if (pos - mem < len) { continue }

    // make sure tail has spaces only
    pos = state.skipSpaces(pos);

    if (pos < max) { continue }

    haveEndMarker = true;
    // found!
    break
  }

  // If a fence has heading spaces, they should be removed from its inner block
  len = state.sCount[startLine];

  state.line = nextLine + (haveEndMarker ? 1 : 0);

  var token   = state.push('fence', 'code', 0);
  token.info    = params;
  token.content = state.getLines(startLine + 1, nextLine, len, true);
  token.markup  = markup;
  token.map     = [startLine, state.line];

  return true
}// Block quotes


function blockquote$2 (state, startLine, endLine, silent) {
  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];

  var oldLineMax = state.lineMax;

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  // check the block quote marker
  if (state.src.charCodeAt(pos) !== 0x3E/* > */) { return false }

  // we know that it's going to be a valid blockquote,
  // so no point trying to find the end of it in silent mode
  if (silent) { return true }

  var oldBMarks  = [];
  var oldBSCount = [];
  var oldSCount  = [];
  var oldTShift  = [];

  var terminatorRules = state.md.block.ruler.getRules('blockquote');

  var oldParentType = state.parentType;
  state.parentType = 'blockquote';
  var lastLineEmpty = false;
  var nextLine;

  // Search the end of the block
  //
  // Block ends with either:
  //  1. an empty line outside:
  //     ```
  //     > test
  //
  //     ```
  //  2. an empty line inside:
  //     ```
  //     >
  //     test
  //     ```
  //  3. another tag:
  //     ```
  //     > test
  //      - - -
  //     ```
  for (nextLine = startLine; nextLine < endLine; nextLine++) {
    // check if it's outdented, i.e. it's inside list item and indented
    // less than said list item:
    //
    // ```
    // 1. anything
    //    > current blockquote
    // 2. checking this line
    // ```
    var isOutdented = state.sCount[nextLine] < state.blkIndent;

    pos = state.bMarks[nextLine] + state.tShift[nextLine];
    max = state.eMarks[nextLine];

    if (pos >= max) {
      // Case 1: line is not inside the blockquote, and this line is empty.
      break
    }

    if (state.src.charCodeAt(pos++) === 0x3E/* > */ && !isOutdented) {
      // This line is inside the blockquote.

      // set offset past spaces and ">"
      var initial = state.sCount[nextLine] + 1;
      var spaceAfterMarker = (void 0);
      var adjustTab = (void 0);

      // skip one optional space after '>'
      if (state.src.charCodeAt(pos) === 0x20 /* space */) {
        // ' >   test '
        //     ^ -- position start of line here:
        pos++;
        initial++;
        adjustTab = false;
        spaceAfterMarker = true;
      } else if (state.src.charCodeAt(pos) === 0x09 /* tab */) {
        spaceAfterMarker = true;

        if ((state.bsCount[nextLine] + initial) % 4 === 3) {
          // '  >\t  test '
          //       ^ -- position start of line here (tab has width===1)
          pos++;
          initial++;
          adjustTab = false;
        } else {
          // ' >\t  test '
          //    ^ -- position start of line here + shift bsCount slightly
          //         to make extra space appear
          adjustTab = true;
        }
      } else {
        spaceAfterMarker = false;
      }

      var offset = initial;
      oldBMarks.push(state.bMarks[nextLine]);
      state.bMarks[nextLine] = pos;

      while (pos < max) {
        var ch = state.src.charCodeAt(pos);

        if (isSpace$e(ch)) {
          if (ch === 0x09) {
            offset += 4 - (offset + state.bsCount[nextLine] + (adjustTab ? 1 : 0)) % 4;
          } else {
            offset++;
          }
        } else {
          break
        }

        pos++;
      }

      lastLineEmpty = pos >= max;

      oldBSCount.push(state.bsCount[nextLine]);
      state.bsCount[nextLine] = state.sCount[nextLine] + 1 + (spaceAfterMarker ? 1 : 0);

      oldSCount.push(state.sCount[nextLine]);
      state.sCount[nextLine] = offset - initial;

      oldTShift.push(state.tShift[nextLine]);
      state.tShift[nextLine] = pos - state.bMarks[nextLine];
      continue
    }

    // Case 2: line is not inside the blockquote, and the last line was empty.
    if (lastLineEmpty) { break }

    // Case 3: another tag found.
    var terminate = false;
    for (var i = 0, l = terminatorRules.length; i < l; i++) {
      if (terminatorRules[i](state, nextLine, endLine, true)) {
        terminate = true;
        break
      }
    }

    if (terminate) {
      // Quirk to enforce "hard termination mode" for paragraphs;
      // normally if you call `tokenize(state, startLine, nextLine)`,
      // paragraphs will look below nextLine for paragraph continuation,
      // but if blockquote is terminated by another tag, they shouldn't
      state.lineMax = nextLine;

      if (state.blkIndent !== 0) {
        // state.blkIndent was non-zero, we now set it to zero,
        // so we need to re-calculate all offsets to appear as
        // if indent wasn't changed
        oldBMarks.push(state.bMarks[nextLine]);
        oldBSCount.push(state.bsCount[nextLine]);
        oldTShift.push(state.tShift[nextLine]);
        oldSCount.push(state.sCount[nextLine]);
        state.sCount[nextLine] -= state.blkIndent;
      }

      break
    }

    oldBMarks.push(state.bMarks[nextLine]);
    oldBSCount.push(state.bsCount[nextLine]);
    oldTShift.push(state.tShift[nextLine]);
    oldSCount.push(state.sCount[nextLine]);

    // A negative indentation means that this is a paragraph continuation
    //
    state.sCount[nextLine] = -1;
  }

  var oldIndent = state.blkIndent;
  state.blkIndent = 0;

  var token_o  = state.push('blockquote_open', 'blockquote', 1);
  token_o.markup = '>';
  var lines = [startLine, 0];
  token_o.map    = lines;

  state.md.block.tokenize(state, startLine, nextLine);

  var token_c  = state.push('blockquote_close', 'blockquote', -1);
  token_c.markup = '>';

  state.lineMax = oldLineMax;
  state.parentType = oldParentType;
  lines[1] = state.line;

  // Restore original tShift; this might not be necessary since the parser
  // has already been here, but just to make sure we can do that.
  for (var i$1 = 0; i$1 < oldTShift.length; i$1++) {
    state.bMarks[i$1 + startLine] = oldBMarks[i$1];
    state.tShift[i$1 + startLine] = oldTShift[i$1];
    state.sCount[i$1 + startLine] = oldSCount[i$1];
    state.bsCount[i$1 + startLine] = oldBSCount[i$1];
  }
  state.blkIndent = oldIndent;

  return true
}// Horizontal rule


function hr$1 (state, startLine, endLine, silent) {
  var max = state.eMarks[startLine];
  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var marker = state.src.charCodeAt(pos++);

  // Check hr marker
  if (marker !== 0x2A/* * */ &&
      marker !== 0x2D/* - */ &&
      marker !== 0x5F/* _ */) {
    return false
  }

  // markers can be mixed with spaces, but there should be at least 3 of them

  var cnt = 1;
  while (pos < max) {
    var ch = state.src.charCodeAt(pos++);
    if (ch !== marker && !isSpace$e(ch)) { return false }
    if (ch === marker) { cnt++; }
  }

  if (cnt < 3) { return false }

  if (silent) { return true }

  state.line = startLine + 1;

  var token  = state.push('hr', 'hr', 0);
  token.map    = [startLine, state.line];
  token.markup = Array(cnt + 1).join(String.fromCharCode(marker));

  return true
}// Lists


// Search `[-+*][\n ]`, returns next pos after marker on success
// or -1 on fail.
function skipBulletListMarker$1 (state, startLine) {
  var max = state.eMarks[startLine];
  var pos = state.bMarks[startLine] + state.tShift[startLine];

  var marker = state.src.charCodeAt(pos++);
  // Check bullet
  if (marker !== 0x2A/* * */ &&
      marker !== 0x2D/* - */ &&
      marker !== 0x2B/* + */) {
    return -1
  }

  if (pos < max) {
    var ch = state.src.charCodeAt(pos);

    if (!isSpace$e(ch)) {
      // " -test " - is not a list item
      return -1
    }
  }

  return pos
}

// Search `\d+[.)][\n ]`, returns next pos after marker on success
// or -1 on fail.
function skipOrderedListMarker$1 (state, startLine) {
  var start = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];
  var pos = start;

  // List marker should have at least 2 chars (digit + dot)
  if (pos + 1 >= max) { return -1 }

  var ch = state.src.charCodeAt(pos++);

  if (ch < 0x30/* 0 */ || ch > 0x39/* 9 */) { return -1 }

  for (;;) {
    // EOL -> fail
    if (pos >= max) { return -1 }

    ch = state.src.charCodeAt(pos++);

    if (ch >= 0x30/* 0 */ && ch <= 0x39/* 9 */) {
      // List marker should have no more than 9 digits
      // (prevents integer overflow in browsers)
      if (pos - start >= 10) { return -1 }

      continue
    }

    // found valid marker
    if (ch === 0x29/* ) */ || ch === 0x2e/* . */) {
      break
    }

    return -1
  }

  if (pos < max) {
    ch = state.src.charCodeAt(pos);

    if (!isSpace$e(ch)) {
      // " 1.test " - is not a list item
      return -1
    }
  }
  return pos
}

function markTightParagraphs$1 (state, idx) {
  var level = state.level + 2;

  for (var i = idx + 2, l = state.tokens.length - 2; i < l; i++) {
    if (state.tokens[i].level === level && state.tokens[i].type === 'paragraph_open') {
      state.tokens[i + 2].hidden = true;
      state.tokens[i].hidden = true;
      i += 2;
    }
  }
}

function list$1 (state, startLine, endLine, silent) {
  var max, pos, start, token;
  var nextLine = startLine;
  var tight = true;

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[nextLine] - state.blkIndent >= 4) { return false }

  // Special case:
  //  - item 1
  //   - item 2
  //    - item 3
  //     - item 4
  //      - this one is a paragraph continuation
  if (state.listIndent >= 0 &&
      state.sCount[nextLine] - state.listIndent >= 4 &&
      state.sCount[nextLine] < state.blkIndent) {
    return false
  }

  var isTerminatingParagraph = false;

  // limit conditions when list can interrupt
  // a paragraph (validation mode only)
  if (silent && state.parentType === 'paragraph') {
    // Next list item should still terminate previous list item;
    //
    // This code can fail if plugins use blkIndent as well as lists,
    // but I hope the spec gets fixed long before that happens.
    //
    if (state.sCount[nextLine] >= state.blkIndent) {
      isTerminatingParagraph = true;
    }
  }

  // Detect list type and position after marker
  var isOrdered;
  var markerValue;
  var posAfterMarker;
  if ((posAfterMarker = skipOrderedListMarker$1(state, nextLine)) >= 0) {
    isOrdered = true;
    start = state.bMarks[nextLine] + state.tShift[nextLine];
    markerValue = Number(state.src.slice(start, posAfterMarker - 1));

    // If we're starting a new ordered list right after
    // a paragraph, it should start with 1.
    if (isTerminatingParagraph && markerValue !== 1) { return false }
  } else if ((posAfterMarker = skipBulletListMarker$1(state, nextLine)) >= 0) {
    isOrdered = false;
  } else {
    return false
  }

  // If we're starting a new unordered list right after
  // a paragraph, first line should not be empty.
  if (isTerminatingParagraph) {
    if (state.skipSpaces(posAfterMarker) >= state.eMarks[nextLine]) { return false }
  }

  // For validation mode we can terminate immediately
  if (silent) { return true }

  // We should terminate list on style change. Remember first one to compare.
  var markerCharCode = state.src.charCodeAt(posAfterMarker - 1);

  // Start list
  var listTokIdx = state.tokens.length;

  if (isOrdered) {
    token       = state.push('ordered_list_open', 'ol', 1);
    if (markerValue !== 1) {
      token.attrs = [['start', markerValue]];
    }
  } else {
    token       = state.push('bullet_list_open', 'ul', 1);
  }

  var listLines = [nextLine, 0];
  token.map    = listLines;
  token.markup = String.fromCharCode(markerCharCode);

  //
  // Iterate list items
  //

  var prevEmptyEnd = false;
  var terminatorRules = state.md.block.ruler.getRules('list');

  var oldParentType = state.parentType;
  state.parentType = 'list';

  while (nextLine < endLine) {
    pos = posAfterMarker;
    max = state.eMarks[nextLine];

    var initial = state.sCount[nextLine] + posAfterMarker - (state.bMarks[nextLine] + state.tShift[nextLine]);
    var offset = initial;

    while (pos < max) {
      var ch = state.src.charCodeAt(pos);

      if (ch === 0x09) {
        offset += 4 - (offset + state.bsCount[nextLine]) % 4;
      } else if (ch === 0x20) {
        offset++;
      } else {
        break
      }

      pos++;
    }

    var contentStart = pos;
    var indentAfterMarker = (void 0);

    if (contentStart >= max) {
      // trimming space in "-    \n  3" case, indent is 1 here
      indentAfterMarker = 1;
    } else {
      indentAfterMarker = offset - initial;
    }

    // If we have more than 4 spaces, the indent is 1
    // (the rest is just indented code block)
    if (indentAfterMarker > 4) { indentAfterMarker = 1; }

    // "  -  test"
    //  ^^^^^ - calculating total length of this thing
    var indent = initial + indentAfterMarker;

    // Run subparser & write tokens
    token        = state.push('list_item_open', 'li', 1);
    token.markup = String.fromCharCode(markerCharCode);
    var itemLines = [nextLine, 0];
    token.map    = itemLines;
    if (isOrdered) {
      token.info = state.src.slice(start, posAfterMarker - 1);
    }

    // change current state, then restore it after parser subcall
    var oldTight = state.tight;
    var oldTShift = state.tShift[nextLine];
    var oldSCount = state.sCount[nextLine];

    //  - example list
    // ^ listIndent position will be here
    //   ^ blkIndent position will be here
    //
    var oldListIndent = state.listIndent;
    state.listIndent = state.blkIndent;
    state.blkIndent = indent;

    state.tight = true;
    state.tShift[nextLine] = contentStart - state.bMarks[nextLine];
    state.sCount[nextLine] = offset;

    if (contentStart >= max && state.isEmpty(nextLine + 1)) {
      // workaround for this case
      // (list item is empty, list terminates before "foo"):
      // ~~~~~~~~
      //   -
      //
      //     foo
      // ~~~~~~~~
      state.line = Math.min(state.line + 2, endLine);
    } else {
      state.md.block.tokenize(state, nextLine, endLine, true);
    }

    // If any of list item is tight, mark list as tight
    if (!state.tight || prevEmptyEnd) {
      tight = false;
    }
    // Item become loose if finish with empty line,
    // but we should filter last element, because it means list finish
    prevEmptyEnd = (state.line - nextLine) > 1 && state.isEmpty(state.line - 1);

    state.blkIndent = state.listIndent;
    state.listIndent = oldListIndent;
    state.tShift[nextLine] = oldTShift;
    state.sCount[nextLine] = oldSCount;
    state.tight = oldTight;

    token        = state.push('list_item_close', 'li', -1);
    token.markup = String.fromCharCode(markerCharCode);

    nextLine = state.line;
    itemLines[1] = nextLine;

    if (nextLine >= endLine) { break }

    //
    // Try to check if list is terminated or continued.
    //
    if (state.sCount[nextLine] < state.blkIndent) { break }

    // if it's indented more than 3 spaces, it should be a code block
    if (state.sCount[nextLine] - state.blkIndent >= 4) { break }

    // fail if terminating block found
    var terminate = false;
    for (var i = 0, l = terminatorRules.length; i < l; i++) {
      if (terminatorRules[i](state, nextLine, endLine, true)) {
        terminate = true;
        break
      }
    }
    if (terminate) { break }

    // fail if list has another type
    if (isOrdered) {
      posAfterMarker = skipOrderedListMarker$1(state, nextLine);
      if (posAfterMarker < 0) { break }
      start = state.bMarks[nextLine] + state.tShift[nextLine];
    } else {
      posAfterMarker = skipBulletListMarker$1(state, nextLine);
      if (posAfterMarker < 0) { break }
    }

    if (markerCharCode !== state.src.charCodeAt(posAfterMarker - 1)) { break }
  }

  // Finalize list
  if (isOrdered) {
    token = state.push('ordered_list_close', 'ol', -1);
  } else {
    token = state.push('bullet_list_close', 'ul', -1);
  }
  token.markup = String.fromCharCode(markerCharCode);

  listLines[1] = nextLine;
  state.line = nextLine;

  state.parentType = oldParentType;

  // mark paragraphs tight if needed
  if (tight) {
    markTightParagraphs$1(state, listTokIdx);
  }

  return true
}function reference$1 (state, startLine, _endLine, silent) {
  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];
  var nextLine = startLine + 1;

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  if (state.src.charCodeAt(pos) !== 0x5B/* [ */) { return false }

  function getNextLine (nextLine) {
    var endLine = state.lineMax;

    if (nextLine >= endLine || state.isEmpty(nextLine)) {
      // empty line or end of input
      return null
    }

    var isContinuation = false;

    // this would be a code block normally, but after paragraph
    // it's considered a lazy continuation regardless of what's there
    if (state.sCount[nextLine] - state.blkIndent > 3) { isContinuation = true; }

    // quirk for blockquotes, this line should already be checked by that rule
    if (state.sCount[nextLine] < 0) { isContinuation = true; }

    if (!isContinuation) {
      var terminatorRules = state.md.block.ruler.getRules('reference');
      var oldParentType = state.parentType;
      state.parentType = 'reference';

      // Some tags can terminate paragraph without empty line.
      var terminate = false;
      for (var i = 0, l = terminatorRules.length; i < l; i++) {
        if (terminatorRules[i](state, nextLine, endLine, true)) {
          terminate = true;
          break
        }
      }

      state.parentType = oldParentType;
      if (terminate) {
        // terminated by another block
        return null
      }
    }

    var pos = state.bMarks[nextLine] + state.tShift[nextLine];
    var max = state.eMarks[nextLine];

    // max + 1 explicitly includes the newline
    return state.src.slice(pos, max + 1)
  }

  var str = state.src.slice(pos, max + 1);

  max = str.length;
  var labelEnd = -1;

  for (pos = 1; pos < max; pos++) {
    var ch = str.charCodeAt(pos);
    if (ch === 0x5B /* [ */) {
      return false
    } else if (ch === 0x5D /* ] */) {
      labelEnd = pos;
      break
    } else if (ch === 0x0A /* \n */) {
      var lineContent = getNextLine(nextLine);
      if (lineContent !== null) {
        str += lineContent;
        max = str.length;
        nextLine++;
      }
    } else if (ch === 0x5C /* \ */) {
      pos++;
      if (pos < max && str.charCodeAt(pos) === 0x0A) {
        var lineContent$1 = getNextLine(nextLine);
        if (lineContent$1 !== null) {
          str += lineContent$1;
          max = str.length;
          nextLine++;
        }
      }
    }
  }

  if (labelEnd < 0 || str.charCodeAt(labelEnd + 1) !== 0x3A/* : */) { return false }

  // [label]:   destination   'title'
  //         ^^^ skip optional whitespace here
  for (pos = labelEnd + 2; pos < max; pos++) {
    var ch$1 = str.charCodeAt(pos);
    if (ch$1 === 0x0A) {
      var lineContent$2 = getNextLine(nextLine);
      if (lineContent$2 !== null) {
        str += lineContent$2;
        max = str.length;
        nextLine++;
      }
    } else if (isSpace$e(ch$1)) ; else {
      break
    }
  }

  // [label]:   destination   'title'
  //            ^^^^^^^^^^^ parse this
  var destRes = state.md.helpers.parseLinkDestination(str, pos, max);
  if (!destRes.ok) { return false }

  var href = state.md.normalizeLink(destRes.str);
  if (!state.md.validateLink(href)) { return false }

  pos = destRes.pos;

  // save cursor state, we could require to rollback later
  var destEndPos = pos;
  var destEndLineNo = nextLine;

  // [label]:   destination   'title'
  //                       ^^^ skipping those spaces
  var start = pos;
  for (; pos < max; pos++) {
    var ch$2 = str.charCodeAt(pos);
    if (ch$2 === 0x0A) {
      var lineContent$3 = getNextLine(nextLine);
      if (lineContent$3 !== null) {
        str += lineContent$3;
        max = str.length;
        nextLine++;
      }
    } else if (isSpace$e(ch$2)) ; else {
      break
    }
  }

  // [label]:   destination   'title'
  //                          ^^^^^^^ parse this
  var titleRes = state.md.helpers.parseLinkTitle(str, pos, max);
  while (titleRes.can_continue) {
    var lineContent$4 = getNextLine(nextLine);
    if (lineContent$4 === null) { break }
    str += lineContent$4;
    pos = max;
    max = str.length;
    nextLine++;
    titleRes = state.md.helpers.parseLinkTitle(str, pos, max, titleRes);
  }
  var title;

  if (pos < max && start !== pos && titleRes.ok) {
    title = titleRes.str;
    pos = titleRes.pos;
  } else {
    title = '';
    pos = destEndPos;
    nextLine = destEndLineNo;
  }

  // skip trailing spaces until the rest of the line
  while (pos < max) {
    var ch$3 = str.charCodeAt(pos);
    if (!isSpace$e(ch$3)) { break }
    pos++;
  }

  if (pos < max && str.charCodeAt(pos) !== 0x0A) {
    if (title) {
      // garbage at the end of the line after title,
      // but it could still be a valid reference if we roll back
      title = '';
      pos = destEndPos;
      nextLine = destEndLineNo;
      while (pos < max) {
        var ch$4 = str.charCodeAt(pos);
        if (!isSpace$e(ch$4)) { break }
        pos++;
      }
    }
  }

  if (pos < max && str.charCodeAt(pos) !== 0x0A) {
    // garbage at the end of the line
    return false
  }

  var label = normalizeReference$3(str.slice(1, labelEnd));
  if (!label) {
    // CommonMark 0.20 disallows empty labels
    return false
  }

  // Reference can not terminate anything. This check is for safety only.
  /* istanbul ignore if */
  if (silent) { return true }

  if (typeof state.env.references === 'undefined') {
    state.env.references = {};
  }
  if (typeof state.env.references[label] === 'undefined') {
    state.env.references[label] = { title: title, href: href };
  }

  state.line = nextLine;
  return true
}// List of valid html blocks names, according to commonmark spec
// https://spec.commonmark.org/0.30/#html-blocks

var block_names$1 = [
  'address',
  'article',
  'aside',
  'base',
  'basefont',
  'blockquote',
  'body',
  'caption',
  'center',
  'col',
  'colgroup',
  'dd',
  'details',
  'dialog',
  'dir',
  'div',
  'dl',
  'dt',
  'fieldset',
  'figcaption',
  'figure',
  'footer',
  'form',
  'frame',
  'frameset',
  'h1',
  'h2',
  'h3',
  'h4',
  'h5',
  'h6',
  'head',
  'header',
  'hr',
  'html',
  'iframe',
  'legend',
  'li',
  'link',
  'main',
  'menu',
  'menuitem',
  'nav',
  'noframes',
  'ol',
  'optgroup',
  'option',
  'p',
  'param',
  'search',
  'section',
  'summary',
  'table',
  'tbody',
  'td',
  'tfoot',
  'th',
  'thead',
  'title',
  'tr',
  'track',
  'ul'
];// Regexps to match html elements

var attr_name$1     = '[a-zA-Z_:][a-zA-Z0-9:._-]*';

var unquoted$1      = '[^"\'=<>`\\x00-\\x20]+';
var single_quoted$1 = "'[^']*'";
var double_quoted$1 = '"[^"]*"';

var attr_value$1  = '(?:' + unquoted$1 + '|' + single_quoted$1 + '|' + double_quoted$1 + ')';

var attribute$1   = '(?:\\s+' + attr_name$1 + '(?:\\s*=\\s*' + attr_value$1 + ')?)';

var open_tag$1    = '<[A-Za-z][A-Za-z0-9\\-]*' + attribute$1 + '*\\s*\\/?>';

var close_tag$1   = '<\\/[A-Za-z][A-Za-z0-9\\-]*\\s*>';
var comment$2     = '<!---?>|<!--(?:[^-]|-[^-]|--[^>])*-->';
var processing$1  = '<[?][\\s\\S]*?[?]>';
var declaration$1 = '<![A-Za-z][^>]*>';
var cdata$1       = '<!\\[CDATA\\[[\\s\\S]*?\\]\\]>';

var HTML_TAG_RE$2 = new RegExp('^(?:' + open_tag$1 + '|' + close_tag$1 + '|' + comment$2 +
                        '|' + processing$1 + '|' + declaration$1 + '|' + cdata$1 + ')');
var HTML_OPEN_CLOSE_TAG_RE$2 = new RegExp('^(?:' + open_tag$1 + '|' + close_tag$1 + ')');// HTML block


// An array of opening and corresponding closing sequences for html tags,
// last argument defines whether it can terminate a paragraph or not
//
var HTML_SEQUENCES$1 = [
  [/^<(script|pre|style|textarea)(?=(\s|>|$))/i, /<\/(script|pre|style|textarea)>/i, true],
  [/^<!--/,        /-->/,   true],
  [/^<\?/,         /\?>/,   true],
  [/^<![A-Z]/,     />/,     true],
  [/^<!\[CDATA\[/, /\]\]>/, true],
  [new RegExp('^</?(' + block_names$1.join('|') + ')(?=(\\s|/?>|$))', 'i'), /^$/, true],
  [new RegExp(HTML_OPEN_CLOSE_TAG_RE$2.source + '\\s*$'),  /^$/, false]
];

function html_block$1 (state, startLine, endLine, silent) {
  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  if (!state.md.options.html) { return false }

  if (state.src.charCodeAt(pos) !== 0x3C/* < */) { return false }

  var lineText = state.src.slice(pos, max);

  var i = 0;
  for (; i < HTML_SEQUENCES$1.length; i++) {
    if (HTML_SEQUENCES$1[i][0].test(lineText)) { break }
  }
  if (i === HTML_SEQUENCES$1.length) { return false }

  if (silent) {
    // true if this sequence can be a terminator, false otherwise
    return HTML_SEQUENCES$1[i][2]
  }

  var nextLine = startLine + 1;

  // If we are here - we detected HTML block.
  // Let's roll down till block end.
  if (!HTML_SEQUENCES$1[i][1].test(lineText)) {
    for (; nextLine < endLine; nextLine++) {
      if (state.sCount[nextLine] < state.blkIndent) { break }

      pos = state.bMarks[nextLine] + state.tShift[nextLine];
      max = state.eMarks[nextLine];
      lineText = state.src.slice(pos, max);

      if (HTML_SEQUENCES$1[i][1].test(lineText)) {
        if (lineText.length !== 0) { nextLine++; }
        break
      }
    }
  }

  state.line = nextLine;

  var token   = state.push('html_block', '', 0);
  token.map     = [startLine, nextLine];
  token.content = state.getLines(startLine, nextLine, state.blkIndent, true);

  return true
}// heading (#, ##, ...)


function heading$3 (state, startLine, endLine, silent) {
  var pos = state.bMarks[startLine] + state.tShift[startLine];
  var max = state.eMarks[startLine];

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  var ch  = state.src.charCodeAt(pos);

  if (ch !== 0x23/* # */ || pos >= max) { return false }

  // count heading level
  var level = 1;
  ch = state.src.charCodeAt(++pos);
  while (ch === 0x23/* # */ && pos < max && level <= 6) {
    level++;
    ch = state.src.charCodeAt(++pos);
  }

  if (level > 6 || (pos < max && !isSpace$e(ch))) { return false }

  if (silent) { return true }

  // Let's cut tails like '    ###  ' from the end of string

  max = state.skipSpacesBack(max, pos);
  var tmp = state.skipCharsBack(max, 0x23, pos); // #
  if (tmp > pos && isSpace$e(state.src.charCodeAt(tmp - 1))) {
    max = tmp;
  }

  state.line = startLine + 1;

  var token_o  = state.push('heading_open', 'h' + String(level), 1);
  token_o.markup = '########'.slice(0, level);
  token_o.map    = [startLine, state.line];

  var token_i    = state.push('inline', '', 0);
  token_i.content  = state.src.slice(pos, max).trim();
  token_i.map      = [startLine, state.line];
  token_i.children = [];

  var token_c  = state.push('heading_close', 'h' + String(level), -1);
  token_c.markup = '########'.slice(0, level);

  return true
}// lheading (---, ===)

function lheading$1 (state, startLine, endLine/*, silent */) {
  var terminatorRules = state.md.block.ruler.getRules('paragraph');

  // if it's indented more than 3 spaces, it should be a code block
  if (state.sCount[startLine] - state.blkIndent >= 4) { return false }

  var oldParentType = state.parentType;
  state.parentType = 'paragraph'; // use paragraph to match terminatorRules

  // jump line-by-line until empty one or EOF
  var level = 0;
  var marker;
  var nextLine = startLine + 1;

  for (; nextLine < endLine && !state.isEmpty(nextLine); nextLine++) {
    // this would be a code block normally, but after paragraph
    // it's considered a lazy continuation regardless of what's there
    if (state.sCount[nextLine] - state.blkIndent > 3) { continue }

    //
    // Check for underline in setext header
    //
    if (state.sCount[nextLine] >= state.blkIndent) {
      var pos = state.bMarks[nextLine] + state.tShift[nextLine];
      var max = state.eMarks[nextLine];

      if (pos < max) {
        marker = state.src.charCodeAt(pos);

        if (marker === 0x2D/* - */ || marker === 0x3D/* = */) {
          pos = state.skipChars(pos, marker);
          pos = state.skipSpaces(pos);

          if (pos >= max) {
            level = (marker === 0x3D/* = */ ? 1 : 2);
            break
          }
        }
      }
    }

    // quirk for blockquotes, this line should already be checked by that rule
    if (state.sCount[nextLine] < 0) { continue }

    // Some tags can terminate paragraph without empty line.
    var terminate = false;
    for (var i = 0, l = terminatorRules.length; i < l; i++) {
      if (terminatorRules[i](state, nextLine, endLine, true)) {
        terminate = true;
        break
      }
    }
    if (terminate) { break }
  }

  if (!level) {
    // Didn't find valid underline
    return false
  }

  var content = state.getLines(startLine, nextLine, state.blkIndent, false).trim();

  state.line = nextLine + 1;

  var token_o    = state.push('heading_open', 'h' + String(level), 1);
  token_o.markup   = String.fromCharCode(marker);
  token_o.map      = [startLine, state.line];

  var token_i    = state.push('inline', '', 0);
  token_i.content  = content;
  token_i.map      = [startLine, state.line - 1];
  token_i.children = [];

  var token_c    = state.push('heading_close', 'h' + String(level), -1);
  token_c.markup   = String.fromCharCode(marker);

  state.parentType = oldParentType;

  return true
}// Paragraph

function paragraph$2 (state, startLine, endLine) {
  var terminatorRules = state.md.block.ruler.getRules('paragraph');
  var oldParentType = state.parentType;
  var nextLine = startLine + 1;
  state.parentType = 'paragraph';

  // jump line-by-line until empty one or EOF
  for (; nextLine < endLine && !state.isEmpty(nextLine); nextLine++) {
    // this would be a code block normally, but after paragraph
    // it's considered a lazy continuation regardless of what's there
    if (state.sCount[nextLine] - state.blkIndent > 3) { continue }

    // quirk for blockquotes, this line should already be checked by that rule
    if (state.sCount[nextLine] < 0) { continue }

    // Some tags can terminate paragraph without empty line.
    var terminate = false;
    for (var i = 0, l = terminatorRules.length; i < l; i++) {
      if (terminatorRules[i](state, nextLine, endLine, true)) {
        terminate = true;
        break
      }
    }
    if (terminate) { break }
  }

  var content = state.getLines(startLine, nextLine, state.blkIndent, false).trim();

  state.line = nextLine;

  var token_o    = state.push('paragraph_open', 'p', 1);
  token_o.map      = [startLine, state.line];

  var token_i    = state.push('inline', '', 0);
  token_i.content  = content;
  token_i.map      = [startLine, state.line];
  token_i.children = [];

  state.push('paragraph_close', 'p', -1);

  state.parentType = oldParentType;

  return true
}/** internal
 * class ParserBlock
 *
 * Block-level tokenizer.
 **/


var _rules$4 = [
  // First 2 params - rule name & source. Secondary array - list of rules,
  // which can be terminated by this one.
  ['table',      table$2,      ['paragraph', 'reference']],
  ['code',       code$2],
  ['fence',      fence$2,      ['paragraph', 'reference', 'blockquote', 'list']],
  ['blockquote', blockquote$2, ['paragraph', 'reference', 'blockquote', 'list']],
  ['hr',         hr$1,         ['paragraph', 'reference', 'blockquote', 'list']],
  ['list',       list$1,       ['paragraph', 'reference', 'blockquote']],
  ['reference',  reference$1],
  ['html_block', html_block$1, ['paragraph', 'reference', 'blockquote']],
  ['heading',    heading$3,    ['paragraph', 'reference', 'blockquote']],
  ['lheading',   lheading$1],
  ['paragraph',  paragraph$2]
];

/**
 * new ParserBlock()
 **/
function ParserBlock$2 () {
  /**
   * ParserBlock#ruler -> Ruler
   *
   * [[Ruler]] instance. Keep configuration of block rules.
   **/
  this.ruler = new Ruler$4();

  for (var i = 0; i < _rules$4.length; i++) {
    this.ruler.push(_rules$4[i][0], _rules$4[i][1], { alt: (_rules$4[i][2] || []).slice() });
  }
}

// Generate tokens for input range
//
ParserBlock$2.prototype.tokenize = function (state, startLine, endLine) {
  var rules = this.ruler.getRules('');
  var len = rules.length;
  var maxNesting = state.md.options.maxNesting;
  var line = startLine;
  var hasEmptyLines = false;

  while (line < endLine) {
    state.line = line = state.skipEmptyLines(line);
    if (line >= endLine) { break }

    // Termination condition for nested calls.
    // Nested calls currently used for blockquotes & lists
    if (state.sCount[line] < state.blkIndent) { break }

    // If nesting level exceeded - skip tail to the end. That's not ordinary
    // situation and we should not care about content.
    if (state.level >= maxNesting) {
      state.line = endLine;
      break
    }

    // Try all possible rules.
    // On success, rule should:
    //
    // - update `state.line`
    // - update `state.tokens`
    // - return true
    var prevLine = state.line;
    var ok = false;

    for (var i = 0; i < len; i++) {
      ok = rules[i](state, line, endLine, false);
      if (ok) {
        if (prevLine >= state.line) {
          throw new Error("block rule didn't increment state.line")
        }
        break
      }
    }

    // this can only happen if user disables paragraph rule
    if (!ok) { throw new Error('none of the block rules matched') }

    // set state.tight if we had an empty line before current tag
    // i.e. latest empty line should not count
    state.tight = !hasEmptyLines;

    // paragraph might "eat" one newline after it in nested lists
    if (state.isEmpty(state.line - 1)) {
      hasEmptyLines = true;
    }

    line = state.line;

    if (line < endLine && state.isEmpty(line)) {
      hasEmptyLines = true;
      line++;
      state.line = line;
    }
  }
};

/**
 * ParserBlock.parse(str, md, env, outTokens)
 *
 * Process input string and push block tokens into `outTokens`
 **/
ParserBlock$2.prototype.parse = function (src, md, env, outTokens) {
  if (!src) { return }

  var state = new this.State(src, md, env, outTokens);

  this.tokenize(state, state.line, state.lineMax);
};

ParserBlock$2.prototype.State = StateBlock$1;// Inline parser state


function StateInline$1 (src, md, env, outTokens) {
  this.src = src;
  this.env = env;
  this.md = md;
  this.tokens = outTokens;
  this.tokens_meta = Array(outTokens.length);

  this.pos = 0;
  this.posMax = this.src.length;
  this.level = 0;
  this.pending = '';
  this.pendingLevel = 0;

  // Stores { start: end } pairs. Useful for backtrack
  // optimization of pairs parse (emphasis, strikes).
  this.cache = {};

  // List of emphasis-like delimiters for current tag
  this.delimiters = [];

  // Stack of delimiter lists for upper level tags
  this._prev_delimiters = [];

  // backtick length => last seen position
  this.backticks = {};
  this.backticksScanned = false;

  // Counter used to disable inline linkify-it execution
  // inside <a> and markdown links
  this.linkLevel = 0;
}

// Flush pending text
//
StateInline$1.prototype.pushPending = function () {
  var token = new Token$4('text', '', 0);
  token.content = this.pending;
  token.level = this.pendingLevel;
  this.tokens.push(token);
  this.pending = '';
  return token
};

// Push new token to "stream".
// If pending text exists - flush it as text token
//
StateInline$1.prototype.push = function (type, tag, nesting) {
  if (this.pending) {
    this.pushPending();
  }

  var token = new Token$4(type, tag, nesting);
  var token_meta = null;

  if (nesting < 0) {
    // closing tag
    this.level--;
    this.delimiters = this._prev_delimiters.pop();
  }

  token.level = this.level;

  if (nesting > 0) {
    // opening tag
    this.level++;
    this._prev_delimiters.push(this.delimiters);
    this.delimiters = [];
    token_meta = { delimiters: this.delimiters };
  }

  this.pendingLevel = this.level;
  this.tokens.push(token);
  this.tokens_meta.push(token_meta);
  return token
};

// Scan a sequence of emphasis-like markers, and determine whether
// it can start an emphasis sequence or end an emphasis sequence.
//
//  - start - position to scan from (it should point at a valid marker);
//  - canSplitWord - determine if these markers can be found inside a word
//
StateInline$1.prototype.scanDelims = function (start, canSplitWord) {
  var max = this.posMax;
  var marker = this.src.charCodeAt(start);

  // treat beginning of the line as a whitespace
  var lastChar = start > 0 ? this.src.charCodeAt(start - 1) : 0x20;

  var pos = start;
  while (pos < max && this.src.charCodeAt(pos) === marker) { pos++; }

  var count = pos - start;

  // treat end of the line as a whitespace
  var nextChar = pos < max ? this.src.charCodeAt(pos) : 0x20;

  var isLastPunctChar = isMdAsciiPunct$2(lastChar) || isPunctChar$2(String.fromCharCode(lastChar));
  var isNextPunctChar = isMdAsciiPunct$2(nextChar) || isPunctChar$2(String.fromCharCode(nextChar));

  var isLastWhiteSpace = isWhiteSpace$2(lastChar);
  var isNextWhiteSpace = isWhiteSpace$2(nextChar);

  var left_flanking =
    !isNextWhiteSpace && (!isNextPunctChar || isLastWhiteSpace || isLastPunctChar);
  var right_flanking =
    !isLastWhiteSpace && (!isLastPunctChar || isNextWhiteSpace || isNextPunctChar);

  var can_open  = left_flanking  && (canSplitWord || !right_flanking || isLastPunctChar);
  var can_close = right_flanking && (canSplitWord || !left_flanking  || isNextPunctChar);

  return { can_open: can_open, can_close: can_close, length: count }
};

// re-export Token class to use in block rules
StateInline$1.prototype.Token = Token$4;// Skip text characters for text token, place those to pending buffer
// and increment current pos

// Rule to skip pure text
// '{}$%@~+=:' reserved for extentions

// !, ", #, $, %, &, ', (, ), *, +, ,, -, ., /, :, ;, <, =, >, ?, @, [, \, ], ^, _, `, {, |, }, or ~

// !!!! Don't confuse with "Markdown ASCII Punctuation" chars
// http://spec.commonmark.org/0.15/#ascii-punctuation-character
function isTerminatorChar$1 (ch) {
  switch (ch) {
    case 0x0A/* \n */:
    case 0x21/* ! */:
    case 0x23/* # */:
    case 0x24/* $ */:
    case 0x25/* % */:
    case 0x26/* & */:
    case 0x2A/* * */:
    case 0x2B/* + */:
    case 0x2D/* - */:
    case 0x3A/* : */:
    case 0x3C/* < */:
    case 0x3D/* = */:
    case 0x3E/* > */:
    case 0x40/* @ */:
    case 0x5B/* [ */:
    case 0x5C/* \ */:
    case 0x5D/* ] */:
    case 0x5E/* ^ */:
    case 0x5F/* _ */:
    case 0x60/* ` */:
    case 0x7B/* { */:
    case 0x7D/* } */:
    case 0x7E/* ~ */:
      return true
    default:
      return false
  }
}

function text$2 (state, silent) {
  var pos = state.pos;

  while (pos < state.posMax && !isTerminatorChar$1(state.src.charCodeAt(pos))) {
    pos++;
  }

  if (pos === state.pos) { return false }

  if (!silent) { state.pending += state.src.slice(state.pos, pos); }

  state.pos = pos;

  return true
}

// Alternative implementation, for memory.
//
// It costs 10% of performance, but allows extend terminators list, if place it
// to `ParserInline` property. Probably, will switch to it sometime, such
// flexibility required.

/*
var TERMINATOR_RE = /[\n!#$%&*+\-:<=>@[\\\]^_`{}~]/;

module.exports = function text(state, silent) {
  var pos = state.pos,
      idx = state.src.slice(pos).search(TERMINATOR_RE);

  // first char is terminator -> empty text
  if (idx === 0) { return false; }

  // no terminator -> text till end of string
  if (idx < 0) {
    if (!silent) { state.pending += state.src.slice(pos); }
    state.pos = state.src.length;
    return true;
  }

  if (!silent) { state.pending += state.src.slice(pos, pos + idx); }

  state.pos += idx;

  return true;
}; */// Process links like https://example.org/

// RFC3986: scheme = ALPHA *( ALPHA / DIGIT / "+" / "-" / "." )
var SCHEME_RE$1 = /(?:^|[^a-z0-9.+-])([a-z][a-z0-9.+-]*)$/i;

function linkify$3 (state, silent) {
  if (!state.md.options.linkify) { return false }
  if (state.linkLevel > 0) { return false }

  var pos = state.pos;
  var max = state.posMax;

  if (pos + 3 > max) { return false }
  if (state.src.charCodeAt(pos) !== 0x3A/* : */) { return false }
  if (state.src.charCodeAt(pos + 1) !== 0x2F/* / */) { return false }
  if (state.src.charCodeAt(pos + 2) !== 0x2F/* / */) { return false }

  var match = state.pending.match(SCHEME_RE$1);
  if (!match) { return false }

  var proto = match[1];

  var link = state.md.linkify.matchAtStart(state.src.slice(pos - proto.length));
  if (!link) { return false }

  var url = link.url;

  // invalid link, but still detected by linkify somehow;
  // need to check to prevent infinite loop below
  if (url.length <= proto.length) { return false }

  // disallow '*' at the end of the link (conflicts with emphasis)
  url = url.replace(/\*+$/, '');

  var fullUrl = state.md.normalizeLink(url);
  if (!state.md.validateLink(fullUrl)) { return false }

  if (!silent) {
    state.pending = state.pending.slice(0, -proto.length);

    var token_o = state.push('link_open', 'a', 1);
    token_o.attrs = [['href', fullUrl]];
    token_o.markup = 'linkify';
    token_o.info = 'auto';

    var token_t = state.push('text', '', 0);
    token_t.content = state.md.normalizeLinkText(url);

    var token_c = state.push('link_close', 'a', -1);
    token_c.markup = 'linkify';
    token_c.info = 'auto';
  }

  state.pos += url.length - proto.length;
  return true
}// Proceess '\n'


function newline$2 (state, silent) {
  var pos = state.pos;

  if (state.src.charCodeAt(pos) !== 0x0A/* \n */) { return false }

  var pmax = state.pending.length - 1;
  var max = state.posMax;

  // '  \n' -> hardbreak
  // Lookup in pending chars is bad practice! Don't copy to other rules!
  // Pending string is stored in concat mode, indexed lookups will cause
  // convertion to flat mode.
  if (!silent) {
    if (pmax >= 0 && state.pending.charCodeAt(pmax) === 0x20) {
      if (pmax >= 1 && state.pending.charCodeAt(pmax - 1) === 0x20) {
        // Find whitespaces tail of pending chars.
        var ws = pmax - 1;
        while (ws >= 1 && state.pending.charCodeAt(ws - 1) === 0x20) { ws--; }

        state.pending = state.pending.slice(0, ws);
        state.push('hardbreak', 'br', 0);
      } else {
        state.pending = state.pending.slice(0, -1);
        state.push('softbreak', 'br', 0);
      }
    } else {
      state.push('softbreak', 'br', 0);
    }
  }

  pos++;

  // skip heading spaces for next line
  while (pos < max && isSpace$e(state.src.charCodeAt(pos))) { pos++; }

  state.pos = pos;
  return true
}// Process escaped chars and hardbreaks


var ESCAPED$1 = [];

for (var i$3 = 0; i$3 < 256; i$3++) { ESCAPED$1.push(0); }

'\\!"#$%&\'()*+,./:;<=>?@[]^_`{|}~-'
  .split('').forEach(function (ch) { ESCAPED$1[ch.charCodeAt(0)] = 1; });

function escape (state, silent) {
  var pos = state.pos;
  var max = state.posMax;

  if (state.src.charCodeAt(pos) !== 0x5C/* \ */) { return false }
  pos++;

  // '\' at the end of the inline block
  if (pos >= max) { return false }

  var ch1 = state.src.charCodeAt(pos);

  if (ch1 === 0x0A) {
    if (!silent) {
      state.push('hardbreak', 'br', 0);
    }

    pos++;
    // skip leading whitespaces from next line
    while (pos < max) {
      ch1 = state.src.charCodeAt(pos);
      if (!isSpace$e(ch1)) { break }
      pos++;
    }

    state.pos = pos;
    return true
  }

  var escapedStr = state.src[pos];

  if (ch1 >= 0xD800 && ch1 <= 0xDBFF && pos + 1 < max) {
    var ch2 = state.src.charCodeAt(pos + 1);

    if (ch2 >= 0xDC00 && ch2 <= 0xDFFF) {
      escapedStr += state.src[pos + 1];
      pos++;
    }
  }

  var origStr = '\\' + escapedStr;

  if (!silent) {
    var token = state.push('text_special', '', 0);

    if (ch1 < 256 && ESCAPED$1[ch1] !== 0) {
      token.content = escapedStr;
    } else {
      token.content = origStr;
    }

    token.markup = origStr;
    token.info   = 'escape';
  }

  state.pos = pos + 1;
  return true
}// Parse backticks

function backtick (state, silent) {
  var pos = state.pos;
  var ch = state.src.charCodeAt(pos);

  if (ch !== 0x60/* ` */) { return false }

  var start = pos;
  pos++;
  var max = state.posMax;

  // scan marker length
  while (pos < max && state.src.charCodeAt(pos) === 0x60/* ` */) { pos++; }

  var marker = state.src.slice(start, pos);
  var openerLength = marker.length;

  if (state.backticksScanned && (state.backticks[openerLength] || 0) <= start) {
    if (!silent) { state.pending += marker; }
    state.pos += openerLength;
    return true
  }

  var matchEnd = pos;
  var matchStart;

  // Nothing found in the cache, scan until the end of the line (or until marker is found)
  while ((matchStart = state.src.indexOf('`', matchEnd)) !== -1) {
    matchEnd = matchStart + 1;

    // scan marker length
    while (matchEnd < max && state.src.charCodeAt(matchEnd) === 0x60/* ` */) { matchEnd++; }

    var closerLength = matchEnd - matchStart;

    if (closerLength === openerLength) {
      // Found matching closer length.
      if (!silent) {
        var token = state.push('code_inline', 'code', 0);
        token.markup = marker;
        token.content = state.src.slice(pos, matchStart)
          .replace(/\n/g, ' ')
          .replace(/^ (.+) $/, '$1');
      }
      state.pos = matchEnd;
      return true
    }

    // Some different length found, put it in cache as upper limit of where closer can be found
    state.backticks[closerLength] = matchStart;
  }

  // Scanned through the end, didn't find anything
  state.backticksScanned = true;

  if (!silent) { state.pending += marker; }
  state.pos += openerLength;
  return true
}// ~~strike through~~
//

// Insert each marker as a separate text token, and add it to delimiter list
//
function strikethrough_tokenize (state, silent) {
  var start = state.pos;
  var marker = state.src.charCodeAt(start);

  if (silent) { return false }

  if (marker !== 0x7E/* ~ */) { return false }

  var scanned = state.scanDelims(state.pos, true);
  var len = scanned.length;
  var ch = String.fromCharCode(marker);

  if (len < 2) { return false }

  var token;

  if (len % 2) {
    token         = state.push('text', '', 0);
    token.content = ch;
    len--;
  }

  for (var i = 0; i < len; i += 2) {
    token         = state.push('text', '', 0);
    token.content = ch + ch;

    state.delimiters.push({
      marker: marker,
      length: 0,     // disable "rule of 3" length checks meant for emphasis
      token: state.tokens.length - 1,
      end: -1,
      open: scanned.can_open,
      close: scanned.can_close
    });
  }

  state.pos += scanned.length;

  return true
}

function postProcess$3 (state, delimiters) {
  var token;
  var loneMarkers = [];
  var max = delimiters.length;

  for (var i = 0; i < max; i++) {
    var startDelim = delimiters[i];

    if (startDelim.marker !== 0x7E/* ~ */) {
      continue
    }

    if (startDelim.end === -1) {
      continue
    }

    var endDelim = delimiters[startDelim.end];

    token         = state.tokens[startDelim.token];
    token.type    = 's_open';
    token.tag     = 's';
    token.nesting = 1;
    token.markup  = '~~';
    token.content = '';

    token         = state.tokens[endDelim.token];
    token.type    = 's_close';
    token.tag     = 's';
    token.nesting = -1;
    token.markup  = '~~';
    token.content = '';

    if (state.tokens[endDelim.token - 1].type === 'text' &&
        state.tokens[endDelim.token - 1].content === '~') {
      loneMarkers.push(endDelim.token - 1);
    }
  }

  // If a marker sequence has an odd number of characters, it's splitted
  // like this: `~~~~~` -> `~` + `~~` + `~~`, leaving one marker at the
  // start of the sequence.
  //
  // So, we have to move all those markers after subsequent s_close tags.
  //
  while (loneMarkers.length) {
    var i$1 = loneMarkers.pop();
    var j = i$1 + 1;

    while (j < state.tokens.length && state.tokens[j].type === 's_close') {
      j++;
    }

    j--;

    if (i$1 !== j) {
      token = state.tokens[j];
      state.tokens[j] = state.tokens[i$1];
      state.tokens[i$1] = token;
    }
  }
}

// Walk through delimiter list and replace text tokens with tags
//
function strikethrough_postProcess (state) {
  var tokens_meta = state.tokens_meta;
  var max = state.tokens_meta.length;

  postProcess$3(state, state.delimiters);

  for (var curr = 0; curr < max; curr++) {
    if (tokens_meta[curr] && tokens_meta[curr].delimiters) {
      postProcess$3(state, tokens_meta[curr].delimiters);
    }
  }
}

var r_strikethrough = {
  tokenize: strikethrough_tokenize,
  postProcess: strikethrough_postProcess
};// Process *this* and _that_
//

// Insert each marker as a separate text token, and add it to delimiter list
//
function emphasis_tokenize (state, silent) {
  var start = state.pos;
  var marker = state.src.charCodeAt(start);

  if (silent) { return false }

  if (marker !== 0x5F /* _ */ && marker !== 0x2A /* * */) { return false }

  var scanned = state.scanDelims(state.pos, marker === 0x2A);

  for (var i = 0; i < scanned.length; i++) {
    var token = state.push('text', '', 0);
    token.content = String.fromCharCode(marker);

    state.delimiters.push({
      // Char code of the starting marker (number).
      //
      marker: marker,

      // Total length of these series of delimiters.
      //
      length: scanned.length,

      // A position of the token this delimiter corresponds to.
      //
      token: state.tokens.length - 1,

      // If this delimiter is matched as a valid opener, `end` will be
      // equal to its position, otherwise it's `-1`.
      //
      end: -1,

      // Boolean flags that determine if this delimiter could open or close
      // an emphasis.
      //
      open: scanned.can_open,
      close: scanned.can_close
    });
  }

  state.pos += scanned.length;

  return true
}

function postProcess$2 (state, delimiters) {
  var max = delimiters.length;

  for (var i = max - 1; i >= 0; i--) {
    var startDelim = delimiters[i];

    if (startDelim.marker !== 0x5F/* _ */ && startDelim.marker !== 0x2A/* * */) {
      continue
    }

    // Process only opening markers
    if (startDelim.end === -1) {
      continue
    }

    var endDelim = delimiters[startDelim.end];

    // If the previous delimiter has the same marker and is adjacent to this one,
    // merge those into one strong delimiter.
    //
    // `<em><em>whatever</em></em>` -> `<strong>whatever</strong>`
    //
    var isStrong = i > 0 &&
               delimiters[i - 1].end === startDelim.end + 1 &&
               // check that first two markers match and adjacent
               delimiters[i - 1].marker === startDelim.marker &&
               delimiters[i - 1].token === startDelim.token - 1 &&
               // check that last two markers are adjacent (we can safely assume they match)
               delimiters[startDelim.end + 1].token === endDelim.token + 1;

    var ch = String.fromCharCode(startDelim.marker);

    var token_o   = state.tokens[startDelim.token];
    token_o.type    = isStrong ? 'strong_open' : 'em_open';
    token_o.tag     = isStrong ? 'strong' : 'em';
    token_o.nesting = 1;
    token_o.markup  = isStrong ? ch + ch : ch;
    token_o.content = '';

    var token_c   = state.tokens[endDelim.token];
    token_c.type    = isStrong ? 'strong_close' : 'em_close';
    token_c.tag     = isStrong ? 'strong' : 'em';
    token_c.nesting = -1;
    token_c.markup  = isStrong ? ch + ch : ch;
    token_c.content = '';

    if (isStrong) {
      state.tokens[delimiters[i - 1].token].content = '';
      state.tokens[delimiters[startDelim.end + 1].token].content = '';
      i--;
    }
  }
}

// Walk through delimiter list and replace text tokens with tags
//
function emphasis_post_process (state) {
  var tokens_meta = state.tokens_meta;
  var max = state.tokens_meta.length;

  postProcess$2(state, state.delimiters);

  for (var curr = 0; curr < max; curr++) {
    if (tokens_meta[curr] && tokens_meta[curr].delimiters) {
      postProcess$2(state, tokens_meta[curr].delimiters);
    }
  }
}

var r_emphasis = {
  tokenize: emphasis_tokenize,
  postProcess: emphasis_post_process
};// Process [link](<to> "stuff")


function link$5 (state, silent) {
  var code, label, res, ref;
  var href = '';
  var title = '';
  var start = state.pos;
  var parseReference = true;

  if (state.src.charCodeAt(state.pos) !== 0x5B/* [ */) { return false }

  var oldPos = state.pos;
  var max = state.posMax;
  var labelStart = state.pos + 1;
  var labelEnd = state.md.helpers.parseLinkLabel(state, state.pos, true);

  // parser failed to find ']', so it's not a valid link
  if (labelEnd < 0) { return false }

  var pos = labelEnd + 1;
  if (pos < max && state.src.charCodeAt(pos) === 0x28/* ( */) {
    //
    // Inline link
    //

    // might have found a valid shortcut link, disable reference parsing
    parseReference = false;

    // [link](  <href>  "title"  )
    //        ^^ skipping these spaces
    pos++;
    for (; pos < max; pos++) {
      code = state.src.charCodeAt(pos);
      if (!isSpace$e(code) && code !== 0x0A) { break }
    }
    if (pos >= max) { return false }

    // [link](  <href>  "title"  )
    //          ^^^^^^ parsing link destination
    start = pos;
    res = state.md.helpers.parseLinkDestination(state.src, pos, state.posMax);
    if (res.ok) {
      href = state.md.normalizeLink(res.str);
      if (state.md.validateLink(href)) {
        pos = res.pos;
      } else {
        href = '';
      }

      // [link](  <href>  "title"  )
      //                ^^ skipping these spaces
      start = pos;
      for (; pos < max; pos++) {
        code = state.src.charCodeAt(pos);
        if (!isSpace$e(code) && code !== 0x0A) { break }
      }

      // [link](  <href>  "title"  )
      //                  ^^^^^^^ parsing link title
      res = state.md.helpers.parseLinkTitle(state.src, pos, state.posMax);
      if (pos < max && start !== pos && res.ok) {
        title = res.str;
        pos = res.pos;

        // [link](  <href>  "title"  )
        //                         ^^ skipping these spaces
        for (; pos < max; pos++) {
          code = state.src.charCodeAt(pos);
          if (!isSpace$e(code) && code !== 0x0A) { break }
        }
      }
    }

    if (pos >= max || state.src.charCodeAt(pos) !== 0x29/* ) */) {
      // parsing a valid shortcut link failed, fallback to reference
      parseReference = true;
    }
    pos++;
  }

  if (parseReference) {
    //
    // Link reference
    //
    if (typeof state.env.references === 'undefined') { return false }

    if (pos < max && state.src.charCodeAt(pos) === 0x5B/* [ */) {
      start = pos + 1;
      pos = state.md.helpers.parseLinkLabel(state, pos);
      if (pos >= 0) {
        label = state.src.slice(start, pos++);
      } else {
        pos = labelEnd + 1;
      }
    } else {
      pos = labelEnd + 1;
    }

    // covers label === '' and label === undefined
    // (collapsed reference link and shortcut reference link respectively)
    if (!label) { label = state.src.slice(labelStart, labelEnd); }

    ref = state.env.references[normalizeReference$3(label)];
    if (!ref) {
      state.pos = oldPos;
      return false
    }
    href = ref.href;
    title = ref.title;
  }

  //
  // We found the end of the link, and know for a fact it's a valid link;
  // so all that's left to do is to call tokenizer.
  //
  if (!silent) {
    state.pos = labelStart;
    state.posMax = labelEnd;

    var token_o = state.push('link_open', 'a', 1);
    var attrs = [['href', href]];
    token_o.attrs  = attrs;
    if (title) {
      attrs.push(['title', title]);
    }

    state.linkLevel++;
    state.md.inline.tokenize(state);
    state.linkLevel--;

    state.push('link_close', 'a', -1);
  }

  state.pos = pos;
  state.posMax = max;
  return true
}// Process ![image](<src> "title")


function image$3 (state, silent) {
  var code, content, label, pos, ref, res, title, start;
  var href = '';
  var oldPos = state.pos;
  var max = state.posMax;

  if (state.src.charCodeAt(state.pos) !== 0x21/* ! */) { return false }
  if (state.src.charCodeAt(state.pos + 1) !== 0x5B/* [ */) { return false }

  var labelStart = state.pos + 2;
  var labelEnd = state.md.helpers.parseLinkLabel(state, state.pos + 1, false);

  // parser failed to find ']', so it's not a valid link
  if (labelEnd < 0) { return false }

  pos = labelEnd + 1;
  if (pos < max && state.src.charCodeAt(pos) === 0x28/* ( */) {
    //
    // Inline link
    //

    // [link](  <href>  "title"  )
    //        ^^ skipping these spaces
    pos++;
    for (; pos < max; pos++) {
      code = state.src.charCodeAt(pos);
      if (!isSpace$e(code) && code !== 0x0A) { break }
    }
    if (pos >= max) { return false }

    // [link](  <href>  "title"  )
    //          ^^^^^^ parsing link destination
    start = pos;
    res = state.md.helpers.parseLinkDestination(state.src, pos, state.posMax);
    if (res.ok) {
      href = state.md.normalizeLink(res.str);
      if (state.md.validateLink(href)) {
        pos = res.pos;
      } else {
        href = '';
      }
    }

    // [link](  <href>  "title"  )
    //                ^^ skipping these spaces
    start = pos;
    for (; pos < max; pos++) {
      code = state.src.charCodeAt(pos);
      if (!isSpace$e(code) && code !== 0x0A) { break }
    }

    // [link](  <href>  "title"  )
    //                  ^^^^^^^ parsing link title
    res = state.md.helpers.parseLinkTitle(state.src, pos, state.posMax);
    if (pos < max && start !== pos && res.ok) {
      title = res.str;
      pos = res.pos;

      // [link](  <href>  "title"  )
      //                         ^^ skipping these spaces
      for (; pos < max; pos++) {
        code = state.src.charCodeAt(pos);
        if (!isSpace$e(code) && code !== 0x0A) { break }
      }
    } else {
      title = '';
    }

    if (pos >= max || state.src.charCodeAt(pos) !== 0x29/* ) */) {
      state.pos = oldPos;
      return false
    }
    pos++;
  } else {
    //
    // Link reference
    //
    if (typeof state.env.references === 'undefined') { return false }

    if (pos < max && state.src.charCodeAt(pos) === 0x5B/* [ */) {
      start = pos + 1;
      pos = state.md.helpers.parseLinkLabel(state, pos);
      if (pos >= 0) {
        label = state.src.slice(start, pos++);
      } else {
        pos = labelEnd + 1;
      }
    } else {
      pos = labelEnd + 1;
    }

    // covers label === '' and label === undefined
    // (collapsed reference link and shortcut reference link respectively)
    if (!label) { label = state.src.slice(labelStart, labelEnd); }

    ref = state.env.references[normalizeReference$3(label)];
    if (!ref) {
      state.pos = oldPos;
      return false
    }
    href = ref.href;
    title = ref.title;
  }

  //
  // We found the end of the link, and know for a fact it's a valid link;
  // so all that's left to do is to call tokenizer.
  //
  if (!silent) {
    content = state.src.slice(labelStart, labelEnd);

    var tokens = [];
    state.md.inline.parse(
      content,
      state.md,
      state.env,
      tokens
    );

    var token = state.push('image', 'img', 0);
    var attrs = [['src', href], ['alt', '']];
    token.attrs = attrs;
    token.children = tokens;
    token.content = content;

    if (title) {
      attrs.push(['title', title]);
    }
  }

  state.pos = pos;
  state.posMax = max;
  return true
}// Process autolinks '<protocol:...>'

/* eslint max-len:0 */
var EMAIL_RE$1    = /^([a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*)$/;
/* eslint-disable-next-line no-control-regex */
var AUTOLINK_RE$1 = /^([a-zA-Z][a-zA-Z0-9+.-]{1,31}):([^<>\x00-\x20]*)$/;

function autolink$1 (state, silent) {
  var pos = state.pos;

  if (state.src.charCodeAt(pos) !== 0x3C/* < */) { return false }

  var start = state.pos;
  var max = state.posMax;

  for (;;) {
    if (++pos >= max) { return false }

    var ch = state.src.charCodeAt(pos);

    if (ch === 0x3C /* < */) { return false }
    if (ch === 0x3E /* > */) { break }
  }

  var url = state.src.slice(start + 1, pos);

  if (AUTOLINK_RE$1.test(url)) {
    var fullUrl = state.md.normalizeLink(url);
    if (!state.md.validateLink(fullUrl)) { return false }

    if (!silent) {
      var token_o   = state.push('link_open', 'a', 1);
      token_o.attrs   = [['href', fullUrl]];
      token_o.markup  = 'autolink';
      token_o.info    = 'auto';

      var token_t   = state.push('text', '', 0);
      token_t.content = state.md.normalizeLinkText(url);

      var token_c   = state.push('link_close', 'a', -1);
      token_c.markup  = 'autolink';
      token_c.info    = 'auto';
    }

    state.pos += url.length + 2;
    return true
  }

  if (EMAIL_RE$1.test(url)) {
    var fullUrl$1 = state.md.normalizeLink('mailto:' + url);
    if (!state.md.validateLink(fullUrl$1)) { return false }

    if (!silent) {
      var token_o$1   = state.push('link_open', 'a', 1);
      token_o$1.attrs   = [['href', fullUrl$1]];
      token_o$1.markup  = 'autolink';
      token_o$1.info    = 'auto';

      var token_t$1   = state.push('text', '', 0);
      token_t$1.content = state.md.normalizeLinkText(url);

      var token_c$1   = state.push('link_close', 'a', -1);
      token_c$1.markup  = 'autolink';
      token_c$1.info    = 'auto';
    }

    state.pos += url.length + 2;
    return true
  }

  return false
}// Process html tags


function isLinkOpen$2 (str) {
  return /^<a[>\s]/i.test(str)
}
function isLinkClose$2 (str) {
  return /^<\/a\s*>/i.test(str)
}

function isLetter$1 (ch) {
  /* eslint no-bitwise:0 */
  var lc = ch | 0x20; // to lower case
  return (lc >= 0x61/* a */) && (lc <= 0x7a/* z */)
}

function html_inline$1 (state, silent) {
  if (!state.md.options.html) { return false }

  // Check start
  var max = state.posMax;
  var pos = state.pos;
  if (state.src.charCodeAt(pos) !== 0x3C/* < */ ||
      pos + 2 >= max) {
    return false
  }

  // Quick fail on second char
  var ch = state.src.charCodeAt(pos + 1);
  if (ch !== 0x21/* ! */ &&
      ch !== 0x3F/* ? */ &&
      ch !== 0x2F/* / */ &&
      !isLetter$1(ch)) {
    return false
  }

  var match = state.src.slice(pos).match(HTML_TAG_RE$2);
  if (!match) { return false }

  if (!silent) {
    var token = state.push('html_inline', '', 0);
    token.content = match[0];

    if (isLinkOpen$2(token.content))  { state.linkLevel++; }
    if (isLinkClose$2(token.content)) { state.linkLevel--; }
  }
  state.pos += match[0].length;
  return true
}// Process html entity - &#123;, &#xAF;, &quot;, ...


var DIGITAL_RE$1 = /^&#((?:x[a-f0-9]{1,6}|[0-9]{1,7}));/i;
var NAMED_RE$1   = /^&([a-z][a-z0-9]{1,31});/i;

function entity$1 (state, silent) {
  var pos = state.pos;
  var max = state.posMax;

  if (state.src.charCodeAt(pos) !== 0x26/* & */) { return false }

  if (pos + 1 >= max) { return false }

  var ch = state.src.charCodeAt(pos + 1);

  if (ch === 0x23 /* # */) {
    var match = state.src.slice(pos).match(DIGITAL_RE$1);
    if (match) {
      if (!silent) {
        var code = match[1][0].toLowerCase() === 'x' ? parseInt(match[1].slice(1), 16) : parseInt(match[1], 10);

        var token   = state.push('text_special', '', 0);
        token.content = isValidEntityCode$1(code) ? fromCodePoint$2(code) : fromCodePoint$2(0xFFFD);
        token.markup  = match[0];
        token.info    = 'entity';
      }
      state.pos += match[0].length;
      return true
    }
  } else {
    var match$1 = state.src.slice(pos).match(NAMED_RE$1);
    if (match$1) {
      var decoded = decodeHTML(match$1[0]);
      if (decoded !== match$1[0]) {
        if (!silent) {
          var token$1   = state.push('text_special', '', 0);
          token$1.content = decoded;
          token$1.markup  = match$1[0];
          token$1.info    = 'entity';
        }
        state.pos += match$1[0].length;
        return true
      }
    }
  }

  return false
}// For each opening emphasis-like marker find a matching closing one
//

function processDelimiters$1 (delimiters) {
  var openersBottom = {};
  var max = delimiters.length;

  if (!max) { return }

  // headerIdx is the first delimiter of the current (where closer is) delimiter run
  var headerIdx = 0;
  var lastTokenIdx = -2; // needs any value lower than -1
  var jumps = [];

  for (var closerIdx = 0; closerIdx < max; closerIdx++) {
    var closer = delimiters[closerIdx];

    jumps.push(0);

    // markers belong to same delimiter run if:
    //  - they have adjacent tokens
    //  - AND markers are the same
    //
    if (delimiters[headerIdx].marker !== closer.marker || lastTokenIdx !== closer.token - 1) {
      headerIdx = closerIdx;
    }

    lastTokenIdx = closer.token;

    // Length is only used for emphasis-specific "rule of 3",
    // if it's not defined (in strikethrough or 3rd party plugins),
    // we can default it to 0 to disable those checks.
    //
    closer.length = closer.length || 0;

    if (!closer.close) { continue }

    // Previously calculated lower bounds (previous fails)
    // for each marker, each delimiter length modulo 3,
    // and for whether this closer can be an opener;
    // https://github.com/commonmark/cmark/commit/34250e12ccebdc6372b8b49c44fab57c72443460
    /* eslint-disable-next-line no-prototype-builtins */
    if (!openersBottom.hasOwnProperty(closer.marker)) {
      openersBottom[closer.marker] = [-1, -1, -1, -1, -1, -1];
    }

    var minOpenerIdx = openersBottom[closer.marker][(closer.open ? 3 : 0) + (closer.length % 3)];

    var openerIdx = headerIdx - jumps[headerIdx] - 1;

    var newMinOpenerIdx = openerIdx;

    for (; openerIdx > minOpenerIdx; openerIdx -= jumps[openerIdx] + 1) {
      var opener = delimiters[openerIdx];

      if (opener.marker !== closer.marker) { continue }

      if (opener.open && opener.end < 0) {
        var isOddMatch = false;

        // from spec:
        //
        // If one of the delimiters can both open and close emphasis, then the
        // sum of the lengths of the delimiter runs containing the opening and
        // closing delimiters must not be a multiple of 3 unless both lengths
        // are multiples of 3.
        //
        if (opener.close || closer.open) {
          if ((opener.length + closer.length) % 3 === 0) {
            if (opener.length % 3 !== 0 || closer.length % 3 !== 0) {
              isOddMatch = true;
            }
          }
        }

        if (!isOddMatch) {
          // If previous delimiter cannot be an opener, we can safely skip
          // the entire sequence in future checks. This is required to make
          // sure algorithm has linear complexity (see *_*_*_*_*_... case).
          //
          var lastJump = openerIdx > 0 && !delimiters[openerIdx - 1].open
            ? jumps[openerIdx - 1] + 1
            : 0;

          jumps[closerIdx] = closerIdx - openerIdx + lastJump;
          jumps[openerIdx] = lastJump;

          closer.open  = false;
          opener.end   = closerIdx;
          opener.close = false;
          newMinOpenerIdx = -1;
          // treat next token as start of run,
          // it optimizes skips in **<...>**a**<...>** pathological case
          lastTokenIdx = -2;
          break
        }
      }
    }

    if (newMinOpenerIdx !== -1) {
      // If match for this delimiter run failed, we want to set lower bound for
      // future lookups. This is required to make sure algorithm has linear
      // complexity.
      //
      // See details here:
      // https://github.com/commonmark/cmark/issues/178#issuecomment-270417442
      //
      openersBottom[closer.marker][(closer.open ? 3 : 0) + ((closer.length || 0) % 3)] = newMinOpenerIdx;
    }
  }
}

function link_pairs (state) {
  var tokens_meta = state.tokens_meta;
  var max = state.tokens_meta.length;

  processDelimiters$1(state.delimiters);

  for (var curr = 0; curr < max; curr++) {
    if (tokens_meta[curr] && tokens_meta[curr].delimiters) {
      processDelimiters$1(tokens_meta[curr].delimiters);
    }
  }
}// Clean up tokens after emphasis and strikethrough postprocessing:
// merge adjacent text nodes into one and re-calculate all token levels
//
// This is necessary because initially emphasis delimiter markers (*, _, ~)
// are treated as their own separate text tokens. Then emphasis rule either
// leaves them as text (needed to merge with adjacent text) or turns them
// into opening/closing tags (which messes up levels inside).
//

function fragments_join$1 (state) {
  var curr, last;
  var level = 0;
  var tokens = state.tokens;
  var max = state.tokens.length;

  for (curr = last = 0; curr < max; curr++) {
    // re-calculate levels after emphasis/strikethrough turns some text nodes
    // into opening/closing tags
    if (tokens[curr].nesting < 0) { level--; } // closing tag
    tokens[curr].level = level;
    if (tokens[curr].nesting > 0) { level++; } // opening tag

    if (tokens[curr].type === 'text' &&
        curr + 1 < max &&
        tokens[curr + 1].type === 'text') {
      // collapse two adjacent text nodes
      tokens[curr + 1].content = tokens[curr].content + tokens[curr + 1].content;
    } else {
      if (curr !== last) { tokens[last] = tokens[curr]; }

      last++;
    }
  }

  if (curr !== last) {
    tokens.length = last;
  }
}/** internal
 * class ParserInline
 *
 * Tokenizes paragraph content.
 **/


// Parser rules

var _rules$3 = [
  ['text',            text$2],
  ['linkify',         linkify$3],
  ['newline',         newline$2],
  ['escape',          escape],
  ['backticks',       backtick],
  ['strikethrough',   r_strikethrough.tokenize],
  ['emphasis',        r_emphasis.tokenize],
  ['link',            link$5],
  ['image',           image$3],
  ['autolink',        autolink$1],
  ['html_inline',     html_inline$1],
  ['entity',          entity$1]
];

// `rule2` ruleset was created specifically for emphasis/strikethrough
// post-processing and may be changed in the future.
//
// Don't use this for anything except pairs (plugins working with `balance_pairs`).
//
var _rules2$1 = [
  ['balance_pairs',   link_pairs],
  ['strikethrough',   r_strikethrough.postProcess],
  ['emphasis',        r_emphasis.postProcess],
  // rules for pairs separate '**' into its own text tokens, which may be left unused,
  // rule below merges unused segments back with the rest of the text
  ['fragments_join',  fragments_join$1]
];

/**
 * new ParserInline()
 **/
function ParserInline$2 () {
  /**
   * ParserInline#ruler -> Ruler
   *
   * [[Ruler]] instance. Keep configuration of inline rules.
   **/
  this.ruler = new Ruler$4();

  for (var i = 0; i < _rules$3.length; i++) {
    this.ruler.push(_rules$3[i][0], _rules$3[i][1]);
  }

  /**
   * ParserInline#ruler2 -> Ruler
   *
   * [[Ruler]] instance. Second ruler used for post-processing
   * (e.g. in emphasis-like rules).
   **/
  this.ruler2 = new Ruler$4();

  for (var i$1 = 0; i$1 < _rules2$1.length; i$1++) {
    this.ruler2.push(_rules2$1[i$1][0], _rules2$1[i$1][1]);
  }
}

// Skip single token by running all rules in validation mode;
// returns `true` if any rule reported success
//
ParserInline$2.prototype.skipToken = function (state) {
  var pos = state.pos;
  var rules = this.ruler.getRules('');
  var len = rules.length;
  var maxNesting = state.md.options.maxNesting;
  var cache = state.cache;

  if (typeof cache[pos] !== 'undefined') {
    state.pos = cache[pos];
    return
  }

  var ok = false;

  if (state.level < maxNesting) {
    for (var i = 0; i < len; i++) {
      // Increment state.level and decrement it later to limit recursion.
      // It's harmless to do here, because no tokens are created. But ideally,
      // we'd need a separate private state variable for this purpose.
      //
      state.level++;
      ok = rules[i](state, true);
      state.level--;

      if (ok) {
        if (pos >= state.pos) { throw new Error("inline rule didn't increment state.pos") }
        break
      }
    }
  } else {
    // Too much nesting, just skip until the end of the paragraph.
    //
    // NOTE: this will cause links to behave incorrectly in the following case,
    //       when an amount of `[` is exactly equal to `maxNesting + 1`:
    //
    //       [[[[[[[[[[[[[[[[[[[[[foo]()
    //
    // TODO: remove this workaround when CM standard will allow nested links
    //       (we can replace it by preventing links from being parsed in
    //       validation mode)
    //
    state.pos = state.posMax;
  }

  if (!ok) { state.pos++; }
  cache[pos] = state.pos;
};

// Generate tokens for input range
//
ParserInline$2.prototype.tokenize = function (state) {
  var rules = this.ruler.getRules('');
  var len = rules.length;
  var end = state.posMax;
  var maxNesting = state.md.options.maxNesting;

  while (state.pos < end) {
    // Try all possible rules.
    // On success, rule should:
    //
    // - update `state.pos`
    // - update `state.tokens`
    // - return true
    var prevPos = state.pos;
    var ok = false;

    if (state.level < maxNesting) {
      for (var i = 0; i < len; i++) {
        ok = rules[i](state, false);
        if (ok) {
          if (prevPos >= state.pos) { throw new Error("inline rule didn't increment state.pos") }
          break
        }
      }
    }

    if (ok) {
      if (state.pos >= end) { break }
      continue
    }

    state.pending += state.src[state.pos++];
  }

  if (state.pending) {
    state.pushPending();
  }
};

/**
 * ParserInline.parse(str, md, env, outTokens)
 *
 * Process input string and push inline tokens into `outTokens`
 **/
ParserInline$2.prototype.parse = function (str, md, env, outTokens) {
  var state = new this.State(str, md, env, outTokens);

  this.tokenize(state);

  var rules = this.ruler2.getRules('');
  var len = rules.length;

  for (var i = 0; i < len; i++) {
    rules[i](state);
  }
};

ParserInline$2.prototype.State = StateInline$1;function reFactory (opts) {
  var re = {};
  opts = opts || {};

  re.src_Any = Any.source;
  re.src_Cc = Cc.source;
  re.src_Z = Z.source;
  re.src_P = P.source;

  // \p{\Z\P\Cc\CF} (white spaces + control + format + punctuation)
  re.src_ZPCc = [re.src_Z, re.src_P, re.src_Cc].join('|');

  // \p{\Z\Cc} (white spaces + control)
  re.src_ZCc = [re.src_Z, re.src_Cc].join('|');

  // Experimental. List of chars, completely prohibited in links
  // because can separate it from other part of text
  var text_separators = '[><\uff5c]';

  // All possible word characters (everything without punctuation, spaces & controls)
  // Defined via punctuation & spaces to save space
  // Should be something like \p{\L\N\S\M} (\w but without `_`)
  re.src_pseudo_letter = '(?:(?!' + text_separators + '|' + re.src_ZPCc + ')' + re.src_Any + ')';
  // The same as abothe but without [0-9]
  // var src_pseudo_letter_non_d = '(?:(?![0-9]|' + src_ZPCc + ')' + src_Any + ')';

  re.src_ip4 =

    '(?:(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)';

  // Prohibit any of "@/[]()" in user/pass to avoid wrong domain fetch.
  re.src_auth = '(?:(?:(?!' + re.src_ZCc + '|[@/\\[\\]()]).)+@)?';

  re.src_port =

    '(?::(?:6(?:[0-4]\\d{3}|5(?:[0-4]\\d{2}|5(?:[0-2]\\d|3[0-5])))|[1-5]?\\d{1,4}))?';

  re.src_host_terminator =

    '(?=$|' + text_separators + '|' + re.src_ZPCc + ')' +
    '(?!' + (opts['---'] ? '-(?!--)|' : '-|') + '_|:\\d|\\.-|\\.(?!$|' + re.src_ZPCc + '))';

  re.src_path =

    '(?:' +
      '[/?#]' +
        '(?:' +
          '(?!' + re.src_ZCc + '|' + text_separators + '|[()[\\]{}.,"\'?!\\-;]).|' +
          '\\[(?:(?!' + re.src_ZCc + '|\\]).)*\\]|' +
          '\\((?:(?!' + re.src_ZCc + '|[)]).)*\\)|' +
          '\\{(?:(?!' + re.src_ZCc + '|[}]).)*\\}|' +
          '\\"(?:(?!' + re.src_ZCc + '|["]).)+\\"|' +
          "\\'(?:(?!" + re.src_ZCc + "|[']).)+\\'|" +

          // allow `I'm_king` if no pair found
          "\\'(?=" + re.src_pseudo_letter + '|[-])|' +

          // google has many dots in "google search" links (#66, #81).
          // github has ... in commit range links,
          // Restrict to
          // - english
          // - percent-encoded
          // - parts of file path
          // - params separator
          // until more examples found.
          '\\.{2,}[a-zA-Z0-9%/&]|' +

          '\\.(?!' + re.src_ZCc + '|[.]|$)|' +
          (opts['---']
            ? '\\-(?!--(?:[^-]|$))(?:-*)|' // `---` => long dash, terminate
            : '\\-+|'
          ) +
          // allow `,,,` in paths
          ',(?!' + re.src_ZCc + '|$)|' +

          // allow `;` if not followed by space-like char
          ';(?!' + re.src_ZCc + '|$)|' +

          // allow `!!!` in paths, but not at the end
          '\\!+(?!' + re.src_ZCc + '|[!]|$)|' +

          '\\?(?!' + re.src_ZCc + '|[?]|$)' +
        ')+' +
      '|\\/' +
    ')?';

  // Allow anything in markdown spec, forbid quote (") at the first position
  // because emails enclosed in quotes are far more common
  re.src_email_name =

    '[\\-;:&=\\+\\$,\\.a-zA-Z0-9_][\\-;:&=\\+\\$,\\"\\.a-zA-Z0-9_]*';

  re.src_xn =

    'xn--[a-z0-9\\-]{1,59}';

  // More to read about domain names
  // http://serverfault.com/questions/638260/

  re.s